/*
 *      RANDOM.CMD - V1.0 C.Langanke 1997
 *
 *      REXX sample for "Animated Mouse Pointer for OS/2"
 *
 *      Syntax: RANDOM [*|n] [/TIMEOUT:n] [/?]
 *
 *      RANDOM loades randomly one out of a list of animation resources.
 *      The animatios available have to be stated in RANDOM.LST,
 *      This file has to reside in the same directory as RANDOM.CMD, where
 *      each line contains an animation name in double quotes and a valid
 *      REXX setup string for the enhanced mouse object.
 *      Commentlines begin with a ";".
 *
 *      *|n         loads new aniamtions either forever or for n times
 *                  [Default:1]
 *      /TIMEOUT:n  timeout value in seconds after which a new resource
 *                  has to be loaded [Default:60]
 *      /?          displays this help text.
#*
 *      NOTE: "Animated Mouse Pointer for OS/2" must be
 *            installed before executing this batch program.
 *
 *      Refer to WPAMPTR.INF for more information about how
 *      to use REXX setup strings to configure
 *      "Animated Mouse Pointer for OS/2"
 */
/* First comment is used as online help text. */

 SIGNAL ON HALT

 TitleLine = STRIP(SUBSTR(SourceLine(2), 3));
 PARSE VAR TitleLine CmdName'.CMD 'Info
 Title     = CmdName Info

 /* OS/2 error codes */
 ERROR.NO_ERROR           =  0;
 ERROR.INVALID_FUNCTION   =  1;
 ERROR.FILE_NOT_FOUND     =  2;
 ERROR.PATH_NOT_FOUND     =  3;
 ERROR.ACCESS_DENIED      =  5;
 ERROR.NOT_ENOUGH_MEMORY  =  8;
 ERROR.INVALID_FORMAT     = 11;
 ERROR.INVALID_DATA       = 13;
 ERROR.NO_MORE_FILES      = 18;
 ERROR.WRITE_FAULT        = 29;
 ERROR.READ_FAULT         = 30;
 ERROR.GEN_FAILURE        = 31;
 ERROR.INVALID_PARAMETER  = 87;

 /* show help */
 ARG Parm .
 IF (POS('?', Parm) > 0) THEN
 DO
    rc = ShowHelp();
    EXIT(ERROR.INVALID_PARAMETER);
 END;

 SAY;
 SAY Title;
 SAY;

 /* get parms */
 Timeout    = 60;
 Iterations = 0;
 ClearLine  = '0d'x'[K';

 /* determine a random seed number */
 Time = TIME();
 PARSE VAR TIME Hours':'Minutes':'Seconds
 SeedNumber = Hours''Minutes''Seconds;
 FirstRandom = Random(1,100, SeedNumber);

 /* get commandline parms */
 PARSE ARG Parms
 rc = ParseCommandLineParms(,Parms);

 DO i = 1 TO Parm.0;
    IF (Parm.i = '') THEN ITERATE;
    ThisParm = Parm.i;
    PARSE VAR ThisParm ThisTag':'ThisValue
    ThisTag = TRANSLATE(ThisTag);

    SELECT
       WHEN (POS(ThisTag, '/TIMEOUT') = 1) THEN
       DO
          Timeout = ThisValue;
       END;

       OTHERWISE
       DO
          IF (Iterations = 0) THEN
             Iterations = ThisParm;
          ELSE
          DO
             SAY CmdName': error: invalid parameter 'ThisParm'.';
             EXIT(ERROR.INVALID_PARAMETER);
          END;
       END;

    END; /* SELECT */
 END; /* DO */

 /* verify commandline parms */
 /* - check iteration count */
 SELECT
    WHEN (Iterations = 0)   THEN Iterations = 1;  /* set default */
    WHEN (Iterations = '*') THEN Iterations = -1; /* loop forever */
    WHEN (DATATYPE(Iterations) \= 'NUM') THEN
    DO
       SAY CmdName': error: iteration count must be numeric or "*" to loop forever.';
       EXIT(ERROR.INVALID_PARAMETER);
    END;
    OTHERWISE;
 END;
 IF (Iterations = -1) THEN
    SAY 'Iterate forever';
 ELSE
    SAY Iterations 'Iterations';

 /* - check timeout value */
 IF (DATATYPE(Timeout) \= 'NUM') THEN
 DO
    SAY CmdName': error: Timeout value not numeric.';
    EXIT(ERROR.INVALID_PARAMETER);
 END;
 SAY 'Timeout value:' Timeout 'sec.';

 /* get the list file */
 ListFile = GetCalldir()'\'CmdName'.LST';
 IF (\FileExist(ListFile)) THEN
 DO
    SAY CmdName': error: list file 'ListFile' not found.';
    EXIT(ERROR.FILE_NOT_FOUND);
 END;

 /* read the listfile */
 Animation.  = '';
 Animation.0 = 0;
 DO WHILE (LINES(ListFile) > 0)
    ThisAnimation = STRIP(LINEIN(ListFile));
    IF (ThisAnimation = '') THEN ITERATE;
    IF (LEFT(ThisAnimation, 1) = ';') THEN ITERATE;
    a           = Animation.0 + 1;
    PARSE VAR ThisAnimation '"'Animation.a.Name'"' Animation.a
    Animation.a = STRIP(Animation.a);
    Animation.0 = a;
 END;
 rc= STREAM(ListFile, 'C', 'CLOSE');
 SAY Animation.0 'Animations found in' ListFile;

 /* now loop n times */
 IF (Iterations = -1) THEN
    LastLoop = 2;
 ELSE
    LastLoop = Iterations;
 CurrentAnimation = 0;
 NextAnimation    = 0;
 AnimationCount   = 0;
 SAY;

 DO i = 1 TO LastLoop

    /* determine the animation to choose */
    /* make sure it is not the same as the current */
    DO WHILE (CurrentAnimation = NextAnimation)
       NextAnimation = RANDOM( 1, Animation.0);
    END;

    /* now setup the mouse object */
    AnimationCount = AnimationCount + 1;
    setup = Animation.NextAnimation;
    rc = SysSetObjectData( '<WP_MOUSE>', setup);
    CurrentAnimation = NextAnimation;
    Prompt = ClearLine'animation #'AnimationCount ''Animation.CurrentAnimation.Name' active for ';

    /* only change one time ? */
    IF (LastLoop = i) THEN
    DO
       SAY Prompt''COPIES('08'x, 4)'[K';
       ITERATE;
    END;

    /* sleep a while */
    DO s = Timeout TO 1 BY -1
       CALL CHAROUT, Prompt s 'more seconds';
       rc = SysSleep(1);
    END;

    /* iterate forever ?*/
    IF (Iterations = -1) THEN
       i = 0;
 END;

 EXIT(ERROR.NO_ERROR);

/* ------------------------------------------------------------------------- */
HALT:
 SAY;
 SAY 'Interrupted by user.';
 EXIT(ERROR.GEN_FAILURE);

/* ------------------------------------------------------------------------- */
ShowHelp: PROCEDURE EXPOSE Title

 SAY;
 SAY Title;
 SAY;

 PARSE SOURCE . . ThisFile

 /* skip header */
 DO i = 1 TO 3
    rc = LINEIN(ThisFile);
 END;

 /* show help */
 ThisLine = LINEIN(Thisfile);
 DO WHILE (ThisLine \= ' */')
    IF (LEFT(ThisLine, 2) = '#*') THEN
    DO
       '@PAUSE';
       SAY '[A'D2C(13)'[K[A';
    END;
    SAY SUBSTR(ThisLine, 7);
    ThisLine = LINEIN(Thisfile);
 END;

 /* close file again */
 rc = LINEOUT(Thisfile);

 RETURN('');

/* ------------------------------------------------------------------------- */
FileExist: PROCEDURE
 PARSE ARG FileName

 RETURN(STREAM(Filename, 'C', 'QUERY EXISTS') > '');

/* ------------------------------------------------------------------------- */
GetCalldir: PROCEDURE
PARSE SOURCE . . CallName
 CallDir = FILESPEC('Drive', CallName)||FILESPEC('Path', CallName);
 RETURN(LEFT(CallDir, LENGTH(CallDir) - 1));

/* ------------------------------------------------------------------------- */
ParseCommandLineParms: PROCEDURE EXPOSE TRUE FALSE Parm.
 PARSE ARG Delimiter,Parms

 /* default values */
 ParmCount = 0;
 DROP(Parm.);
 Parm.  = '';
 Parm.0 = 0;

 /* check parameter */
 Parms = STRIP(Parms);
 IF (Parms = '') THEN
    RETURN(Parm.0);

 Delimter = TRANSLATE(STRIP(Delimiter));

 /* convert tabs */
 Parms = TRANSLATE(Parms, D2C(32), D2C(9));

 /* split em up */
 DO WHILE (Parms \= '')
    /* handle quotes */
    QuotesPos = POS('"', WORD(Parms, 1));
    IF (QuotesPos > 0) THEN
    DO
       DoubleQuote = TRUE;
       ThisParmEnd = POS('"', Parms, QuotesPos + 1);
       IF (ThisParmEnd = 0) THEN
          ThisParmEnd = LENGTH(Parms) + 1;

       DO WHILE (DoubleQuote)
          /* delete double quotes */
          DoubleQuote = (SUBSTR(Parms, ThisParmEnd + 1, 1) = '"');
          IF (DoubleQuote) THEN
          DO
             Parms = DELSTR(Parms, ThisParmEnd, 1);
             /* skip a single one */
             ThisParmEnd = POS('"', Parms, ThisParmEnd + 1);
             IF (ThisParmEnd = 0) THEN
                ThisParmEnd = LENGTH(Parms) + 1;
          END;
       END;

       IF (QuotesPos > 1) THEN
          ThisParm = LEFT(Parms, QuotesPos - 1);
       ELSE
          ThisParm = '';
       ThisParm = ThisParm''SUBSTR(Parms, QuotesPos + 1, ThisParmEnd - QuotesPos - 1);
       NextParmPos = ThisParmEnd + 1;

       IF (NextParmPos > LENGTH(Parms)) THEN
          NextParmPos = 0;

    END;
    ELSE
    /* normal parm without quote */
    DO
       /* is it a delimiter ? */
       IF (POS(Delimiter, Parms) = 1) THEN
       DO
          ThisParm = Parms;
          NextParmPos = 0;
       END;
       ELSE
       DO
          ThisParm = STRIP(WORD(Parms, 1));
          NextParmPos = LENGTH(ThisParm) + 1;
       END;
    END;

    /* store this parm and cut it off */
    i      = Parm.0 + 1;
    Parm.i = ThisParm;
    Parm.0 = i;

    IF (NextParmPos > 0) THEN
    DO
       Parms = SUBSTR(Parms, NextParmPos);
       Parms = STRIP(Parms);
    END;
    ELSE
       Parms = '';
 END;

 RETURN(Parm.0);

