/*****************************************************************************/
/*	       Copyright (c) 1994 by Jyrki Salmi <jytasa@jyu.fi>             */
/*        You may modify, recompile and distribute this file freely.         */
/*****************************************************************************/

/*
   Some common utility functions
*/

#include <stdio.h>
#define INCL_KBD
#define INCL_DOSPROCESS
#include <os2.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <io.h>
#include "typedefs.h"
#include "common.h"
#include "error.h"
#include "modules.h"
#include "global.h"

/* Returns time difference in verbal form. timer parameter is the difference */
/* in number of seconds. */

U8 *d_time(U32 timer) {

  static U8 ret[32];

  if (timer < 60)
    sprintf(ret, "%lu s", timer);
  else if (timer < 60 * 60)
    sprintf(ret, "%lu min %lu s", timer / 60, timer % 60);
  else if (timer < 60 * 60 * 24)
    sprintf(ret, "%lu h %lu min %lu s", timer / (60 * 60), (timer / 60) % 60, timer % 60);
  else
    sprintf(ret, "%lu d %lu h %lu min %lu s", timer / (60 * 60 * 24),
	    timer / (60 * 60) % 24, (timer / 60) % 60, timer % 60);
  return(ret);
}

/* Displays a string to the screen. If attr is MSG_NONE, a space will */
/* separate this string from the next one. If MSG_CR then this string will */
/* overwrite the possible previous message if it did not have MSG_LF attr. */
/* MSG_LF adds a newline to the end of the string. */

void msg(U32 attr, U8 *fmt, ...) {

  static U32 prev_attr = MSG_LF;
  static U32 line_len = 0;
  U32 line_len_tmp;
  U32 i;
  va_list arg_ptr;

  va_start(arg_ptr, fmt);
  if (attr & MSG_CR) {
    printf("\r");
  } else if (!(prev_attr & MSG_LF))
    printf(" ");

  line_len_tmp = vprintf(fmt, arg_ptr);
  if (line_len > line_len_tmp) { /* This line has data more than we just */
				 /* printed, let's wipe that old data away */
    for (i = line_len_tmp; i < line_len; i++)
      printf(" ");
    for (i = line_len_tmp; i < line_len; i++)
      printf("\b");
  }
  if (attr & MSG_LF) {
    printf("\n");
    line_len = 0;
  } else {
    line_len = line_len_tmp;
    fflush(stdout);
  }
  prev_attr = attr;
  va_end(arg_ptr);
}

/* Pauses until a key is pressed. Call to this function is installed to the */
/* exit functions list if -pause option is specified. */

void wait_for_keypress(void) {

  APIRET rc;
  KBDKEYINFO kki;

  printf("Press any key to continue...");
  fflush(stdout);
  rc = KbdCharIn(&kki, IO_WAIT, 0);
  if (!rc)
    os2_error(P_ERROR_KBDCHARIN, rc,
	      MODULE_COMMON, __LINE__,
	      NULL);
  printf("\n");
}

/* Makes a beep if -quiet option is not specified. Installed to the exit */
/* functions list. */

void make_noise(void) {

  if (!opt_quiet) {
    DosBeep(500, 100);
    DosSleep(250);
    DosBeep(1500, 50);
  }
}

/* Adds the receive directory specified with -directory option to the */
/* beginning of file name being received */

void add_recv_dir_to_path(U8 **path) {

  U8 new_path[4096];
  U32 i;
  U32 l;

  if (opt_directory == NULL || opt_paths)
    return;

  l = strlen(*path) + strlen(opt_directory) + 2; /* 2 is for possible */
						    /* '\' and the null */
  strcpy(new_path, opt_directory);
  i = strlen(new_path);
  if (new_path[i - 1] != '\\')
    new_path[i++] = '\\';
  strcpy(&new_path[i], *path);
  memcpy(*path, new_path, 4096);
}

/* Gets the length of directory portion of path */

U32 get_dir_len(U8 *path) {

  S32 i;

  i = strlen(path);
  while (--i >= 0) {
    if (path[i] == '\\' || path[i] == '/' || path[i] == ':')
      break;
  }
  return(i + 1);
}

/* Strips drive and directory information from a file path */

void strip_drive_and_dir(U8 *path) {

  U8 new_path[4096];

  strcpy(new_path, &path[get_dir_len(path)]);
  strcpy(path, new_path);
}

/* Creates a directory structure if it does not already exist */

U32 create_dirs(U8 *path) {

  U8 dir[4096];
  U32 path_idx = 0;

  while (path[path_idx] != '\0') {
    if (path[path_idx] == '\\' || path[path_idx] == '/') {
      if (path_idx == 0 || path[path_idx - 1] == ':') {
	memcpy(dir, path, path_idx + 1);
	dir[path_idx + 1] = '\0';
      } else {
	memcpy(dir, path, path_idx);
	dir[path_idx] = '\0';
      }
      if (access(dir, 0) != 0) {
#ifdef __EMX__
	if (mkdir(dir, 0) == -1)
	  return(1);
#else /* __EMX__ */
	if (_mkdir(dir, 0) == -1)
	  return(1);
#endif /* __EMX__ */
      }
    }
    path_idx++;
  }
  return(0);
}

/* Sets a priority of current process */

void set_priority(U32 priority_class, U32 priority_delta) {

  APIRET rc;

  rc = DosSetPriority(PRTYS_PROCESS,
		      priority_class,
		      priority_delta,
		      0);
  if (rc)
    os2_error(P_ERROR_DOSSETPRIORITY, rc,
	      MODULE_COMMON, __LINE__,
	      NULL);
}

/* Handler for SIGINT signals. Sets the aborted variable to non-zero, it */
/* will be checked in status_func() and a proper return value will be */
/* returned to the DLL. */

void interrupt_handler(int sig) {

  if (aborted)
    msg(MSG_LF, "Please wait, transfer is being aborted...");
  else {
    msg(MSG_LF, "Ctrl-C pressed");
    aborted = 1;
  }
}

/* Installs a handler for SIGINT signals */

void install_interrupt_handler(void) {

  if (signal(SIGINT, interrupt_handler) == SIG_ERR) { /* Install our own */
						      /* handler for CTRL-C */
    fprintf(stderr, "Failed to install an interrupt handler\n");
    exit(1);
  }
}

