/*
//   djp.c -- DJGPP V2 executable file compressor with the LZO1 algorithm
//
//   Copyright (C) 1996 MOLNAR Laszlo, see COPYING.DJ for details
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stubinfo.h>
#include <process.h>
#include <unistd.h>
#include <dos.h>

#include "lzo1f.h"
#include "lzo1.h"
#define SIMAGIC "go32stub"

#define EOFLEN 3                 /* for LZO1F */

/* options */
#define OPTD 0x01                /* decompress */
#define OPTS 0x02                /* use stub */
#define OPTT 0x04                /* test */
#define OPTB 0x08                /* keep backup */
#define OPTQ 0x10                /* quiet */
#define OPTC 0x20                /* coff output */
#define OPT0 0x40                /* ignore default options */
#define OPTH 0x80                /* usage */

static FILE *fi,*fo;             /* input,output */

static char wrkmem[LZO1F_999_MEM_COMPRESS];  /* for lzo1f_999 compression*/
static const unsigned char stub[]={          /* stubbed decompressor */
#include "djp.h"
};
static const unsigned char decomph[]={       /* stubless decompressor */
#include "decomp.h"
};

static const char *packerr;      /* error message */
static char tmp_file[200];
static char line[200];           /* line buffer */
static char *chksumpos;          /* checksum position */
static int  options=0;
static int  exetype;

static char *findbytes (const void *b,int blen,char *what,int wlen)
{
    int ic,firstc=what[0];
    char *base=(char*)b;

    for (ic=0; ic<=blen-wlen; ic++)
        if (base[ic]==firstc && memcmp (base+ic,what,wlen)==0)
            return base+ic;

    return NULL;
}

static void printline (const char *s,int pos,unsigned mlen)
{
    if (strlen (s) > mlen)
        s+=mlen-strlen (s);
    sprintf (line+pos,"%-*.*s",79-pos,79-pos,s);
    if (isatty (1) && (!(options & OPTQ)))
    {
        printf ("\r%s",line);
        fflush (stdout);
    }
}

/* this code is from stubify.c */
static void maketempname (char *ofilename,const char *ifilename,const char *ext,int force)
{
    char *ofext = NULL,*ofname;
    int  ofile;
    strcpy (ofilename,ifilename);
    
    for (ofname=ofilename; *ofname; ofname++)
    {
        if (strchr("/\\:", *ofname))
            ofext = NULL;
        if (*ofname == '.')
            ofext = ofname;
    }
    if (ofext == NULL)
        ofext = ofilename + strlen(ofilename);
    strcpy(ofext, ext);
    if (force==0 && __file_exists (ofilename))
        for (ofile=0; ofile<999; ofile++)
        {
            sprintf(ofext, ".%03d", ofile);
            if (!__file_exists (ofilename))
                break;
        }
}

static int patch_decomp (char *s,long oldw,long neww)
{
    char *p;
    if ((p=findbytes (s,sizeof(decomph),(char*)&oldw,4))==NULL)
        return 1;
    *(long*) p = neww;
    return 0;
}

static unsigned inpsize;

int lzo1f_999_compress_callback (const lzo_byte *,lzo_uint,lzo_byte *,lzo_uint *,
                                 lzo_voidp wrkmem,lzo_progress_callback_t);
static void djpcb (unsigned texts,unsigned codes)
{
    static int perc=-1;
    int  ic;
    char temp[20];

    if ((ic=(texts*100)/inpsize) != perc)
    {
        sprintf (temp,"%02d%%",perc=ic);
        printline (temp,51,28);
    }
}

static int pack_sections (char *in,char *out,unsigned foffset,unsigned size)
{
    unsigned packedsize,new_size;
    unsigned adler1,adler2;
    char buf[80];

    packerr=NULL;
    if (in==NULL || out==NULL)
        return packerr="out of memory",-1;      /* hmm.. ;-) */
    fseek (fi,foffset,0);

    if (fread (in,size,1,fi)!=1)
        return packerr="read error",-1;
    packedsize=0;

    /* compute checksum of uncompressed data */
    adler1 = lzo_adler32 (0,NULL,0);
    adler1 = lzo_adler32 (adler1,in,size);

    if (lzo1f_999_compress_callback (in,inpsize=size,out,&packedsize,wrkmem,djpcb) || size <= packedsize)
        return packerr="compression failed",-1;

    sprintf (buf,"%8d -> %8d ratio:%d%%",size,packedsize,(packedsize*100)/size);
    printline (buf,40,30);
    /* decompress and verify */

    new_size=size; /* safe decompression */
    if ((adler2=lzo1f_decompress_x (out,packedsize,in,&new_size,wrkmem)) != 0 || new_size != size)
        return printf ("\n%d\n",new_size),packerr="decompression failed",-1;

    /* compute checksum of decompressed data */
    adler2 = lzo_adler32(0, NULL, 0);
    adler2 = lzo_adler32(adler2, in, new_size);
    if (adler1 != adler2)
        return packerr="decompression data error",-1;

    /* put the checksum of the compressed data into the stub */
    adler2 = lzo_adler32(0, NULL, 0);
    adler2 = lzo_adler32(adler2, out, packedsize);
    if (chksumpos!=NULL)
        if (!(options & OPTS))
        {
            *(long*) chksumpos=adler2;
        }
        else
        {
            sprintf (buf,"%08X",adler2);
            memcpy (chksumpos,buf,8);
        }
    return packedsize;
}

static const char *dorenames (const char *program,char *tmp_file,int cond)
{
    struct ftime t;

    getftime (fileno (fi),&t);
    fclose (fi);
    fclose (fo);
    fi=fo=NULL;

    if ((exetype!=0) ^ (!(options & OPTC)))
        strcpy (wrkmem,program);
    else
    {
        maketempname (wrkmem,program,".~",0);   /* backup file name */

        /* if rename() were here, the compressor would be 10 kbyte bigger */
        if (_rename (program,wrkmem)!=0)
            return "can't rename";
    }
    
    if (cond && !(options & OPTC))
    {
        if (spawnlp (P_WAIT,"stubify.exe","stubify.exe",tmp_file,NULL))
        {
            _rename (wrkmem,program);           /* undo rename */
            return packerr="stubify error";
        }
        maketempname (wrkmem+1000,program,".exe",1);
    }
    else
    {
        maketempname (wrkmem+1000,program,(options & OPTC) ? ".cof" : ".exe",1);
        if (_rename (tmp_file,wrkmem+1000)!=0)
        {
            _rename (wrkmem,program);           /* undo rename */
            return packerr="can't rename .~ file";
        }
    }

    if (!(options & OPTB))
        remove (wrkmem);
        
    if ((fo=fopen (wrkmem+1000,"rb"))!=NULL)     /* copy timestamp */
        setftime (fileno (fo),&t);
    return NULL;
}

static void stripdbg (long *coffhdr)
{
    /* "strip" debug info */
    coffhdr[12+6]=coffhdr[12+7]=coffhdr[12+8]=coffhdr[2]=coffhdr[3]=0;
    coffhdr[4]=(coffhdr[4]&0xFFFF)|(0x010F0000);
}

#define textfpos coffhdr[12+5]
#define textsize coffhdr[12+4]
#define textvirt coffhdr[12+3]
#define datafpos coffhdr[22+5]
#define datasize coffhdr[22+4]
#define datavirt coffhdr[22+3]
#define bsssize  coffhdr[32+4]
#define bssvirt  coffhdr[32+3]

static const char *v2packimage(const char *program)
{
    unsigned short header[0x1c/2];
    char     magic[8],*p1,*in,*out;

    unsigned coff_offset=0,size;
    unsigned long coffhdr[42];
    
    exetype=0;
    printline (program,0,38);
    printline ("packing...",40,30);
    if ((fi = fopen (program,"rb"))==NULL)
        return "can't open";
    if (fread (header,sizeof(header),1,fi)!=1)
        return "read error";
    if (header[0] == 0x5a4d)                    /* MZ exe signature, stubbed? */
    {
        coff_offset = (long)header[2]*512L;
        if (header[1])
            coff_offset += (long)header[1] - 512L;
        fseek (fi, 512, 0);                     /* Position of V2 stubinfo */
        if (fread (magic,8,1,fi)!=1)
            return "read error";
        if (memcmp (SIMAGIC,magic,8) != 0)
            return "bad MAGIC";                 /* Not V2 image, show failure */
        exetype=1;
        
        fseek (fi,0,0);
        if (fread (wrkmem,coff_offset,1,fi)!=1)
            return "can't read stub";
    }
    /* Check for real image, (for now assume imbedded) */
    fseek (fi,coff_offset,0);
    if (fread (coffhdr,0x0a8,1,fi)!=1)
        return "can't read COFF header";
    if (*(short*)&coffhdr[0] != 0x014c)         /* COFF? */
        return "not V2 image";                  /* Not V2 image, show failure */
    if ((coffhdr[4] & 0x00020000)==0)
        return "COFF not executable";           /* Not executable */

    if (textsize+datasize != coffhdr[6]+coffhdr[7])
        return "already packed";
    if (textvirt+textsize != datavirt ||
        datavirt+datasize != bssvirt)
    {
        if (textvirt+textsize < datavirt &&
            datavirt-textvirt == datafpos-textfpos)
        {
            coffhdr[6]=textsize=datavirt-textvirt;
            /* This needs to compress Quake! */
        }
        else
            return "already packed";
    }

    maketempname (tmp_file,program,".\xfe",0);

    if ((fo=fopen (tmp_file,"wb"))==NULL)
        return "can't open temp file";
    if (!(options & OPTS))
    {
        if (exetype && !(options & OPTC))       /* keep original stub */
            if (fwrite (wrkmem,coff_offset,1,fo)!=1)
                return "write error";
    }

    stripdbg (coffhdr);

    size=((textsize+511)&~511)+datasize;
    in=(char*) malloc (size+512);
    out=(char*) malloc (size+size/64+512);
    textsize=datasize=pack_sections(in,out,coff_offset,size);

    if (in!=NULL)
        free (in);

    if (packerr!=NULL)
    {
        if (out!=NULL)
            free (out);
        return packerr;
    }

    textfpos=0xa8;
    if (!(options & OPTS))
    {
        unsigned char decomp2[sizeof(decomph)];
        unsigned      t1,ic;

        memcpy (decomp2,decomph,sizeof(decomph));
        if (bsssize < 0x200)
            bsssize = 0x200;            /* give it a .bss */

        textsize=sizeof(decomph);       /* new size of .text */
        while (1)
        {
            datafpos=textfpos+textsize; /* offset of raw data */
            if ((t1=((datafpos+datasize)&0x1ff)) >= EOFLEN && t1 < 0x1fc)
               break;
            /*printf ("\nEOFLEN\n");*/
            textsize+=4;
        } 

        datavirt=bssvirt+t1-datasize;
        if (patch_decomp (decomp2,0xdeadbee0,0x1000) ||
            patch_decomp (decomp2,0xdeadbee1,datasize) ||
            patch_decomp (decomp2,0xdeadbee2,datavirt) ||
            patch_decomp (decomp2,0xdeadbee3,bssvirt) ||
            patch_decomp (decomp2,0xdeadbee4,0x200/4) ||  
            patch_decomp (decomp2,0xdeadbee5,coffhdr[9]))
        {
            printline ("bad decompressor file",40,30);
            exit (1);
        }
        textvirt=coffhdr[9]=0xa8;
        bssvirt+=0x200;
        bsssize-=0x200;

        /* because of a feature (bug?) in stub.asm we need some padding */
        t1=datasize & 3;
        memset (out+datasize,4-t1,4-t1);
        datasize+=4-t1;      
        if (fwrite (coffhdr,sizeof(coffhdr),1,fo)!=1 ||
            fwrite (decomp2,textsize,1,fo)!=1 ||
            fwrite (out,datasize,1,fo)!=1)
        {
            free (out);
            return "write error";
        }
    }
    else
    {
        datafpos=datasize=0;
        if (fwrite (stub,sizeof(stub),1,fo)!=1 ||
            fwrite (coffhdr,sizeof(coffhdr),1,fo)!=1 ||
            fwrite (out,textsize,1,fo)!=1) 
        {
            free (out);
            return "write error";
        }
    }
    free (out);

    return dorenames (program,tmp_file,exetype==0 && (!(options & OPTS)));
}

static int checksumgood (char *in,unsigned size)
{
    char     buf [20];
    unsigned adler1;

    if (chksumpos==NULL)
    {
        if (options & OPTT)
            printline ("no checksum",40,30);
        return 1;
    }
    adler1 = lzo_adler32 (0,NULL,0);
    adler1 = lzo_adler32 (adler1,in,size);
    sprintf (buf,"%08X",adler1);                
    if (memcmp (buf,chksumpos,8)==0)
        return 1;
    /*printf ("%8.8s != %8.8s\n",chksumpos,buf);*/
    packerr="bad checksum";
    return 0;
}

static int unpack_section (char *in,char *out,unsigned foffset,unsigned isize,int method)
{
    unsigned unpackedsize;
    char buf[80];

    packerr=NULL;
    if (in==NULL || out==NULL)
        return packerr="out of memory",-1;
    fseek (fi,foffset,0);

    if (fread (in,isize,1,fi)!=1)
        return packerr="read error",-1;

    if (exetype!=2)                     /* remove padding */
        isize-=in[isize-1];
    if (checksumgood (in,isize)==0)
        return -1;
    if ((options & OPTD)==0)
    {
        if (chksumpos!=NULL)
            printline ("ok!",40,30);
        return 0;
    }
    if (method!='/' && method!='F')
        return packerr="unknown compression algorithm",-1;
    if ((method=='F' && lzo1f_decompress (in,isize,out,&unpackedsize,NULL)) ||
        (method=='/' && lzo1_decompress (in,isize,out,&unpackedsize,NULL)))
    {
        return packerr="decompression failed",-1;
    }

    sprintf (buf,"%8d -> %8d",isize,unpackedsize);
    printline (buf,40,30);

    if (fwrite (out,unpackedsize,1,fo)!=1)
        return packerr="write error",-1;
    return unpackedsize;
}

static const char *v2unpackimage (const char *program)
{
    char     magic[9],*p1,*in,*out;
    unsigned coff_offset=0,size,chksm;
    unsigned long coffhdr[42];
    unsigned short header[512/2];
    int      method='/';                        /* for LZO1/5-99 */

    exetype=0;
    printline (program,0,38);
    if ((fi = fopen (program,"rb"))==NULL)
        return "can't open";
    if (fread (header,sizeof(header),1,fi)!=1)
        return "read error";
    if (header[0] == 0x5a4d)                    /* MZ exe signature, stubbed? */
    {
        if (header[0x1a/2] == 0x6a64)           /* 'dj' */
            exetype++;
        coff_offset = (long)header[2]*512L;
        if (header[1])
            coff_offset += (long)header[1] - 512L;
        fseek (fi, 512, 0);                     /* Position of V2 stubinfo */
        if (fread (magic,8,1,fi)!=1)
            return "read error";
        if (memcmp (SIMAGIC,magic,8) != 0)
            return "bad MAGIC";                 /* Not V2 image, show failure */
        exetype++;
        fseek (fi,0,0);
        if (fread (wrkmem,coff_offset,1,fi)!=1)
            return "can't read stub";
    }

    fseek (fi,coff_offset,0);
    if (fread (coffhdr,0x0a8,1,fi)!=1)
        return "can't read coff header";
    if (*(short*)&coffhdr[0] != 0x014c)         /* COFF? */
        return "not V2 image";                  /* Not V2 image, show failure */

    maketempname (tmp_file,program,".\xfe",0);
    if ((fo=fopen ((options & OPTD) ? tmp_file : "nul","wb"))==NULL)
        return "can't open temp file";

    if (exetype==2)                             /* stubbed decompressor */
    {
        if ((chksumpos=findbytes (header,sizeof(header),"Checksum: ",10))!=NULL)
             chksumpos+=10;
        if ((p1=findbytes (header,sizeof(header),"LZO1",4))!=NULL)
            method=p1[4];                           /* 'F' for LZO1F-999 */
    }
    else
    {
        if (fread (wrkmem+coff_offset,256,1,fi)!=1)
            return "read error";
        if ((p1=findbytes (wrkmem+coff_offset,256,"DJP ",4))==NULL)
            return "not compressed with DJP";
        if ((p1=(char*) memchr (p1,0,10))==NULL)
            return "not DJP header";
        method=*(p1+1);                         /* decompression algorithm */
        sprintf (chksumpos=magic,"%08X",*(unsigned*)(p1+2)); /* checksum */

        if (exetype && !(options & OPTC) && fwrite (wrkmem,coff_offset,1,fo)!=1) /* copy the stub */
            return "write error";
    }
    
    size=textsize > datasize ? textsize : datasize;
    in=(char*)malloc (size+512);
    out=(char*)malloc (coffhdr[6]+coffhdr[7]+512); /* original size from the a.out header */

    if (exetype==2)
    {
        if (unpack_section (in,out,coff_offset+0xa8,textsize,method)==-1 ||
            (datasize && unpack_section (in,out,coff_offset+0xa8+textsize,datasize,method)==-1))
        {}
        /* restore original size and file offset */
        textsize=coffhdr[6];
        datasize=coffhdr[7];
        datafpos=datavirt-0x1000;
    }
    else
    {
        unpack_section (in,out,coff_offset+datafpos,datasize,method);
        memcpy (coffhdr,out,sizeof (coffhdr)); /* restore original coff header */
        stripdbg (coffhdr);
    }

    if (in!=NULL)
        free (in);
    if (out!=NULL)
        free (out);
    if (packerr!=NULL)
        return packerr;

    fseek (fo,exetype==2 || (options & OPTC) ? 0 : coff_offset,0);
    if (fwrite (coffhdr,sizeof(coffhdr),1,fo)!=1)
        return "write error";

    if ((options & OPTD)==0) 
        return packerr;                         /* test */

    return dorenames (program,tmp_file,exetype!=1);
}

/* Don't link this function from libc.a ==> -1600 bytes */
int _is_executable (const char *p,int i,const char *q)
{
    return 0;
}

#define BAD_OPTION 0x40000000
static int checkoption (int ch)
{
    static char *onames="dstbqc0h";
    char *p;
    
    if ((p=strchr (onames,ch|0x20))==NULL)
        return BAD_OPTION;
    return 1 << (p-onames);
}

int main (int argc, char **argv)
{
    int waserror=0,ic,jc,opt2=0;
    const char *p;
    char  buf[80];

    if ((p=getenv ("DJP_OPTIONS"))!=NULL)  /* get the default options */
        for (ic=0; ic<strlen(p); ic++)      /* from the environment */
            opt2 |= checkoption (p[ic]);
    opt2 &= OPTS|OPTC|OPTB|OPTQ;            /* use only these options */

    for (ic=1; ic<argc && argv[ic][0]=='-'; ic++)
        for (jc=1; jc<strlen(argv[ic]); jc++)
            if ((options |= checkoption (argv[ic][jc])) >= BAD_OPTION)
            {
                printf ("-%c unknown option!\n",argv[ic][jc]);
                return 1;
            }

    if (!(options & OPTQ))
        printf ("\nDJP 1.04 executable file compressor for DJGPP V2 programs."
                " (C) 1996 ML1050\n"
                "LZO data compression (C) 1996 Markus Franz Xaver Johannes Oberhumer\n\n");

    if (argc < 2 || (options & OPTH))
    {
        if (options & OPTQ)
            printf ("\n-qh ?? ok, here is the help in quiet mode :-)\n");
        else
            printf ("usage: %s [options] file [files]\n"
                    "options:\n  -d decompress\t\t\t"
                    "-s put the decompression code into %s\n"
                    "  -t integrity test\t\t"
                    "-c %s output\n  "
                    "-b %s backup files  \t-q %s\n  -h this help\t\t\t"
                    "-0 ignore default options from DJP_OPTIONS\n"
                    "file can be .exe or COFF\n\n",argv[0],
                    (opt2 & OPTS ) ? ".text" : "the stub",
                    (opt2 & OPTC ) ? "EXE" : "COFF",
                    (opt2 & OPTB ) ? "delete" : "keep",
                    (opt2 & OPTQ ) ? "verbose" : "quiet"
                    );
        return 1;
    }
    
    if (!(options & OPT0))
        options ^= opt2;   /* apply the default options */
    if ((options & (OPTC|OPTS|OPTT|OPTD))==(OPTC|OPTS))
    {
        printf ("-sc ?? sorry, this won't work.\n");
        return 1;
    }

    if (!(options & OPTS))
        chksumpos=findbytes (decomph,sizeof (decomph),"CHKS",4);
    else if ((chksumpos=findbytes (stub,sizeof (stub),"Checksum: ",10))!=NULL)
        chksumpos+=10;

    close (2); /* disable error messages from stubify */
    for (; ic<argc; ic++)
    {
        fi=fo=NULL;
        tmp_file[0]=0;
        if (options & (OPTD|OPTT))
            p=v2unpackimage(argv[ic]);
        else
            p=v2packimage(argv[ic]);
        if (p!=NULL)
        {
            sprintf (buf,"%s!",p);
            printline (buf,40,30);
            waserror++;
            if (options & OPTQ)         /* print error in quiet mode */
                printf ("%s\n",line);
        }
        if (!(options & OPTQ))
        {
            if (!isatty (1))            /* print full line if !isatty(1) */
                printf ("%s",line);
            printf ("\n");
        }
        if (fi!=NULL)
            fclose (fi);
        if (fo!=NULL)
            fclose (fo);
        if (tmp_file[0])
            remove (tmp_file);
    }
    return waserror;
}

/*
Todo:

-  keep stubinfo with -s

*/