/* lzo_mchw.ch -- matching functions using a window

   This file is part of the LZO real-time data compression library.

   Copyright (C) 1996 Markus Franz Xaver Johannes Oberhumer

   The LZO library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   The LZO library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the LZO library; see the file COPYING.LIB.
   If not, write to the Free Software Foundation, Inc.,
   675 Mass Ave, Cambridge, MA 02139, USA.

   Markus F.X.J. Oberhumer
   markus.oberhumer@jk.uni-linz.ac.at
 */




/***********************************************************************
//
************************************************************************/

typedef struct
{
	int init;

	lzo_uint look;			/* bytes in lookahead buffer */

	lzo_uint match_length;
	lzo_uint match_offset;

	const lzo_byte *ip;
	const lzo_byte *in;
	const lzo_byte *in_end;

	lzo_uint textsize;		/* text size counter */
	lzo_uint codesize;		/* code size counter */
	lzo_uint printcount;	/* counter for reporting progress every 1K bytes */

	lzo_progress_callback_t cb;

#if defined(LZO1C)
	lzo_uint r1_match_len;
	lzo_byte *m3;

	/* some stats */
	long r1_r, m3_r, m2_m, m3_m, lazy;
#endif

#if defined(LZO1D)
	/* some stats */
	long lit, m1, m2, m3, m4, lazy;
#endif

#if defined(LZO1F)
	lzo_uint r1_lit_len;
	lzo_uint r1_match_len;

	/* some stats */
	long r1_r, m2_m, m3_m, lazy;
#endif
}
LZO_COMPRESS_T;



#define getbyte(c)		((c).ip < (c).in_end ? *((c).ip)++ : -1)

#include "lzo_swd.ch"



/***********************************************************************
//
************************************************************************/

static int
match_init ( LZO_COMPRESS_T *c, lzo_swd_t *s )
{
	s->c = c;

	c->init = 1;  
	c->textsize = c->codesize = c->printcount = 0;

	return swd_init(s);
}


/***********************************************************************
//
************************************************************************/

static int
find_match ( LZO_COMPRESS_T *c, lzo_swd_t *s,
			 lzo_uint this_len, lzo_uint skip )
{
	int r;

	if (!c->init)
	{
		assert(this_len == 0);
		assert(skip == 0);
		r = match_init(c,s);
		if (r != 0)
		{
			c->look = 0;
			c->match_length = 0;
			return r;
		}
	}
	else
	{
		assert(this_len > 0);
	}

	if (skip > 0)
	{
		assert(this_len >= skip);
		swd_accept(s, this_len - skip);
		c->textsize += this_len - skip + 1;
	}
	else
	{
		assert(this_len <= 1);
		c->textsize += this_len - skip;
	}
		
	s->swd_len = THRESHOLD;
	swd_findbest(s);
	c->match_length = s->swd_len;
	c->match_offset = s->swd_bpos + 1;

	if (s->swd_char < 0)
	{
		c->look = 0;
		c->match_length = 0;
	}
	else
	{
		c->look = s->swd_look + 1;
	}
 
	if (c->cb && c->textsize > c->printcount) 
	{ 
		(*c->cb)(c->textsize,c->codesize);
		c->printcount += 1024;
	}

	return LZO_E_OK;
}


/*
vi:ts=4
*/

