/* lzo_swd.c -- sliding window dictionary 

   This file is part of the LZO real-time data compression library.

   Copyright (C) 1996 Markus Franz Xaver Johannes Oberhumer

   The LZO library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   The LZO library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the LZO library; see the file COPYING.LIB.
   If not, write to the Free Software Foundation, Inc.,
   675 Mass Ave, Cambridge, MA 02139, USA.

   Markus F.X.J. Oberhumer
   markus.oberhumer@jk.uni-linz.ac.at
 */



/***********************************************************************
// This file is adapted from HA, a general purpose file archiver.
// HA is Copyright (C) 1995 Harri Hirvola and is distributed it
// under the terms of the GNU General Public License.
// see: HA 0.999 (beta), file swdict.c
************************************************************************/

#define MINLEN		(THRESHOLD + 1)

#define HSIZE		16384

#if (MINLEN >= 3)
#  define HASH(b,p)	(((40799u*((((b[p]<<5)^b[p+1])<<5)^b[p+2]))>>5)&(HSIZE-1))
#  define MAXCNT	1024
#elif (MINLEN == 2)
#  define HASH(b,p)	(b[p]^(b[p+1]<<6))
#  define MAXCNT	1024
#else
#  error
#endif


typedef struct
{
	lzo_uint swd_bpos, swd_len;
	lzo_uint swd_look;
	int swd_char;

	lzo_uint blen;
	lzo_uint bbf, inptr;
	lzo_uint binb;

	unsigned char b [ N + F + F - 1];

	unsigned short ll [ N + F ];
	unsigned short best [ N + F ];
	unsigned short ccnt [ HSIZE ];
	unsigned short cr [ HSIZE ];

	LZO_COMPRESS_T *c;
}
lzo_swd_t;



static
int swd_init(lzo_swd_t *s)
{
	lzo_uint i;
	int c;

	s->blen = N + F;
	for (i = 0; i < HSIZE; ++i)
		s->ccnt[i] = 0;
	s->binb = N;
	s->bbf = s->swd_look = s->inptr = 0;
	while (s->swd_look < F)
	{
		if ((c = getbyte(*(s->c))) < 0)
			break;
		s->b[s->inptr++] = c;
		s->swd_look++;
	}
	s->swd_len = MINLEN - 1;
	return LZO_E_OK;
}


static 
void swd_getbyte(lzo_swd_t *s)
{
	int c;

	if (++s->bbf == s->blen)
		s->bbf = 0;
	if ((c = getbyte(*(s->c))) < 0)
	{
		--s->swd_look;
		if (++s->inptr == s->blen)
			s->inptr = 0;
	}
	else if (s->inptr < F - 1)
	{
		s->b[s->inptr + s->blen] = s->b[s->inptr] = c;
		++s->inptr;
	}
	else
	{
		s->b[s->inptr] = c;
		if (++s->inptr == s->blen)
			s->inptr = 0;
	}
}


static
void swd_accept(lzo_swd_t *s, lzo_uint j)
{
	lzo_uint i;

	while (j--)
	{
		if (s->binb == 0)
			--s->ccnt[HASH(s->b,s->inptr)];
		else
			--s->binb;

		i = HASH(s->b,s->bbf);
		s->ll[s->bbf] = s->cr[i];
		s->cr[i] = s->bbf;
		s->best[s->bbf] = F + 1;
		s->ccnt[i]++;

		swd_getbyte(s);
	}

	s->swd_len = MINLEN - 1;
}


static
lzo_uint _swd_findbest(lzo_swd_t *s, lzo_uint cnt, lzo_uint ptr)
{
	unsigned char *b = s->b;
	unsigned char *pbbf = b + s->bbf;
	lzo_uint swd_len = s->swd_len;
	unsigned char ref;
	unsigned char * const pxx = pbbf + s->swd_look;

	for (ref = pbbf[swd_len - 1]; cnt--; ptr = s->ll[ptr])
	{
		assert(swd_len < s->swd_look);
		if (b[ptr + swd_len - 1] == ref &&
		    b[ptr + swd_len] == pbbf[swd_len] &&
#if (MINLEN >= 3)
			b[ptr] == pbbf[0] && b[ptr + 1] == pbbf[1])
#else
			b[ptr] == pbbf[0])
#endif
		{
			register unsigned char *p1 = pbbf + MINLEN;
			register unsigned char *p2 = b + ptr + MINLEN;
			register unsigned char *px = pxx;
			lzo_uint i;

			while (p1 < px && *p1 == *p2)
				p1++, p2++;

			i = p1 - pbbf;
			if (i > swd_len)
			{
				s->swd_bpos = ptr;
				if ((swd_len = i) == s->swd_look || s->best[ptr] < i)
					break;
				ref = pbbf[swd_len - 1];
			}
		}
	}
	return swd_len;
}


static
void swd_findbest(lzo_swd_t *s)
{
	lzo_uint i;
	lzo_uint cnt, ptr, start_len;
	const lzo_uint bbf = s->bbf;

	i = HASH(s->b,bbf);
	if ((cnt = s->ccnt[i]++) > MAXCNT)
		cnt = MAXCNT;
	ptr = s->ll[bbf] = s->cr[i];
	s->cr[i] = bbf;
	s->swd_char = s->b[bbf];
	if ((start_len = s->swd_len) >= s->swd_look)
	{
		if (s->swd_look == 0)
			s->swd_char = -1;
		s->best[bbf] = F + 1;
	}
	else
	{
		s->swd_bpos = N + 1;
		s->best[bbf] = s->swd_len = _swd_findbest(s,cnt,ptr);
		if (s->swd_len > start_len)
		{
			if (s->swd_bpos < bbf)
				s->swd_bpos = bbf - s->swd_bpos - 1;
			else
				s->swd_bpos = s->blen - 1 - s->swd_bpos + bbf;
		}
	}

	if (s->binb == 0)
		--s->ccnt[HASH(s->b,s->inptr)];
	else
		--s->binb;
		
	swd_getbyte(s);
}


/*
vi:ts=4
*/

