/******************************************************************** ITEMS.H
 *                                                                          *
 *                     Display Item Class Definition                        *
 *                                                                          *
 ****************************************************************************/

#ifndef ITEMS_H
#define ITEMS_H

#ifndef OS2_INCLUDED
   #define INCL_BASE
   #define INCL_PM
   #include <os2.h>
#endif

#include <time.h>

#include "Dde.h"
#include "ReString.h"

extern void Log ( char *Message, ... ) ;

class Item {

   private:
      USHORT Id ;                  // Item ID.
      BOOL   Flag ;                // Flag: Show this item at this time?
      char   Name [80] ;           // Text for items profile name.
      char   DefaultLabel [80] ;   // Text to display on left part of line (default).
      Dde_Server *pDdeServer ;     // -> DDE Server object
      char Topic [80] ;            // DDE Topic name
      Dde_Item *pDdeItem ;         // -> DDE Item object

   protected:
      char   CurrentLabel [80] ;   // Text to display on left part of line.
      ULONG  Value ;               // Current value.
      int    LevelSense ;          // Flag: If +1, level sensing has Error > Warning > Normal.
      ULONG  MinLevel, MaxLevel ;  //       If -1, Error < Warning < Normal.  If zero, level checks are disabled.
      ULONG  WarningLevel, ErrorLevel, DefaultLevels[2] ;
      COLOR  NormalColors[2], WarningColors[2], ErrorColors[2] ;

   public:
      Item ( USHORT id, char *pName, char *pCurrentLabel, char *pDefaultLabel, Dde_Server *ddeserver, char *topic ) :
         Id(id), Flag(TRUE), Value(0), pDdeServer(ddeserver), LevelSense(0) {
         strcpy ( Name, pName ) ;
         strcpy ( CurrentLabel, pCurrentLabel ) ;
         strcpy ( DefaultLabel, pDefaultLabel ) ;
         strcpy ( Topic, topic ) ;
         pDdeServer->AddItem ( Topic, DefaultLabel, DDEFMT_TEXT, "", 1 ) ;
         pDdeItem = pDdeServer->FindItem ( Topic, DefaultLabel ) ;
         NormalColors [0] = 0xFFFFFF ;
         NormalColors [1] = 0x000000 ;
         WarningColors[0] = 0xFFFF00 ;
         WarningColors[1] = 0x000000 ;
         ErrorColors  [0] = 0xFF0000 ;
         ErrorColors  [1] = 0xFFFFFF ;
      } /* endmethod */

      ~Item ( ) {
         pDdeServer->RemoveItem ( Topic, DefaultLabel ) ;
      } /* endmethod */

      void GetProfile ( HINI IniHandle ) ;
      void PutProfile ( HINI IniHandle ) ;

      USHORT QueryId           ( void ) { return ( Id   ) ; }
      BOOL   QueryFlag         ( void ) { return ( Flag ) ; }
      PCHAR  QueryName         ( void ) { return ( Name ) ; }
      PCHAR  QueryCurrentLabel ( void ) { return ( CurrentLabel ) ; }
      PCHAR  QueryDefaultLabel ( void ) { return ( DefaultLabel ) ; }
      ULONG  QueryValue        ( void ) { return ( Value ) ; }

      void SetLabel ( char *label ) { strcpy ( CurrentLabel, label ) ;  Value = 0 ; }

      void SetNormalColors ( COLOR Background, COLOR Foreground ) {
         NormalColors[0] = Background ;
         NormalColors[1] = Foreground ;
      } /* endmethod */

      ULONG QueryMinLevel   ( ) { return ( MinLevel ) ; }
      ULONG QueryMaxLevel   ( ) { return ( MaxLevel ) ; }
      int   QueryLevelSense ( ) { return ( LevelSense ) ; }

      void SetMinLevel   ( ULONG Level ) { MinLevel = Level ; }
      void SetMaxLevel   ( ULONG Level ) { MaxLevel = Level ; }
      void SetLevelSense ( int   Sense ) { LevelSense = Sense ; }

      void QueryDefaultLevels ( ULONG &Warning, ULONG &Error ) {
         Warning = DefaultLevels [0] ;
         Error = DefaultLevels [1] ;
      } /* endmethod */

      ULONG QueryWarningLevel ( ) { return ( WarningLevel ) ; }
      void SetWarningLevel ( ULONG Level ) { WarningLevel = Level ; }
      void SetWarningColors ( COLOR Background, COLOR Foreground ) {
         WarningColors[0] = Background ;
         WarningColors[1] = Foreground ;
      } /* endmethod */

      ULONG QueryErrorLevel ( ) { return ( ErrorLevel ) ; }
      void SetErrorLevel ( ULONG Level ) { ErrorLevel = Level ; }
      void SetErrorColors ( COLOR Background, COLOR Foreground ) {
         ErrorColors[0] = Background ;
         ErrorColors[1] = Foreground ;
      } /* endmethod */

      void SetFlag   ( void ) { Flag = TRUE ; }
      void ResetFlag ( void ) { Flag = FALSE ; }

      void DdeUpdate ( char *Text ) ;

      void Paint ( HPS hPS, RECTL &Rectangle, 
         char *Label, char *Text, ULONG NewValue, int Scale=1 ) ;

      // The following functions must be provided by all subclasses.

      virtual int Measure ( HPS hPS, RECTL &Rectangle ) = 0 ;

      virtual ULONG NewValue ( void ) = 0 ;

      virtual void FormatText ( char *Label, char *Text, ULONG Value ) = 0 ;

      virtual void FormatLine ( char *Buffer, int MaxWidth ) ;

      virtual void Repaint ( HPS hPS, RECTL &Rectangle, BOOL Mandatory ) = 0 ;
} ;

class Clock : public Item {

   private:
      COUNTRYINFO CountryInfo ;
      char szAm [8] ;
      char szPm [8] ;
      ResourceString *DaysOfWeek ;
      BOOL ShowSeconds ;

   public:
      Clock ( USHORT id, char *pName, char *pCurrentLabel, char *pDefaultLabel, 
         Dde_Server *pDdeServer, char *Topic, COUNTRYINFO &countryinfo,
         char *szam, char *szpm, ResourceString *daysofweek, BOOL showseconds )
         : Item ( id, pName, pCurrentLabel, pDefaultLabel, pDdeServer, Topic ),
         DaysOfWeek(daysofweek), ShowSeconds(showseconds) {
         CountryInfo = countryinfo ;
         memcpy ( szAm, szam, min(strlen(szam),sizeof(szAm)-1) ) ;
         szAm [ sizeof(szAm) - 1 ] = 0 ;
         memcpy ( szPm, szpm, min(strlen(szpm),sizeof(szPm)-1) ) ;
         szPm [ sizeof(szPm) - 1 ] = 0 ;
      } /* endmethod */

      void SetShowSeconds ( BOOL showseconds ) { ShowSeconds = showseconds ; Value = 0 ; }

      int Measure ( HPS hPS, RECTL &Rectangle ) ;
      ULONG NewValue ( void ) ;
      void FormatText ( char *Label, char *Text, ULONG Value ) ;
      void FormatLine ( char *Buffer, int MaxWidth ) ;
      void Repaint ( HPS hPS, RECTL &Rectangle, BOOL Mandatory ) ;
} ;

class ElapsedTime : public Item {

   private:
      COUNTRYINFO CountryInfo ;
      ResourceString *Day ;
      ResourceString *Days ;
      BOOL ShowSeconds ;

   public:
      ElapsedTime ( USHORT id, char *pName, char *pCurrentLabel, char *pDefaultLabel, 
         Dde_Server *pDdeServer, char *Topic, COUNTRYINFO &countryinfo,
         ResourceString *day, ResourceString *days, BOOL showseconds )
         : Item ( id, pName, pCurrentLabel, pDefaultLabel, pDdeServer, Topic ),
         Day(day), Days(days), ShowSeconds(showseconds) {
         CountryInfo = countryinfo ;
      } /* endmethod */

      void SetShowSeconds ( BOOL showseconds ) { ShowSeconds = showseconds ;  Value = 0 ; }

      int Measure ( HPS hPS, RECTL &Rectangle ) ;
      ULONG NewValue ( void ) ;
      void FormatText ( char *Label, char *Text, ULONG Value ) ;
      void Repaint ( HPS hPS, RECTL &Rectangle, BOOL Mandatory ) ;
} ;

class SwapFree : public Item {

   private:
      COUNTRYINFO CountryInfo ;
      USHORT ShowK ;
      PSZ SwapPath ;
      ULONG MinFree ;

   public:
      SwapFree ( USHORT id, char *pName, char *pCurrentLabel, char *pDefaultLabel, 
         Dde_Server *pDdeServer, char *Topic, COUNTRYINFO &countryinfo,
         USHORT sk, PSZ swappath, ULONG minfree )
         : Item ( id, pName, pCurrentLabel, pDefaultLabel, pDdeServer, Topic ),
         ShowK(sk), MinFree(minfree) {
         CountryInfo = countryinfo ;
         SwapPath = new BYTE [ strlen(PCHAR(swappath)) + 1 ] ;
         strcpy ( PCHAR(SwapPath), PCHAR(swappath) ) ;
         LevelSense = -1 ;
         MinLevel = 0 ;
         MaxLevel = 64*1024*1024 ;
         DefaultLevels [0] = WarningLevel = 8*1024*1024 ;
         DefaultLevels [1] = ErrorLevel = 4*1024*1024 ;
      } /* endmethod */

      ~SwapFree ( void ) {
         delete [] SwapPath ;
      } /* endmethod */

      void SetShowK ( USHORT flag ) { ShowK = flag ;  Value = 0 ; }

      int Measure ( HPS hPS, RECTL &Rectangle ) ;
      ULONG NewValue ( void ) ;
      void FormatText ( char *Label, char *Text, ULONG Value ) ;
      void Repaint ( HPS hPS, RECTL &Rectangle, BOOL Mandatory ) ;
} ;

class MemoryFree : public Item {

   private:
      COUNTRYINFO CountryInfo ;
      USHORT ShowK ;

   public:
      MemoryFree ( USHORT id, char *pName, char *pCurrentLabel, char *pDefaultLabel, 
         Dde_Server *pDdeServer, char *Topic, COUNTRYINFO &countryinfo, USHORT sk ) 
         : Item ( id, pName, pCurrentLabel, pDefaultLabel, pDdeServer, Topic ), 
         ShowK(sk) {
         CountryInfo = countryinfo ;
         LevelSense = -1 ;
         MinLevel = 0 ;
         ULONG TotalPhysicalMemory ;
         DosQuerySysInfo ( QSV_TOTPHYSMEM, QSV_TOTPHYSMEM, &TotalPhysicalMemory, sizeof(TotalPhysicalMemory) ) ;
         MaxLevel = TotalPhysicalMemory ;
         DefaultLevels [0] = WarningLevel = TotalPhysicalMemory / 8 ;
         DefaultLevels [1] = ErrorLevel = TotalPhysicalMemory / 16 ;
      } /* endmethod */

      void SetShowK ( USHORT flag ) { ShowK = flag ;  Value = 0 ; }

      int Measure ( HPS hPS, RECTL &Rectangle ) ;
      ULONG NewValue ( void ) ;
      void FormatText ( char *Label, char *Text, ULONG Value ) ;
      void Repaint ( HPS hPS, RECTL &Rectangle, BOOL Mandatory ) ;
} ;

class VirtualFree : public Item {

   private:
      COUNTRYINFO CountryInfo ;
      USHORT ShowK ;

   public:
      VirtualFree ( USHORT id, char *pName, char *pCurrentLabel, char *pDefaultLabel, 
         Dde_Server *pDdeServer, char *Topic, COUNTRYINFO &countryinfo, USHORT sk ) 
         : Item ( id, pName, pCurrentLabel, pDefaultLabel, pDdeServer, Topic ), 
         ShowK(sk) {
         CountryInfo = countryinfo ;
         LevelSense = -1 ;
         MinLevel = 0 ;
         MaxLevel = 64*1024*1024 ;
         DefaultLevels [0] = WarningLevel = 8*1024*1024 ;
         DefaultLevels [1] = ErrorLevel = 4*1024*1024 ;
      } /* endmethod */

      void SetShowK ( USHORT flag ) { ShowK = flag ;  Value = 0 ; }

      int Measure ( HPS hPS, RECTL &Rectangle ) ;
      ULONG NewValue ( void ) ;
      void FormatText ( char *Label, char *Text, ULONG Value ) ;
      void Repaint ( HPS hPS, RECTL &Rectangle, BOOL Mandatory ) ;
} ;

class SwapSize : public Item {

   private:
      COUNTRYINFO CountryInfo ;
      USHORT ShowK ;
      PSZ SwapPath ;

   public:
      SwapSize ( USHORT id, char *pName, char *pCurrentLabel, char *pDefaultLabel, 
         Dde_Server *pDdeServer, char *Topic, COUNTRYINFO countryinfo, 
         USHORT sk, PSZ swappath )
         : Item ( id, pName, pCurrentLabel, pDefaultLabel, pDdeServer, Topic ), 
         ShowK(sk) {
         CountryInfo = countryinfo ;
         SwapPath = new BYTE [ strlen(PCHAR(swappath)) + 1 ] ;
         strcpy ( PCHAR(SwapPath), PCHAR(swappath) ) ;
         LevelSense = +1 ;
         MinLevel = 0 ;
         MaxLevel = 64*1024*1024 ;
         DefaultLevels [0] = WarningLevel = 16*1024*1024 ;
         DefaultLevels [1] = ErrorLevel = 32*1024*1024 ;
      } /* endmethod */

      ~SwapSize ( void ) {
         delete [] SwapPath ;
      } /* endmethod */

      void SetShowK ( USHORT flag ) { ShowK = flag ;  Value = 0 ; }

      int Measure ( HPS hPS, RECTL &Rectangle ) ;
      ULONG NewValue ( void ) ;
      void FormatText ( char *Label, char *Text, ULONG Value ) ;
      void Repaint ( HPS hPS, RECTL &Rectangle, BOOL Mandatory ) ;
} ;

class SpoolSize : public Item {

   private:
      COUNTRYINFO CountryInfo ;
      USHORT ShowK ;
      PSZ SpoolPath ;

   public:
      SpoolSize ( USHORT id, char *pName, char *pCurrentLabel, char *pDefaultLabel, 
         Dde_Server *pDdeServer, char *Topic, COUNTRYINFO &countryinfo, 
         USHORT sk, PSZ spoolpath )
         : Item ( id, pName, pCurrentLabel, pDefaultLabel, pDdeServer, Topic ), 
         ShowK(sk) {
         CountryInfo = countryinfo ;
         SpoolPath = new BYTE [ strlen(PCHAR(spoolpath)) + 1 ] ;
         strcpy ( PCHAR(SpoolPath), PCHAR(spoolpath) ) ;
         LevelSense = +1 ;
         MinLevel = 0 ;
         MaxLevel = 64*1024*1024 ;
         DefaultLevels [0] = WarningLevel = 8*1024*1024 ;
         DefaultLevels [1] = ErrorLevel = 16*1024*1024 ;
      } /* endmethod */

      ~SpoolSize ( void ) {
         delete [] SpoolPath ;
      } /* endmethod */

      void SetShowK ( USHORT flag ) { ShowK = flag ;  Value = 0 ; }

      int Measure ( HPS hPS, RECTL &Rectangle ) ;
      ULONG NewValue ( void ) ;
      void FormatText ( char *Label, char *Text, ULONG Value ) ;
      void Repaint ( HPS hPS, RECTL &Rectangle, BOOL Mandatory ) ;
} ;

class CpuLoad : public Item {

   private:
      PULONG IdleCount ;
      ULONG MaxCount ;

   public:
      CpuLoad ( USHORT id, char *pName, char *pCurrentLabel, char *pDefaultLabel, 
         Dde_Server *pDdeServer, char *Topic, ULONG maxcount, PULONG idlecount )
         : Item ( id, pName, pCurrentLabel, pDefaultLabel, pDdeServer, Topic ), 
         MaxCount(maxcount), IdleCount(idlecount) { 
         LevelSense = +1 ;
         MinLevel = 0 ;
         MaxLevel = 100 ;
         DefaultLevels [0] = WarningLevel = 75 ;
         DefaultLevels [1] = ErrorLevel = 90 ;
      } /* endmethod */

      int Measure ( HPS hPS, RECTL &Rectangle ) ;
      ULONG NewValue ( void ) ;
      void FormatText ( char *Label, char *Text, ULONG Value ) ;
      void Repaint ( HPS hPS, RECTL &Rectangle, BOOL Mandatory ) ;
} ;

class TaskCount : public Item {

   private:
      HAB Anchor ;

   public:
      TaskCount ( USHORT id, char *pName, char *pCurrentLabel, char *pDefaultLabel, 
         Dde_Server *pDdeServer, char *Topic, HAB anchor )
         : Item ( id, pName, pCurrentLabel, pDefaultLabel, pDdeServer, Topic ), 
         Anchor(anchor) { 
         LevelSense = +1 ;
         MinLevel = 0 ;
         MaxLevel = 256 ;     
         DefaultLevels [0] = WarningLevel = 30 ;  
         DefaultLevels [1] = ErrorLevel = 40 ;    
      } /* endmethod */

      int Measure ( HPS hPS, RECTL &Rectangle ) ;
      ULONG NewValue ( void ) ;
      void FormatText ( char *Label, char *Text, ULONG Value ) ;
      void Repaint ( HPS hPS, RECTL &Rectangle, BOOL Mandatory ) ;
} ;

class DriveFree : public Item {

   private:
      COUNTRYINFO CountryInfo ;
      USHORT ShowK ;
      ResourceString *DriveError ;
      USHORT DriveNumber ;
      BOOL ShowFileSystemName ;
      BYTE FileSystem [80] ;
      BOOL ShowDiskLabel ;
      BYTE DiskLabel [12] ;
      BOOL Error ;

   public:
      DriveFree ( USHORT id, char *pName, char *pCurrentLabel, char *pDefaultLabel, 
         Dde_Server *pDdeServer, char *Topic, COUNTRYINFO &countryinfo, 
         USHORT sk, USHORT drivenumber, ResourceString *driveerror, BOOL showfilesystemname, 
         PSZ filesystem, BOOL showdisklabel, PSZ disklabel ) 
         : Item ( id, pName, pCurrentLabel, pDefaultLabel, pDdeServer, Topic ), 
         ShowK(sk), DriveError(driveerror), DriveNumber(drivenumber),
         ShowFileSystemName(showfilesystemname), ShowDiskLabel(showdisklabel), 
         Error(FALSE) {
         CountryInfo = countryinfo ;
         strcpy ( PCHAR(FileSystem), PCHAR(filesystem) ) ;
         strcpy ( PCHAR(DiskLabel), PCHAR(disklabel) ) ;
         LevelSense = -1 ;
         MinLevel = 0 ;
         DosError ( FERR_DISABLEHARDERR ) ;
         FSALLOCATE Allocation ;
         DosQueryFSInfo ( DriveNumber, FSIL_ALLOC, (PBYTE)&Allocation, sizeof(Allocation) ) ;
         DosError ( FERR_ENABLEHARDERR ) ;
         MaxLevel = Allocation.cUnit*Allocation.cSectorUnit*Allocation.cbSector ;
         DefaultLevels [0] = WarningLevel = MaxLevel / 5 ;
         DefaultLevels [1] = ErrorLevel = MaxLevel / 10 ;
      } /* endmethod */

      void SetShowK ( USHORT flag ) { ShowK = flag ;  Value = 0 ; }
      void SetShowFileSystemName ( BOOL showfilesystemname ) { ShowFileSystemName = showfilesystemname ;  Value = 0 ; }
      void SetShowDiskLabel ( BOOL showdisklabel ) { ShowDiskLabel = showdisklabel ;  Value = 0 ; }
      void ResetError ( ) { Error = FALSE ; } 

      int Measure ( HPS hPS, RECTL &Rectangle ) ;
      ULONG NewValue ( void ) ;
      void FormatText ( char *Label, char *Text, ULONG Value ) ;
      void Repaint ( HPS hPS, RECTL &Rectangle, BOOL Mandatory ) ;
} ;

class TotalFree : public Item {

   private:
      COUNTRYINFO CountryInfo ;
      USHORT ShowK ;
      ULONG Drives ;

   public:
      TotalFree ( USHORT id, char *pName, char *pCurrentLabel, char *pDefaultLabel, 
         Dde_Server *pDdeServer, char *Topic, COUNTRYINFO &countryinfo, 
         USHORT sk, ULONG drives )
         : Item ( id, pName, pCurrentLabel, pDefaultLabel, pDdeServer, Topic ), 
         ShowK(sk), Drives(drives) {
         CountryInfo = countryinfo ;
         LevelSense = -1 ;
         MinLevel = 0 ;
         MaxLevel = 64*1024*1024 ;    
         DefaultLevels [0] = WarningLevel = 8*1024*1024 ;
         DefaultLevels [1] = ErrorLevel = 4*1024*1024 ;
      } /* endmethod */

      void SetShowK ( USHORT flag ) { ShowK = flag ;  Value = 0 ; }
      void ResetMask ( ULONG drives ) { Drives = drives ; }

      int Measure ( HPS hPS, RECTL &Rectangle ) ;
      ULONG NewValue ( void ) ;
      void FormatText ( char *Label, char *Text, ULONG Value ) ;
      void Repaint ( HPS hPS, RECTL &Rectangle, BOOL Mandatory ) ;
} ;

class SwapSlack : public Item {

   private:
      COUNTRYINFO  CountryInfo ;
      USHORT       ShowK ;
      VirtualFree *pVirtualFree ;
      SwapFree    *pSwapFree ;
      MemoryFree  *pMemFree ;

   public:
      SwapSlack ( USHORT id, char *pName, char *pCurrentLabel, char *pDefaultLabel, 
         Dde_Server *pDdeServer, char *Topic, COUNTRYINFO &countryinfo, 
         USHORT sk, VirtualFree *pvf, SwapFree *psf, MemoryFree *pmf ) 
         : Item ( id, pName, pCurrentLabel, pDefaultLabel, pDdeServer, Topic ), 
         ShowK(sk), pVirtualFree(pvf), pSwapFree(psf), pMemFree(pmf) {
         CountryInfo = countryinfo ;
      } /* endmethod */

      void SetShowK ( USHORT flag ) { ShowK = flag ;  Value = 0 ; }

      int Measure ( HPS hPS, RECTL &Rectangle ) ;
      ULONG NewValue ( void ) ;
      void FormatText ( char *Label, char *Text, ULONG Value ) ;
      void Repaint ( HPS hPS, RECTL &Rectangle, BOOL Mandatory ) ;
} ;

class ProcessCount : public Item {

   private:
      PULONG DQPS_Buffer ;

   public:
      ProcessCount ( USHORT id, char *pName, char *pCurrentLabel, char *pDefaultLabel,
         Dde_Server *pDdeServer, char *Topic )
         : Item ( id, pName, pCurrentLabel, pDefaultLabel, pDdeServer, Topic ) { 
         DosAllocMem ( (PPVOID)&DQPS_Buffer, 0x10000, PAG_READ | PAG_WRITE | PAG_COMMIT | OBJ_TILE ) ;
         LevelSense = +1 ;
         MinLevel = 0 ;
         MaxLevel = 256 ;    
         DefaultLevels [0] = WarningLevel = 30 ; 
         DefaultLevels [1] = ErrorLevel = 40 ;   
      } /* endmethod */

      ~ProcessCount ( ) { DosFreeMem ( DQPS_Buffer ) ; }

      int Measure ( HPS hPS, RECTL &Rectangle ) ;
      ULONG NewValue ( void ) ;
      void FormatText ( char *Label, char *Text, ULONG Value ) ;
      void Repaint ( HPS hPS, RECTL &Rectangle, BOOL Mandatory ) ;
} ;

extern PSZ ScanSystemConfig ( HAB Anchor, PSZ Keyword ) ;

class ThreadCount : public Item {

   private:
      PULONG DQPS_Buffer ;

   public:
      ThreadCount ( USHORT id, char *pName, char *pCurrentLabel, char *pDefaultLabel,
         Dde_Server *pDdeServer, char *Topic )
         : Item ( id, pName, pCurrentLabel, pDefaultLabel, pDdeServer, Topic ) { 
         DosAllocMem ( (PPVOID)&DQPS_Buffer, 0x10000, PAG_READ | PAG_WRITE | PAG_COMMIT | OBJ_TILE ) ;
         LevelSense = +1 ;
         MinLevel = 0 ;
         PSZ Threads = ScanSystemConfig ( 0, "THREADS" ) ;
         if ( Threads == 0 ) 
            Threads = (PSZ) "64" ;
         int MaxThreads = atoi ( Threads ) ;
         MaxLevel = MaxThreads ;
         DefaultLevels [0] = WarningLevel = ( MaxThreads * 3 ) / 4 ; 
         DefaultLevels [1] = ErrorLevel = ( MaxThreads * 9 ) / 10 ;   
      } /* endmethod */

      ~ThreadCount ( ) { DosFreeMem ( DQPS_Buffer ) ; }

      int Measure ( HPS hPS, RECTL &Rectangle ) ;
      ULONG NewValue ( void ) ;
      void FormatText ( char *Label, char *Text, ULONG Value ) ;
      void Repaint ( HPS hPS, RECTL &Rectangle, BOOL Mandatory ) ;
} ;

#endif
