#include "../defs.h"
#include "../mountrec.h"
#include "local_btree.h"
#include "../errors.h"


static int   BitMapRead(MOUNT_RECORD *mnt,
			BT_HEADER    *head,
			LONGWORD     node,
			BYTE         *buffer,
			BYTE        **bitmap,
			LONGWORD     *first_in_bitmap,
			int          *bimap_length,
			LONGWORD     *bitmap_node_no,
			LONGWORD     *next_bitmap);


			      /* Allocate a node in the bitmap; doesn't deal 
                               * with extending the bitmap */
int  BT_BitMapAllocNode(MOUNT_RECORD *mnt,
			BT_HEADER *head,
			LONGWORD *node,
			BYTE * buffer)     {
					/* I'm sure here is enough space */
  int nr_nodes;
  int err;
  BYTE n,mask, *bitmap;
  int i, j;
  int node_length;
  LONGWORD base,current_node,bitmap_node_no, next_to_read;
  
  nr_nodes = head->bth.bthNNodes;	/* Total number of nodes in tree */

  current_node     = 0;	      /* Start from node 0 */
  base  = 0;		      /* I want to start from the beginning */

  while(current_node < nr_nodes)      {

    CHKERR(BitMapRead(mnt, head, current_node, buffer, &bitmap, &base,
		      &node_length, &bitmap_node_no, &next_to_read));
    
					/* Search for one bit 0 */
    for(i=0, j=0; i < node_length ; i++, (mask >>= 1))   {
			      /* Next byte */
      if(!(i & 7))  {
	mask = 0x80;
	n = bitmap[j++];
      }

      if(!(n & mask))
	break;
    }
			      /* One free slot found */
    if(i < node_length)   {
      *node = base + i;
      bitmap[j-1] |= mask;    /* Mark the node allocated */

			      /* Write the bitmap node only if base is not 0 
                               * because otherwise the bitmap is in the header 
                               * */
      if(bitmap_node_no != 0) {
	CHKERR(BT_WriteNode(mnt, head, bitmap_node_no, buffer));
      }
      head->bth.bthFree --;
      return 0;
    }

			      /* Here if we need to search the next bitmap 
                               * node  */
    current_node = base + node_length;
  }

			      /* Here only if I visited all the nodes */
  return(BT_NOTALLOC);
}



int  BT_BitMapFreeNode(MOUNT_RECORD *mnt,
		       BT_HEADER *head,
		       LONGWORD node,
		       BYTE * buffer)     {
				
  int nr_nodes;
  int err;
  BYTE mask, *bitmap;
  int len_node;
  LONGWORD  base, bitmap_node_no, next_bitmap_node;

  nr_nodes = head->bth.bthNNodes;	/* Total number of nodes in tree */

  if(node >= nr_nodes)
    return(BT_BADNODE);

  base = 0;		      /* I want to start from the beginning */

			      /* Read the bitmap node associated with node */
  CHKERR(BitMapRead(mnt, head, node, buffer, &bitmap, &base, &len_node,
		    &bitmap_node_no, &next_bitmap_node));

  mask = 0x80 >> ((node-base) & 7);
  
  bitmap[(node-base) >> 3] &= ~mask;

			      /* Write the bitmap only if not in header */
  if(bitmap_node_no != 0)
    CHKERR(BT_WriteNode(mnt, head, bitmap_node_no, buffer));

  head->bth.bthFree++; 
  return 0;
}


			      /* Test if the node is marked allocated in the 
                               * bitmap
			       */
int  BT_BitMapNodeBusy(MOUNT_RECORD *mnt,
		       BT_HEADER *head,
		       LONGWORD node,
		       BYTE * buffer)     {
				
  int nr_nodes;
  int err;
  BYTE mask, *bitmap;
  int len_node;
  LONGWORD base, bitmap_node_no, next_bitmap_node;

  nr_nodes = head->bth.bthNNodes;	/* Total number of nodes in tree */

  if(node >= nr_nodes)
    return(BT_BADNODE);

  base = 0;		      /* I want to start from the beginning */

			      /* Read the bitmap node associated with node */
  CHKERR(BitMapRead(mnt, head, node, buffer, &bitmap, &base, &len_node,
		    &bitmap_node_no, &next_bitmap_node));

  mask = 0x80 >> ((node-base) & 7);
  
  return(bitmap[(node-base) >> 3] & mask);
}


			      /* Extends the bitmap for additional nodes . I 
                               * ASSUME that I don't need to allocate more 
                               * than one bitmap node */

int BT_BitMapExtend(MOUNT_RECORD *mnt,
		    BT_HEADER    *head,
		    LONGWORD     nodes,
		    BYTE         *buffer)   {

  LONGWORD nr_nodes;
  LONGWORD base, bitmap_node, next_bitmap_node;
  int len_node, err;
  int part;
  BYTE *bitmap;
  
			      /* Get the old number of nodes */
  nr_nodes = head->bth.bthNNodes;

			      /* Read the bitmap for the last node in file */
  base = 0;		      /* Start from the beginning */

  CHKERR(BitMapRead(mnt, head, (LONGWORD)(nr_nodes - 1), buffer, &bitmap,
		    &base, &len_node, &bitmap_node, &next_bitmap_node));

			      /* Maybe I can fit part of the new nodes bit 
                               * right here */
  part = (base == 0 ? (256*8) : (494*8)) - len_node;

			      /* If yes I can get out because I know that all 
                               * the bits are reset */
  if(part >= nodes)
    return 0;


			      /* Allocate one more node for the bitmap */
			      /* In order to avoid infinite loop I won't call 
                               * BT_BitMapAllocNode to alloc the next block 
                               * but instead I will allocate the node number 
                               * nr_nodes for the bitmap */
  head->bth.bthFree--;
			      /* Put the pointer in the old node */
  PutLongWord( nr_nodes , & buffer[ND_FLINK]);
  
			      /* Put it's bit 1 if part >= 1 */
  if(part >= 1)  {

    bitmap[(len_node + 1) >> 3] &= ~( 0x80 >> (len_node & 7));
  }
  
			      /* And write the previous node */
  if(bitmap_node != 0)
    CHKERR(BT_WriteNode(mnt, head, bitmap_node, buffer));
  

			      /* Read in the new node */
  CHKERR(BT_ReadNode(mnt, head, nr_nodes, buffer));

			      /* Initialize it's descriptor */
  bzero(buffer , BT_NODESIZE);
  PutByte( NODE_MAP, & buffer[ND_TYPE]);
  PutWord( (WORD)1, & buffer[ND_NRECS]);

			      /* and the backpointers */
  PutWord( 14, & buffer[ BT_NODESIZE - 2]);
  PutWord( BT_NODESIZE - 4, & buffer[ BT_NODESIZE - 4]);

			      /* If necessary mark it's bit here */
  if(part == 0)
    *(buffer+14) = 0x80;

			      /* Write the new node */
  return(BT_WriteNode(mnt, head, nr_nodes, buffer));
  
}
  

			      /* Reads the bitmap node for the node in buffer 
                               * and returns: the number of the next bitmap 
                               * node (a convenient way of chaining if the 
                               * user wants to read all the bitmap), a pointer 
                               * to where in the buffer is the bitmap, the 
                               * bitmap length in bits and the number of the 
                               * node that corresponds to the forst bit in the 
                               * bitmap. If all three next_bitmap, 
                               * first_in_bitmap and bitmap_length are non 0 
                               * they are assumed as from the previous call 
                               * and I start searching at that position */

static int   BitMapRead(MOUNT_RECORD *mnt,
			BT_HEADER    *head,
			LONGWORD     node,
			BYTE         *buffer,
			BYTE        **bitmap,
			LONGWORD     *first_in_bitmap,
			int          *bitmap_length,
			LONGWORD     *bitmap_node_no,
			LONGWORD     *next_bitmap)     {



  int err;

  if(*next_bitmap == 0 || *first_in_bitmap == 0 || *bitmap_length == 0)  {


			      /* Start from the bitmap in the header */
    *first_in_bitmap = 0;
    *bitmap_length   = (head->bth.bthNNodes < 256*8) ?
                        head->bth.bthNNodes : (256*8);
    *next_bitmap     = head->bth.bthAddMap;
    *bitmap          = head->bth.btMap;
    *bitmap_node_no  = 0;
  }

  while(*first_in_bitmap + *bitmap_length <= node)   {

			      /* Read the next bitmap node */
    if(*next_bitmap == 0)
      return(BT_BADNODE);

    CHKERR(BT_ReadNode(mnt, head, *next_bitmap, buffer));

    *bitmap_node_no  = *next_bitmap;
    *first_in_bitmap += *bitmap_length;
    *next_bitmap     = GetLongWord(& buffer[ND_FLINK]);
    *bitmap          = & buffer[14];
    *bitmap_length   = (head->bth.bthNNodes < (*first_in_bitmap + 494*8)) ?
                       (head->bth.bthNNodes - *first_in_bitmap) : (494*8);
  }
  return 0;
}









