//
// $Header: D:/ext2-os2/RCS/fs_rw.c,v 8.0 1996/05/31 00:22:42 Willm Exp Willm $
//

// Linux ext2 file system driver for OS/2 2.x and WARP - Allows OS/2 to     
// access your Linux ext2fs partitions as normal drive letters.
// Copyright (C) 1995, 1996 Matthieu WILLM 
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#define INCL_DOSERRORS
#define INCL_NOPMAPI
#include <os2.h>                // From the "Developer Connection Device Driver Kit" version 2.0

#include <fsd.h>
#include <fsh.h>

#include <os2/types.h>
#include <os2/os2misc.h>
#include <os2/os2proto.h>
#include <os2/volume.h>
#include <os2/errors.h>
#include <os2/log.h>

#include <linux/fs.h>
#include <linux/fs_proto.h>
#include <linux/stat.h>

#include <os2/cdfsd.h>

#ifndef MINIFSD
extern char trace_FS_READ;
extern char trace_FS_WRITE;
#endif

/**********************************************************************************************/
/*** FS_READ() : This is the IFS entry point for the DosRead() system call                  ***/
/**********************************************************************************************/
_FS_RET _FS_ENTRY FS_READ(
                          struct sffsi   _FS_PTR psffsi,
                          struct sffsd   _FS_PTR psffsd,
                          char           _FS_PTR pData,
                          unsigned short _FS_PTR pLen,
                          unsigned short         IOflag
                         )
{
    int     rc, rc2, err;
    struct file        *filp;
    struct super_block *sb;
    UINT32 BytesRead;
    char          lock[12];
    unsigned long lock_lin;

#ifndef MINIFSD
    if (trace_FS_READ) {
        kernel_printf("FS_READ( len = %d ) pre-invocation",  *pLen);
    }
#endif

    /*
     * Locks the user buffer to check write access and to prevent other threads from freing it
     * behind us when we are sleeping. (this is a long term verify lock)
     */
    if ((rc = LockUserBuffer(pData, *pLen, lock, LOCK_WRITE, &lock_lin)) == NO_ERROR) {
        /*
         * Gets the superblock from psffsi
         */
        if ((sb = getvolume(psffsi->sfi_hVPB))) {
            /*
             * Gets the file structure from psffsd
             */
            if ((filp = ((_sffsd _FS_PTR)psffsd)->p_file)) {
                /*
                 * Tests if it is a regular file
                 */
                if (S_ISREG(filp->f_inode->i_mode)) {
                    /*
                     * If the file position in psffsi is not consistent with the
                     * one in the file structure in psffsd, something went wrong => panic
                     */
                    if (psffsi->sfi_position == (unsigned long)filp->f_pos) {
                        /*
                         * Now we do the actual read operation
                         */
                        err = VFS_read(filp, pData, (UINT32)*pLen, &BytesRead);
                        *pLen                 = (UINT16)BytesRead;
                        psffsi->sfi_tstamp   |= ST_PREAD;
                        psffsi->sfi_position  = filp->f_pos;
                        rc = err;

                    } else {
                        kernel_printf("FS_READ( %lu ) very weird : psffsi->sfi_position != filp->f_pos : %ld %ld", filp->f_inode->i_ino, psffsi->sfi_position, filp->f_pos);
                        *pLen = 0;
                        rc =  ERROR_READ_FAULT;        // Maybe we should FSH_INTERR ?
                    } /* sfi_position != f_pos */

                } else {
                    kernel_printf("Can't FS_READ( %lu ) - Not a regular file", filp->f_inode->i_ino);
                    *pLen = 0;
                    rc = ERROR_ACCESS_DENIED;
                } /* !S_ISREG */

            } else {
                kernel_printf("FS_READ() - filp is NULL");
                rc = ERROR_INVALID_PARAMETER;
            } /* filp = NULL */

        } else {
            kernel_printf("FS_READ - Couldn't get the superblock");
            rc = ERROR_INVALID_PARAMETER;
        } /* sb = NULL */

        if ((rc2 = VMUnlock(lock_lin)) == NO_ERROR) {
           /* Nothing else to do */
        } else {
            kernel_printf("FS_READ : VMUnlock() returned %d", rc2);
            rc = rc2;
        } /* VMUnlock failed */
    } else {
        kernel_printf("FS_READ : LockUserBuffer() returned %d", rc);
    }


    /*
     * This is the unique return point of FS_READ
     */
#ifndef MINIFSD
    if (trace_FS_READ) {
        kernel_printf("FS_READ( len = %u ) post-inocatio (rc = %d)", *pLen, rc);
    }
#endif
    return rc;

}


#ifndef MINIFSD

/**********************************************************************************************/
/*** FS_WRITE() : This is the IFS entry point for the DosWrite() system call                ***/
/**********************************************************************************************/
_FS_RET _FS_ENTRY FS_WRITE(
                           struct sffsi   _FS_PTR psffsi,
                           struct sffsd   _FS_PTR psffsd,
                           char           _FS_PTR pData,
                           unsigned short _FS_PTR pLen,
                           unsigned short         IOflag
                          )
{
    int                 rc, rc2, err;
    struct file        *filp;
    struct super_block *sb;
    long   BytesWritten;
    char          lock[12];        // Memory lock handle
    unsigned long lock_lin;        // FLAT address of memory lock handle

    if (trace_FS_WRITE) {
        kernel_printf("FS_WRITE( len = %u ) pre-invocation", *pLen);
    }

    /*
     * Locks the user buffer to check read access and to prevent other threads from freing it
     * behind us when we are sleeping. (this is a long term verify lock)
     */
    if (Read_Write) {
        if ((rc = LockUserBuffer(pData, *pLen, lock, LOCK_READ, &lock_lin)) == NO_ERROR) {
            /*
             * Gets the superblock from psffsi
             */
            if ((sb = getvolume(psffsi->sfi_hVPB))) {
                /*
                 * Gets the file structure from psffsd
                 */
                if ((filp = ((_sffsd _FS_PTR)psffsd)->p_file)) {
                    /*
                     * Tests if it is a regular file
                     */
                    if (S_ISREG(filp->f_inode->i_mode)) {

                        /*
                         * If the file position in psffsi is not consistent with the
                         * one in the file structure in psffsd, something went wrong => panic
                         */
                        if (psffsi->sfi_position == (unsigned long)filp->f_pos) {
                            /*
                             * Now we do the actual write operation
                             */
                            err   = VFS_write(filp, pData, (loff_t)*pLen, &BytesWritten);
                            *pLen = (unsigned short)BytesWritten;
                            psffsi->sfi_tstamp  |= ST_PWRITE;
                            psffsi->sfi_size     = filp->f_inode->i_size;
                            psffsi->sfi_position = filp->f_pos;
                            rc = err;

                        } else {
                            kernel_printf("FS_WRITE( %lu ) very weird : psffsi->sfi_position != filp->f_pos : %ld %ld", filp->f_inode->i_ino, psffsi->sfi_position, filp->f_pos);
                            *pLen = 0;
                            rc =  ERROR_WRITE_FAULT;        // Maybe we should FSH_INTERR ?
                        } /* sfi_position != f_pos */

                    } else {
                        kernel_printf("Can't FS_WRITE( %lu ) - Not a regular file", filp->f_inode->i_ino);
                        *pLen = 0;
                        rc = ERROR_ACCESS_DENIED;
                    } /* !S_ISREG */

                } else {
                    kernel_printf("FS_WRITE() - filp is NULL");
                    rc = ERROR_INVALID_PARAMETER;
                } /* filp = NULL */

            } else {
                kernel_printf("FS_WRITE - Couldn't get the superblock");
                rc = ERROR_INVALID_PARAMETER;
            } /* sb = NULL */

            if ((rc2 = VMUnlock(lock_lin)) == NO_ERROR) {
               /* Nothing else to do */
            } else {
                kernel_printf("FS_WRITE : VMUnlock() returned %d", rc2);
                rc = rc2;
            } /* VMUnlock failed */

        } else {
            kernel_printf("FS_WRITE : LockUserBuffer() returned %d", rc);
        } /* LockUserBuffer()  failed */
    } else {
        /*
         * ext2-os2.ifs in read only mode (-rw not set on the IFS command line)
         */
        kernel_printf("ERROR ! FS_WRITE() called and write access not enabled");
        rc = ERROR_WRITE_PROTECT;
    } /* Read_Write  = 0 */


    /*
     * This is the unique return point of FS_WRITE
     */
    if (trace_FS_WRITE) {
        kernel_printf("FS_WRITE( len = %u ) post-invocation (rc = %d)", *pLen, rc);
    }
    return rc;

}

#endif
