//
// $Header: D:/ext2-os2/RCS/fs_fsctl.c,v 8.0 1996/05/31 00:22:33 Willm Exp Willm $
//

// Linux ext2 file system driver for OS/2 2.x and WARP - Allows OS/2 to     
// access your Linux ext2fs partitions as normal drive letters.
// Copyright (C) 1995, 1996 Matthieu WILLM 
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#define INCL_DOSERRORS
#define INCL_NOPMAPI
#include <os2.h>                // From the "Developer Connection Device Driver Kit" version 2.0
#include <fsd.h>                // From the "IFS toolkit"
#include <fsh.h>                // From the "IFS toolkit"

#include <os2/types.h>
#include <os2/os2misc.h>
#include <os2/os2proto.h>
#include <linux/fs.h>
#include <linux/fs_proto.h>
#include <os2/vfsapi.h>
#include <os2/ifsdbg.h>
#include <os2/log.h>
#include <os2/volume.h>
#include <os2/files.h>
#include <os2/cdfsd.h>



/*
 * Error messages to return for func. FSCTL_FUNC_NEW_INFO
 */
static struct fsctl_msg ext2_os2_magic   = {sizeof(magic_msg), magic_msg};
static struct fsctl_msg ext2_os2_default = {sizeof(default_msg), default_msg};

#define ERROR_DEVICE_NOT_OPEN    0xEE01                // Log facility not open

extern int sys_bdflush(int func, long data);

static unsigned long shrink_cache_wait;

_FS_RET _FS_ENTRY FS_FSCTL(
                           union argdat   _FS_PTR pArgdat,
                           unsigned short         iArgType,
                           unsigned short         func,
                           char           _FS_PTR pParm,
                           unsigned short         lenParm,
                           unsigned short _FS_PTR plenParmOut,
                           char           _FS_PTR pData,
                           unsigned short         lenData,
                           unsigned short _FS_PTR plenDataOut
                          )
{
    int rc, error;
    int i;
    struct super_block *sb;
    struct vpfsi *pvpfsi;
    struct vpfsd *pvpfsd;
    char          lockdata[12];
    unsigned long lockdata_lin;

    if ((rc = FSH_PROBEBUF(PB_OPWRITE, (char *)plenParmOut, sizeof(int))) != NO_ERROR) {
        kernel_printf("FS_FSCTL - can't access pLenParmOut");
        return rc;
    }
    if ((rc = FSH_PROBEBUF(PB_OPWRITE, (char *)plenDataOut, sizeof(int))) != NO_ERROR) {
        kernel_printf("FS_FSCTL - can't access pLenDataOut");
        return rc;
    }

    switch(func) {
        //
        // First standard DosFsCtl function : get extended error info.
        // We use this from the vfsapi library to check that a path name or
        // file handle refers to an ext2-os2 volume : we simply return a magic
        // string for error number 0 (NO_ERROR). For other error numbers, we return
        // a default message.
        //
        case FSCTL_FUNC_NEW_INFO :
            if ((rc = FSH_PROBEBUF(PB_OPREAD, pParm, sizeof(int))) != NO_ERROR) {
                return rc;
            }
            error = *((int *)pParm);
            if ((rc = FSH_PROBEBUF(PB_OPWRITE, pData, sizeof(struct fsctl_msg))) != NO_ERROR) {
                return rc;
            }
            if (error == NO_ERROR) {
                memcpy(pData, &ext2_os2_magic, sizeof(ext2_os2_magic));
            } else {
                memcpy(pData, &ext2_os2_default, sizeof(ext2_os2_default));
            }
            if ((rc = FSH_PROBEBUF(PB_OPWRITE, (char *)plenDataOut, sizeof(int))) != NO_ERROR) {
                return rc;
            }
            *plenDataOut = sizeof(struct fsctl_msg);
            if ((rc = FSH_PROBEBUF(PB_OPWRITE, (char *)plenParmOut, sizeof(int))) != NO_ERROR) {
                return rc;
            }
            *plenParmOut = 0;
            return NO_ERROR;




        case IFSDBG_OPEN :
            if (BufOpen == 0) {
                BufOpen = 1;
                if (BufPtr == 0) {
                    if ((rc = FSH_SEMSET(&BufSem)) == ERROR_INTERRUPT) {
                        BufOpen = 0;
                        return rc;
                    } /* end if */
                } /* end if */
                return NO_ERROR;
            } else {
                return ERROR_DEVICE_IN_USE;
            } /* end if */

        case IFSDBG_CLOSE:
            if (BufOpen == 1) {
                BufOpen = 0;
                return NO_ERROR;
            } else {
                return ERROR_DEVICE_NOT_OPEN;
            } /* end if */


        case IFSDBG_READ:
            if (BufOpen == 1) {
                /*************************************************************/
                /*** Waits on the log semaphore                            ***/
                /*************************************************************/
                if ((rc = FSH_SEMWAIT(&BufSem, TO_INFINITE)) == ERROR_INTERRUPT) {
                    BufOpen = 0;
                    return rc;
                } /* end if */
                /*************************************************************/
                /*** verify access to the user buffer                      ***/
                /*************************************************************/
                if ((rc = FSH_PROBEBUF(1, pData, lenData)) != NO_ERROR) {
                    return rc;
                }
                /*************************************************************/
                /*** vefify access to the user lenght to be returned       ***/
                /*************************************************************/
                if ((rc = FSH_PROBEBUF(1, (char _FS_PTR)plenDataOut, sizeof(unsigned int))) != NO_ERROR) {
                    return rc;
                }
                /*************************************************************/
                /*** If no data is present, simply set the semaphore       ***/
                /*************************************************************/
                if (BufPtr == 0) {
                    *plenDataOut = 0;
                    if ((rc = FSH_SEMSET(&BufSem)) == ERROR_INTERRUPT) {
                        BufOpen = 0;
                        return rc;
                    } /* end if */
                    return NO_ERROR;
                } /* end if */

                /*************************************************************/
                /*** If the log data is smaller than the requested amount  ***/
                /*** we copy them all                                      ***/
                /*************************************************************/
                if (BufPtr < lenData) {
                    memcpy(pData, BufMsg, BufPtr + 1);
                    *plenDataOut = BufPtr + 1;
                    BufPtr = 0;
                    if ((rc = FSH_SEMSET(&BufSem)) == ERROR_INTERRUPT) {
                        BufOpen = 0;
                        return rc;
                    } /* end if */
                    return NO_ERROR;
                } /* end if */
                /*************************************************************/
                /*** We set the log semaphore                              ***/
                /*** ext2-os2.exe will wait on it until some more data is  ***/
                /*** present                                               ***/
                /*************************************************************/
                if ((rc = FSH_SEMSET(&BufSem)) == ERROR_INTERRUPT) {
                    BufOpen = 0;
                    return rc;
                } /* end if */

                return NO_ERROR;
            } else {
                return ERROR_DEVICE_NOT_OPEN;
            } /* end if */


        //
        // VFSAPI library entry point : bdflush() system call (buffer management - lazy write)
        // Input :
        //      pData (ignored)
        //      pParm (ignored)
        // Output :
        //      pData (ignored)
        //      pParm (ignored)
        //
        case EXT2_OS2_BDFLUSH :
            /*
             * We should never return from this call.
             */
            sys_bdflush(0, 0);
            break;

        //
        // Shrink cache captive thread - we do this because we can't hack OS/2 virtual memory
        // manager to do it itself when needed.
        //
        case EXT2_OS2_SHRINK_CACHE :
            while(1) {
                if (buffermem)
                    shrink_buffers(3);
                ProcBlock((unsigned long)&shrink_cache_wait, 10000, 1);
            }

        //
        // This one is used to retrieve some data from ext2-os2.ifs
        //
        case EXT2_OS2_GETDATA :
            {
                struct ext2_os2_data *d;

                if ((rc = LockUserBuffer(pData, lenData, lockdata, LOCK_WRITE, &lockdata_lin)) == NO_ERROR) {
                    d = (struct ext2_os2_data *)pData;

                    d->b.buffer_mem      = buffermem;
                    d->b.cache_size      = cache_size;
                    d->b.nr_buffer_heads = nr_buffer_heads;

                    memcpy(&(d->b.nr_free), nr_free, sizeof(nr_free));
                    memcpy(&(d->b.nr_buffers_type), nr_buffers_type, sizeof(nr_buffers_type));

                    d->i.nr_inodes      = nr_inodes;
                    d->i.nr_free_inodes = nr_free_inodes;
                    d->i.nr_iget        = nr_iget;
                    d->i.nr_iput        = nr_iput;

                    d->f.nhfiles     = nr_files;
                    d->f.nfreehfiles = nr_free_files;
                    d->f.nusedhfiles = nr_used_files;

                    *plenDataOut = sizeof(struct ext2_os2_data);

                    rc = VMUnlock(lockdata_lin);
                }
                return rc;
            }
        //
        // VFSAPI Library entry point : this is the vfs_sync() routine (standart sync() system call)
        // Input :
        //      pData (ignored)
        //      pParm (ignored)
        // Output :
        //      pData (ignored)
        //      pParm (ignored)
        //
        case EXT2_OS2_SYNC :
            if (Read_Write) {
                for (i = 0 ; i < NB_MAX_VOLS ; i++) {
                    if (volglobdat.listvol[i].status == VOL_STATUS_MOUNTED) {
                        FSH_GETVOLPARM(volglobdat.listvol[i].hVPB, &pvpfsi, &pvpfsd);
                        sb = ((hvolume *)pvpfsd)->p_volume;

                        if (sb) {
                            kernel_printf("Flushing volume 0x%0X", volglobdat.listvol[i].hVPB);
                            sync_buffers(sb->s_dev, 0);
                            sync_inodes(sb->s_dev);
                            sync_buffers(sb->s_dev, 0);
                        }
                    }
                }
                return NO_ERROR;
            } else {
                return ERROR_NOT_SUPPORTED;
            }

        //
        // VFSAPI Library entry point : this is the vfs_stat() routine (standart stat() system call)
        // Input :
        //         pData must point to a buffer large enough to hold struct vfs_stat
        //      pParm (ignored)
        // Output :
        //         pData contains the struct vfs_stat if success, or garbage if failed
        //      pParm (ignored)
        //
        case EXT2_OS2_STAT :
            *plenParmOut = 0;
            *plenDataOut = sizeof(struct new_stat);
            //
            // Verifies that this is a path name directed call
            //
            if (iArgType != FSCTL_ARG_CURDIR) {
                kernel_printf("FS_FSCTL(EXT2_OS2_FSTAT) - Not a path name directed call");
                return ERROR_INVALID_PARAMETER;
            }
            kernel_printf("FS_FSCTL(EXT2_OS2_STAT) - Path is %s", pArgdat->cd.pPath);
            //
            // Verifies that struct new_stat fits in pData
            //
            if (lenData < sizeof(struct new_stat)) {
                kernel_printf("FS_FSCTL(EXT2_OS2_STAT) - buffer too small to hold stat struct");
                return ERROR_BUFFER_OVERFLOW;
            }
            //
            // Verifies that we can write struct new_stat in pData
            //
            if ((rc = FSH_PROBEBUF(PB_OPWRITE, pData, sizeof(struct new_stat))) != NO_ERROR) {
                kernel_printf("FS_FSCTL(EXT2_OS2_STAT) - can't access pData");
                return rc;
            }
            {
                 struct file       *f;
                 struct new_stat   *s;
                 unsigned long      DOSmode;
                //
                // Gets the superblock
                //
                sb = getvolume(pArgdat->cd.pcdfsi->cdi_hVPB);
                DOSmode = (is_case_retensive() ? OPENMODE_DOSBOX : 0);

                if ((f = _open_by_name(sb, pArgdat->cd.pPath, OPENMODE_READONLY | DOSmode)) == NULL) {
                    kernel_printf("FS_FSCTL() - path %s not found", pArgdat->cd.pPath);
                    return ERROR_PATH_NOT_FOUND;
                } /* end if */
                s = (struct new_stat *)pData;
                memset(s, 0, sizeof(struct new_stat));
                s->st_dev = f->f_inode->i_dev;
                s->st_ino = f->f_inode->i_ino;
                s->st_mode = f->f_inode->i_mode;
                s->st_nlink = f->f_inode->i_nlink;
                s->st_uid = f->f_inode->i_uid;
                s->st_gid = f->f_inode->i_gid;
                s->st_rdev = f->f_inode->i_rdev;
                s->st_size = f->f_inode->i_size;
//        if (inode->i_pipe)
//                tmp.st_size = PIPE_SIZE(*inode);
                s->st_atime = f->f_inode->i_atime;
                s->st_mtime = f->f_inode->i_mtime;
                s->st_ctime = f->f_inode->i_ctime;
                s->st_blocks = f->f_inode->i_blocks;
                s->st_blksize = f->f_inode->i_blksize;
                vfs_close(f);
            }
            return NO_ERROR;

        //
        // VFSAPI Library entry point : this is the vfs_fstat() routine (standart fstat() system call)
        // Input :
        //         pData must point to a buffer large enough to hold struct vfs_stat
        //      pParm (ignored)
        // Output :
        //         pData contains the struct vfs_stat if success, or garbage if failed
        //      pParm (ignored)
        //
        case EXT2_OS2_FSTAT :
            *plenParmOut = 0;
            *plenDataOut = sizeof(struct new_stat);
            //
            // Verifies that this is a file handle directed call
            //
            if (iArgType != FSCTL_ARG_FILEINSTANCE) {
                kernel_printf("FS_FSCTL(EXT2_OS2_FSTAT) - Not a file handle directed call");
                return ERROR_INVALID_PARAMETER;
            }
            //
            // Verifies that struct new_stat fits in pData
            //
            if (lenData < sizeof(struct new_stat)) {
                kernel_printf("FS_FSCTL(EXT2_OS2_FSTAT) - buffer too small to hold stat struct");
                return ERROR_BUFFER_OVERFLOW;
            }
            //
            // Verifies that we can write struct new_stat in pData
            //
            if ((rc = FSH_PROBEBUF(PB_OPWRITE, pData, sizeof(struct new_stat))) != NO_ERROR) {
                kernel_printf("FS_FSCTL(EXT2_OS2_FSTAT) - can't access pData");
                return rc;
            }
            {
                struct new_stat *s;
                struct inode    *inode;

                inode = ((_sffsd *)pArgdat->sf.psffsd)->p_file->f_inode;
                s = (struct new_stat *)pData;

                memset(s, 0, sizeof(struct new_stat));
                s->st_dev = inode->i_dev;
                s->st_ino = inode->i_ino;
                s->st_mode = inode->i_mode;
                s->st_nlink = inode->i_nlink;
                s->st_uid = inode->i_uid;
                s->st_gid = inode->i_gid;
                s->st_rdev = inode->i_rdev;
                s->st_size = inode->i_size;
//        if (inode->i_pipe)
//                tmp.st_size = PIPE_SIZE(*inode);
                s->st_atime = inode->i_atime;
                s->st_mtime = inode->i_mtime;
                s->st_ctime = inode->i_ctime;
                s->st_blocks = inode->i_blocks;
                s->st_blksize = inode->i_blksize;
            }
            return NO_ERROR;

        default:
#ifdef FS_TRACE
           kernel_printf("FS_FSCTL() - Invalid function %d", func);
#endif
           return ERROR_NOT_SUPPORTED;

    } /* end switch */
}
