/* fseek.c (emx+gcc) -- Copyright (c) 1990-1995 by Eberhard Mattes */

#include <sys/emx.h>
#include <stdio.h>
#include <io.h>
#include <errno.h>

int fseek (FILE *stream, long offset, int origin)
{
  long cur_pos;
  int fflush_result;

  if (!(stream->flags & _IOOPEN) || origin < 0 || origin > 2
      || (stream->flags & _IOSPECIAL))
    {
      errno = EINVAL;
      return (EOF);
    }

  cur_pos = -1;                 /* Not yet computed */

  /* fflush() is not required if all of the following conditions are met:
     - the stream is a read-only file
     - the buffer has not been modified by ungetc()
     - the stream is buffered
     - there are no pushed-back characters
     - the new position is within buffer */

  if ((stream->flags & (_IORW|_IOREAD|_IOWRT|_IOUNGETC)) == _IOREAD
      && bbuf (stream) && stream->_ungetc_count == 0)
    {
      long file_pos, end_pos, buf_pos;
      int text_mode, n;

      file_pos = tell (stream->handle);
      if (file_pos == -1)
        return (EOF);
      cur_pos = ftell (stream);
      if (origin == SEEK_CUR)
        {
          offset += cur_pos;
          origin = SEEK_SET;
        }
      else if (origin == SEEK_END)
        {
          end_pos = lseek (stream->handle, 0L, SEEK_END);
          lseek (stream->handle, file_pos, SEEK_SET);
          if (end_pos == -1)
            return (EOF);
          offset += end_pos;
          origin = SEEK_SET;
        }
      text_mode =  (stream->handle >= 0 && stream->handle < _nfiles
                    && (_files[stream->handle] & F_CRLF));
      n = stream->ptr - stream->buffer;
      if (text_mode)
        {
          /* Use lower bound on the buffer position for a quick check
             if the new position can at all be inside the buffer.  The
             lower bound is the exact buffer position if all
             characters in the buffer are newline characters. */

          buf_pos = cur_pos - 2 * n;
        }
      else
        buf_pos = cur_pos - n;

      if (offset >= buf_pos && offset < file_pos)
        {
          if (text_mode)
            {
              const char *p;

              /* Compute exact buffer position.  This is only required
                 if the new position is smaller than the current
                 position. */

              if (offset >= cur_pos)
                buf_pos = cur_pos; /* Wrong value doesn't matter */
              else
                {
                  p = stream->ptr;
                  buf_pos = cur_pos - n;
                  while (n > 0)
                    {
                      if (*--p == '\n')
                        --buf_pos;
                      --n;
                    }
                }

              if (offset >= buf_pos)
                {
                  long tmp_pos;

                  /* The new position is within the buffer.  Adjust
                     the new position for newline characters.  If
                     offset >= cur_pos, we can start at the current
                     position. */

                  if (offset >= cur_pos)
                    {
                      /* Optimization. */

                      p = stream->ptr;
                      tmp_pos = cur_pos;
                    }
                  else
                    {
                      p = stream->buffer;
                      tmp_pos = buf_pos;
                    }
                  while (tmp_pos < offset)
                    {
                      if (*p == '\n')
                        {
                          ++tmp_pos;
                          if (tmp_pos >= offset)
                            break;
                        }
                      ++p; ++tmp_pos;
                    }
                  stream->rcount -= p - stream->ptr;
                  stream->ptr = (char *)p;
                  stream->flags &= ~_IOEOF;
                  return (0);
                }
            }
          else if (offset >= buf_pos)
            {
              stream->ptr = stream->buffer + (offset - buf_pos);
              stream->rcount = file_pos - offset;
              stream->flags &= ~_IOEOF;
              return (0);
            }
        }
    }

#if 0
  /* Get file position indicator before calling fflush().  ANSI
     X3.159-1989 is not precise on this. */

  if (origin == SEEK_CUR && cur_pos == -1)
    cur_pos = ftell (stream);
#endif

  fflush_result = fflush (stream);
  stream->flags &= ~_IOEOF;
  if (stream->flags & _IORW)
    stream->flags &= ~(_IOREAD|_IOWRT);

  if (origin == SEEK_CUR)
    {
      if (cur_pos == -1)
        cur_pos = ftell (stream);
      if (cur_pos == -1) return (EOF);
      offset += cur_pos;
      origin = SEEK_SET;
    }

  if (lseek (stream->handle, offset, origin) == -1)
    return (EOF);
  else
    return (fflush_result);
}
