//	LAWFILE.C
//	Patrick Tennberg, 1994,95,96

#define	STRICT
#include	"includes.h"
#pragma	hdrstop
#include	"resource.h"
#include	"calvin.h"
#include	"lawfile.h"
#include	"tools.h"

//	End of line character
#define	EOL	   0x0D

//	Exceptions used by the module
#define	SE_LAWFILEACCESS_FAILED		MAKESOFTWAREEXCEPTION(3,0,1)
#define	SE_INDEXFILECREATE_FAILED	MAKESOFTWAREEXCEPTION(3,0,2)
#define	SE_HEAPALLOC_FAILED				MAKESOFTWAREEXCEPTION(3,0,3)
#define SE_LAWTOLONG              MAKESOFTWAREEXCEPTION(3,0,4)

//	Constants
static LPCSTR szLawFileName 	= "CALVIN.LAW";
static LPCSTR szIndexFileName	=	"CALVIN.NDX";

//	We we build an index we need to save a time stamp so we
//	know if we must rebuild the index file. The time stamp we
//	save in the index file is the time and date of the CALVIN.LAW
//	file.
typedef struct
  {
	UINT			nNoIndex;
	FILETIME	ftLastWrite;
  }
TIndexHeader, *PTIndexHeader;

//	Local function prototypes
LONG FileExceptionFilter(LPEXCEPTION_POINTERS lpEP,PTFileData fd);
void ReadIndex(PTFileData fd);
void BuildIndex(HANDLE *hIndexFile,PTIndexHeader ih,PTFileData fd);

//	Load the law file and create a memory map of it
void LoadLawFile(PTFileData fd)
  {
	char	szFullPath[MAXPATH];

	//	Create a full path name to the law file
	ModulePath(szFullPath);
	lstrcat(szFullPath,szLawFileName);
	// Init the FileData structure to zero
	memset(fd,0,sizeof(TFileData));
	__try
	  {
		//	Open the law file for reading only. The file can't be shared and must exist.
		fd->hFile = CreateFile(szFullPath,GENERIC_READ | GENERIC_WRITE,0,NULL,OPEN_EXISTING,FILE_ATTRIBUTE_NORMAL,NULL);
		if (INVALID_HANDLE_VALUE == fd->hFile)
			RaiseException(SE_LAWFILEACCESS_FAILED,0,1,(LPDWORD)fd);
		//	This call is needed beacuse of some editors behviour, more
		//	info later in the function.
		GetFileTime(fd->hFile,NULL,NULL,&(fd->ftLastWrite));
		//	Get size of the file, we need this information for the
		//	file mapping.
		fd->dwFileSize = GetFileSize(fd->hFile,NULL);
		if (0 == fd->dwFileSize)
			RaiseException(SE_LAWFILEACCESS_FAILED,0,1,(LPDWORD)fd);
		//	Create the file mapping. Note, we add an EOL character in the end of the
		//	memory mapped file, just to simplify matters for ExtractLaw.
		fd->hFileMap = CreateFileMapping(fd->hFile,NULL,PAGE_READWRITE,0,fd->dwFileSize + 1,NULL);
		if (NULL == fd->hFileMap)
			RaiseException(SE_LAWFILEACCESS_FAILED,0,1,(LPDWORD)fd);
		//	Get the address where the first byte of the file is
		//	mapped into memory.
		fd->lpsFile = MapViewOfFile(fd->hFileMap,FILE_MAP_WRITE,0,0,0);
		if (NULL == fd->lpsFile)
			RaiseException(SE_LAWFILEACCESS_FAILED,0,1,(LPDWORD)fd);
		//	Some text editors never adds an end of line character
		//	therefore we add one temporary. This leads to a lot of
		//	trouble when we want to maintain the last write stamp on
		//	the law file. Therefore we read the file time first and
		//	resets the time stamp after we removed the EOL and closed
		//	the file.
		fd->lpsFile[fd->dwFileSize] = EOL;
	  }
	__except (FileExceptionFilter(GetExceptionInformation(),fd))
	  {
		//	Because we fix everything inside the exception filter
		//	we just idle inside the exception block.
	  }
  }

//	Unmap the law file and close all handles
void UnloadLawFile(PTFileData fd)
  {
	//	Kill the memory mapped file
	UnmapViewOfFile(fd->lpsFile);
	CloseHandle(fd->hFileMap);
	//	Remove the trailing EOL by positioning the file pointer
	//	at the end of the file, not including the EOL, and setting
	//	the end-of-file.
	SetFilePointer(fd->hFile,fd->dwFileSize,NULL,FILE_BEGIN);
	SetEndOfFile(fd->hFile);
	//	Restoring the file stamp
	SetFileTime(fd->hFile,NULL,NULL,&(fd->ftLastWrite));
	//	Close the file handle
	CloseHandle(fd->hFile);
  }

//	Extract the law from the law file
LPSTR ExtractLaw(DWORD dwIndex,LPSTR szLaw,PTFileData fd)
  {
	LPSTR	lpsFilePtr	= (LPSTR)(fd->lpsFile + dwIndex);
	LPSTR	szOrgPtr 		= szLaw;

	while ((*szLaw++ = *lpsFilePtr++) != EOL);
	*(szLaw - 1) = '\0';
	return szOrgPtr;
  }

//	Get a law, if we runned out of laws or
//	if we couldnt find any law file, return FALSE
//	else return TRUE and a law in szLaw
BOOL GetRandomLaw(PTFileData fd,LPSTR szLaw)
  {
	if (FALSE == fd->failed)
	  {
		//	Initialize the random seed
		srand((unsigned int)time(NULL));
		//	The index file is not opened or created.
		if (NULL == fd->lpdwIndex)
			ReadIndex(fd);
		//	If every thing went well lets get the law
		if (FALSE == fd->failed)
		  {
			if (0 == fd->nNoLaws)
			  {
				HMODULE hModule = GetModuleHandle(szAppName);

				LoadString(hModule,IDS_ALLAWSDISPLAYED,szLaw,MAX_LAWLEN);
				return FALSE;
			  }
			else
			  {
			  //  According to all text about using random number this is
				//  a very bad method because a linear congruential generator
				//  often have their low-order bits much less random and its bad
				//  in many other ways to, but for crist sake we only want to 
				//  display some text.
			  int nIndex = (rand() % fd->nNoLaws);

				lstrcpy(szLaw,ExtractLaw(fd->lpdwIndex[nIndex],szLaw,fd));
				//	Swap index between the last index and the choosen
				//	index. We only want's to display a law once.
				fd->lpdwIndex[nIndex] = fd->lpdwIndex[--fd->nNoLaws];
			  }
		  }
	  }
	//	Ops, we can't access the law or the index file, lets
	//	inform the user.
	if (TRUE == fd->failed)
	  {
		HMODULE hModule = GetModuleHandle(szAppName);

    switch (fd->nErrorCode)
      {
      case ERR_GENERIC:
		    LoadString(hModule,IDS_LAWFILENOTFOUND,szLaw,MAX_LAWLEN);
        break;
      case ERR_LAWTOLONG:
        {
        char szBuffer[MAX_LAWLEN + 1];

        LoadString(hModule,IDS_LAWTOLONG,szBuffer,MAX_LAWLEN);
        wsprintf(szLaw,szBuffer,MAX_LAWLEN,fd->nNoLaws,fd->nStringLength);
        }
        break;
      }
		return FALSE;
	  }
	return TRUE;
  }

//	Creates an index file for fast access to the law file.
//	This index file is saved for later use. If the law file
//	has changed the index will be rebuilt.
void ReadIndex(PTFileData fd)
  {
	TIndexHeader	ih;
	BOOL					fStatus;
	DWORD					dwDummy;
	HANDLE				hIndexFile = NULL;
	char					szFullPath[MAXPATH];

	//	Create a full path name to the law file
	ModulePath(szFullPath);
	lstrcat(szFullPath,szIndexFileName);
	//	Reset the index header
	memset(&ih,0,sizeof(TIndexHeader));
	__try // Finally block
	  {
		//	Open the index file.
		hIndexFile = CreateFile(szFullPath,GENERIC_READ | GENERIC_WRITE,0,NULL,OPEN_ALWAYS,FILE_ATTRIBUTE_NORMAL,NULL);
   	//	Lets read the header
		ReadFile(hIndexFile,&ih,sizeof(TIndexHeader),&dwDummy,NULL);
		//	We now compare the file date saved in the index header with
		//	the time stamp on the law file. If the differs than we must
		//	rebuild the index	file.
		if (CompareFileTime(&(fd->ftLastWrite),&(ih.ftLastWrite)) != 0)
		  {
			BuildIndex(&hIndexFile,&ih,fd);
			if (TRUE == fd->failed)
				return;
		  }
    // The number of laws are stored in the index header, so
    // we initialize fd->nNoLaws to this value
    fd->nNoLaws = ih.nNoIndex;
		//	Allocate memory for the index array
		__try
		  {
			fd->lpdwIndex = HeapAlloc(GetProcessHeap(),0,sizeof(DWORD) * fd->nNoLaws);
			if (NULL == fd->lpdwIndex)
				RaiseException(SE_HEAPALLOC_FAILED,0,1,(LPDWORD)fd);
		  }
		__except (FileExceptionFilter(GetExceptionInformation(),fd))
		  {
			//	Let's split
			return;
		  }
		//	We have now done all the grunt work lets read the index file
		fStatus = ReadFile(hIndexFile,fd->lpdwIndex,sizeof(DWORD) * fd->nNoLaws,&dwDummy,NULL);
		if (FALSE == fStatus)
			return;
	  }
	__finally
	  {
		if (hIndexFile != INVALID_HANDLE_VALUE)
      {
			CloseHandle(hIndexFile);
      // Delete the index file if we failed
      // and the reason for failing was the law length
      if (TRUE == fd->failed && ERR_LAWTOLONG == fd->nErrorCode)
        DeleteFile(szFullPath);      
      }
		if (TRUE == fd->failed && fd->lpdwIndex != NULL)
			HeapFree(GetProcessHeap(),0,fd->lpdwIndex);
	  }
  }

//	Rebuilt index file
void BuildIndex(HANDLE *hIndexFile,PTIndexHeader ih,PTFileData fd)
  {
	BOOL		fStatus;
	DWORD		dwIndex = 0;
	DWORD		dwDummy;
  LPSTR   lpsPrevFilePtr;
	LPSTR		lpsFilePtr = fd->lpsFile;

	fd->nNoLaws = 0;
	__try
	  {
		//	Empty the index file
		SetFilePointer(*hIndexFile,0,NULL,FILE_BEGIN);
		SetEndOfFile(*hIndexFile);
		//	Write the index header. The header will be written once more
		//	when we have all the information.
		fStatus = WriteFile(*hIndexFile,ih,sizeof(TIndexHeader),&dwDummy,NULL);
		if (FALSE == fStatus)
			RaiseException(SE_INDEXFILECREATE_FAILED,0,1,(LPDWORD)fd);
		//	Create the index file. We have prepared the law file so
		//	we are garanted to have at least one legal entry.
		do
		  {
      // Calculate the number of laws
			fd->nNoLaws++; 
      // Write the index
			fStatus = WriteFile(*hIndexFile,&dwIndex,sizeof(DWORD),&dwDummy,NULL);
			if (FALSE == fStatus)
				RaiseException(SE_INDEXFILECREATE_FAILED,0,1,(LPDWORD)fd);
      lpsPrevFilePtr = lpsFilePtr;
			while (*lpsFilePtr++ != EOL);
      if ((fd->nStringLength = (int)(lpsFilePtr - lpsPrevFilePtr - 1)) >= MAX_LAWLEN)
        RaiseException(SE_LAWTOLONG,0,1,(LPDWORD)fd);   
      lpsFilePtr++;
      dwIndex = (lpsFilePtr - fd->lpsFile);
		  }
		while (dwIndex < fd->dwFileSize);
		//	Almost done but first we need to write the index header, again
    ih->nNoIndex = fd->nNoLaws;
		ih->ftLastWrite = fd->ftLastWrite;
		SetFilePointer(*hIndexFile,0,NULL,FILE_BEGIN);
		fStatus = WriteFile(*hIndexFile,ih,sizeof(TIndexHeader),&dwDummy,NULL);
		if (FALSE == fStatus)
			RaiseException(SE_INDEXFILECREATE_FAILED,0,1,(LPDWORD)fd);
	  }
	__except (FileExceptionFilter(GetExceptionInformation(),fd))
	  {
		//	Because we fix everything inside the exception filter
		//	we just idle inside the exception block.
	  }
  }

//	Expception filter for the file handling functions
LONG FileExceptionFilter(LPEXCEPTION_POINTERS lpEP,PTFileData fd)
  {
	fd->failed = TRUE;
  fd->nErrorCode = ERR_GENERIC;
	//	If the exception not belongs to use, then pass it
	//	further.
	switch (lpEP->ExceptionRecord->ExceptionCode)
	  { 
		case SE_LAWFILEACCESS_FAILED:
			//	Close open handlers
			if (fd->hFile != NULL)
				CloseHandle(fd->hFile);
			fd->hFile = NULL;
			if (fd->hFileMap != NULL)
				CloseHandle(fd->hFileMap);
			fd->hFileMap = NULL;
			break;
    case SE_LAWTOLONG:
      fd->nErrorCode = ERR_LAWTOLONG;
      break;
		case SE_HEAPALLOC_FAILED:
		case SE_INDEXFILECREATE_FAILED:
			break;
		default:
			return EXCEPTION_CONTINUE_SEARCH;
	  }
	return EXCEPTION_EXECUTE_HANDLER;
  }
