package jere.applets.onthisday;

import java.util.*;

/**
 * Represents an OTDML event tag.
 *
 * @author Jere Kpyaho (jere@sci.fi)
 */
public class Event extends Object implements Sortable {
    private Date date;
    private String topic;
    private String description;

    /**
     * Constructs an event with default field values.
     */
    public Event() {
        this(new Date(), "(none)", "(none)");
    }

    /**
     * Constructs an event with given field values.
     *
     * @param date The date of the event
     * @param topic The topic of the event
     * @param description The description of the event
     */
    public Event(Date date, String topic, String description) {
        setDate(date);
        setTopic(topic);
        setDescription(description);
    }

    /**
     * Sets the date of the event.
     *
     * @param date The date of the event
     */
    public void setDate(Date date) {
        this.date = date;
    }

    /**
     * Sets the topic of the event.
     *
     * @param topic The topic of the event
     */
    public void setTopic(String topic) {
        this.topic = topic;
    }

    /**
     * Sets the description of the event.
     *
     * @param description The description of the event
     */
    public void setDescription(String description) {
        this.description = description;
    }

    /**
     * Gets the date of the event.
     *
     * @return The date of the event
     */
    public Date getDate() {
        return date;
    }

    /**
     * Convenience method to extract year.
     *
     * @return The year of the event
     */
    public int getYear() {
        Calendar cal = GregorianCalendar.getInstance();
        cal.setTime(date);
        return cal.get(Calendar.YEAR);
    }

    /**
     * Gets the topic of the event.
     *
     * @return The topic of the event
     */
    public String getTopic() {
        return topic;
    }

    /**
     * Gets the description of the event.
     *
     * @return The description of the event
     */
    public String getDescription() {
        return description;
    }

    /**
     * Returns a string representation of the event.
     *
     * @return The event as a printable string
     */
    public String toString() {
        return topic + ":" + date + ":" + description;
    }

    /**
     * Compares this event to another event. Used when the
     * events are sorted.
     *
     * @param other The other event to compare to
     * @return Zero if the events are equal, &lt; 0 if this event
     *         is earlier than the other, %gt; 0 if this event is
     *         later than the other
     */
    public int compareTo(Object other) {
        int value = 0;
        Event otherEvent = (Event) other;

        if (this.getYear() > otherEvent.getYear()) {
            value = 1;
        }
        else if (this.getYear() == otherEvent.getYear()) {
            value = 0;
        }
        else {
            value = -1;
        }
        return value;
    }
}

