package jere.applets;

import java.applet.*;
import java.util.*;
import java.awt.*;
import java.net.*;

/**
  * Helper class for parsing applet parameters.
  * Contains static methods for parsing parameter values 
  * that represent strings, integers, and colors.
  * @version 1.1
  * @author Jere Kpyaho (jere@sci.fi)
 **/
public class AppletParameter {

  /**
    * Parses a string parameter.
    * @param applet The applet whose parameters are parsed
    * @param name The name of the parameter
    * @param defaultValue The default value of the parameter
    * @return The parameter value if found, or the default value
   **/
  public static String getString(Applet applet, String name, String defaultValue) {
    String value = applet.getParameter(name);
    return (value != null) ? value : defaultValue;
  }

  /**
    * Parses an integer parameter.
    * @param applet The applet whose parameters are parsed
    * @param name The name of the parameter
    * @param defaultValue The default value of the parameter
    * @return The parameter value if found, or the default value
   **/
  public static int getInteger(Applet applet, String name, int defaultValue) {
    String stringValue = applet.getParameter(name);
    int value = defaultValue;
    if (stringValue != null) {
      try {
        value = Integer.parseInt(stringValue);
      }
      catch (NumberFormatException nfe) {
        value = defaultValue;
      }
    }
    return value;
  }

  /**
    * Parses a string containing a hex RGB color (#RRGGBB)
    * or one of the 16 standard named colors of HTML 3.2 and CSS Level 1.
    * @param applet The applet whose parameters are parsed
    * @param name The name of the parameter
    * @param defaultValue The default value of the parameter
    * @return The parameter value if found, or the default value
   **/
  public static Color getColor(Applet applet, String name, Color defaultValue) {
    Color colorValue = null;
    String colorString = applet.getParameter(name);
    if (colorString != null) {
      if (colorString.charAt(0) == '#' && colorString.length() == 7) {    // #RRGGBB, hex RGB color
        try {
          /*
          // The Java 1.0 method:
          int rgb = Integer.parseInt(colorString.substring(1), 16);
          colorValue = new Color(rgb);
          */
          
          // The Java 1.1 method:
          colorValue = Color.decode("0x" + colorString.substring(1));
        }
        catch (NumberFormatException nfe) {
          applet.showStatus("Bad parameter: '" + name + "'. Using default value.");
          colorValue = defaultValue;
        }
      }
      else {
        colorValue = (Color) standardColors.get(colorString.toLowerCase());  // null if color not found
      }
    }
    else {
      colorValue = defaultValue;
    }
    return colorValue;
  }

  // A hash table to store the Color objects that correspond
  // to the standard color names.
  private static Hashtable standardColors = new Hashtable(16);

  // Initialize the hash table with the 16 standard HTML 3.2 colors.
  // Use the java.awt.Color constants where appropriate.
  // Since this is done in a static initializer, the colors are 
  // always there when the getColor() method is called.
  static {
    standardColors.put("black",   Color.black                );
    standardColors.put("silver",  new Color(0xC0, 0xC0, 0xC0));  
    standardColors.put("gray",    new Color(0x80, 0x80, 0x80));
    standardColors.put("white",   Color.white                );
    standardColors.put("maroon",  new Color(0x80, 0x00, 0x00));
    standardColors.put("red",     Color.red                  );
    standardColors.put("purple",  new Color(0x80, 0x00, 0x80));
    standardColors.put("fuchsia", new Color(0xFF, 0x00, 0xFF));
    standardColors.put("green",   Color.green                );
    standardColors.put("lime",    new Color(0x00, 0xFF, 0x00));
    standardColors.put("olive",   new Color(0x80, 0x80, 0x00));
    standardColors.put("yellow",  Color.yellow               );
    standardColors.put("navy",    new Color(0x00, 0x00, 0x80));
    standardColors.put("blue",    Color.blue                 );
    standardColors.put("teal",    new Color(0x00, 0x80, 0x80));
    standardColors.put("aqua",    new Color(0x00, 0xFF, 0xFF));
  }

  /**
    * Parses a string representing a URL.
    * @param applet The applet whose parameters are parsed
    * @param name The name of the parameter
    * @param defaultValue The default value of the parameter
    * @return The parameter value if found, or the default value
    * @since 1.1
   **/
  public static URL getURL(Applet applet, String name, URL defaultValue) {
    URL url = null;
    String urlString = applet.getParameter(name);
    if (urlString != null) {
      try {
        url = new URL(urlString);      
      }
      catch (MalformedURLException mue) {
        applet.showStatus("Bad parameter: '" + name + "'. Using default value.");
        url = defaultValue;
      }
    }
    else {
      url = defaultValue;    
    }
    return url; 
  }
  
  /**
    * Parses a string representing a font.
    * The string is parsed using Font.decode, so we should have at least
    * the font name in the string. The default is 12 point plain, but we can 
    * pass our own default font.
    * @param applet The applet whose parameters are parsed
    * @param name The name of the parameter
    * @param defaultValue The default value of the parameter
    * @return The parameter value if found, or the default value
    * @since 1.1
   **/
  public static Font getFont(Applet applet, String name, Font defaultValue) {
    Font font = null;
    String fontString = applet.getParameter(name);
    if (fontString != null) {
      font = Font.decode(fontString);
    }
    else {
      font = defaultValue;
    }
    return font;
  }
}

