unit Main;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  RxGrdCpt, ComCtrls, ToolWin, ExtCtrls, LCD_Lab, Menus, MPlayer, Placemnt,
  AppEvent, RXShell, Engine, MMSystem;

resourcestring
  SUnknownArtist = '(tuntematon esittj)';
  SUnknownTitle = '(tuntematon levy)';
  SNoDisc = 'Asemassa ei ole levy';
  SInsertDisc = 'Syt ni-CD asemaan';
  SNoTrack = '';

  SRetrievingData = 'Haetaan levyn tietoja...';
  SStopped = 'Pysytetty';
  SPlaying = 'Toistaa';
  SPaused = 'Tauko toistossa';

type
  TTimeMode = (tmNone, tmDiscTotal, tmTrackElapsed, tmTrackRemaining,
               tmDiscElapsed, tmDiscRemaining);

const
  TimeModes: array [TTimeMode] of string =
    ('No disc', 'Disc length', 'Track elapsed', 'Track remaining',
     'Disc elapsed', 'Disc remaining');

type
  TTrackNumber = 1..99;  // maximum number of tracks on CD
  TTrackSet = set of TTrackNumber;
  TPlayList = array [TTrackNumber] of Boolean;

type
  TFrmMain = class(TForm)
    RxGradientCaption1: TRxGradientCaption;
    StatusBar: TStatusBar;
    TbrControls: TToolBar;
    ImlControls: TImageList;
    BtnPlay: TToolButton;
    BtnPause: TToolButton;
    BtnStop: TToolButton;
    BtnPrevious: TToolButton;
    BtnBack: TToolButton;
    BtnForward: TToolButton;
    BtnNext: TToolButton;
    BtnEject: TToolButton;
    PnlDisplay: TPanel;
    LblArtist: TLCDLabel;
    LblTitle: TLCDLabel;
    LblTrackInfo: TLCDLabel;
    LblTrackName: TLCDLabel;
    MainMenu: TMainMenu;
    MnuDisc: TMenuItem;
    MnuDiscEdit: TMenuItem;
    MnuDiscSep1: TMenuItem;
    MnuDiscExit: TMenuItem;
    MnuHelp: TMenuItem;
    MnuHelpAbout: TMenuItem;
    Player: TMediaPlayer;
    TmrStatus: TTimer;
    FormPlacement1: TFormPlacement;
    ToolButton1: TToolButton;
    BtnTimeMode: TToolButton;
    RxTrayIcon1: TRxTrayIcon;
    AppEvents1: TAppEvents;
    BtnContinuousPlay: TToolButton;
    BtnPlayList: TToolButton;
    MnuOptions: TMenuItem;
    MnuOptionsRepeat: TMenuItem;
    MnuOptionsPlaylist: TMenuItem;
    MnuOptionsShuffle: TMenuItem;
    procedure MnuDiscExitClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure BtnPlayClick(Sender: TObject);
    procedure BtnStopClick(Sender: TObject);
    procedure BtnPauseClick(Sender: TObject);
    procedure TmrStatusTimer(Sender: TObject);
    procedure BtnEjectClick(Sender: TObject);
    procedure MnuDiscEditClick(Sender: TObject);
    procedure BtnNextClick(Sender: TObject);
    procedure BtnPreviousClick(Sender: TObject);
    procedure BtnBackClick(Sender: TObject);
    procedure BtnForwardClick(Sender: TObject);
    procedure MnuHelpAboutClick(Sender: TObject);
    procedure BtnTimeModeClick(Sender: TObject);
    procedure AppEvents1Minimize(Sender: TObject);
    procedure AppEvents1Restore(Sender: TObject);
    procedure RxTrayIcon1DblClick(Sender: TObject);
    procedure BtnContinuousPlayClick(Sender: TObject);
    procedure BtnPlayListClick(Sender: TObject);
    procedure MnuOptionsRepeatClick(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure FormDestroy(Sender: TObject);
    procedure MnuOptionsPlaylistClick(Sender: TObject);
    procedure MnuOptionsShuffleClick(Sender: TObject);
  private
    MyPlayer: TCDPlayer;

    Paused: Boolean;
    HaveInfo: Boolean;
    CurrentTrack: Longint;
    InfoChanged: Boolean;
    TimeMode: TTimeMode;
    ContinuousPlay: Boolean;
    Programmed: Boolean;
    NextPlayListTrack: Integer;
    PlayList: TPlayList;
    Shuffle: Boolean;
    procedure UpdateStatus;
    procedure WriteDiscInfo;
    procedure BeginPlay(StartTrack: Integer);
    procedure EnablePlayButtons;
    procedure DisablePlayButtons;
    procedure MakeRandomPlaylist(NumTracks: Longint; var Playlist: TPlayList);
    function GetNextPlayListTrack: Longint;
  protected
    procedure MMNotify(var Msg: TMessage); message mm_MciNotify;
  public
    { Public declarations }
  end;

var
  FrmMain: TFrmMain;

implementation

uses
  Disc, IniFiles, DiscEdit, About, PlayList, Log;

{$R *.DFM}

var
  DiscInfo: TDiscInfo;
  IniFileName: string;
  WinDir: array [0..256] of Char;

{ Closes the program's main window. }
procedure TFrmMain.MnuDiscExitClick(Sender: TObject);
begin
  Close;  // close the program's main window
end;

(**
 * Called when the main form is created.
 * Initializes the playlist and sets some important variables.
 **)
procedure TFrmMain.FormCreate(Sender: TObject);
var
  I: Integer;
begin
  try
    MyPlayer := TCDPlayer.Create(Self.Handle);
  except
    on E: EMCIDeviceError do
    begin
      MessageDlg(E.Message, mtError, [mbOK], 0);
    end;
  end;


  // Zap the playlist.
  for I := 1 to 99 do
  begin
    PlayList[I] := False;
  end;
  Programmed := False;
  NextPlayListTrack := 0;

  FillChar(DiscInfo, SizeOf(DiscInfo), 0);   // reset the disc information

  GetWindowsDirectory(@WinDir, SizeOf(WinDir));
  IniFileName := WinDir + '\cdplayer.ini';

  InfoChanged := False;
  CurrentTrack := 0;
  TimeMode := tmNone;
  ContinuousPlay := False;

  Paused := False;
  HaveInfo := False;
end;

{ Called the the Play button is clicked. }
procedure TFrmMain.BtnPlayClick(Sender: TObject);
var
  I: Integer;
begin
  FrmMessages.LstMessages.Items.Add('PLAY');

  if MyPlayer.MediaPresent then
  begin
    BeginPlay(1);
  end;
end;

procedure TFrmMain.BtnStopClick(Sender: TObject);
begin
  FrmMessages.LstMessages.Items.Add('STOP');

  if MyPlayer.MediaPresent then
  begin
    Paused := False;
    CurrentTrack := 0;
    TimeMode := tmDiscTotal;
    MyPlayer.Stop;
  end;
end;

procedure TFrmMain.BtnPauseClick(Sender: TObject);
begin
  FrmMessages.LstMessages.Items.Add('PAUSE');

  if MyPlayer.MediaPresent then
  begin
    if Paused then
    begin
      Paused := False;
      MyPlayer.Resume;
    end
    else
    begin
      if MyPlayer.GetMode = mci_Mode_Play then
      begin
        Paused := True;
        MyPlayer.Pause;
      end;
    end;
  end;
end;

procedure TFrmMain.BtnEjectClick(Sender: TObject);
begin
  if not HaveInfo then
  begin
    MyPlayer.Load;
    TimeMode := tmDiscTotal;
    EnablePlayButtons;
  end
  else
  begin
    MyPlayer.Eject;
    DisablePlayButtons;
  end;
  TimeMode := tmNone;
end;

procedure TFrmMain.BtnPreviousClick(Sender: TObject);
var
  StartTrack: Longint;
begin
  if Programmed then
  begin
    NextPlayListTrack := CurrentTrack;
    repeat
      Dec(NextPlayListTrack);
      if NextPlayListTrack < 1 then
        NextPlayListTrack := MyPlayer.GetTrackCount;
    until PlayList[NextPlayListTrack];
    StartTrack := NextPlayListTrack;
  end
  else
  begin
    if CurrentTrack <= 1 then
      StartTrack := MyPlayer.GetTrackCount
    else
      StartTrack := CurrentTrack - 1;
  end;
  BeginPlay(StartTrack);
end;

procedure TFrmMain.BtnNextClick(Sender: TObject);
var
  StartTrack: Longint;
begin
  if Programmed then
    StartTrack := GetNextPlayListTrack
  else
  begin
    if CurrentTrack = MyPlayer.GetTrackCount then
      StartTrack := 1
    else
      StartTrack := CurrentTrack + 1;
  end;

  if StartTrack <> 0 then
    BeginPlay(StartTrack);
end;

procedure TFrmMain.UpdateStatus;
var
  Track, Min, Sec: Longint;
  DisplayTime: Longint;
  TimeTemplate: string;
  TrackName: string;
  TrackLen: Longint;
begin
  DisplayTime := MyPlayer.GetPosition;
  Track := MyPlayer.GetCurrentTrack;

  case TimeMode of
    tmNone:
    begin
      Min := 0;
      Sec := 0;
      Track := 0;
    end;

    tmDiscTotal:
    begin
      MyPlayer.SetMSF;
      DisplayTime := MyPlayer.GetLength;
      Min := MCI_MSF_MINUTE(DisplayTime);
      Sec := MCI_MSF_SECOND(DisplayTime);
      Track := MyPlayer.GetTrackCount;
    end;

    tmTrackElapsed:
    begin
      MyPlayer.SetTMSF;
      DisplayTime := MyPlayer.GetPosition;
      Min := MCI_TMSF_MINUTE(DisplayTime);
      Sec := MCI_TMSF_SECOND(DisplayTime);
      Track := MyPlayer.GetCurrentTrack;
      CurrentTrack := Track;
    end;

    tmTrackRemaining:
    begin
      MyPlayer.SetMS;
      (* the player's position should be further than the track starting point *)
      Track := MyPlayer.GetCurrentTrack;
      DisplayTime := MyPlayer.GetPosition; // TrackLength[Track];  // unfinished
          // - (MyPlayer.GetPosition - MyPlayer.GetTrackPosition[Track]);
      MillisecToMinSec(DisplayTime, Min, Sec);
      CurrentTrack := Track;
    end;

    tmDiscElapsed:
    begin
      MyPlayer.SetMSF;
      DisplayTime := MyPlayer.GetPosition;
      Min := MCI_MSF_MINUTE(DisplayTime);
      Sec := MCI_MSF_SECOND(DisplayTime);
      Track := MyPlayer.GetCurrentTrack;
      CurrentTrack := Track;
    end;

    tmDiscRemaining:
    begin
      MyPlayer.SetMS;
      DisplayTime := MyPlayer.GetLength - MyPlayer.GetPosition;
      MillisecToMinSec(DisplayTime, Min, Sec);
      Track := MyPlayer.GetCurrentTrack;
      CurrentTrack := Track;
    end;
  end;

  TimeTemplate := '%.2d  %.2d:%.2d';

  if (TimeMode in [tmTrackRemaining, tmDiscRemaining]) and
     (MyPlayer.GetMode <> mci_Mode_Stop) then
  begin
    TimeTemplate := '%.2d -%.2d:%.2d';
  end;

  LblTrackInfo.Caption := Format(TimeTemplate,
                                 [Track, Min, Sec]);

  if MyPlayer.GetMode = mci_Mode_Stop then
    TrackName := SNoTrack
  else
    TrackName := DiscInfo.TrackNames[CurrentTrack];

  (* comparing strings should be faster than updating the display *)
  if LblTrackName.Caption <> TrackName then
    LblTrackName.Caption := TrackName;
end;

procedure TFrmMain.TmrStatusTimer(Sender: TObject);
var
  Ini: TIniFile;
  TrackNumber: Longint;
begin
  (* A CD was inserted since we last were here. *)
  if (not HaveInfo) and MyPlayer.MediaPresent then
  begin
    StatusBar.Panels[0].Text := SRetrievingData;
    HaveInfo := True;
    Paused := False;
    CurrentTrack := 0;

    DiscInfo.DiscID := MakeDiscID(MyPlayer);
    FrmMessages.LstMessages.Items.Add('ID = ' + DiscInfo.DiscID);

    (* Must read from INI file *)
    Ini := TIniFile.Create(IniFileName);
    try
      DiscInfo.Artist := Ini.ReadString(DiscInfo.DiscID, 'artist', SUnknownArtist);
      DiscInfo.Title := Ini.ReadString(DiscInfo.DiscID, 'title', SUnknownTitle);

      MyPlayer.SetTMSF;

      (* CDPLAYER.INI stores track names with zero based offsets. *)
      for TrackNumber := 0 to MyPlayer.GetTrackCount - 1 do
      begin
        DiscInfo.TrackNames[TrackNumber + 1] :=
          Ini.ReadString(DiscInfo.DiscID,
                         IntToStr(TrackNumber),
                         'Track ' + IntToStr(TrackNumber + 1));
      end;

      TimeMode := tmDiscTotal;

      MnuDiscEdit.Enabled := True;
      EnablePlayButtons;
    finally
      Ini.Free;
    end;

    LblArtist.Caption := DiscInfo.Artist;
    LblTitle.Caption := DiscInfo.Title;
    UpdateStatus;
    StatusBar.Panels[0].Text := SStopped;
  end
 (* CD was removed since we last were here *)
  else if HaveInfo and (not MyPlayer.MediaPresent) then
  begin
    HaveInfo := False;
    Paused := False;
    FillChar(DiscInfo, SizeOf(DiscInfo), 0);
    MnuDiscEdit.Enabled := False;
    DisablePlayButtons;
    Exit;
  end
  (* no disc in drive *)
  else if (not HaveInfo) and
          (not MyPlayer.MediaPresent) then
  begin
    FillChar(DiscInfo, SizeOf(DiscInfo), 0);
    LblArtist.Caption := SNoDisc;
    LblTitle.Caption := SInsertDisc;
    UpdateStatus;
    Paused := False;
    StatusBar.Panels[0].Text := SNoDisc;
    MnuDiscEdit.Enabled := False;
    Exit;
  end;

  (* If we get here, there is a disc in the drive and we
     have its information. *)

  CurrentTrack := 1;
  
  if (MyPlayer.GetMode = mci_Mode_Play) or Paused then
  begin
    UpdateStatus;
    if MyPlayer.GetMode = mci_Mode_Play then
      StatusBar.Panels[0].Text := SPlaying
    else
      StatusBar.Panels[0].Text := SPaused;
  end
  else if MyPlayer.GetMode = mci_Mode_Stop then
  begin
    MyPlayer.SetMSF;
    UpdateStatus;
    StatusBar.Panels[0].Text := SStopped;
  end
  else (* maybe skip pressed? *)
  begin
    MyPlayer.SetMSF;
    UpdateStatus;
  end;

  if MyPlayer.GetErrorNumber <> 0 then
  begin
    StatusBar.Panels[1].Text := MyPlayer.GetErrorMessage;
  end
  else
  begin
    StatusBar.Panels[1].Text := '';
  end;
end;

procedure TFrmMain.MnuDiscEditClick(Sender: TObject);
var
  I: Longint;
begin
  FrmDiscEditor.TxtArtist.Text := LblArtist.Caption;
  FrmDiscEditor.TxtTitle.Text := LblTitle.Caption;

  FrmDiscEditor.LstTracks.Items.Clear;

  for I := 1 to MyPlayer.GetTrackCount do
  begin
    FrmDiscEditor.LstTracks.Items.Add(
      DiscInfo.TrackNames[I]
    );
  end;

  if FrmDiscEditor.ShowModal = mrOK then
  begin
    InfoChanged := FrmDiscEditor.HasChanged;

    (* Zap the track name list and fill it again with the
       information from the form. *)
    FillChar(DiscInfo.TrackNames, SizeOf(DiscInfo.TrackNames), 0);
    for I := 0 to FrmDiscEditor.LstTracks.Items.Count - 1 do
    begin
      DiscInfo.TrackNames[I + 1] :=
        FrmDiscEditor.LstTracks.Items[I];
    end;

    LblArtist.Caption := FrmDiscEditor.TxtArtist.Text;
    LblTitle.Caption := FrmDiscEditor.TxtTitle.Text;

    WriteDiscInfo;
  end;
end;

procedure TFrmMain.WriteDiscInfo;
var
  Ini: TIniFile;
  I: Longint;
  Order: string;
begin
  if not InfoChanged then Exit;

  (* First erase the old section for this disc *)
  Ini := TIniFile.Create(IniFileName);
  try
    Ini.EraseSection(DiscInfo.DiscID);

    with Ini do
    begin
      WriteInteger(DiscInfo.DiscID, 'EntryType', 1);
      WriteString(DiscInfo.DiscID, 'artist', LblArtist.Caption);
      WriteString(DiscInfo.DiscID, 'title', LblTitle.Caption);
      WriteInteger(DiscInfo.DiscID, 'numtracks', MyPlayer.GetTrackCount);

      Order := '';
      for I := 1 to MyPlayer.GetTrackCount do
      begin
        Order := Order + IntToStr(I - 1) + ' ';
        WriteString(DiscInfo.DiscID, IntToStr(I - 1),
          DiscInfo.TrackNames[I]);
      end;
      WriteString(DiscInfo.DiscID, 'order', Trim(Order));
      WriteInteger(DiscInfo.DiscID, 'numplay', MyPlayer.GetTrackCount);
    end;
  finally
    Ini.Free;
  end;

  InfoChanged := False;
end;


procedure TFrmMain.BtnBackClick(Sender: TObject);
begin
{
  Player.Frames := 10;
  Player.Notify := True;
  Player.Back;
}
end;

procedure TFrmMain.BtnForwardClick(Sender: TObject);
begin
{
  if not (mpCanStep in Player.Capabilities) then
    StatusBar.Panels[1].Text := 'Unable to step!';

  Player.Frames := 10;
  Player.Notify := True;
  Player.Step;
}  
end;

procedure TFrmMain.MnuHelpAboutClick(Sender: TObject);
begin
  FrmAbout.ShowModal;
end;

procedure TFrmMain.BtnTimeModeClick(Sender: TObject);
begin
  if MyPlayer.GetMode <> mci_Mode_Stop then
  begin
    if TimeMode = tmDiscRemaining then
    begin
      TimeMode := tmTrackElapsed;
    end
    else
    begin
      TimeMode := TTimeMode(Succ(Ord(TimeMode)));
    end;
    StatusBar.Panels[1].Text := TimeModes[TimeMode];
  end;
end;

procedure TFrmMain.AppEvents1Minimize(Sender: TObject);
begin
  RxTrayIcon1.Active := True;
  FrmMain.Hide;
end;

procedure TFrmMain.AppEvents1Restore(Sender: TObject);
begin
  FrmMain.Show;
  Application.BringToFront;
  RxTrayIcon1.Active := False;
end;

procedure TFrmMain.RxTrayIcon1DblClick(Sender: TObject);
begin
  Application.Restore;
end;

procedure TFrmMain.MakeRandomPlaylist(NumTracks: Longint; var Playlist: TPlayList);
var
  I: Integer;
  TrackNumber: TTrackNumber;
  List: set of TTrackNumber;
begin
  List := [];

  for I := 1 to NumTracks do
  begin
    TrackNumber := Random(NumTracks) + 1;
    while not TrackNumber in List do
    begin
      List := List + [TrackNumber];
      TrackNumber := Random(NumTracks) + 1;
    end;
  end;

end;

{ Handle click of Repeat button. }
procedure TFrmMain.BtnContinuousPlayClick(Sender: TObject);
begin
  ContinuousPlay := not ContinuousPlay;
  MnuOptionsRepeat.Checked := ContinuousPlay;
end;

const
  ModeStr: array[TMPModes] of string =
    ('Not ready', 'Stopped', 'Playing',
    'Recording', 'Seeking', 'Paused', 'Open');


procedure TFrmMain.BeginPlay(StartTrack: Integer);
begin
  Paused := False;
  if Programmed then
    StartTrack := GetNextPlayListTrack;
  CurrentTrack := StartTrack;

  TimeMode := tmTrackElapsed;
  if CurrentTrack < MyPlayer.GetTrackCount then
    MyPlayer.PlayDisc(CurrentTrack, CurrentTrack + 1)
  else
    MyPlayer.PlayTrack(CurrentTrack);
end;

procedure TFrmMain.BtnPlayListClick(Sender: TObject);
var
  I: Integer;
  S: string;
begin
  //FrmPlayList.LstTracks.Items.Clear;

  FrmPlayList.LblDiscInfo.Caption :=
    DiscInfo.Artist + ' - ' + DiscInfo.Title;

  { Don't add if the names are already on the list. }
  { All the items are initially checked. }
  if FrmPlayList.LstTracks.Items.Count = 0 then
  begin
    for I := 1 to MyPlayer.GetTrackCount do
    begin
      FrmPlayList.LstTracks.Items.Add(DiscInfo.TrackNames[I]);
      FrmPlayList.LstTracks.Checked[I - 1] := True;
      PlayList[I] := True;
    end;
  end;

  if FrmPlayList.ShowModal = mrOK then
  begin
    S := '';
    for I := 1 to MyPlayer.GetTrackCount do
    begin
      PlayList[I] := FrmPlayList.LstTracks.Checked[I - 1];
      if PlayList[I] then
        S := S + IntToStr(I) + ' ';
    end;
    //MessageDlg(S, mtInformation, [mbOk], 0);
    Programmed := True;
    NextPlayListTrack := 1;
    while not PlayList[NextPlayListTrack] do
    begin
      Inc(NextPlayListTrack);
      if NextPlayListTrack > MyPlayer.GetTrackCount then
        NextPlayListTrack := 1;
    end;
    CurrentTrack := NextPlayListTrack;
  end;
end;

procedure TFrmMain.DisablePlayButtons;
begin
  BtnPlay.Enabled := False;
  BtnPause.Enabled := False;
  BtnStop.Enabled := False;
  //BtnBack.Enabled := False;
  //BtnForward.Enabled := False;
  BtnNext.Enabled := False;
  BtnPrevious.Enabled := False;
end;

procedure TFrmMain.EnablePlayButtons;
begin
  BtnPlay.Enabled := True;
  BtnPause.Enabled := True;
  BtnStop.Enabled := True;
  //BtnBack.Enabled := True;
  //BtnForward.Enabled := True;
  BtnNext.Enabled := True;
  BtnPrevious.Enabled := True;
end;

procedure TFrmMain.MnuOptionsRepeatClick(Sender: TObject);
begin
  ContinuousPlay := not ContinuousPlay;
  MnuOptionsRepeat.Checked := ContinuousPlay;
  BtnContinuousPlay.Down := ContinuousPlay;
end;

procedure TFrmMain.FormClose(Sender: TObject; var Action: TCloseAction);
begin
  { If a disc is currently playing, stop it before closing down. }
  if MyPlayer.GetMode = mci_Mode_Play then
  begin
    MyPlayer.Stop;
  end;
end;

procedure TFrmMain.FormDestroy(Sender: TObject);
begin
  MyPlayer.Free;
end;

procedure TFrmMain.MMNotify(var Msg: TMessage);
var
  StartTrack: Longint;
begin
  case Msg.wParam of
    mci_Notify_Superseded: begin
      FrmMessages.LstMessages.Items.Add('Superseded');
      //MyPlayer.Stop;
    end;
    mci_Notify_Aborted: begin
      FrmMessages.LstMessages.Items.Add('Aborted');
      //MyPlayer.Stop;
    end;

    mci_Notify_Successful:
    begin
      FrmMessages.LstMessages.Items.Add('Successful');

      if MyPlayer.GetMode = mci_Mode_Stop then
      begin
        if ContinuousPlay then
        begin
          if Programmed then
          begin
            CurrentTrack := 1;
            StartTrack := GetNextPlayListTrack
          end
          else
          begin
            StartTrack := 1;
          end;
        end
        else { normal play }
        begin
          if CurrentTrack = MyPlayer.GetTrackCount then
            StartTrack := 0
          else
            Inc(StartTrack);
        end;
      end;
      if StartTrack <> 0 then
        BeginPlay(StartTrack);
    end;
  end;
end;

procedure TFrmMain.MnuOptionsPlaylistClick(Sender: TObject);
begin
  BtnPlayListClick(Self);
end;

procedure TFrmMain.MnuOptionsShuffleClick(Sender: TObject);
begin
  Shuffle := not Shuffle;
  MnuOptionsShuffle.Checked := Shuffle;

  MessageDlg('Satunnaissoitto ei sittenkn toimi viel. Aika loppui kesken!',
    mtInformation, [mbOK], 0);
    
  //MakeRandomPlayList(MyPlayer.GetTrackCount, PlayList);
  //Programmed := True;
  //NextPlayListTrack := 1;
end;

function TFrmMain.GetNextPlayListTrack: Longint;
var
  TrackNumber: Longint;
begin
  TrackNumber := CurrentTrack;
  repeat
    Inc(TrackNumber);
    { If it falls off from the end, set next playlist track to zero
      and bail out. }
    if TrackNumber > MyPlayer.GetTrackCount then
    begin
      TrackNumber := 0;
      Break;
    end;
  until PlayList[TrackNumber];
  Result := TrackNumber;
end;

end.
