/*
** Copyright (C) 1999 Juhan Aslak Nkkljrvi <j.aslak@kolumbus.fi>
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
** 
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
** 
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software 
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#include <libintl.h>
#define _(Strinkula) gettext (Strinkula)

#include <fcntl.h>
#include <signal.h>
#include <unistd.h>
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <sys/time.h>
#include <errno.h>

#include "version.h"
#include "command.h"
#include "wmode.h"

unsigned char XOR=0x00;

#define SOCKET "/tmp/led_socket"
#define MAXMSG 2048
volatile sig_atomic_t faterr = 0;
// 10 times/second
int delay=100000;
int interval=1;
int current_wmode=0;
int wmode_num=0;
struct WMode **wmode=0;
int norc=0,quiet=0;
char *rcname=NULL;

int on=1;
int chg=1;
int uses_cpu=0,uses_mem=0;

#define HANDLE_MESSAGE_SIZE 2048
char *handle_message;
int handle_message_len;

long cpu[4]={0,0,0,0},last[4];

void WriteLED(int t)
{
	FILE *f;
	f=fopen("/dev/port","wb");
	if (!f)
		return;
	fseek(f,0x378,SEEK_SET);
	fputc(t^XOR,f);
	fclose(f);
}

int AddWMode(struct WMode *w,int n)
{
	struct WMode** t;
	int k;
	
	if (n<wmode_num)
	{
		if (wmode[n])
			free(wmode[n]);
		wmode[n]=w;
		return 0;
	}
	t=(struct WMode**)malloc((n+1)*sizeof(struct WMode*));
	for (k=0;k<wmode_num;k++)
		t[k]=wmode[k];
	for (;k<n;k++)
		t[k]=0;
	t[n]=w;

	if (wmode)
		free(wmode);
		
	wmode=t;
	wmode_num=n+1;
	
	return 0;
}

inline int ReadInt(int* a,char* s)
{
	memcpy(a,s,sizeof(int));
	return sizeof(int);
}

int make_named_socket(char* filename)	// Stolen!
{
	struct sockaddr_un name;
	int sock;
	size_t size;

	sock=socket(PF_UNIX,SOCK_DGRAM,0);
	fcntl(sock,F_SETFL,O_NONBLOCK);
	if (sock<0)
	{
		perror("socket");
		return 1;
	}
	name.sun_family=AF_UNIX;
	strcpy(name.sun_path,filename);

	size = (offsetof (struct sockaddr_un, sun_path) + strlen (name.sun_path) + 1);

	if (bind(sock,(struct sockaddr *)&name,size)<0)
	{
		perror("bind");
		exit(1);
	}
	return sock;
}

struct timeval tsub (struct timeval x,struct timeval y)	// Stolen!
{
	struct timeval result;
	/* Perform the carry for the later subtraction by updating Y. */
	if (x.tv_usec < y.tv_usec)
	{
		int nsec = (y.tv_usec - x.tv_usec) / 1000000 + 1;
		y.tv_usec -= 1000000 * nsec;
		y.tv_sec += nsec;
	}
	if (x.tv_usec - y.tv_usec > 1000000)
	{
		int nsec = (y.tv_usec - x.tv_usec) / 1000000;
		y.tv_usec += 1000000 * nsec;
		y.tv_sec -= nsec;
	}

	/* Compute the time remaining to wait.
	   `Tv_usec' is certainly positive. */
	result.tv_sec = x.tv_sec - y.tv_sec;
	result.tv_usec = x.tv_usec - y.tv_usec;
	
	return result;
}


void pois(int sig)
{
	if (faterr)
		raise(sig);
	faterr=1;
	
	printf(_("Got shutdown message, deleting %s ...\n"),SOCKET);
	unlink(SOCKET);
	
	signal(sig,SIG_DFL);	// Otherwise at least on my system it would infinitely recurse itself
	raise(sig);
}

#define hprintf(format...) handle_message_len+=sprintf(handle_message+handle_message_len,##format)
int HandleMessage(char* s,int nbytes)
{
	int p=0;
	int type;
	int t;
	struct WMode* a;
	int k,i,j;

	while (p<nbytes)
	{
		p+=ReadInt(&type,s+p);
		switch (type)
		{
			case CMD_NONE:
				hprintf(_("Having fun?\n"));
				break;
			case CMD_WMODE:
				p+=ReadInt(&t,s+p);
				if (t>=0 && t<wmode_num)
					current_wmode=t;
				else if (t==-1)
				{
					current_wmode++;
					if (current_wmode>=wmode_num)
						current_wmode=0;
				}
				else if (t==-2)
				{
					current_wmode--;
					if (current_wmode<0)
						current_wmode=wmode_num-1;
				}
				else
				{
					hprintf(_("Illegal working mode: %d (%d is max)\n"),t,wmode_num-1);
					break;
				}
				chg=1;
				hprintf(_("Working mode set to %d.\n"),current_wmode);
				break;
			case CMD_OFF:
				if (on)
				{
					on=0;
					chg=1;
					hprintf(_("%s turned off.\n"),PROG_NAME_SHORT);
				}
				break;
			case CMD_ON:
				if (!on)
				{
					on=1;
					chg=1;
					hprintf(_("%s turned on.\n"),PROG_NAME_SHORT);
				}
				break;
			case CMD_DEFWMODE:
				chg=1;
				p+=ReadInt(&t,s+p);
				
				a=(struct WMode *)malloc(sizeof(struct WMode));
				p+=ReadInt(&(a->gauge_num),s+p);
				a->gauge=(struct Gauge**)malloc(a->gauge_num*sizeof(struct Gauge*));
				for (k=0;k<a->gauge_num;k++)
				{
					a->gauge[k]=(struct Gauge*)malloc(sizeof(struct Gauge));
					p+=ReadInt(&(a->gauge[k]->modenum),s+p);
					a->gauge[k]->mode=(int*)malloc(a->gauge[k]->modenum*sizeof(int));
					for (i=0;i<a->gauge[k]->modenum;i++)
						p+=ReadInt(&(a->gauge[k]->mode[i]),s+p);
					p+=ReadInt(&(a->gauge[k]->size),s+p);
					a->gauge[k]->position=(int*)malloc(a->gauge[k]->size*sizeof(int));
					for (i=0;i<a->gauge[k]->size;i++)
						p+=ReadInt(&(a->gauge[k]->position[i]),s+p);
				}
				AddWMode(a,t);
				hprintf(_("Working mode %d defined.\n"),t);
				break;
			case CMD_XOR:
				p+=ReadInt(&t,s+p);
				XOR=t;
				hprintf(_("XOR status set to %02X.\n"),XOR);
				break;
			case CMD_LISTWMODES:
				t=0;
				for (k=0;k<wmode_num;k++)
					if (wmode[k])
						t++;
				hprintf(t==1 ? _("%d working mode:\n") : _("%d working modes:\n"),t);
				for (k=0;k<wmode_num;k++)
					if (wmode[k])
					{
						hprintf("\n");
						hprintf(_("Working Mode %d:\n"),k);
						hprintf(wmode[k]->gauge_num==1 ? _("%d gauge\n") : _("%d gauges\n"),wmode[k]->gauge_num);
						for (i=0;i<wmode[k]->gauge_num;i++)
						{
							hprintf(_("Gauge %d: Mode"),i);
							for (j=0;j<wmode[k]->gauge[i]->modenum;j++)
								hprintf(" %d",wmode[k]->gauge[i]->mode[j]);
							hprintf(_(", Size %d, Positions"),wmode[k]->gauge[i]->size);
							for (j=0;j<wmode[k]->gauge[i]->size;j++)
								hprintf(" %d",wmode[k]->gauge[i]->position[j]);
							hprintf("\n");
						}
					}
				break;
			case CMD_DELAY:
				p+=ReadInt(&t,s+p);
				delay=t;
				hprintf(_("Delay set to %d.\n"),delay);
				break;
			default:
				hprintf(_("Unknown command `%d'!\n"),type);
				return 1;
		}
	}
	if (p>nbytes)
		hprintf(_("Warning: Block was %d bytes too short!\n"),p-nbytes);
}

void HandleState()
{
	int k,i;
	
	if (!on)
		WriteLED(0);
	for (k=0;k<wmode[current_wmode]->gauge_num;k++)
		for (i=0;i<wmode[current_wmode]->gauge[k]->modenum;i++)
			switch (wmode[current_wmode]->gauge[k]->mode[i])
			{
				case GAUGE_USER:
				case GAUGE_NICE:
				case GAUGE_SYS:
					uses_cpu=1;
					break;
				case GAUGE_RAM:
				case GAUGE_SWAP:
				case GAUGE_MEM:
					uses_mem=1;
					break;
			}
}

void Init()
{
	struct WMode *a;

	signal(SIGTERM,pois);
	signal(SIGINT,pois);
	signal(SIGQUIT,pois);

	a=malloc(sizeof(struct WMode));
	a->gauge_num=0;
	a->gauge=(struct Gauge**)malloc(a->gauge_num*sizeof(struct Gauge*));
	
	AddWMode(a,0);
}

void HandleData()
{
	int a;
	int k,i,j;
	int LED[8];
	long max;
	float usr,nice,sys,idle;
	float ram,swap,mem;
	long ramtotal,ramused,swaptotal,swapused;
	long t;
	float b;
	int F;
	FILE *f;
	char buff[400];
	
	for (k=0;k<8;k++)
		LED[k]=0;

	if (uses_cpu)
	{
		f=fopen("/proc/stat","rt");
		if (f)
		{
			fgets(buff,400,f);
			fclose(f);
			last[0]=cpu[0];last[1]=cpu[1];last[2]=cpu[2];last[3]=cpu[3];
			sscanf(buff,"cpu  %ld %ld %ld %ld",&cpu[0],&cpu[1],&cpu[2],&cpu[3]);
			max=cpu[0]-last[0];
			max+=cpu[1]-last[1];
			max+=cpu[2]-last[2];
			max+=cpu[3]-last[3];
			usr=(float)(cpu[0]-last[0])/(float)max;
			nice=(float)(cpu[1]-last[1])/(float)max;
			sys=(float)(cpu[2]-last[2])/(float)max;
//			idle=(float)(cpu[3]-last[3])/(float)max;
		}
	}
	if (uses_mem)
	{
		f=fopen("/proc/meminfo","rt");
		if (f)
		{
			while (strncmp(fgets(buff,400,f),"MemTotal:",9));
			ramtotal=strtol(buff+11,NULL,0);
			fgets(buff,400,f);
			fclose(f);
			ramused=ramtotal-strtol(buff+11,NULL,0);

			while (strncmp(fgets(buff,400,f),"SwapTotal:",10));
			swaptotal=strtol(buff+11,NULL,0);
			fgets(buff,400,f);
			swapused=swaptotal-strtol(buff+11,NULL,0);
			
			ram=(float)ramused/(float)ramtotal;
			swap=(float)swapused/(float)swaptotal;
			mem=(float)(ramused+swapused)/(float)(ramtotal+swaptotal);
		}
	}
	for (k=0;k<wmode[current_wmode]->gauge_num;k++)
	{
		b=0;
		F=0;
		for  (j=0;j<wmode[current_wmode]->gauge[k]->modenum;j++)
			switch (wmode[current_wmode]->gauge[k]->mode[j])
			{
				case GAUGE_USER:
					b+=usr*wmode[current_wmode]->gauge[k]->size;
					F=1;
					break;
				case GAUGE_NICE:
					b+=nice*wmode[current_wmode]->gauge[k]->size;
					F=1;
					break;
				case GAUGE_SYS:
					b+=sys*wmode[current_wmode]->gauge[k]->size;
					F=1;
					break;
				case GAUGE_RAM:
					a=(int) (ram*wmode[current_wmode]->gauge[k]->size+0.5);
					F=1;
					break;
				case GAUGE_SWAP:
					a=(int) (swap*wmode[current_wmode]->gauge[k]->size+0.5);
					F=1;
					break;
				case GAUGE_MEM:
					a=(int) (mem*wmode[current_wmode]->gauge[k]->size+0.5);
					F=1;
					break;
			}
		if (F)
			a=(int)(b+0.5);
		if (a>wmode[current_wmode]->gauge[k]->size)
			a=wmode[current_wmode]->gauge[k]->size;
		for (i=0;i<a;i++)
			LED[wmode[current_wmode]->gauge[k]->position[i]]=1;
	}

	a=0;
	for (k=0;k<8;k++)
		if (LED[k])
			a|=1<<k;
	WriteLED(a);
}

int HandleArgs(int argc,char *argv[])
{
	int p;

	for (p=1;p<argc;p++)
		if (argv[p][0]=='-')
		{
			if (argv[p][1]=='h' || !strcmp(argv[p]+1,"-help"))
			{
				printf(_(
"%s %d.%d.%d %s server\n"
"\n"
"Sits behind, obeys led_client's commands and does what it's told to do.\n"
"\n"
"Options:\n"
"\n"
"  -n, --no-rc                  Don't use .ledrc file.\n"
"  -r, --rc FILE                Use FILE as rc-file.\n"
"  -q, --quiet                  Doesn't spam the terminal with echos to client.\n"
"\n"
"  -v, --version                Displays the version.\n"
"  -h, --help                   Displays this help.\n")
,PROG_NAME,MAJOR_VER,MINOR_VER,REV_VER,VER_NOTE);
				return 1;
			}
			else if (argv[p][1]=='v' || !strcmp(argv[p]+1,"-version"))
			{
				printf("%s %d.%d.%d %s server\n",PROG_NAME,MAJOR_VER,MINOR_VER,REV_VER,VER_NOTE);
				return 1;
			}
			else if (argv[p][1]=='n' || !strcmp(argv[p]+1,"-no-rc"))
				norc=1;
			else if (argv[p][1]=='r' || !strcmp(argv[p]+1,"-rc"))
			{
				p++;
				if (p==argc)
				{
					fprintf(stderr,_("Parameter -r requires argument!\n"));
					return 1;
				}
				rcname=strdup(argv[p]);
			}
			else if (argv[p][1]=='q' || !strcmp(argv[p]+1,"-quiet"))
				quiet=1;
		}
	return 0;
}

int main(int argc,char *argv[])
{
	struct sockaddr_un name;
	int sock;
	size_t size;
	int nbytes;
	char message[MAXMSG];
	int k;
	int r;

	setlocale (LC_ALL, "");
	bindtextdomain (PACKAGE, LOCALEDIR);
	textdomain (PACKAGE);
	
	if (r=HandleArgs(argc,argv))
		return r;
	
	Init();
	sock=make_named_socket(SOCKET);

	if (!norc)
	{
		pid_t pid;
		pid=fork();
		if (pid<0)
			fprintf(stderr,_("Can't fork, use .ledrc from script\n"));
		else if (!pid)
		{
			FILE *f;
			char buff[400];
		
			if (!rcname)
				sprintf(buff,"%s/.ledrc",getenv("HOME"));
			else
				strcpy(buff,rcname);
			f=fopen(buff,"rt");
			if (!f)
			{
				fprintf(stderr,_("Can't open %s!\n"),buff);
				return;
			}
			fprintf(stderr,_("** Applying %s ...\n"),buff);
			sprintf(buff,"led_client -q ");
			while (!feof(f))
			{
				if (!fgets(buff+14,386,f))
					break;
				fprintf(stderr,_("** Executing %s\n"),buff);
				system(buff);
			}
			fprintf(stderr,_("** Finished.\n"));
			return;
		}
	}
	while (1)
	{
		if (chg)
		{
			chg=0;
			HandleState();
		}
		if (on)
			HandleData();
			
		size = sizeof (name);
		nbytes = recvfrom (sock, message, MAXMSG, 0,
		                   (struct sockaddr *) & name, (int *)&size);
		if (nbytes>=0 || errno!=EWOULDBLOCK)
		{
			if (nbytes < 0)
			{

				exit (EXIT_FAILURE);
			}
			handle_message_len=0;
			handle_message=(char *)malloc(HANDLE_MESSAGE_SIZE*sizeof(char));
			
			HandleMessage(message,nbytes);
			if (!quiet)
				puts(handle_message);
			
			nbytes = sendto (sock, handle_message, handle_message_len+1, 0,
			                 (struct sockaddr *) & name, size);
			if (nbytes < 0)
			{
				perror ("sendto (server)");
				exit (EXIT_FAILURE);
			}
			free(handle_message);
		}
		if (delay)
			usleep(delay);
	}
}
