/****************************************************************************

    Module  : DockUtil.c

    Version : v1.0alpha

    Date    : 20/8/93

    Changes : 20/8/93 - Added use of RePaintSlot function to selectively
                        repaint a single slot instead of using invalidate
                        clientrect to paint whole window when only a single
                        slot changed.
              24/6/94 - rewrote code to position dock & abstracted code into
                        function SetDockPos()

*****************************************************************************/


#include <windows.h>
#include <commdlg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <direct.h>
#include <ctype.h>
#include <shellapi.h>
#include "freedock.h"

BOOL         bFoundWindow;  // Global var for EnumWindows Callback

/****************************************************************
   ReadINIFile()

   Read in and process all slot data, also carry out any setup
   which is required.
****************************************************************/


int ReadINIFile(HWND hwnd, HANDLE hInst, SLOT_ENTRY * Slot)
{
    char            SlotSection[MAX_FPATH_LEN], TmpBuffer[MAX_FPATH_LEN];
    int             StartSlot, SlotNum;
    int             i;
                           
	/*********************************
		Initialise Mail box Flag
	*********************************/                           
    DockOptions.bMailInBox = FALSE;          
    
    /**********************************************************************
           Start by initialising the first slot to be the title icon
    ***********************************************************************/
    Slot[0].SlotType = SLOT_SPECIAL_TITLE;
    UtilLoadIcon( &Slot[0] );

    /**********************************************************************
       Set the index to the first free slot to the slot after the title
       slot
    ***********************************************************************/

    StartSlot = 1;

    /**********************************************************************
        Setup the special built in options in their own slots if configured
        and adjust StartSlot if required
    ***********************************************************************/

    GetPrivateProfileString("Main Options", "WinExitActive", "No", TmpBuffer, MAX_FPATH_LEN, IniFile);
    if (toupper(TmpBuffer[0]) == 'Y') {
        DockOptions.WinExitActive = TRUE;
        Slot[StartSlot].SlotType = SLOT_SPECIAL_EXIT;
        UtilLoadIcon( &Slot[StartSlot] );
        StartSlot++;
        GetPrivateProfileString("Main Options", "WinExitConfirm", "Yes", TmpBuffer, MAX_FPATH_LEN, IniFile);
        if (toupper(TmpBuffer[0]) == 'N')
            DockOptions.WinExitConfirm = FALSE;
        else
            DockOptions.WinExitConfirm = TRUE;
    } 
    else
        DockOptions.WinExitActive = FALSE;

    GetPrivateProfileString("Main Options", "Position", "Horizontal", TmpBuffer, MAX_FPATH_LEN, IniFile);
    if (toupper(TmpBuffer[0]) == 'H') {
        DockOptions.Position = DOCK_HORZ;
    } 
    else
        DockOptions.Position = DOCK_VERT;


    GetPrivateProfileString("Main Options", "ClockActive", "No", TmpBuffer, MAX_FPATH_LEN, IniFile);
    if (toupper(TmpBuffer[0]) == 'Y') {
        DockOptions.ClockActive = TRUE;
        Slot[StartSlot].SlotType = SLOT_SPECIAL_CLOCK;
        UtilLoadIcon( &Slot[StartSlot] );
        StartSlot++;
    } 
    else
        DockOptions.ClockActive = FALSE;

    GetPrivateProfileString("Main Options", "MailActive", "No", TmpBuffer, MAX_FPATH_LEN, IniFile);
    if (toupper(TmpBuffer[0]) == 'Y') {
        DockOptions.MailActive = TRUE;
        GetPrivateProfileString("Main Options", "MailSound", "Yes", TmpBuffer, MAX_FPATH_LEN, IniFile);
        if (toupper(TmpBuffer[0]) == 'N')
            DockOptions.MailSound = FALSE;
        else
            DockOptions.MailSound = TRUE;

        GetPrivateProfileString("Main Options", "MailPath", "c:\\mail.box", DockOptions.MailPath, MAX_FPATH_LEN, IniFile);
        /* GetPrivateProfileString Cannot handle long, so do it manually */
        GetPrivateProfileString("Main Options", "MailBoxSize", "0", TmpBuffer, MAX_FPATH_LEN, IniFile);
        sscanf(TmpBuffer, "%lu", &DockOptions.MailBoxSize);

        GetPrivateProfileString("Main Options", "MailBoxDateTime", "0", TmpBuffer, MAX_FPATH_LEN, IniFile);
        sscanf(TmpBuffer, "%lu", &DockOptions.MailBoxDateTime);

        DockOptions.MailFreq = GetPrivateProfileInt("Main Options", "MailFreq", 10, IniFile);

        Slot[StartSlot].SlotType = SLOT_SPECIAL_MAIL;
        UtilLoadIcon( &Slot[StartSlot] );
        StartSlot++;
    } 
    else
        DockOptions.MailActive = FALSE;

    DockOptions.DockLeft = GetPrivateProfileInt("Main Options", "DockLeft", 0, IniFile);
    DockOptions.DockTop = GetPrivateProfileInt("Main Options", "DockTop", 0, IniFile);

    DockOptions.DockSize = GetPrivateProfileInt("Main Options", "DockSize", 10, IniFile);

    GetPrivateProfileString("Main Options", "AlwaysOnTop", "No", TmpBuffer, MAX_FPATH_LEN, IniFile);
    if (toupper(TmpBuffer[0]) == 'Y')
        DockOptions.AlwaysOnTop = TRUE;
    else
        DockOptions.AlwaysOnTop = FALSE;

    GetPrivateProfileString("Main Options", "SingleClickStart", "No", TmpBuffer, MAX_FPATH_LEN, IniFile);
    if (toupper(TmpBuffer[0]) == 'Y')
        DockOptions.SingleClickStart = TRUE;
    else
        DockOptions.SingleClickStart = FALSE;

    GetPrivateProfileString("Main Options", "MaxView", "No", TmpBuffer, MAX_FPATH_LEN, IniFile);
    if (toupper(TmpBuffer[0]) == 'Y')
        DockOptions.MaxView = TRUE;
    else
        DockOptions.MaxView = FALSE;


    /*******************************************************************************
       SlotNum is used to index the sections in the .INI file, each slot is
       uniquely numbered in a section of the type [Slot xx] where xx is the slot
       number.
       The for loop attempts to read the AppName field from each slot, if this
       fails the slot is empty so the slot type is set to SLOT_FREE and the next
       slot is read. If the reading of AppName succeeds, the slot is used and
       the rest of the details are read in.
    *******************************************************************************/
    SlotNum = 1;

    for (i = StartSlot; i < DOCK_MAX_SIZE; i++) {

       	UtilEmptySlot( &Slot[i] );	// Ensure Default Empty struct is in Slot 

        Slot[i].ID = SlotNum;

        sprintf(SlotSection, "Slot %d", SlotNum++);

        GetPrivateProfileString(SlotSection, "AppName", "", Slot[i].AppName, MAX_FPATH_LEN, IniFile);
        if (Slot[i].AppName[0] != '\0') {
            GetPrivateProfileString(SlotSection, "CmdLine", "", Slot[i].CmdLine, MAX_CMDLINE_LEN, IniFile);
            GetPrivateProfileString(SlotSection, "RunTimeDir", "", Slot[i].RunTimeDir, MAX_FPATH_LEN, IniFile);

            Slot[i].StartState = GetPrivateProfileInt(SlotSection, "StartState", START_NORMAL, IniFile);

            GetPrivateProfileString(SlotSection, "StartOnTop", "No", TmpBuffer, MAX_FPATH_LEN, IniFile);
            if (toupper(TmpBuffer[0]) == 'N')
                Slot[i].StartOnTop = FALSE;
            else
                Slot[i].StartOnTop = TRUE;

            Slot[i].WinX = GetPrivateProfileInt(SlotSection, "WinX", 0, IniFile);
            Slot[i].WinY = GetPrivateProfileInt(SlotSection, "WinY", 0, IniFile);
            Slot[i].WinWidth = GetPrivateProfileInt(SlotSection, "WinWidth", DEFAULT_WIN_W, IniFile);
            Slot[i].WinHeight = GetPrivateProfileInt(SlotSection, "WinHeight", DEFAULT_WIN_H, IniFile);

            GetPrivateProfileString(SlotSection, "IconFile", "", Slot[i].IconFile, MAX_FPATH_LEN, IniFile);

            Slot[i].IconPos = GetPrivateProfileInt(SlotSection, "IconPos", 0, IniFile);

            // Load the Icon into the slot's own bitmap & update the IconTotal Field
            UtilLoadIcon( &Slot[i] );
      
            Slot[i].SlotType = SLOT_USED;
        } 
        else
            Slot[i].SlotType = SLOT_FREE;
    }

    return 1;
}

/* Updates a slot entry within an INI file */
int 
UpdateINIFile(SLOT_ENTRY * Slot)
{
    char            TmpBuffer[MAX_FPATH_LEN], SlotName[MAX_FPATH_LEN];
    char            PosStr[2][11] =
    {
        {"Horizontal"},
        {"Vertical"}};

    /* Flush any cached .INI file to disk */
    WritePrivateProfileString(NULL, NULL, NULL, IniFile);

    /* if Slot is a special, write out the main options */
    if( (Slot->SlotType != SLOT_USED) && (Slot->SlotType != SLOT_FREE) ){

        WritePrivateProfileString("Main Options", "Position", PosStr[DockOptions.Position], IniFile);
        WritePrivateProfileString("Main Options", "MaxView", (DockOptions.MaxView) ? "Yes" : "No", IniFile);
        WritePrivateProfileString("Main Options", "AlwaysOnTop", (DockOptions.AlwaysOnTop) ? "Yes" : "No", IniFile);
        WritePrivateProfileString("Main Options", "SingleClickStart", (DockOptions.SingleClickStart) ? "Yes" : "No", IniFile);

        WritePrivateProfileString("Main Options", "ClockActive", (DockOptions.ClockActive) ? "Yes" : "No", IniFile);

        WritePrivateProfileString("Main Options", "MailActive", (DockOptions.MailActive) ? "Yes" : "No", IniFile);
        WritePrivateProfileString("Main Options", "MailPath", DockOptions.MailPath, IniFile);
        WritePrivateProfileString("Main Options", "MailSound", (DockOptions.MailSound) ? "Yes" : "No", IniFile);
        sprintf(TmpBuffer, "%d", DockOptions.MailFreq);
        WritePrivateProfileString("Main Options", "MailFreq", TmpBuffer, IniFile);

        sprintf(TmpBuffer, "%d", DockOptions.DockSize);
        WritePrivateProfileString("Main Options", "DockSize", TmpBuffer, IniFile);

        sprintf(TmpBuffer, "%d", DockOptions.DockLeft);
        WritePrivateProfileString("Main Options", "DockLeft", TmpBuffer, IniFile);
        sprintf(TmpBuffer, "%d", DockOptions.DockTop);
        WritePrivateProfileString("Main Options", "DockTop", TmpBuffer, IniFile);

        sprintf(TmpBuffer, "%lu", DockOptions.MailBoxDateTime);
        WritePrivateProfileString("Main Options", "MailBoxDateTime", TmpBuffer, IniFile);
        sprintf(TmpBuffer, "%lu", DockOptions.MailBoxSize);
        WritePrivateProfileString("Main Options", "MailBoxSize", TmpBuffer, IniFile);


        WritePrivateProfileString("Main Options", "WinExitActive", (DockOptions.WinExitActive) ? "Yes" : "No", IniFile);
        WritePrivateProfileString("Main Options", "WinExitConfirm", (DockOptions.WinExitConfirm) ? "Yes" : "No", IniFile);
    }
    else{   // Slot is Used or Free

        if ((Slot->SlotType == SLOT_USED) || (Slot->SlotType == SLOT_FREE)){
            sprintf(SlotName, "Slot %d", Slot->ID);
            WritePrivateProfileString(SlotName, NULL, NULL, IniFile);
        }
        if (Slot->SlotType == SLOT_USED) {
            WritePrivateProfileString(SlotName, NULL, NULL, IniFile);
            WritePrivateProfileString(SlotName, "AppName", Slot->AppName, IniFile);
            WritePrivateProfileString(SlotName, "CmdLine", Slot->CmdLine, IniFile);
            WritePrivateProfileString(SlotName, "RunTimeDir", Slot->RunTimeDir, IniFile);

            sprintf(TmpBuffer, "%d", Slot->StartState);
            WritePrivateProfileString(SlotName, "StartState", TmpBuffer, IniFile);
            WritePrivateProfileString(SlotName, "StartOnTop", (Slot->StartOnTop) ? "Yes" : "No", IniFile);
            sprintf(TmpBuffer, "%d", Slot->WinX);
            WritePrivateProfileString(SlotName, "WinX", TmpBuffer, IniFile);
            sprintf(TmpBuffer, "%d", Slot->WinY);
            WritePrivateProfileString(SlotName, "WinY", TmpBuffer, IniFile);
            sprintf(TmpBuffer, "%d", Slot->WinWidth);
            WritePrivateProfileString(SlotName, "WinWidth", TmpBuffer, IniFile);
            sprintf(TmpBuffer, "%d", Slot->WinHeight);
            WritePrivateProfileString(SlotName, "WinHeight", TmpBuffer, IniFile);

            WritePrivateProfileString(SlotName, "IconFile", Slot->IconFile, IniFile);

            sprintf(TmpBuffer, "%d", Slot->IconPos);
            WritePrivateProfileString(SlotName, "IconPos", TmpBuffer, IniFile);
        }
    }
    return 1;
}

int FindSlotHit(int DockPosition, int XHit, int YHit)
{
    if (DockPosition == DOCK_HORZ) {
        return (XHit / SLOT_BUTTON_WIDTH);
    } 
    else {
        return (YHit / SLOT_BUTTON_HEIGHT);
    }


}

BOOL IsCursorOutOfDock(int X, int Y)
{
    if (DockOptions.Position == DOCK_HORZ) {
        if( (X < (0-(SLOT_BUTTON_WIDTH/2))) ||
            (Y < (0-(SLOT_BUTTON_HEIGHT/2))) ||
            (Y > (SLOT_BUTTON_HEIGHT+(SLOT_BUTTON_HEIGHT/2))) ||
            (X > ((DockOptions.DockSize * SLOT_BUTTON_WIDTH)+(SLOT_BUTTON_WIDTH/2))) ){
            return TRUE;
        }
    } 
    else {
        if( (X < (0-(SLOT_BUTTON_HEIGHT/2))) ||
            (Y < (0-(SLOT_BUTTON_WIDTH/2))) ||
            (X > (SLOT_BUTTON_WIDTH+(SLOT_BUTTON_WIDTH/2)) ) ||
            (Y > ((DockOptions.DockSize * SLOT_BUTTON_HEIGHT)+(SLOT_BUTTON_HEIGHT/2))) ){
            return TRUE;
        }
    }

    return( FALSE );
}


BOOL FAR PASCAL
    AppOptionsDlgProc(HWND hDlg, UINT iMessage, WPARAM wParam, LPARAM lParam)
{
    static BOOL     Grabbing, bWasEmpty;
    static int      NumIcons;
    RECT            rect;
    HWND            hwndGrab, hwndTmp;
    POINT           pt;
    char            TmpBuffer[MAX_FPATH_LEN], *TmpCharPtr;
    int             Status;
    short int       TmpX, TmpY;
    static HCURSOR  hGrabCur;
    static FARPROC  lpfnChooseIconDlgProc, lpfnPreviewerDlgProc;
    DRAWITEMSTRUCT *ItemStruct;
           RECT     rcItem;
           HWND     hwndItem;
           HDC      hdcGrab;
    static HBITMAP  hbmGrab;

    switch (iMessage) {
        case WM_INITDIALOG:

            SendDlgItemMessage(hDlg, QX_FILE, EM_LIMITTEXT, MAX_FPATH_LEN, 0L);
            SendDlgItemMessage(hDlg, QX_CMDLINE, EM_LIMITTEXT, MAX_CMDLINE_LEN, 0L);
            SendDlgItemMessage(hDlg, QX_RUNTIMEDIR, EM_LIMITTEXT, MAX_FPATH_LEN, 0L);
            SendDlgItemMessage(hDlg, QX_ICONFILE, EM_LIMITTEXT, MAX_FPATH_LEN, 0L);

            lpfnChooseIconDlgProc = MakeProcInstance((FARPROC) ChooseIconDlgProc, hAppInst);
            lpfnPreviewerDlgProc  = MakeProcInstance((FARPROC) PreviewerDlgProc,  hAppInst);

			// Copy Selected Slot into Workspace
            memcpy(&TmpSlot, &Slot[CurSlot], sizeof(SLOT_ENTRY));

            // Initialise TmpSlots Scratch Space
            TmpSlot.IconIndex = (gIconWidth * (DOCK_MAX_SIZE));
            // Load the current Iocn into the Scratch Space
            UtilLoadIcon( &TmpSlot );

            switch (TmpSlot.StartState) {

                case START_MINIMUM:
                    CheckRadioButton(hDlg, QX_NORMAL, QX_STORE, QX_MINIMUM);
                    break;

                case START_MAXIMUM:
                    CheckRadioButton(hDlg, QX_NORMAL, QX_STORE, QX_MAXIMUM);
                    break;

                case START_STORE:
                    CheckRadioButton(hDlg, QX_NORMAL, QX_STORE, QX_STORE);
                    break;

                default:
                    CheckRadioButton(hDlg, QX_NORMAL, QX_STORE, QX_NORMAL);
                    break;
            }

            if( TmpSlot.StartOnTop ){
                    CheckRadioButton(hDlg, QX_START_TOP, QX_START_TOP, QX_START_TOP);
            }

            if ( TmpSlot.SlotType == SLOT_USED ){
                NumIcons = TmpSlot.IconTotal;
            }
            else{
                NumIcons = 0;
            }

            SetDlgItemInt(hDlg, QX_ICONNUM, TmpSlot.IconPos + 1, TRUE);
            SetDlgItemInt(hDlg, QX_ICONTOTAL, NumIcons, TRUE);

            /* Set starting values. */
            SetDlgItemText(hDlg, QX_FILE, TmpSlot.AppName);
            SetDlgItemText(hDlg, QX_CMDLINE, TmpSlot.CmdLine);
            SetDlgItemText(hDlg, QX_RUNTIMEDIR, TmpSlot.RunTimeDir);
            SetDlgItemText(hDlg, QX_ICONFILE, TmpSlot.IconFile);

            Grabbing = FALSE;

			/********************************************************
				There is a special condition when an empty slot is
				left clicked and the App name specified
			********************************************************/
			bWasEmpty = (TmpSlot.AppName[0] == '\0');

            return (TRUE);

        case WM_PAINT:
            /** Paint the Grabber bitmap **/
            hwndTmp = GetDlgItem( hDlg, QX_GRABWIN );
            hdcGrab = GetDC( hwndTmp );
            hbmGrab = LoadBitmap(hAppInst, "GRABBER");
            DrawBitmap(hdcGrab, hbmGrab, 0, 0);
            ReleaseDC( hwndTmp, hdcGrab );
            DeleteObject( hbmGrab );
            /** return false to force the default message handler to paint **/
            /** the rest of the diaog box **/
            return(FALSE);

        case WM_DRAWITEM :  // Draw the Current Icon
            ItemStruct = (DRAWITEMSTRUCT *)lParam;
            UtilDrawIcon( ItemStruct->hDC, &TmpSlot, 0, 0 );
            return (TRUE);


        case WM_HSCROLL:
            if (wParam == SB_LINEDOWN)
                if (NumIcons > 0)
                    TmpSlot.IconPos = min(NumIcons - 1, TmpSlot.IconPos + 1);

            if (wParam == SB_LINEUP)
                if (NumIcons > 0)
                    if (TmpSlot.IconPos > 0)
                        TmpSlot.IconPos = TmpSlot.IconPos - 1;

            // Update icon number
            SetDlgItemInt(hDlg, QX_ICONNUM, TmpSlot.IconPos + 1, TRUE);
            // Load the Icon into the slot's own bitmap  & update the IconTotal Field
            UtilLoadIcon( &TmpSlot );
            // ReDraw Icon by invalidating the button area
            hwndItem = GetDlgItem( hDlg, QX_CHOOSEICON );
            GetClientRect( hwndItem, &rcItem );
            InvalidateRect( hwndItem, &rcItem, TRUE );

            return (TRUE);

        case WM_COMMAND:
            switch (wParam) {
                case QX_OK:
                    if (SendDlgItemMessage(hDlg, QX_ICONFILE, EM_GETMODIFY, 0, 0L)) {
                        SendDlgItemMessage(hDlg, QX_ICONFILE, EM_SETMODIFY, FALSE, 0L);
                        GetDlgItemText(hDlg, QX_ICONFILE, TmpSlot.IconFile, MAX_FPATH_LEN);
                        // Load the Icon into the slot's own bitmap & Initialise IconTotal Field
                        TmpSlot.IconPos = 0;
                        UtilLoadIcon( &TmpSlot );
                        if (TmpSlot.IconTotal == 0) {
                            MessageBox(hDlg, "This file contains no icons",
                                "FreeDock", MB_OK | MB_ICONSTOP);
                            TmpSlot.IconPos = Slot[CurSlot].IconPos;
                            strcpy(TmpSlot.IconFile, Slot[CurSlot].IconFile);
                            SetDlgItemText(hDlg, QX_ICONFILE, TmpSlot.IconFile);
	                        UtilLoadIcon( &TmpSlot );
                        } 

                        NumIcons = TmpSlot.IconTotal;

                        SetDlgItemInt(hDlg, QX_ICONNUM, TmpSlot.IconPos + 1, TRUE);
                        SetDlgItemInt(hDlg, QX_ICONTOTAL, NumIcons, TRUE);
                        // ReDraw Icon by invalidating the button area
                        hwndItem = GetDlgItem( hDlg, QX_CHOOSEICON );
                        GetClientRect( hwndItem, &rcItem );
                        InvalidateRect( hwndItem, &rcItem, TRUE );
                        return (TRUE);
                    }

                    /*****************************************
                        Try to fully qualify the pathname 
                    *****************************************/
                    GetDlgItemText(hDlg, QX_FILE, TmpBuffer, MAX_FPATH_LEN);

                    Status = (UINT)FindExecutable(TmpBuffer,
                                                  TmpSlot.RunTimeDir,
                                                  TmpSlot.AppName);
                    if( Status <= 31 ){
                            MessageBox(hDlg, "Error, File or Path Not Found",
                                       "FreeDock Error", MB_OK | MB_ICONSTOP);
                            return(TRUE);
                    }
                    else{
                        SetDlgItemText(hDlg, QX_FILE, TmpSlot.AppName);
                    }
                    
                    GetDlgItemText(hDlg, QX_FILE, TmpSlot.AppName, MAX_FPATH_LEN);
                    GetDlgItemText(hDlg, QX_CMDLINE, TmpSlot.CmdLine, MAX_CMDLINE_LEN);
                    GetDlgItemText(hDlg, QX_RUNTIMEDIR, TmpSlot.RunTimeDir, MAX_FPATH_LEN);
                    GetDlgItemText(hDlg, QX_ICONFILE, TmpSlot.IconFile, MAX_FPATH_LEN);

					/*****************************************************************
						If this is an empty slot which is being filled out by hand
					*****************************************************************/
					if(bWasEmpty){
                        TmpSlot.SlotType = SLOT_USED;
                        TmpSlot.StartState = START_NORMAL;
                        TmpSlot.WinX = DEF_STORE_X;
                        TmpSlot.WinY = DEF_STORE_Y;
                        TmpSlot.WinWidth = DEF_STORE_W;
                        TmpSlot.WinHeight = DEF_STORE_H;
						if( TmpSlot.RunTimeDir[0] == '\0' ){
	                        strcpy(TmpSlot.RunTimeDir, TmpSlot.AppName);
						}
						if( TmpSlot.IconFile[0] == '\0' ){
	                        strcpy(TmpSlot.IconFile, TmpSlot.AppName);
						}
                        TmpSlot.IconPos = 0;

                        /************************************************
                           Set Run Time Dir to be same path as filename
                        ************************************************/
                        TmpCharPtr = strrchr(TmpSlot.RunTimeDir, '\\');
						if (TmpCharPtr != NULL ){
	                        if (*(TmpCharPtr-1) == ':')
    	                        *(TmpCharPtr+1) = '\0';
        	                else{
            	                *TmpCharPtr = '\0';
							}
						}

                        /*****************************************************************
                           Check What type of file has been dropped, is it a .PIF, .COM
                           or a .BAT ?
                        *****************************************************************/
                        if (!stricmp(&TmpSlot.AppName[strlen(TmpSlot.AppName) - 4], ".PIF") ||
                            !stricmp(&TmpSlot.AppName[strlen(TmpSlot.AppName) - 4], ".COM") ||
                            !stricmp(&TmpSlot.AppName[strlen(TmpSlot.AppName) - 4], ".BAT")) {


		                    Status = (UINT)FindExecutable("progman.exe",
    		                                              NULL,
        		                                          TmpSlot.IconFile);
            		        if( Status <= 31 ){
                		            MessageBox(hDlg, "Program Manager not found, cannot select default Icon.",
                    		                   "FreeDock Error", MB_OK | MB_ICONSTOP);
		                    }
                            TmpSlot.IconPos = 1;
                            // Load the Icon into the slot's own bitmap
                            UtilLoadIcon( &TmpSlot );
                        }
                        /******************************************************************
                            Is is a .EXE, if so is it a Windows EXE or a DOS EXE ?
                        ******************************************************************/
                        else if (!stricmp(&TmpSlot.AppName[strlen(TmpSlot.AppName) - 4], ".EXE")) {
                            // Load the Icon into the slot's own bitmap
                            UtilLoadIcon( &TmpSlot );
                            if (TmpSlot.IconTotal == 0) {

			                    Status = (UINT)FindExecutable("progman.exe",
    			                                              NULL,
        			                                          TmpSlot.IconFile);
            			        if( Status <= 31 ){
                			            MessageBox(hDlg, "Program Manager not found, cannot select default Icon.",
                    			                   "FreeDock Error", MB_OK | MB_ICONSTOP);
		                    	}

                                TmpSlot.IconPos = 1;
                                // Load the Icon into the slot's own bitmap
                                UtilLoadIcon( &TmpSlot );

                            }
                        }
                        /******************************************************************
                            It must be a document file, try to find an association for it
                        ******************************************************************/
                        else {
                            strcpy(TmpSlot.CmdLine, TmpSlot.AppName);
                            // Read the executable name in,
                            // then call SearchPath to get entire path for application.
                            Status = (UINT)FindExecutable(TmpSlot.CmdLine,
                                						  TmpSlot.RunTimeDir,
                                						  TmpSlot.AppName);
                            
                            // Now setup the parameter which is still in TmpBuffer
                            strcpy(TmpSlot.CmdLine, TmpBuffer);
                                
                            // Now call find executable to ensure we have a complete
                            // path to the application
                            if( FindExecutable(TmpSlot.AppName,
                                TmpSlot.RunTimeDir,
                                TmpBuffer) > (HINSTANCE)32 ){
	                          	strcpy( TmpSlot.AppName, TmpBuffer);
	                        }
                            
                            if (Status > 32) {
                                strcpy(TmpSlot.IconFile, TmpSlot.AppName);
                                TmpSlot.IconPos = 0;
                                // Load the Icon into the slot's own bitmap
                                UtilLoadIcon( &TmpSlot );
                            } 
                            else {
                                sprintf(TmpBuffer, "%s\nis not an executable or an associated file", TmpSlot.CmdLine);
                                MessageBox(hDlg, TmpBuffer, "FreeDock", MB_OK | MB_ICONSTOP);
                                TmpSlot.SlotType = SLOT_FREE;
                            }
                        }
					}


                    /*************************************************************************
                        If the user has just entered an application name into an empty slot
                        then mark this slot as being in use.
                    *************************************************************************/
                    if (TmpSlot.AppName[0] != '\0'){
                        TmpSlot.SlotType = SLOT_USED;
                    }

                    if (IsDlgButtonChecked(hDlg, QX_MINIMUM))
                        TmpSlot.StartState = START_MINIMUM;
                    else if (IsDlgButtonChecked(hDlg, QX_MAXIMUM))
                        TmpSlot.StartState = START_MAXIMUM;
                    else if (IsDlgButtonChecked(hDlg, QX_STORE))
                        TmpSlot.StartState = START_STORE;
                    else
                        TmpSlot.StartState = START_NORMAL;

                    if (IsDlgButtonChecked(hDlg, QX_START_TOP))
                        TmpSlot.StartOnTop = TRUE;
                    else
                        TmpSlot.StartOnTop = FALSE;

					// Restore Icon Index to it's original value
		            TmpSlot.IconIndex = Slot[CurSlot].IconIndex;

					// Copy modified WorkSpace into Selected Slot
                    memcpy(&Slot[CurSlot], &TmpSlot, sizeof(SLOT_ENTRY));

					// Reload the Icon incase it was changed
                    UtilLoadIcon( &TmpSlot );

                    RePaintSlot(CurSlot, FALSE);
                    FreeProcInstance(lpfnChooseIconDlgProc);
                    EndDialog(hDlg, TRUE);
                    return(TRUE);

                case QX_REMOVE:
                	UtilEmptySlot( &Slot[CurSlot] );
					UpdateINIFile( &Slot[CurSlot] );	// Delete entry in ini file
                    RePaintSlot(CurSlot, TRUE);
                    EndDialog(hDlg, TRUE);
                    return (TRUE);

                case QX_BROWSEICON :
                    Status = BrowseIconFile( hDlg, TmpSlot.IconFile );
                    if(Status){
                        TmpSlot.IconPos = 0;
                        // Load the Icon into the slot's own bitmap  & update the IconTotal Field
                        UtilLoadIcon( &TmpSlot );
                        if (TmpSlot.IconTotal == 0) {
                            MessageBox(hDlg, "This file contains no icons",
                                "FreeDock", MB_OK | MB_ICONSTOP);
                            TmpSlot.IconPos = Slot[CurSlot].IconPos;
                            strcpy(TmpSlot.IconFile, Slot[CurSlot].IconFile);
                            SetDlgItemText(hDlg, QX_ICONFILE, TmpSlot.IconFile);
	                        UtilLoadIcon( &TmpSlot );
                        } 
                        NumIcons = TmpSlot.IconTotal;
                        SetDlgItemInt(hDlg, QX_ICONNUM, (TmpSlot.IconPos+1), TRUE);
                        SetDlgItemInt(hDlg, QX_ICONTOTAL, NumIcons, TRUE);
                        SetDlgItemText(hDlg, QX_ICONFILE, TmpSlot.IconFile);
                        // ReDraw Icon by invalidating the button area
                        hwndItem = GetDlgItem( hDlg, QX_CHOOSEICON );
                        GetClientRect( hwndItem, &rcItem );
                        InvalidateRect( hwndItem, &rcItem, TRUE );
                    }
                    return (TRUE);

                case QX_BROWSEFILE :
                    Status = BrowseFileName( hDlg, TmpSlot.AppName );
                    if(Status){
                        SetDlgItemText(hDlg, QX_FILE, TmpSlot.AppName);
                        strcpy( TmpSlot.IconFile, TmpSlot.AppName );
                        TmpSlot.IconPos = 0;
                        // Load the Icon into the slot's own bitmap & update the IconTotal Field
                        UtilLoadIcon( &TmpSlot );
                        if( TmpSlot.IconTotal == 0 ){
                            // File has no Icons so select PROGMAN.EXE : Icon (2)

		                    Status = (UINT)FindExecutable("progman.exe",
    		                                              NULL,
        		                                          TmpSlot.IconFile);
            		        if( Status <= 31 ){
                		            MessageBox(hDlg, "Program Manager not found, cannot select default Icon.",
                    		                   "FreeDock Error", MB_OK | MB_ICONSTOP);
		                    }
	                        TmpSlot.IconPos = 1;
	                        // Load the Icon into the slot's own bitmap & update the IconTotal Field
    	                    UtilLoadIcon( &TmpSlot );
                        }

                        NumIcons = TmpSlot.IconTotal;
                        SetDlgItemInt(hDlg, QX_ICONNUM, (TmpSlot.IconPos+1), TRUE);
                        SetDlgItemInt(hDlg, QX_ICONTOTAL, NumIcons, TRUE);
                        SetDlgItemText(hDlg, QX_ICONFILE, TmpSlot.IconFile);
                        // ReDraw Icon by invalidating the button area
                        hwndItem = GetDlgItem( hDlg, QX_CHOOSEICON );
                        GetClientRect( hwndItem, &rcItem );
                        InvalidateRect( hwndItem, &rcItem, TRUE );
                    }
                    return (TRUE);

                case QX_CANCEL:
                    FreeProcInstance(lpfnChooseIconDlgProc);
                    EndDialog(hDlg, FALSE);
                    return (TRUE);

				case QX_PREVIEW :
                    Status = DialogBox(hAppInst, "PREVIEWERDLG", hDlg, lpfnPreviewerDlgProc);
					if(Status){
						TmpSlot.StartState = START_STORE;
    	    	        CheckRadioButton(hDlg, QX_NORMAL, QX_STORE, QX_STORE);
					}
					return (TRUE);

                case QX_CHOOSEICON :
                    /*************************************************************
                    	If we are showing a dialog on an empty slot, then don't
                    	allow an icon to be chosen
                    *************************************************************/
                    if ( NumIcons == 0 ) return(TRUE);

                    Status = DialogBox(hAppInst, "CHOOSEICON", hDlg, lpfnChooseIconDlgProc);
                    if(Status){
                        UtilLoadIcon( &TmpSlot );
                        hwndItem = GetDlgItem( hDlg, QX_CHOOSEICON );
                        GetClientRect( hwndItem, &rcItem );
                        InvalidateRect( hwndItem, &rcItem, TRUE );
                        SetDlgItemInt(hDlg, QX_ICONNUM, TmpSlot.IconPos + 1, TRUE);
                    }
                    return (TRUE);
            }
            return (FALSE);
 
  		case WM_LBUTTONDOWN:
                GetWindowRect( hDlg, &rect );
                /** Correct for position of dialogbox       **/
				/** and correct for dialog border & frame **/
                TmpX = (short int)LOWORD(lParam) + (short int)rect.left + gDlgFrameWidth;
                TmpY = (short int)HIWORD(lParam) + (short int)rect.top + gDlgFrameHeight + gDlgTitleHeight;
                /** Get the location of the Grabber Item **/
                hwndTmp = GetDlgItem( hDlg, QX_GRABWIN );
                GetWindowRect( hwndTmp, &rect );
                /** Next see if the button was pressed here **/
                if( (TmpX <= rect.right && TmpX >= rect.left ) &&
                    (TmpY <= rect.bottom && TmpY >= rect.top ) ){
                    /** Then click hit Grabber Item **/
                        SetCapture(hDlg);
                        hGrabCur = LoadCursor(hAppInst, "GrabberCur");
                        SetCursor(hGrabCur);
                        Grabbing = TRUE;
						/** Hide the Application Options Dialog window from the grabber **/
                        SetWindowPos( hDlg, 0, 0,0,0,0,
                                      SWP_HIDEWINDOW|SWP_NOSIZE|SWP_NOMOVE|SWP_NOZORDER );
  			    }
                return (TRUE);
  
        case WM_LBUTTONUP:
            if (!Grabbing)
                return TRUE;
            /********************************************************
               Button click positions are reported relative to the
               Dialog window, so correct for this by adding the x,y
               coordinates of the top left corner of the dialog to
               the click position. Also correct for the dialog
			   frame and title bar sizes.
            ********************************************************/
            GetWindowRect(hDlg, &rect);
            pt.x = rect.left + (short int)LOWORD(lParam) + gDlgFrameWidth;
            pt.y = rect.top + (short int)HIWORD(lParam) + gDlgFrameHeight + gDlgTitleHeight;
            hwndGrab = WindowFromPoint(pt);

			/** Show the the dialog window again **/
            SetWindowPos( hDlg, 0, 0,0,0,0,
                          SWP_SHOWWINDOW|SWP_NOSIZE|SWP_NOMOVE|SWP_NOZORDER );

            /* Climb back up to the top of a parent-child list */
            hwndTmp = GetParent(hwndGrab);
            while (hwndTmp != NULL) {
                hwndGrab = hwndTmp;
                hwndTmp = GetParent(hwndGrab);
            }
            GetWindowRect(hwndGrab, &rect);
            TmpSlot.WinX = rect.left;
            TmpSlot.WinY = rect.top;
            TmpSlot.WinWidth = rect.right - rect.left;
            TmpSlot.WinHeight = rect.bottom - rect.top;

            if( hwndGrab == GetDesktopWindow() ){
                MessageBox(hDlg, "Oops, that's the Desktop window, please try again.",
                    "FreeDock", MB_OK | MB_ICONINFORMATION);
            }
            else if( hwndGrab == hwndDock ){
                MessageBox(hDlg, "Oops, that's a FreeDock window, please try again.",
                    "FreeDock", MB_OK | MB_ICONINFORMATION);
            }
            else{
                MessageBox(hDlg, "Window Grabbed Successfully.",
                    "FreeDock", MB_OK | MB_ICONINFORMATION);
				TmpSlot.StartState = START_STORE;
                CheckRadioButton(hDlg, QX_NORMAL, QX_STORE, QX_STORE);
            }

            Grabbing = FALSE;
            ReleaseCapture();
            SetCursor(LoadCursor(NULL, IDC_ARROW));
            DestroyCursor(hGrabCur);

            return (TRUE);
    }

    return FALSE;
}


BOOL FAR PASCAL 
    MainOptionsDlgProc(HWND hDlg, UINT iMessage, WPARAM wParam, LPARAM lParam)
{

    static int 		OrigPosn;
    static int 		OrigDockSize;
	extern FARPROC  lpfnAboutDockDlgProc;	// Defined in dockwin.c

    switch (iMessage) {

        case WM_INITDIALOG:
            OrigPosn = DockOptions.Position + QX_HORZ;  // Save original position incase user cancels
            OrigDockSize = DockOptions.DockSize;

            CheckDlgButton(hDlg, QX_MAILCHECK, DockOptions.MailActive);
            CheckDlgButton(hDlg, QX_WINEXIT, DockOptions.WinExitActive);
            CheckDlgButton(hDlg, QX_CLOCK, DockOptions.ClockActive);
            CheckDlgButton(hDlg, QX_ON_TOP, DockOptions.AlwaysOnTop);
            CheckDlgButton(hDlg, QX_MAX_VIEW, DockOptions.MaxView);
            CheckDlgButton(hDlg, QX_SINGLECLICK, DockOptions.SingleClickStart);
            CheckRadioButton(hDlg, QX_HORZ, QX_VERT, OrigPosn);
            SetDlgItemInt(hDlg, QX_DOCK_SIZE, DockOptions.DockSize, TRUE);
            return (TRUE);

        case WM_VSCROLL:
            if (wParam == SB_LINEUP)
                if ( DockOptions.DockSize < DOCK_MAX_SIZE){
                    DockOptions.DockSize++;
                    SetDlgItemInt(hDlg, QX_DOCK_SIZE, DockOptions.DockSize, TRUE);
                }

            if (wParam == SB_LINEDOWN)
                if ( DockOptions.DockSize > 1){
                    DockOptions.DockSize--;
                    SetDlgItemInt(hDlg, QX_DOCK_SIZE, DockOptions.DockSize, TRUE);
                }

			SetDockWinPos( DockOptions.Position + QX_HORZ );
            return (TRUE);

        case WM_COMMAND:
            switch (wParam) {
                case QX_OK:
                    DockOptions.MailActive = IsDlgButtonChecked(hDlg, QX_MAILCHECK);
                    DockOptions.WinExitActive = IsDlgButtonChecked(hDlg, QX_WINEXIT);
                    DockOptions.ClockActive = IsDlgButtonChecked(hDlg, QX_CLOCK);
                    DockOptions.AlwaysOnTop = IsDlgButtonChecked(hDlg, QX_ON_TOP);
                    DockOptions.MaxView = IsDlgButtonChecked(hDlg, QX_MAX_VIEW);
                    DockOptions.SingleClickStart = IsDlgButtonChecked(hDlg, QX_SINGLECLICK);

                    /**************************************************************
                        Handle Always On Top Option
                    **************************************************************/
                    if (DockOptions.AlwaysOnTop) {
                        SetWindowPos(hwndDock, HWND_TOPMOST, 0, 0, 0, 0,
                            SWP_NOMOVE | SWP_NOSIZE | SWP_NOACTIVATE);
                    } 
                    else {
                        SetWindowPos(hwndDock, HWND_BOTTOM, 0, 0, 0, 0,
                            SWP_NOMOVE | SWP_NOSIZE | SWP_NOACTIVATE);
                    }

                    EndDialog(hDlg, TRUE);
                    break;

                    /****************************************************************
                        Handle Dock Position Options, for updating dock pos as the
                        user choses the radio button.
                    ****************************************************************/
                case QX_HORZ:
                case QX_VERT:
                    SetDockWinPos( wParam );
                    break;

                case QX_CANCEL:
                    DockOptions.DockSize = OrigDockSize ;
					SetDockWinPos( OrigPosn ); // restore original position
                    EndDialog(hDlg, FALSE);
                    break;

				case QX_ABOUT:
                    DialogBox(hAppInst, "ABOUTDOCKDLG", hDlg, lpfnAboutDockDlgProc);
					break;
            }
            break;

    }

    return FALSE;
}


BOOL FAR PASCAL 
    WinExitOptionsDlgProc(HWND hDlg, UINT iMessage, WPARAM wParam, LPARAM lParam)
{

    switch (iMessage) {
        case WM_INITDIALOG:
            CheckDlgButton(hDlg, QX_WINEXITCONFIRM, DockOptions.WinExitConfirm);
            return (TRUE);

        case WM_COMMAND:
            switch (wParam) {
                case QX_OK:
                    DockOptions.WinExitConfirm = IsDlgButtonChecked(hDlg, QX_WINEXITCONFIRM);
                    EndDialog(hDlg, TRUE);
                    break;

                case QX_CANCEL:
                    EndDialog(hDlg, FALSE);
                    break;
            }
            break;
    }
    return FALSE;
}


BOOL FAR PASCAL 
    MailOptionsDlgProc(HWND hDlg, UINT iMessage, WPARAM wParam, LPARAM lParam)
{
    BOOL            bTxlated;

    switch (iMessage) {
        case WM_INITDIALOG:
            SendDlgItemMessage(hDlg, QX_MAILPATH, EM_LIMITTEXT, MAX_FPATH_LEN, 0L);
            SetDlgItemText(hDlg, QX_MAILPATH, DockOptions.MailPath);
            SetDlgItemInt(hDlg, QX_MAILFREQ, DockOptions.MailFreq, TRUE);
            CheckDlgButton(hDlg, QX_MAILSOUND, DockOptions.MailSound);
            return (TRUE);

        case WM_COMMAND:
            switch (wParam) {
                case QX_OK:
                    GetDlgItemText(hDlg, QX_MAILPATH, DockOptions.MailPath, MAX_FPATH_LEN);
                    DockOptions.MailFreq = GetDlgItemInt(hDlg, QX_MAILFREQ, &bTxlated, FALSE);
                    if (!bTxlated) {
                        MessageBox(hDlg, "Not a valid integer!", "FreeDock", MB_OK | MB_ICONSTOP);
                        SendDlgItemMessage(hDlg, QX_MAILFREQ, EM_SETSEL, 0, MAKELONG(0, -1));
                        return (TRUE);
                    }
                    DockOptions.MailSound = IsDlgButtonChecked(hDlg, QX_MAILSOUND);
                    KillTimer(hwndDock, MAIL_TIMER);
                    SetTimer(hwndDock, MAIL_TIMER, DockOptions.MailFreq * 1000 * 60, NULL);
                    EndDialog(hDlg, TRUE);
                    break;

                case QX_CANCEL:
                    EndDialog(hDlg, FALSE);
                    break;
            }
            break;
    }
    return FALSE;
}


BOOL FAR PASCAL 
    ClockOptionsDlgProc(HWND hDlg, UINT iMessage, WPARAM wParam, LPARAM lParam)
{
    switch (iMessage) {
        case WM_INITDIALOG:
            return (TRUE);

        case WM_COMMAND:
            if (wParam == QX_OK)
                EndDialog(hDlg, TRUE);
            break;
    }
    return FALSE;
}


BOOL FAR PASCAL 
    AboutDockDlgProc(HWND hDlg, UINT iMessage, WPARAM wParam, LPARAM lParam)
{
	static HICON hIcon;

    switch (iMessage) {
        case WM_INITDIALOG:
            SetDlgItemText(hDlg, QX_VERSION, VersionText);
            SetDlgItemText(hDlg, QX_OS_BITS, OS_Bits);
#ifdef WIN32
			hIcon = LoadIcon(hAppInst, "FREEDOCK32");
#else
			hIcon = LoadIcon(hAppInst, "FREEDOCK16");
#endif
            SendDlgItemMessage(hDlg, QX_ICON1, STM_SETICON, (WPARAM)hIcon, 0L);
            return (TRUE);

        case WM_COMMAND:
            switch (wParam) {

                case QX_OK:
                    EndDialog(hDlg, TRUE);
                    break;

				case QX_EXIT:
                    //            nItem = MessageBox(hDlg, "Do you really want to exit FreeDock ?",
                    //                               "FreeDock", MB_OKCANCEL | MB_ICONSTOP);

                    //            if (nItem == IDOK) {
                    UpdateINIFile( &Slot[0]);
                    PostQuitMessage(0);
                    //            }
                    return (TRUE);
            }
    }
    return FALSE;
}


BOOL FAR PASCAL BrowseIconFile( HWND hwnd, char *FileName )
{
    OPENFILENAME ofn;
    char DirName[256], FileTitle[256], TmpFileName[MAX_FPATH_LEN];
    char Filter[] = "Executables (*.EXE)\0*.EXE\0Icon Files (*.ICO)\0*.ICO\0DLL's (*.DLL)\0*.DLL\0\0";
    BOOL Status;
    char *TmpPtr;

    memset( &ofn, 0, sizeof(OPENFILENAME) );

    strcpy( TmpFileName, FileName );
    strcpy( DirName, FileName );
    TmpPtr = strrchr(DirName, '\\');
    if(TmpPtr != NULL){
        *TmpPtr = '\0';
    }

    ofn.lpstrTitle   = "Select an Icon File";
    ofn.lStructSize  = sizeof(OPENFILENAME);
    ofn.hwndOwner    = hwnd;
    ofn.lpstrFilter  = Filter;
    ofn.nFilterIndex = 1;
    ofn.lpstrFile = TmpFileName;
    ofn.nMaxFile = MAX_FPATH_LEN;
    ofn.lpstrFileTitle = FileTitle;
    ofn.nMaxFileTitle = sizeof(FileTitle);
    ofn.lpstrInitialDir = DirName;
    ofn.Flags = OFN_HIDEREADONLY | OFN_PATHMUSTEXIST | OFN_FILEMUSTEXIST;

    Status = GetOpenFileName(&ofn);

    // If no change made, this is the same as cancel
    if(!stricmp( FileName, TmpFileName ))
        return(FALSE);

    if(Status){
        strcpy( FileName, TmpFileName );
    }

    return(Status);
}


BOOL FAR PASCAL BrowseFileName( HWND hwnd, char *FileName )
{
    OPENFILENAME ofn;
    char DirName[256], FileTitle[256], TmpFileName[MAX_FPATH_LEN];
    char Filter[] = "Executables (*.EXE)\0*.EXE\0Batch Files (*.BAT)\0*.BAT\0PIF Files (*.PIF)\0*.PIF\0DOS .COM Files (*.COM)\0*.COM\0\0";
    BOOL Status;
    char *TmpPtr;

    memset( &ofn, 0, sizeof(OPENFILENAME) );

    strcpy( TmpFileName, FileName );
    strcpy( DirName, FileName );
    TmpPtr = strrchr(DirName, '\\');
    if(TmpPtr != NULL){
        *TmpPtr = '\0';
    }

    ofn.lpstrTitle   = "Select a File";
    ofn.lStructSize  = sizeof(OPENFILENAME);
    ofn.hwndOwner    = hwnd;
    ofn.lpstrFilter  = Filter;
    ofn.nFilterIndex = 1;
    ofn.lpstrFile = TmpFileName;
    ofn.nMaxFile = MAX_FPATH_LEN;
    ofn.lpstrFileTitle = FileTitle;
    ofn.nMaxFileTitle = sizeof(FileTitle);
    ofn.lpstrInitialDir = DirName;
    ofn.Flags = OFN_HIDEREADONLY | OFN_PATHMUSTEXIST | OFN_FILEMUSTEXIST;

    Status = GetOpenFileName(&ofn);

    // If no change made, this is the same as cancel
    if(!stricmp( FileName, TmpFileName ))
        return(FALSE);

    if(Status){
        strcpy( FileName, TmpFileName );
    }

    return(Status);
}


BOOL FAR PASCAL 
    ChooseIconDlgProc(HWND hDlg, UINT iMessage, WPARAM wParam, LPARAM lParam)
{
    // Slot info passed in via global TmpSlot struct
    // CurIcon and CurPage are 0 relative
    static int NumIcons, CurIcon, NumPages, CurPage;
    static HANDLE *Icon;          // pointer to array of icon handles
    RECT rect;
    DRAWITEMSTRUCT *ItemStruct;
    int i, ItemNum;

    switch (iMessage) {
        case WM_INITDIALOG:
            // find number of icons & create array to hold ALL handles
            NumIcons = TmpSlot.IconTotal;
            Icon = (HANDLE *)malloc( NumIcons * sizeof(HANDLE) );

            // Read all Icon handles
            for( i=0; i< NumIcons ; i++){
                Icon[i] = ExtractIcon(hAppInst, TmpSlot.IconFile, i);
            }

            CurIcon = TmpSlot.IconPos;
            NumPages = (NumIcons / 50) + 1;
            CurPage = (CurPage / 50);

            SetDlgItemInt(hDlg, QX_ICONNUM, (CurIcon+1), TRUE);
            SetDlgItemInt(hDlg, QX_ICONTOTAL, NumIcons, TRUE);
            SetDlgItemInt(hDlg, QX_PAGENUM, (CurPage+1), TRUE);
            SetDlgItemInt(hDlg, QX_PAGETOTAL, NumPages, TRUE);
            SendDlgItemMessage(hDlg, QX_ICONBOX, STM_SETICON, (WPARAM)Icon[CurIcon], 0L);
            return (TRUE);

        case WM_DRAWITEM :
            ItemStruct = (DRAWITEMSTRUCT *)lParam;
            ItemNum = (ItemStruct->CtlID-QX_ICON1) + (CurPage*50);
            if(ItemNum < NumIcons){
                DrawIcon( ItemStruct->hDC, 0, 0, Icon[ItemNum]);
            }
            return (TRUE);

        case WM_HSCROLL:
            if (wParam == SB_LINEDOWN)
                if (CurPage < (NumPages-1)){
                    CurPage = CurPage + 1;
                    GetClientRect( hDlg, &rect );
                    for(i = QX_ICON1; i < QX_ICON1+50; i++){
                        InvalidateRect( GetDlgItem( hDlg, i) , &rect, TRUE );
                    }
                }

            if (wParam == SB_LINEUP)
                if (CurPage > 0){
                    CurPage = CurPage - 1;
                    GetClientRect( hDlg, &rect );
                    for(i = QX_ICON1; i < QX_ICON1+50; i++){
                        InvalidateRect( GetDlgItem( hDlg, i) , &rect, TRUE );
                    }
                }

            SetDlgItemInt(hDlg, QX_PAGENUM, CurPage + 1, TRUE);

            return (TRUE);

        case WM_COMMAND:
            if (wParam == QX_OK){
                TmpSlot.IconPos = CurIcon;
                free( Icon );
                EndDialog(hDlg, TRUE);
            }
            else if (wParam == QX_CANCEL){
                free( Icon );
                EndDialog(hDlg, FALSE);
            } 
            else if( (wParam >= QX_ICON1) && (wParam <= QX_ICON1+50) ){
                CurIcon = min( (int)(NumIcons-1), (int)((wParam - QX_ICON1) + (CurPage*50)));
                SetDlgItemInt(hDlg, QX_ICONNUM, (CurIcon+1), TRUE);
                SendDlgItemMessage(hDlg, QX_ICONBOX, STM_SETICON, (WPARAM)Icon[CurIcon], 0L);
            }
            break;
    }
    return FALSE;
}

/**************************************************************************
    ExecSlot.

    This function handles the execution of a program associated
    with a slot, it takes the Slot index so it can find the program
    to execute, it also takes the command line to pass, to enable
    this function to be used to execute a program as a result of
    a drag and drop action.
    In the case of drag&drop, the caller must create the command line
    before calling this function.
**************************************************************************/

void ExecSlot( int SlotHit, char *CmdLine ){

    int                 StartState;
    LPARAM              hStartAppInst;
    UINT                nItem;
    BOOL                bStartedApp = FALSE;

#ifdef WIN32
	STARTUPINFO         StartupInfo;
    PROCESS_INFORMATION ProcInfo;
    char    TmpBuffer[MAX_FPATH_LEN + MAX_CMDLINE_LEN + 1];
#endif

    switch (Slot[SlotHit].StartState) {

        case START_MINIMUM:
            StartState = SW_SHOWMINIMIZED;
            break;

        case START_MAXIMUM:
            StartState = SW_SHOWMAXIMIZED;
            break;

        case START_NORMAL:
        case START_STORE :
            StartState = SW_SHOWNORMAL;
            break;

        default: return;
    }

    if(Slot[SlotHit].StartOnTop){
        StartState = StartState | SW_SHOWMAXIMIZED;
    }


	/**********************************************
		Check if the working directory exists
		if not, return.
	**********************************************/
#ifdef WIN32
	if( !SetCurrentDirectory( Slot[SlotHit].RunTimeDir ) ){
#else
	if( _chdir( Slot[SlotHit].RunTimeDir ) ){
#endif
        MessageBox( NULL, "The Runtime Directory for this application cannot be found.\r\r\n Cannot Start Application",
					"FreeDock Error",
                    MB_ICONSTOP | MB_OK); 
		return;
	}

    SetCursor(LoadCursor(NULL, IDC_WAIT));
    /**********************************************
        Need to start up the application in a
        diferent manner under NT so we can 
        keep track of it and find the window
        it creates inorder to re-position
        it or set it on top
    *********************************************/
#ifdef WIN32

    memset( &StartupInfo, '\0', sizeof(STARTUPINFO) );
    memset( &ProcInfo,    '\0', sizeof(PROCESS_INFORMATION) );
    StartupInfo.cb = sizeof(StartupInfo);

	/***********************************************************
		Setup the window Pos, size at startup time
		if required (only WIN32 supports this option)
		Can't specify that a window is always on top though,
		so we still use the EnumWindows function for this,
	***********************************************************/    
    if (Slot[SlotHit].StartState == START_STORE){
		StartupInfo.dwX		= Slot[SlotHit].WinX;
		StartupInfo.dwY		= Slot[SlotHit].WinY;
		StartupInfo.dwXSize	= Slot[SlotHit].WinWidth;
		StartupInfo.dwYSize	= Slot[SlotHit].WinHeight;
		StartupInfo.dwFlags = STARTF_USEPOSITION | STARTF_USESIZE;
	}

    // Create Full Command Line
    sprintf( TmpBuffer, "%s %s", Slot[SlotHit].AppName, CmdLine );
	/*****************************************************************
		For some wierd reason, the help for CreateProcess states
		that if the program is a windows app and has params then
		the program name and command line should both be passed
		as the command line with module name set to NULL. But if
		there is no command line, then the module name is passed
		in the module name field and the command line is NULL
	*****************************************************************/
    bStartedApp = CreateProcess( NULL, TmpBuffer,
    						//	 (CmdLine[0]=='\0')?TmpBuffer:NULL,
                            //     (CmdLine[0]=='\0')?NULL:TmpBuffer,
                                 NULL,                 // default process security
                                 NULL,                 // default thread security
                                 FALSE,                // don't inherit handles
                                 0L,			       // No special startup flags
                                 NULL,                 // Inherit the ENVIRONMENT
                                 Slot[SlotHit].RunTimeDir,  // Current Dir
                                 &StartupInfo,         // Process Startup Info
                                 &ProcInfo );          // Recieves Process data
    /** Save the process ID to pass into the EnumWindows func later if reqd **/
    hStartAppInst = (LPARAM)ProcInfo.dwProcessId;
    
#else
    hStartAppInst = (LPARAM)ShellExecute(GetDesktopWindow(),
        "open",
        Slot[SlotHit].AppName,
        CmdLine,
        Slot[SlotHit].RunTimeDir,
        StartState);

    if (hStartAppInst > 33){
        bStartedApp = TRUE;
    }
#endif

    /** No Sleep() in 16bit windows, and it does not seem to be reqd **/
#ifdef WIN32
    /** Have a kip to give the window a chance to appear **/
    Sleep(500);
#endif

    /** Check for an error starting the application **/
    if( !bStartedApp ){
        MessageBox( hwndDock, "Error starting application.",
                    "FreeDock Error", MB_OK | MB_ICONSTOP);
        SetCursor(LoadCursor(NULL, IDC_ARROW));
        return;
    }
    
	/** No need to try to reposition window in NT, it's done already **/
#ifdef WIN32
    if( Slot[SlotHit].StartOnTop ){               
#else
    if( (Slot[SlotHit].StartState == START_STORE) || 
        (Slot[SlotHit].StartOnTop)){               
#endif   
        /** Copy Slot into TmpSlot to pass to callback fuction **/
        memcpy( &TmpSlot, &Slot[SlotHit], sizeof(SLOT_ENTRY) );

        bFoundWindow = FALSE;	// Set Call back function success flag to fail

        do{
            nItem = IDCANCEL;
               EnumWindows( UtilSetupStartedWin, (LPARAM)hStartAppInst );
               if(!bFoundWindow){
                    nItem = MessageBox(hwndDock, "Could not locate window for started application. Retry ?",
                            "Freedock Error", MB_RETRYCANCEL | MB_ICONSTOP);
               }
        }while( nItem != IDCANCEL );
    }

    SetCursor(LoadCursor(NULL, IDC_ARROW));

    return;
}


void SetDockWinPos( int DockPos )
{

	// This fucntion correctly positions the dock on the desktop, 
	// allowing for the current orientation

	// Limit dock position to actual screen
	DockOptions.DockLeft = max(DockOptions.DockLeft, 0);
	DockOptions.DockTop = max(DockOptions.DockTop, 0);

	DockOptions.DockLeft = min(DockOptions.DockLeft, (gScreenWidth - SLOT_BUTTON_WIDTH));
	DockOptions.DockTop = min(DockOptions.DockTop, (gScreenHeight - SLOT_BUTTON_HEIGHT));

    switch( DockPos ){

        case QX_HORZ:
            DockOptions.Position = DOCK_HORZ;
            SetWindowPos(hwndDock,
                HWND_TOP,
                DockOptions.DockLeft,
                DockOptions.DockTop,
                DockOptions.DockSize * SLOT_BUTTON_WIDTH + 2,
                SLOT_BUTTON_HEIGHT + 2,
                SWP_NOACTIVATE | SWP_SHOWWINDOW);
            break;

        default :
            DockOptions.Position = DOCK_VERT;
            SetWindowPos(hwndDock,
                HWND_TOP,
                DockOptions.DockLeft,
                DockOptions.DockTop,
                SLOT_BUTTON_WIDTH + 2,
                DockOptions.DockSize * SLOT_BUTTON_HEIGHT + 2,
                SWP_NOACTIVATE | SWP_SHOWWINDOW);
            break;

    }
}


void UtilLoadIcon( SLOT_ENTRY *Slot ){

    RECT	rect;
    HICON   IconHandle;

    rect.top = 0;
    rect.left = Slot->IconIndex;
    rect.bottom = gIconHeight;
    rect.right = Slot->IconIndex + gIconWidth;

    Slot->IconTotal = 0;
    
    switch( Slot->SlotType ){

        case SLOT_SPECIAL_TITLE :
#ifdef WIN32
            IconHandle = LoadIcon(hAppInst, "FREEDOCK32");
#else
            IconHandle = LoadIcon(hAppInst, "FREEDOCK16");
#endif
            break;

        case SLOT_SPECIAL_MAIL :
        	if( DockOptions.bMailInBox ){
	            IconHandle = LoadIcon(hAppInst, "MAILICON");
        	}
        	else{
            	IconHandle = LoadIcon(hAppInst, "NOMAILICON");
            }
            break;

        case SLOT_SPECIAL_CLOCK :
            IconHandle = LoadIcon(hAppInst, "CLOCKICON");
            break;

        case SLOT_SPECIAL_EXIT :
            IconHandle = LoadIcon(hAppInst, "EXITICON");
            break;
    
        default:
            Slot->IconTotal  = (int)ExtractIcon(hAppInst, Slot->IconFile, -1);
            IconHandle = (HICON)ExtractIcon(hAppInst, Slot->IconFile, Slot->IconPos);
            break;
    }

	FillRect( DockOptions.hdcIconCache, &rect, GetStockObject(LTGRAY_BRUSH) );
    DrawIcon( DockOptions.hdcIconCache, Slot->IconIndex, 0, IconHandle );
}


/**********************************************************************
	Funrtion to set all the variable fields in an emptied slot to
	their default values. Does not change ID or IconIndex fields
	since they are assigned at startup and cannot be changed.

	Also erases the Icon in the IconCache.
**********************************************************************/

void UtilEmptySlot( SLOT_ENTRY *Slot ){

    RECT	rect;

    rect.top = 0;
    rect.left = Slot->IconIndex;
    rect.bottom = gIconHeight;
    rect.right = Slot->IconIndex + gIconWidth;
 
    Slot->SlotType		= SLOT_FREE;
    Slot->AppName[0]	= '\0';
    Slot->CmdLine[0]	= '\0';
    Slot->RunTimeDir[0]	= '\0';
    Slot->StartState	= START_NORMAL;
    Slot->WinX			= 0;
    Slot->WinY			= 0;
    Slot->WinWidth		= DEFAULT_WIN_W;
    Slot->WinHeight		= DEFAULT_WIN_H;
    Slot->IconFile[0]	= '\0';
    Slot->IconPos		= 0;
    Slot->IconTotal		= 0;

	FillRect( DockOptions.hdcIconCache, &rect, GetStockObject(LTGRAY_BRUSH) );
}


BOOL CALLBACK UtilSetupStartedWin( HWND hwnd, LPARAM lParam ){


	/****************************************************
		Because Application Instances are not unique
		under WIN32 / Win NT we must locate the app
		just started in a diferent manner depending on
		the version of Windows in use
	****************************************************/

#ifdef WIN32
	{
    DWORD     ThreadId;
    DWORD     ProcessId = 1L;       // Must be non-NULL to recieve Process ID

    ThreadId = GetWindowThreadProcessId( hwnd, &ProcessId );
    if( ProcessId == (DWORD)lParam ){
        bFoundWindow = TRUE;
    }
    }
#else
    if( GetWindowWord( hwnd, GWW_HINSTANCE ) == (HINSTANCE)lParam ){
        bFoundWindow = TRUE;
    }
#endif

    if( bFoundWindow ){

		// Window position is setup by CreateProcess if requried
        if(TmpSlot.StartState == START_STORE){
            /*****************************
            	Re-position the window 
            *****************************/
            SetWindowPos(hwnd, HWND_NOTOPMOST, TmpSlot.WinX, TmpSlot.WinY,
                    TmpSlot.WinWidth, TmpSlot.WinHeight, SWP_NOZORDER);
        }

        if( TmpSlot.StartOnTop ){
            /*****************************
                Put the window on top 
            *****************************/
            SetWindowPos(hwnd, HWND_TOPMOST, 0, 0, 0, 0, 
                         SWP_NOMOVE | SWP_NOSIZE | SWP_NOACTIVATE);
        }
    }
    // returning FALSE keeps enumeration going, TRUE stops it
    return (!bFoundWindow);
}



BOOL FAR PASCAL PreviewerDlgProc(HWND hDlg, UINT iMessage, WPARAM wParam, LPARAM lParam)
{

	static RECT rect;

    switch (iMessage) {
        case WM_INITDIALOG:
			SetWindowPos(hDlg, 0L, 
						 TmpSlot.WinX,
						 TmpSlot.WinY,
						 TmpSlot.WinWidth,
						 TmpSlot.WinHeight,
						 SWP_SHOWWINDOW | SWP_NOZORDER);
            return (TRUE);

        case WM_COMMAND:
            switch (wParam) {
                case QX_OK:
		            GetWindowRect(hDlg, &rect);
    	        	TmpSlot.WinX = rect.left;
        		    TmpSlot.WinY = rect.top;
        	    	TmpSlot.WinWidth = rect.right - rect.left;
		            TmpSlot.WinHeight = rect.bottom - rect.top;
                    EndDialog(hDlg, TRUE);
                    break;

                case QX_CANCEL:
                    EndDialog(hDlg, FALSE);
                    break;
            }
            break;
    }
    return FALSE;
}


/***************************************
	Swaps the contents of two slots
	(Only swaps non-variable fields)
	not ID
***************************************/
void SwapSlots( SLOT_ENTRY *Slot1, SLOT_ENTRY *Slot2 ){

	int 		Slot1_ID, Slot2_ID;
	SLOT_ENTRY  WorkingSlot;

	/****************************************
		Store original Slot IDs for later
	****************************************/
	Slot1_ID = Slot1->ID;
	Slot2_ID = Slot2->ID;

	/*********************************
		Copy Slot 1 to Working Slot
	*********************************/
	memcpy( &WorkingSlot, Slot1, sizeof(SLOT_ENTRY) );

	/*********************************
		Copy Slot 2 to  Slot1
	*********************************/
    memcpy( Slot1, Slot2, sizeof(SLOT_ENTRY) );

	/*********************************
		Copy Working Slot to Slot 2
	*********************************/
    memcpy( Slot2, &WorkingSlot, sizeof(SLOT_ENTRY) );

	/***************************************
		Now restore the correct values
		to the ID fields in the Slots
	***************************************/
	Slot1->ID = Slot1_ID;
	Slot2->ID = Slot2_ID;

	return;
}
