/* lzo_util.c -- utilities for the the LZO library

   This file is part of the LZO real-time data compression library.

   Copyright (C) 1996 Markus Franz Xaver Johannes Oberhumer

   The LZO library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   The LZO library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the LZO library; see the file COPYING.LIB.
   If not, write to the Free Software Foundation, Inc.,
   675 Mass Ave, Cambridge, MA 02139, USA.

   Markus F.X.J. Oberhumer
   markus.oberhumer@jk.uni-linz.ac.at
 */


#include <stdio.h>

#define NDEBUG
#include <assert.h>
#include "lzo_conf.h"
#include "lzo_util.h"


LZO_EXTERN(lzo_uint)
_lzo_fread(FILE *f, lzo_voidp s, lzo_uint len);


/***********************************************************************
// slow but portable <string.h> stuff, only used in assertions
************************************************************************/

int lzo_memcmp(const lzo_voidp s1, const lzo_voidp s2, lzo_uint len)
{
#if (LZO_UINT_MAX <= UINT_MAX)
	return memcmp(s1,s2,len);
#else
	const lzo_byte *p1 = (const lzo_byte *) s1;
	const lzo_byte *p2 = (const lzo_byte *) s2;
	int d;

	if (len > 0) do
	{
		d = *p1 - *p2;
		if (d != 0)
			return d;
		p1++;
		p2++;
	}
	while (--len > 0);
	return 0;
#endif
}


lzo_voidp lzo_memcpy(lzo_voidp dest, const lzo_voidp src, lzo_uint len)
{
#if (LZO_UINT_MAX <= UINT_MAX)
	return memcpy(dest,src,len);
#else
	lzo_byte *p1 = (lzo_byte *) dest;
	const lzo_byte *p2 = (const lzo_byte *) src;

	if (len > 0) do
		*p1++ = *p2++;
	while (--len > 0);
	return dest;
#endif
}


lzo_voidp lzo_memmove(lzo_voidp dest, const lzo_voidp src, lzo_uint len)
{
#if (LZO_UINT_MAX <= UINT_MAX)
	return memmove(dest,src,len);
#else
	lzo_byte *p1 = (lzo_byte *) dest;
	const lzo_byte *p2 = (const lzo_byte *) src;

	if (len <= 0 || p1 == p2)
		return dest;

	if (p1 < p2)
	{
		do
			*p1++ = *p2++;
		while (--len > 0);
	}
	else
	{
		p1 += len;
		p2 += len;
		do
			*--p1 = *--p2;
		while (--len > 0);
	}
	return dest;
#endif
}


lzo_voidp lzo_memset(lzo_voidp s, int c, lzo_uint len)
{
#if (LZO_UINT_MAX <= UINT_MAX)
	return memset(s,c,len);
#else
	lzo_byte *p = (lzo_byte *) s;

	if (len > 0) do
		*p++ = LZO_BYTE(c);
	while (--len > 0);
	return s;
#endif
}


/***********************************************************************
// adler32 checksum - adapted from free code by Mark Adler
// http://quest.jpl.nasa.gov/zlib
************************************************************************/

#define BASE 65521u /* largest prime smaller than 65536 */
#define NMAX 5552
/* NMAX is the largest n such that 255n(n+1)/2 + (n+1)(BASE-1) <= 2^32-1 */

#define DO1(buf,i)  {s1 += buf[i]; s2 += s1;}
#define DO2(buf,i)  DO1(buf,i); DO1(buf,i+1);
#define DO4(buf,i)  DO2(buf,i); DO2(buf,i+2);
#define DO8(buf,i)  DO4(buf,i); DO4(buf,i+4);
#define DO16(buf)   DO8(buf,0); DO8(buf,8);

lzo_uint lzo_adler32(lzo_uint adler, const lzo_byte *buf, lzo_uint len)
{
	lzo_uint s1 = adler & 0xffff;
	lzo_uint s2 = (adler >> 16) & 0xffff;
	int k;

	if (buf == NULL)
		return 1;

	while (len > 0)
	{
		k = len < NMAX ? (int) len : NMAX;
		len -= k;
		while (k >= 16) 
		{
			DO16(buf);
			buf += 16;
			k -= 16;
		}
		if (k != 0) do 
		{
			s1 += *buf++;
			s2 += s1;
        } while (--k);
		s1 %= BASE;
		s2 %= BASE;
	}
	return (s2 << 16) | s1;
}

#undef BASE
#undef NMAX
#undef DO1
#undef DO2
#undef DO4
#undef DO8
#undef DO16


/***********************************************************************
// read from a file
************************************************************************/

lzo_uint _lzo_fread(FILE *f, lzo_voidp s, lzo_uint len)
{
#if (LZO_UINT_MAX <= UINT_MAX)
	return fread(s,1,len,f);
#else
	/* seems to work with len > 64k with Borland C 4.0 */
	lzo_byte *p = (lzo_byte *) s;
	lzo_uint l = 0, k;

	while (l < len)
	{
		k = len - l > 32768u ? 32768u : len - l;
		k = fread(p,1,(unsigned)k,f);
		if (k <= 0)
			break;
		p += k;
		l += k;
	}
	return l;
#endif
}


/***********************************************************************
//
************************************************************************/

int lzo_assert(int expr)
{
	return (expr) ? 1 : 0;
}


int _lzo_config_check(void)
{
	int r = 1;
	int i;
	char wrkmem[8 * sizeof(lzo_byte *)];
	const lzo_bytepp const dict = (const lzo_bytepp) wrkmem;

	r &= lzo_assert(sizeof(lzo_uint) >= 4);
	r &= lzo_assert(sizeof(lzo_uint) >= sizeof(unsigned));

	r &= lzo_assert(sizeof(lzo_ptrdiff_t) >= 4);
	r &= lzo_assert(sizeof(lzo_ptrdiff_t) >= sizeof(ptrdiff_t));

	r &= lzo_assert(sizeof(lzo_voidp) >= 4);
	r &= lzo_assert(sizeof(lzo_voidp) == sizeof(lzo_byte *));
	r &= lzo_assert(sizeof(lzo_voidp) == sizeof(lzo_voidpp));
	r &= lzo_assert(sizeof(lzo_voidp) == sizeof(lzo_bytepp));
	r &= lzo_assert(sizeof(lzo_voidp) == sizeof(dict[0]));
#if 0
	r &= lzo_assert(sizeof(lzo_voidp) == sizeof(lzo_ptr_t));
#endif


	/* sanity check of memory model */
	if (r == 1)
	{
		for (i = 0; i < 8; i++)
			r &= lzo_assert((lzo_voidp) (&dict[i]) ==
			                (lzo_voidp) (&wrkmem[i * sizeof(lzo_byte *)]));
	}
	/* assert NULL == 0 */
	if (r == 1)
	{
		BZERO8_PTR(dict,8);
		for (i = 0; i < 8; i++)
			r &= lzo_assert(dict[i] == NULL);
	}

	return r == 1 ? LZO_E_OK : LZO_E_ERROR;
}


/***********************************************************************
//
************************************************************************/

/*
  If you use the LZO library in a product, an acknowledgment is welcome
  in the documentation of your product. If for some reason you cannot
  include such an acknowledgment, I would appreciate that you keep this
  copyright string in the executable of your product.
 */

const lzo_byte __lzo_copyright[] = 
	"\n\n"
	"LZO real-time data compression library.\n"
	"(C) 1996 Markus Franz Xaver Johannes Oberhumer\n"
	"<markus.oberhumer@jk.uni-linz.ac.at>\n"
	"\n";


int _lzo_copyright(void)
{
	lzo_uint adler;

	adler = lzo_adler32(0, NULL, 0);
	adler = lzo_adler32(adler, __lzo_copyright, sizeof(__lzo_copyright));
	return (adler == 0xe86f2b99L) ? LZO_E_OK : LZO_E_ERROR;
}


/***********************************************************************
//
************************************************************************/

unsigned lzo_version(void)
{
	return LZO_VERSION;
}

const char *lzo_version_string(void)
{
	return LZO_VERSION_STRING;
}


/*
vi:ts=4
*/
