/***************************************************************************

Jr. Pac Man memory map (preliminary)

0000-3fff ROM
4000-47ff Video RAM (also color RAM)
4800-4fff RAM
8000-dfff ROM

memory mapped ports:

read:
5000      IN0
5040      IN1
5080      DSW1

*
 * IN0 (all bits are inverted)
 * bit 7 : CREDIT
 * bit 6 : COIN 2
 * bit 5 : COIN 1
 * bit 4 : RACK TEST
 * bit 3 : DOWN player 1
 * bit 2 : RIGHT player 1
 * bit 1 : LEFT player 1
 * bit 0 : UP player 1
 *
*
 * IN1 (all bits are inverted)
 * bit 7 : TABLE or UPRIGHT cabinet select (1 = UPRIGHT)
 * bit 6 : START 2
 * bit 5 : START 1
 * bit 4 : TEST SWITCH
 * bit 3 : DOWN player 2 (TABLE only)
 * bit 2 : RIGHT player 2 (TABLE only)
 * bit 1 : LEFT player 2 (TABLE only)
 * bit 0 : UP player 2 (TABLE only)
 *
*
 * DSW1 (all bits are inverted)
 * bit 7 :  ?
 * bit 6 :  difficulty level
 *                       1 = Normal  0 = Harder
 * bit 5 :\ bonus pac at xx000 pts
 * bit 4 :/ 00 = 10000  01 = 15000  10 = 20000  11 = 30000
 * bit 3 :\ nr of lives
 * bit 2 :/ 00 = 1  01 = 2  10 = 3  11 = 5
 * bit 1 :\ play mode
 * bit 0 :/ 00 = free play   01 = 1 coin 1 credit
 *          10 = 1 coin 2 credits   11 = 2 coins 1 credit
 *

write:
4ff2-4ffd 6 pairs of two bytes:
          the first byte contains the sprite image number (bits 2-7), Y flip (bit 0),
		  X flip (bit 1); the second byte the color
5000      interrupt enable
5001      sound enable
5002      unused
5003      flip screen
5004      unused
5005      unused
5006      unused
5007      coin counter
5040-5044 sound voice 1 accumulator (nibbles) (used by the sound hardware only)
5045      sound voice 1 waveform (nibble)
5046-5049 sound voice 2 accumulator (nibbles) (used by the sound hardware only)
504a      sound voice 2 waveform (nibble)
504b-504e sound voice 3 accumulator (nibbles) (used by the sound hardware only)
504f      sound voice 3 waveform (nibble)
5050-5054 sound voice 1 frequency (nibbles)
5055      sound voice 1 volume (nibble)
5056-5059 sound voice 2 frequency (nibbles)
505a      sound voice 2 volume (nibble)
505b-505e sound voice 3 frequency (nibbles)
505f      sound voice 3 volume (nibble)
5062-506d Sprite coordinates, x/y pairs for 6 sprites
5070      palette bank
5071      colortable bank
5073      background priority over sprites
5074      char gfx bank
5075      sprite gfx bank
5080      scroll
50c0      Watchdog reset

I/O ports:
OUT on port $0 sets the interrupt vector


***************************************************************************/

#include "driver.h"
#include "vidhrdw/generic.h"


void jrpacman_init_machine(void);
int jrpacman_interrupt(void);

extern unsigned char *jrpacman_scroll,*jrpacman_bgpriority;
extern unsigned char *jrpacman_charbank,*jrpacman_spritebank;
extern unsigned char *jrpacman_palettebank,*jrpacman_colortablebank;
void jrpacman_vh_convert_color_prom(unsigned char *palette, unsigned char *colortable,const unsigned char *color_prom);
int jrpacman_vh_start(void);
void jrpacman_vh_stop(void);
void jrpacman_videoram_w(int offset,int data);
void jrpacman_palettebank_w(int offset,int data);
void jrpacman_colortablebank_w(int offset,int data);
void jrpacman_charbank_w(int offset,int data);
void jrpacman_vh_screenrefresh(struct osd_bitmap *bitmap);

extern unsigned char *pengo_soundregs;
void pengo_sound_enable_w(int offset,int data);
void pengo_sound_w(int offset,int data);
void pengo_sh_update(void);



static struct MemoryReadAddress readmem[] =
{
	{ 0x4000, 0x4fff, MRA_RAM },	/* including video and color RAM */
	{ 0x0000, 0x3fff, MRA_ROM },
	{ 0x8000, 0xdfff, MRA_ROM },
	{ 0x5000, 0x503f, input_port_0_r },	/* IN0 */
	{ 0x5040, 0x507f, input_port_1_r },	/* IN1 */
	{ 0x5080, 0x50bf, input_port_2_r },	/* DSW1 */
	{ -1 }	/* end of table */
};

static struct MemoryWriteAddress writemem[] =
{
	{ 0x4800, 0x4fef, MWA_RAM },
	{ 0x4000, 0x47ff, jrpacman_videoram_w, &videoram, &videoram_size },
	{ 0x5040, 0x505f, pengo_sound_w, &pengo_soundregs },
	{ 0x4ff0, 0x4fff, MWA_RAM, &spriteram, &spriteram_size },
	{ 0x5060, 0x506f, MWA_RAM, &spriteram_2 },
	{ 0x5080, 0x5080, MWA_RAM, &jrpacman_scroll },
	{ 0x5070, 0x5070, jrpacman_palettebank_w, &jrpacman_palettebank },
	{ 0x5071, 0x5071, jrpacman_colortablebank_w, &jrpacman_colortablebank },
	{ 0x5073, 0x5073, MWA_RAM, &jrpacman_bgpriority },
	{ 0x5074, 0x5074, jrpacman_charbank_w, &jrpacman_charbank },
	{ 0x5075, 0x5075, MWA_RAM, &jrpacman_spritebank },
	{ 0x5000, 0x5000, interrupt_enable_w },
	{ 0x50c0, 0x50c0, MWA_NOP },
	{ 0x5001, 0x5001, pengo_sound_enable_w },
	{ 0x5002, 0x5007, MWA_NOP },
	{ 0x0000, 0x3fff, MWA_ROM },
	{ 0x8000, 0xdfff, MWA_ROM },
	{ -1 }	/* end of table */
};



static struct IOWritePort writeport[] =
{
	{ 0, 0, interrupt_vector_w },
	{ -1 }	/* end of table */
};



static struct InputPort input_ports[] =
{
	{	/* IN0 */
		0xff,
		{ OSD_KEY_UP, OSD_KEY_LEFT, OSD_KEY_RIGHT, OSD_KEY_DOWN,
				OSD_KEY_F1, 0, 0, OSD_KEY_3 },
		{ OSD_JOY_UP, OSD_JOY_LEFT, OSD_JOY_RIGHT, OSD_JOY_DOWN,
				0, 0, 0, 0 }
	},
	{	/* IN1 */
		0xff,
		{ 0, 0, 0, 0, OSD_KEY_F2, OSD_KEY_1, OSD_KEY_2, 0 },
		{ 0, 0, 0, 0, 0, 0, 0, 0 }
	},
	{	/* DSW1 */
		0xe9,
		{ 0, 0, 0, 0, 0, OSD_KEY_F1, 0, 0 },
		{ 0, 0, 0, 0, 0, 0, 0, 0 }
	},
	{ -1 }	/* end of table */
};

static struct TrakPort trak_ports[] =
{
        { -1 }
};


static struct KEYSet keys[] =
{
        { 0, 0, "MOVE UP" },
        { 0, 1, "MOVE LEFT"  },
        { 0, 2, "MOVE RIGHT" },
        { 0, 3, "MOVE DOWN" },
        { -1 }
};


static struct DSW dsw[] =
{
	{ 2, 0x0c, "LIVES", { "1", "2", "3", "5" } },
	{ 2, 0x30, "BONUS", { "10000", "15000", "20000", "30000" } },
	{ 2, 0x40, "DIFFICULTY", { "HARD", "NORMAL" }, 1 },
	{ 2, 0x80, "SW7", { "ON", "OFF" }, 1 },
	{ -1 }
};



static struct GfxLayout charlayout =
{
	8,8,	/* 8*8 characters */
	512,	/* 512 characters */
	2,	/* 2 bits per pixel */
	{ 0, 4 },	/* the two bitplanes for 4 pixels are packed into one byte */
	{ 7*8, 6*8, 5*8, 4*8, 3*8, 2*8, 1*8, 0*8 }, /* characters are rotated 90 degrees */
	{ 8*8+0, 8*8+1, 8*8+2, 8*8+3, 0, 1, 2, 3 },	/* bits are packed in groups of four */
	16*8	/* every char takes 16 bytes */
};
static struct GfxLayout spritelayout =
{
	16,16,	/* 16*16 sprites */
	128,	/* 128 sprites */
	2,	/* 2 bits per pixel */
	{ 0, 4 },	/* the two bitplanes for 4 pixels are packed into one byte */
	{ 39 * 8, 38 * 8, 37 * 8, 36 * 8, 35 * 8, 34 * 8, 33 * 8, 32 * 8,
			7 * 8, 6 * 8, 5 * 8, 4 * 8, 3 * 8, 2 * 8, 1 * 8, 0 * 8 },
	{ 8*8, 8*8+1, 8*8+2, 8*8+3, 16*8+0, 16*8+1, 16*8+2, 16*8+3,
			24*8+0, 24*8+1, 24*8+2, 24*8+3, 0, 1, 2, 3 },
	64*8	/* every sprite takes 64 bytes */
};



static struct GfxDecodeInfo gfxdecodeinfo[] =
{
	{ 1, 0x0000, &charlayout,      0, 128 },
	{ 1, 0x2000, &spritelayout,    0, 128 },
	{ -1 } /* end of array */
};



static unsigned char color_prom[] =
{
	/* 9E - palette low bits */
	0x00,0x07,0x0D,0x0F,0x08,0x08,0x0A,0x0F,0x02,0x0F,0x0D,0x09,0x08,0x0A,0x0F,0x0F,
	0x00,0x05,0x07,0x09,0x00,0x03,0x06,0x0D,0x00,0x0F,0x0B,0x0D,0x00,0x0F,0x0F,0x08,
	/* 9F - palette high bits */
	0x00,0x00,0x01,0x0E,0x01,0x0F,0x0E,0x02,0x01,0x03,0x02,0x0C,0x03,0x0A,0x0A,0x0F,
	0x00,0x00,0x00,0x00,0x01,0x01,0x02,0x02,0x03,0x03,0x05,0x0A,0x0C,0x0F,0x0E,0x0F,
	/* 9P - color lookup table */
	0x00,0x00,0x00,0x00,0x00,0x0F,0x0B,0x01,0x00,0x00,0x00,0x00,0x00,0x0F,0x0B,0x03,
	0x00,0x00,0x00,0x00,0x00,0x0F,0x0B,0x05,0x00,0x00,0x00,0x00,0x00,0x0F,0x0B,0x07,
	0x00,0x08,0x0A,0x09,0x00,0x0B,0x01,0x09,0x00,0x0E,0x0B,0x07,0x00,0x09,0x06,0x07,
	0x00,0x09,0x02,0x06,0x00,0x0F,0x04,0x09,0x00,0x0F,0x00,0x0E,0x00,0x01,0x0C,0x0F,
	0x00,0x0E,0x00,0x0B,0x00,0x0C,0x0B,0x0E,0x00,0x0C,0x0F,0x01,0x00,0x04,0x0C,0x0F,
	0x00,0x01,0x02,0x0F,0x00,0x07,0x0C,0x02,0x00,0x09,0x06,0x0F,0x00,0x0D,0x0C,0x0F,
	0x00,0x05,0x03,0x09,0x00,0x0F,0x0B,0x00,0x00,0x0E,0x00,0x0B,0x00,0x0E,0x00,0x0B,
	0x00,0x0F,0x02,0x07,0x00,0x0F,0x0E,0x01,0x00,0x0F,0x0B,0x0E,0x00,0x0E,0x00,0x0F,
	0x00,0x00,0x00,0x00,0x00,0x0D,0x0C,0x02,0x00,0x04,0x08,0x0D,0x00,0x05,0x06,0x07,
	0x00,0x0B,0x0D,0x0A,0x00,0x0A,0x0D,0x01,0x00,0x05,0x0A,0x03,0x00,0x06,0x0C,0x0D,
	0x00,0x05,0x07,0x09,0x00,0x0C,0x02,0x09,0x00,0x0D,0x0C,0x0E,0x00,0x0D,0x0C,0x0F,
	0x00,0x0D,0x0C,0x06,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
};



/* waveforms for the audio hardware */
/* I don't know if these are correct. I'm using the Pac Man ones. */
static unsigned char samples[8*32] =
{
	0xff,0x11,0x22,0x33,0x44,0x55,0x55,0x66,0x66,0x66,0x55,0x55,0x44,0x33,0x22,0x11,
	0xff,0xdd,0xcc,0xbb,0xaa,0x99,0x99,0x88,0x88,0x88,0x99,0x99,0xaa,0xbb,0xcc,0xdd,

	0xff,0x44,0x66,0x66,0x55,0x33,0x11,0x22,0x33,0x33,0x22,0x11,0xee,0xcc,0xbb,0xdd,
	0xff,0x11,0x33,0x22,0x00,0xdd,0xcc,0xbb,0xbb,0xcc,0xdd,0xbb,0x99,0x88,0x88,0xaa,

	0xff,0x22,0x44,0x55,0x66,0x55,0x44,0x22,0xff,0xcc,0xaa,0x99,0x88,0x99,0xaa,0xcc,
	0xff,0x33,0x55,0x66,0x55,0x33,0xff,0xbb,0x99,0x88,0x99,0xbb,0xff,0x66,0xff,0x88,

	0xff,0x55,0x33,0x00,0x33,0x55,0x11,0xee,0x33,0x66,0x44,0xff,0x11,0x22,0xee,0xaa,
	0xff,0x44,0x00,0xcc,0xdd,0xff,0xaa,0x88,0xbb,0x00,0xdd,0x99,0xbb,0xee,0xbb,0x99,

	0x88,0x00,0x77,0xff,0x99,0x00,0x66,0xff,0xaa,0x00,0x55,0xff,0xbb,0x00,0x44,0xff,
	0xcc,0x00,0x33,0xff,0xdd,0x00,0x22,0xff,0xee,0x00,0x11,0xff,0xff,0x00,0x00,0xff,

	0xff,0x00,0xee,0x11,0xdd,0x22,0xcc,0x33,0xbb,0x44,0xaa,0x55,0x99,0x66,0x88,0x77,
	0x88,0x77,0x99,0x66,0xaa,0x55,0xbb,0x44,0xcc,0x33,0xdd,0x22,0xee,0x11,0xff,0x00,

	0x88,0x99,0xaa,0xbb,0xcc,0xdd,0xee,0xff,0x00,0x11,0x22,0x33,0x44,0x55,0x66,0x77,
	0x77,0x66,0x55,0x44,0x33,0x22,0x11,0x00,0xff,0xee,0xdd,0xcc,0xbb,0xaa,0x99,0x88,

	0x88,0x99,0xaa,0xbb,0xcc,0xdd,0xee,0xff,0x00,0x11,0x22,0x33,0x44,0x55,0x66,0x77,
	0x88,0x99,0xaa,0xbb,0xcc,0xdd,0xee,0xff,0x00,0x11,0x22,0x33,0x44,0x55,0x66,0x77
};



static struct MachineDriver machine_driver =
{
	/* basic machine hardware */
	{
		{
			CPU_Z80,
			3072000,	/* 3.072 Mhz */
			0,
			readmem,writemem,0,writeport,
			jrpacman_interrupt,1
		}
	},
	60,
	1,	/* single CPU, no need for interleaving */
	jrpacman_init_machine,

	/* video hardware */
	28*8, 36*8, { 0*8, 28*8-1, 0*8, 36*8-1 },
	gfxdecodeinfo,
	32,128*4,
	jrpacman_vh_convert_color_prom,

	VIDEO_TYPE_RASTER,
	0,
	jrpacman_vh_start,
	jrpacman_vh_stop,
	jrpacman_vh_screenrefresh,

	/* sound hardware */
	samples,
	0,
	0,
	0,
	pengo_sh_update
};



/***************************************************************************

  Game driver(s)

***************************************************************************/

ROM_START( jrpacman_rom )
	ROM_REGION(0x10000)	/* 64k for code */
	ROM_LOAD( "jrp8d.bin", 0x0000, 0x2000, 0xc55c5c2c )
	ROM_LOAD( "jrp8e.bin", 0x2000, 0x2000, 0xc0c8f2b6 )
	ROM_LOAD( "jrp8h.bin", 0x8000, 0x2000, 0x4ec948b1 )
	ROM_LOAD( "jrp8j.bin", 0xa000, 0x2000, 0x2099a009 )
	ROM_LOAD( "jrp8k.bin", 0xc000, 0x2000, 0x3744a458 )

	ROM_REGION(0x4000)	/* temporary space for graphics (disposed after conversion) */
	ROM_LOAD( "jrp2c.bin", 0x0000, 0x2000, 0xdf0ea63e )
	ROM_LOAD( "jrp2e.bin", 0x2000, 0x2000, 0x15b224bc )
ROM_END



static void jrpacman_decode(void)
{
	/* The encryption PALs garble bits 0, 2 and 7 of the ROMs. The encryption */
	/* scheme is complex (basically it's a state machine) and can only be */
	/* faithfully emulated at run time. To avoid the performance hit that would */
	/* cause, here we have a table of the values which must be XORed with */
	/* each memory region to obtain the decrypted bytes. */
	/* Decryption table provided by David Caldwell (david@indigita.com) */
	/* For an accurate reproduction of the encryption, see jrcrypt.c */
	struct {
	    int count;
	    int value;
	} table[] =
	{
		{ 0x00C1, 0x00 },{ 0x0002, 0x80 },{ 0x0004, 0x00 },{ 0x0006, 0x80 },
		{ 0x0003, 0x00 },{ 0x0002, 0x80 },{ 0x0009, 0x00 },{ 0x0004, 0x80 },
		{ 0x9968, 0x00 },{ 0x0001, 0x80 },{ 0x0002, 0x00 },{ 0x0001, 0x80 },
		{ 0x0009, 0x00 },{ 0x0002, 0x80 },{ 0x0009, 0x00 },{ 0x0001, 0x80 },
		{ 0x00AF, 0x00 },{ 0x000E, 0x04 },{ 0x0002, 0x00 },{ 0x0004, 0x04 },
		{ 0x001E, 0x00 },{ 0x0001, 0x80 },{ 0x0002, 0x00 },{ 0x0001, 0x80 },
		{ 0x0002, 0x00 },{ 0x0002, 0x80 },{ 0x0009, 0x00 },{ 0x0002, 0x80 },
		{ 0x0009, 0x00 },{ 0x0002, 0x80 },{ 0x0083, 0x00 },{ 0x0001, 0x04 },
		{ 0x0001, 0x01 },{ 0x0001, 0x00 },{ 0x0002, 0x05 },{ 0x0001, 0x00 },
		{ 0x0003, 0x04 },{ 0x0003, 0x01 },{ 0x0002, 0x00 },{ 0x0001, 0x04 },
		{ 0x0003, 0x01 },{ 0x0003, 0x00 },{ 0x0003, 0x04 },{ 0x0001, 0x01 },
		{ 0x002E, 0x00 },{ 0x0078, 0x01 },{ 0x0001, 0x04 },{ 0x0001, 0x05 },
		{ 0x0001, 0x00 },{ 0x0001, 0x01 },{ 0x0001, 0x04 },{ 0x0002, 0x00 },
		{ 0x0001, 0x01 },{ 0x0001, 0x04 },{ 0x0002, 0x00 },{ 0x0001, 0x01 },
		{ 0x0001, 0x04 },{ 0x0002, 0x00 },{ 0x0001, 0x01 },{ 0x0001, 0x04 },
		{ 0x0001, 0x05 },{ 0x0001, 0x00 },{ 0x0001, 0x01 },{ 0x0001, 0x04 },
		{ 0x0002, 0x00 },{ 0x0001, 0x01 },{ 0x0001, 0x04 },{ 0x0002, 0x00 },
		{ 0x0001, 0x01 },{ 0x0001, 0x04 },{ 0x0001, 0x05 },{ 0x0001, 0x00 },
		{ 0x01B0, 0x01 },{ 0x0001, 0x00 },{ 0x0002, 0x01 },{ 0x00AD, 0x00 },
		{ 0x0031, 0x01 },{ 0x005C, 0x00 },{ 0x0005, 0x01 },{ 0x604E, 0x00 },
	    { 0,0 }
	};
	int i,j,A;


	A = 0;
	i = 0;
	while (table[i].count)
	{
		for (j = 0;j < table[i].count;j++)
		{
			RAM[A] ^= table[i].value;
			A++;
		}
		i++;
	}
}


static int hiload(void)
{
	static int resetcount;


	/* during a reset, leave time to the game to clear the screen */
	if (++resetcount < 60) return 0;

	/* wait for "HIGH SCORE" to be on screen */
	if ((memcmp(&RAM[0x476d],"\x40\x40\x40\x40",4) == 0) &&
            (memcmp(&RAM[0x4751],"\x48\x47\x49\x48",4) == 0))
	{
		void *f;


		resetcount = 0;

		if ((f = osd_fopen(Machine->gamedrv->name,0,OSD_FILETYPE_HIGHSCORE,0)) != 0)
		{
			char buf[10];
			int hi;


			osd_fread(f,&RAM[0x4e88],4);
			/* also copy the high score to the screen, otherwise it won't be */
			/* updated */
			hi = (RAM[0x4e88] & 0x0f) +
			     (RAM[0x4e88] >> 4) * 10 +
			     (RAM[0x4e89] & 0x0f) * 100 +
			     (RAM[0x4e89] >> 4) * 1000 +
			     (RAM[0x4e8a] & 0x0f) * 10000 +
			     (RAM[0x4e8a] >> 4) * 100000 +
			     (RAM[0x4e8b] & 0x0f) * 1000000 +
			     (RAM[0x4e8b] >> 4) * 10000000;
			if (hi)
			{
				sprintf(buf,"%8d",hi);
				if (buf[2] != ' ') jrpacman_videoram_w(0x0772,buf[2]-'0');
				if (buf[3] != ' ') jrpacman_videoram_w(0x0771,buf[3]-'0');
				if (buf[4] != ' ') jrpacman_videoram_w(0x0770,buf[4]-'0');
				if (buf[5] != ' ') jrpacman_videoram_w(0x076f,buf[5]-'0');
				if (buf[6] != ' ') jrpacman_videoram_w(0x076e,buf[6]-'0');
				jrpacman_videoram_w(0x076d,buf[7]-'0');
			}
			osd_fclose(f);
		}

		return 1;
	}
	else return 0;	/* we can't load the hi scores yet */
}



static void hisave(void)
{
	void *f;

	if ((f = osd_fopen(Machine->gamedrv->name,0,OSD_FILETYPE_HIGHSCORE,1)) != 0)
	{
		osd_fwrite(f,&RAM[0x4e88],4);
		osd_fclose(f);
	}
}



struct GameDriver jrpacman_driver =
{
	"Jr. Pac Man",
	"jrpacman",
	"DAVID CALDWELL\nNICOLA SALMORIA",
	&machine_driver,

	jrpacman_rom,
	jrpacman_decode, 0,
	0,

	input_ports, 0, trak_ports, dsw, keys,

	color_prom, 0, 0,
	ORIENTATION_DEFAULT,

	hiload, hisave
};
