//
// $Header: D:/ext2-os2/minifsd/vfs/RCS/inode.c,v 1.1 1996/05/27 23:51:26 Willm Exp Willm $
//

// Linux ext2 file system driver for OS/2 2.x and WARP - Allows OS/2 to
// access your Linux ext2fs partitions as normal drive letters.
// Copyright (C) 1995, 1996 Matthieu WILLM 
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.


#ifndef OS2
#include <linux/stat.h>
#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/mm.h>
#include <linux/string.h>

#include <asm/system.h>
#else
#define INCL_DOSERRORS
#define INCL_NOPMAPI
#include <os2.h>                // From the "Developer Connection Device Driver Kit" version 2.0

#include <os2/types.h>
#include <os2/os2proto.h>
#include <linux/sched.h>
#include <linux/fs.h>
#include <linux/fs_proto.h>
#include <linux/stat.h>
#include <os2/errors.h>
extern unsigned long event;
#define const
#endif


#ifdef OS2
#define Panic(x) FSH_INTERR((x), sizeof(x))
#define inline _inline
#endif

struct inode_hash_entry {
        struct inode * inode;
        int updating;
} hash_table[NR_IHASH];

#ifndef OS2
static struct inode * first_inode;
static struct wait_queue * inode_wait = NULL;
static int nr_inodes = 0, nr_free_inodes = 0;
#else

#define NR_INODES 100		// Should be enough for booting
struct inode stage1_inode[NR_INODES];

struct inode * first_inode;
unsigned long inode_wait = NULL;
long nr_inodes = 0, nr_free_inodes = 0;
#endif

#ifndef OS2
static inline int const hashfn(dev_t dev, unsigned int i)
{
        return (dev ^ i) % NR_IHASH;
}
#else
static inline long const hashfn(dev_t dev, unsigned long i)
{
        return (long)(((unsigned long)dev ^ i) % NR_IHASH);
}
#endif

#ifndef OS2
static inline struct inode_hash_entry * const hash(dev_t dev, int i)
{
        return hash_table + hashfn(dev, i);
}
#else
static inline struct inode_hash_entry * const hash(dev_t dev, long i)
{
        return hash_table + hashfn(dev, (unsigned long)i);
}
#endif
static void insert_inode_free(struct inode *inode)
{
        _disable();
        inode->i_next = first_inode;
        inode->i_prev = first_inode->i_prev;
        inode->i_next->i_prev = inode;
        inode->i_prev->i_next = inode;
        first_inode = inode;
        _enable();
}

static void remove_inode_free(struct inode *inode)
{
        _disable();
        if (first_inode == inode)
                first_inode = first_inode->i_next;
        if (inode->i_next)
                inode->i_next->i_prev = inode->i_prev;
        if (inode->i_prev)
                inode->i_prev->i_next = inode->i_next;
        inode->i_next = inode->i_prev = NULL;
        _enable();
}

void insert_inode_hash(struct inode *inode)
{
        struct inode_hash_entry *h;

        _disable();
        h = hash(inode->i_dev, inode->i_ino);

        inode->i_hash_next = h->inode;
        inode->i_hash_prev = NULL;
        if (inode->i_hash_next)
                inode->i_hash_next->i_hash_prev = inode;
        h->inode = inode;
        _enable();
}

static void remove_inode_hash(struct inode *inode)
{
        struct inode_hash_entry *h;

        _disable();
        h = hash(inode->i_dev, inode->i_ino);

        if (h->inode == inode)
                h->inode = inode->i_hash_next;
        if (inode->i_hash_next)
                inode->i_hash_next->i_hash_prev = inode->i_hash_prev;
        if (inode->i_hash_prev)
                inode->i_hash_prev->i_hash_next = inode->i_hash_next;
        inode->i_hash_prev = inode->i_hash_next = NULL;
        _enable();
}

static void put_last_free(struct inode *inode)
{
        remove_inode_free(inode);
        _disable();
        inode->i_prev = first_inode->i_prev;
        inode->i_prev->i_next = inode;
        inode->i_next = first_inode;
        inode->i_next->i_prev = inode;
        _enable();
}

#ifndef OS2
void grow_inodes(void)
{
        struct inode * inode;
        int i;

        if (!(inode = (struct inode*) get_free_page(GFP_KERNEL)))
                return;

        i=PAGE_SIZE / sizeof(struct inode);
        nr_inodes += i;
        nr_free_inodes += i;

        if (!first_inode)
                inode->i_next = inode->i_prev = first_inode = inode++, i--;

        for ( ; i ; i-- )
                insert_inode_free(inode++);

}
#else

static int grown = 0;

void grow_inodes(void) {
        struct inode * inode;
        long i;

        if (!grown) 
		grown = 1;
        else
		return;

        inode = stage1_inode;
        i     = NR_INODES;

        kernel_printf("********* grow_inode() from %lu to %lu", nr_inodes, i);
        nr_inodes      += i;
        nr_free_inodes += i;
#if 0
        if (!first_inode)
                inode->i_next = inode->i_prev = first_inode = inode++, i--;
#else
//
// If I don't do it that way with MS Visual C++, first_inode->i_prev = NULL ... TRAP D !!!!
//
        if (!first_inode) {
                first_inode   = inode;
                inode->i_next = inode;
                inode->i_prev = inode;
                inode++;
                i--;
        }
        if ((!first_inode)         ||
            (!first_inode->i_prev) ||
            (!first_inode->i_next))
                Panic("grow_inodes - 1");
#endif

        for ( ; i ; i-- ) {
                insert_inode_free(inode++);
                if ((!first_inode) || (!first_inode->i_prev) || (!first_inode->i_next)) Panic("grow_inodes - 2");
        }
        
}
#endif
void inode_init(void) {
        memset(hash_table, 0, sizeof(hash_table));
        first_inode = NULL;
}

void __wait_on_inode(struct inode *);

extern void (*wait_on_inode)();
extern void (*lock_inode)();
extern void (*unlock_inode)();

void stage2_wait_on_inode(struct inode * inode)
{
        if (inode->i_lock)
                __wait_on_inode(inode);
}

void stage2_lock_inode(struct inode * inode)
{
        wait_on_inode(inode);
        inode->i_lock = 1;
}

void stage2_unlock_inode(struct inode * inode)
{
        inode->i_lock = 0;

        wake_up(&inode->i_wait);
}

void clear_inode(struct inode * inode)
{
        unsigned long wait;

        wait_on_inode(inode);
        remove_inode_hash(inode);
        remove_inode_free(inode);
        wait = ((volatile struct inode *) inode)->i_wait;
        if (inode->i_count)
                nr_free_inodes++;
        memset(inode,0,sizeof(*inode));
        ((volatile struct inode *) inode)->i_wait = wait;
        if (!inode) Panic("clear_inode() - insert_inode_free(NULL)");
        insert_inode_free(inode);
}



static void read_inode(struct inode * inode)
{
//	printk("\tread_inode(%ld)", inode->i_ino);

#ifdef OS2
        if (inode->i_ino == INODE_DASD) {
                inode->i_mode    = S_IFREG | S_IRUSR | S_IWUSR;
                inode->i_size    = inode->i_sb->sector_size * inode->i_sb->nb_sectors;
                inode->i_blksize = inode->i_sb->s_blocksize;
                inode->i_blocks  = inode->i_size / inode->i_sb->s_blocksize;
                inode->i_op      = &ext2_file_inode_operations;
                return;
        }
#endif
        lock_inode(inode);
        if (inode->i_sb && inode->i_sb->s_op && inode->i_sb->s_op->read_inode)
                inode->i_sb->s_op->read_inode(inode);
        unlock_inode(inode);
}




void iput(struct inode * inode)
{

        if (!inode)
                return;
#ifdef FS_TRACE
        kernel_printf("iput(%lu) - prev. i_count = %d", inode->i_ino, inode->i_count);
#endif
        wait_on_inode(inode);
        if (!inode->i_count) {
                ext2_os2_panic(0, "VFS: iput: trying to free free inode %lu", inode->i_ino);
        }
repeat:
        if (inode->i_count>1) {
                inode->i_count--;
                return;
        }
        wake_up(&inode_wait);

#ifdef OS2
        if (inode->i_ino != INODE_DASD) {
#endif
        if (inode->i_sb && inode->i_sb->s_op && inode->i_sb->s_op->put_inode) {
                inode->i_sb->s_op->put_inode(inode);
                if (!inode->i_nlink)
                        return;
        }
#ifdef OS2
        }
#endif
        inode->i_count--;
        nr_free_inodes++;
        return;
}

struct inode * get_empty_inode(void)
{
        struct inode * inode, * best;
        int i;

        if (nr_inodes < NR_INODE && nr_free_inodes < (nr_inodes >> 2))
                grow_inodes();
repeat:
        inode = first_inode;
        best = NULL;
        for (i = 0; i<nr_inodes; inode = inode->i_next, i++) {
                if (!inode->i_count) {
                        if (!best)
                                best = inode;
                        if (!inode->i_dirt && !inode->i_lock) {
                                best = inode;
                                break;
                        }
                }
        }
        if (!best || best->i_dirt || best->i_lock)
                if (nr_inodes < NR_INODE) {
                        grow_inodes();
                        goto repeat;
                }
        inode = best;

        if (!inode) {
                printk("VFS: No free inodes - contact Linus\n");
                sleep_on(&inode_wait);
                goto repeat;
        }

        if (inode->i_lock) {
                wait_on_inode(inode);
                goto repeat;
        }
        if (inode->i_count)
                goto repeat;
        clear_inode(inode);
        inode->i_count = 1;
        inode->i_nlink = 1;
        inode->i_version = ++event;
        inode->i_sem.count = 1;
        nr_free_inodes--;
        if (nr_free_inodes < 0) {
                printk ("VFS: get_empty_inode: bad free inode count.\n");
                nr_free_inodes = 0;
        }
        return inode;
}


#ifndef OS2
struct inode * __iget(struct super_block * sb, int nr, int crossmntp)
{
        static struct wait_queue * update_wait = NULL;
#else
struct inode * __iget(struct super_block * sb, ino_t nr, int crossmntp)
{
        static unsigned long update_wait = 0;
#endif
        struct inode_hash_entry * h;
        struct inode * inode;
        struct inode * empty = NULL;

//        printk("\tiget(%lu)", nr);


        if (!sb)
                ext2_os2_panic(0, "VFS: iget with sb==NULL");
        h = hash(sb->s_dev, nr);
repeat:
        for (inode = h->inode; inode ; inode = inode->i_hash_next)
                if (inode->i_dev == sb->s_dev && inode->i_ino == nr)
                        goto found_it;
        if (!empty) {
                h->updating++;
                empty = get_empty_inode();
                if (!--h->updating)
                        wake_up(&update_wait);
                if (empty)
                        goto repeat;
                return (NULL);
        }
        inode = empty;
        inode->i_sb = sb;
        inode->i_dev = sb->s_dev;
        inode->i_ino = nr;
        inode->i_flags = sb->s_flags;
        put_last_free(inode);
        insert_inode_hash(inode);
        read_inode(inode);
        goto return_it;

found_it:
        if (!inode->i_count)
                nr_free_inodes--;
        inode->i_count++;
        wait_on_inode(inode);
#ifdef OS2
        if (!inode || (SELECTOROF(inode) == -1)) Panic("__iget - 1");
#endif
        if (inode->i_dev != sb->s_dev || inode->i_ino != nr) {
                printk("Whee.. inode changed from under us. Tell Linus\n");
                iput(inode);
                goto repeat;
        }
        if (empty)
                iput(empty);

return_it:
        while (h->updating)
                sleep_on(&update_wait);
        return inode;
}

void __wait_on_inode(struct inode * inode)
{
    _disable();
    while (inode->i_lock) {
        ProcBlock((unsigned long)(&(inode->i_wait)), -1, 1);
        _disable();
    }
    _enable();
}

void stage1_wait_on_inode() {}
void stage1_lock_inode() {}
void stage1_unlock_inode() {}

void (*wait_on_inode)() = stage1_wait_on_inode;
void (*lock_inode)()    = stage1_lock_inode;
void (*unlock_inode)()  = stage1_unlock_inode;

void inode_stage1_to_stage2(void) {
    int i;

    memset(hash_table, 0, sizeof(hash_table));

    for (i = 0; i < nr_inodes; i++) {
        stage1_inode[i].i_ino       = 0;
        stage1_inode[i].i_lock      = 0;
        stage1_inode[i].i_hash_prev = 0;
        stage1_inode[i].i_hash_next = 0;
    }
}
