//
// $Header: D:/ext2-os2/vfs/RCS/f_table.c,v 8.0 1996/05/31 16:43:31 Willm Exp Willm $
//

// Linux ext2 file system driver for OS/2 2.x and WARP - Allows OS/2 to
// access your Linux ext2fs partitions as normal drive letters.
// Copyright (C) 1995, 1996 Matthieu WILLM
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

/*
 *  INSPIRED FROM : linux/fs/file_table.c
 *
 *  Copyright (C) 1991, 1992  Linus Torvalds
 */

#define INCL_DOS
#define INCL_DOSERRORS
#define INCL_NOPMAPI
#include <os2.h>

#include <os2/types.h>
#include <linux/fs.h>
#include <os2/os2proto.h>
#include <os2/files.h>
#include <os2/minifsd.h>

extern unsigned long event;

long nr_files           = 0;
long nr_free_files      = 0;
long nr_used_files      = 0;

struct file *free_files = 0;
struct file *used_files = 0;

/*
 * MUST be called with interrupts disabled
 */
static void remove_from_list(struct file *f, struct file **list_head, long *counter) {

    if (!f)
        ext2_os2_panic(1, "remove_from_list - f is NULL");
    if (f->f_magic != FILE_MAGIC)
        ext2_os2_panic(1, "remove_from_list - invalid magic number");
    if (!list_head)
        ext2_os2_panic(1, "remove_from_list - list_head is NULL");
    if (!(*list_head) || !(*counter))
        ext2_os2_panic(1, "remove_from_list - used list empty");
    if (!(f->f_prev) || !(f->f_next))
        ext2_os2_panic(1, "remove_from_list - used file list corrupted");
    if (f->f_list != list_head)
        ext2_os2_panic(1, "remove_from_list - f in wrong list");


    f->f_prev->f_next = f->f_next;
    f->f_next->f_prev = f->f_prev;

    if (*list_head == f)
        *list_head = f->f_next;
    if(*list_head == f)
         *list_head = 0;
    f->f_next = 0;
    f->f_prev = 0;
    f->f_list = 0;
    (*counter) --;

}


/*
 * MUST be called with interrupts disabled
 */
static void put_last_in_list(struct file *f, struct file **list_head, long *counter) {
    if (!f)
        ext2_os2_panic(1, "put_last_in_list - f is NULL");
    if (f->f_magic != FILE_MAGIC)
        ext2_os2_panic(1, "put_last_in_list - invalid magic number");
    if (f->f_prev || f->f_next)
        ext2_os2_panic(1, "put_last_in_list - used block list corrupted");
    if (!list_head) 
        ext2_os2_panic(1, "put_last_in_list - list_head is NULL");
    if (f->f_list)
        ext2_os2_panic(1, "put_last_in_list - f already in a list");


    if(!(*list_head)) {
        *list_head = f;
        (*list_head)->f_prev = f;
    };

    f->f_next = *list_head;
    f->f_prev = (*list_head)->f_prev;
    (*list_head)->f_prev->f_next = f;
    (*list_head)->f_prev = f;
    f->f_list = list_head;
    (*counter)++;

}

int put_filp(struct file *f) {
    _disable();
    remove_from_list(f, &used_files, &nr_used_files);
    memset(f, 0, sizeof(struct file));
    f->f_magic = FILE_MAGIC;
    put_last_in_list(f, &free_files, &nr_free_files);
    _enable();
    return NO_ERROR;
}

static int grow_files(void) {
    struct file *f;
    long         i;

    i = 61440 / sizeof(struct file);

    /*
     * Swappable memory allocated here
     */
    f = (struct file *) G_malloc(i * sizeof(struct file));

    if (!f)
	return 0;

    nr_files += i;

    if (!free_files) {
        _disable();
        memset(f, 0, sizeof(struct file));
        f->f_magic = FILE_MAGIC;
        f->f_next = f;
        f->f_prev = f;
	f->f_list = &free_files;
        free_files = f;
        nr_free_files ++;
        f ++;
        i --;
        _enable();
    }

    for (; i ; i--) {
	_disable();
        memset(f, 0, sizeof(struct file));
        f->f_magic = FILE_MAGIC;
	put_last_in_list(f++, &free_files, &nr_free_files);
        _enable();
    }

    return 1;
}

struct file *get_empty_filp(void) {
    struct file *f;

    /*
     * NB: free_files is in a non swappable segment
     */
    do {
	_disable();
        f = free_files;
        if (f) {
            remove_from_list(f, &free_files, &nr_free_files);
	    memset(f, 0, sizeof(struct file));
            f->f_magic   = FILE_MAGIC;
            f->f_count   = 1;
            f->f_version = ++event;
            put_last_in_list(f, &used_files, &nr_used_files);
	    _enable();
            return f;
        }
	_enable();
    } while (grow_files());

    return NULL;
}


/*
 * The following routines are useless for a mini FSD
 */
#ifndef MINIFSD

unsigned long file_table_init(unsigned long start, unsigned long end) {
    return start;
}

void inherit_minifsd_files(struct minifsd_to_fsd_data  *mfs_data) {
    struct file  *f;
    struct file **mfs_free;
    struct file **mfs_used;
    long         *nr_mfs_used;
    long         *nr_mfs_free;

    nr_mfs_free   = mfs_data->nfreehfiles;
    nr_mfs_used   = mfs_data->nusedhfiles;

    nr_files     += *(mfs_data->nhfiles);

    mfs_used      = mfs_data->used_hfiles;
    mfs_free      = mfs_data->free_hfiles;


    while((f = *mfs_used) != 0) {
        remove_from_list(f, mfs_used, nr_mfs_used);
        f->f_op = f->f_inode->i_op->default_file_ops;
        put_last_in_list(f, &used_files, &nr_used_files);
    }
    if (*nr_mfs_used)
	ext2_os2_panic(1, "nr_mfs_used != 0");

    while((f = *mfs_free) != 0) {
        remove_from_list(f, mfs_free, nr_mfs_free);
        put_last_in_list(f, &free_files, &nr_free_files);
    }
    if (*nr_mfs_free)
	ext2_os2_panic(1, "nr_mfs_free != 0");

}

void invalidate_files(struct super_block *sb) {
    struct file *f;
    struct file  token;
    int          found;
    int          pass;

    memset(&token, 0, sizeof(struct file));
    token.f_magic = FILE_MAGIC;
    put_last_in_list(&token, &used_files, &nr_used_files);


    found = 1;
    pass  = 0;
    while (found) {
        pass ++;
        found = 0;
        f = used_files;
        while (((f = used_files) != 0) && (f != &token)) {
            if (f->f_inode->i_sb == sb) {
                vfs_close(f);
                found ++;
            } else {
#if 0		
                /*
                 * This will guarantee *f to be present in memory, so that we don't implicitly sleep
                 */
                if (FSH_PROBEBUF(PB_OPREAD, (char *)f, sizeof(struct file)) != NO_ERROR)
		    ext2_os2_panic(1, "invalidate_files - FSH_PROBEBUF failed");
#endif
		_disable();
                remove_from_list(f, &used_files, &nr_used_files);
                put_last_in_list(f, &used_files, &nr_used_files);
		_enable();
            }
        }
    }
    _disable();
    remove_from_list(&token, &used_files, &nr_used_files);
    _enable();
}

void invalidate_all_files(void) {
    while(used_files) {
       vfs_close(used_files);
       kernel_printf("\tinvalidate_all_files() - Found file in use");
    }
}
#endif /* MINIFSD not defined */

