/*
 * Border Line
 *
 * This program allows players to place a limited number of border lines
 * between their planets.  The border line's only purpose is to detect
 * the movement of a cloaked ship in and out of the player's space.
 * Movement is detected by Gravametric Sensor technology.
 *
 * Currently players can define 4 border lines using special planetary  
 * friendly codes.  The border lines are defined as follows where ? can
 * be any alphanumeric value.
 *
 * Line   Fcode Plan 1   Fcode Plan 2
 *   1       ?B1            ?B2
 *   2       ?B2            ?B3
 *   3       ?B3            ?B4
 *   4       ?B4            ?B5
 *
 * The lines are linked, allowing a player to almost encircle the entire
 * empire.  The only limit is that a line cannot be longer than 500 lyrs.
 * Also the line has a 60% chance of detecting a cloaked ship crossing.
 * These are the default config values.  They can be changed by adding
 * -l(value) and -d(value) during phase 2 (-2). See below.
 *
 * The program needs to be excecuted in both the AUXHOST1 and AUXHOST2
 * phase.  It also requires a -1 or -2 parameter to let the program
 * know which phase its running in.
 *
 * For HOST 3.2
 *
 *    AUXHOST1.BAT should contain
 *       border.exe -1 [directory]
 *
 *    AUXHOST2.BAT should contain
 *       border.exe -2 [directory] 
 *
 * For PHOST 3.2
 *
 *    phost -1 [directory]
 *    border -1 [directory]
 *    phost -2 [directory]
 *    border -2 [directory]
 *    phost -3 [directory]
 *
 * Please note that for HOST users, at least 255K of conventional
 * memory will be required during the HOST shell.  Since HOST keeps
 * 330K for itself, you need 585K before HOST runs.
 *
 * This file is copyrighted by Rick Rogers. You may copy and
 * distribute this file freely as long as you retain this notice.
 * If you make any changes or have any comments I'd appreciate an
 * email.  I can be reached at
 *
 *     frogers@nlnet.nf.ca
 * or  rrogers@minnie.imd.nrc.ca after Dec 11, 1995.
 *
 * A note to programmers.  This file uses the PDK library.  For
 * more information on this excellent aid to host addon programming
 * see
 *      http://www-personal.engin.umich.edu/~asterian/phost.html
 *
 * Many thanks to Andrew Sterian for some programming style suggestions
 * and for the time in getting the PDK to support the Microsoft 
 * Visual C++ compiler.
 */

#define BORDER_VERSION_MAJOR 1
#define BORDER_VERSION_MINOR 1

#include <stdio.h>
#include <stdlib.h>
#include "phostpdk.h"
#include "pgetopt.h"

#define  AND   &&
#define  OR    ||
#define  EQ    ==
#define  NEQ   !=
#define  GT    >
#define  GE    >=
#define  LT 	<
#define  LE    <=

typedef struct {
    Uns16 LineLength;
    Uns16 Detection;
} Options_Struct;

static Options_Struct gConfig = {
    500,
    60
};

Uns16 lPlan1[56],lPlan2[56];
Uns16 sOwner,lOwner;
Uns16 BorderP[6][12];
char MessageLine[20][40];
char Message[300];

double frandom(int n); /* Public domain Random number generator - see end */

static void
CheckfCode(int i)
{
   char fcode[4];
   Uns16 pOwner;
   PlanetFCode(i,fcode);
   pOwner=PlanetOwner(i);
   if (pOwner GT 0 AND fcode[1] EQ 'B' AND fcode[2] GE '1' AND fcode[2] LE '5') {
      BorderP[fcode[2]-'0'][pOwner] = i;
   }
}

static void
Border_Cross(int iLine)
{
   int j;
   char planName[21];
   lOwner=PlanetOwner(lPlan1[iLine]);
   if (frandom(-1) LT (double)gConfig.Detection/100.) {
      sprintf(MessageLine[0],"(-0000) <<Borderline Alert!>>\r");
      sprintf(MessageLine[1],"\r");
      sprintf(MessageLine[2],"\r");
      sprintf(MessageLine[3]," Gravametric sensors have\r");
      sprintf(MessageLine[4]," detected a cloaked ship\r");
      sprintf(MessageLine[5]," crossing our lines.\r");
      sprintf(MessageLine[6],"\r");
      sprintf(MessageLine[7]," The crossing occurred\r");
      sprintf(MessageLine[8]," between planets\r");
      sprintf(MessageLine[9],"\r");
      PlanetName(lPlan1[iLine], planName);
      sprintf(MessageLine[10],"     %3u - %s\r",lPlan1[iLine],planName);
      sprintf(MessageLine[11],"\r");
      PlanetName(lPlan2[iLine], planName);
      sprintf(MessageLine[12]," and %3u - %s\r",lPlan2[iLine],planName);
      for (j=13;j<=20;j++) sprintf(MessageLine[j],"\r");
      strcpy(Message,MessageLine[0]);
      for (j=1; j<=20; j++) strcat(Message,MessageLine[j]);
      /* Write Borderline cross message to player */
      WriteAUXHOSTMessage(lOwner, Message);
   }
}

static void
usage(char *argv[])
{
    fprintf(stderr,
"Usage: %s option [GameDirectory [RootDirectory]]\n"
"\n"
"options:\n"
"        -1        -- Parameter required for phase 1 execution\n"
"        -2        -- Parameter required for phase 2 execution\n"
"        -l(value) -- Maximum Line Length (default 500) with -2\n"
"        -d(value) -- Chance of cross detection (default 60%) with -2\n"
"        -h        -- prints this help summary and exits\n"
"        -v        -- prints program version and exits\n"
        ,argv[0]);
    exit(1);
}

static int
processOptions(int argc, char *argv[])
{
    int c,Phase;

/* Check for a required parameter */
    if('-' NEQ argv[1][0] AND ('1' NEQ argv[1][1] OR '2' NEQ argv[1][1]
       OR 'h' NEQ argv[1][1] OR 'v' NEQ argv[1][1])) {
        usage(argv);
    }

    while ((c=pgetopt(argc, argv, "h12vl:d:")) NEQ EOF) {
        switch (c) {
         case 'h': default:
            usage(argv);
         case '1':
            Phase=1;
            break;
         case '2':
            Phase=2;
            break;
         case 'l':
            gConfig.LineLength=atoi(poptarg);
            break;
         case 'd':
            gConfig.Detection=atoi(poptarg);
            break;
         case 'v':
            printf("Border Line version %u.%u\n", BORDER_VERSION_MAJOR, BORDER_VERSION_MINOR);
            exit(0);
        }
    }

    if (poptind < argc) {
        gGameDirectory = argv[poptind++];
    }
    if (poptind < argc) {
        gRootDirectory = argv[poptind++];
    }
    if (poptind < argc) {
        fprintf(stderr,"Too many parameters. First extraneous parameter is '%s'\n"
                       "Type '%s -h' for a usage summary.\n",
                       argv[poptind], argv[0]);
        exit(1);
    }
    return(Phase);
}

static Boolean
Between(int x1,int y1, int x2, int y2, int xi, int yi)
{
   if ( ((x1 LT xi AND xi LT x2) OR (x1 GT xi AND xi GT x2)) AND
        ((y1 LT yi AND yi LT y2) OR (y1 GT yi AND yi GT y2)) ) return(1);
   else return(0);
}

static Boolean
BetweenAndInc(int x1,int y1, int x2, int y2, int xi, int yi)
{
   if ( ((x1 LE xi AND xi LE x2) OR (x1 GE xi AND xi GE x2)) AND
        ((y1 LE yi AND yi LE y2) OR (y1 GE yi AND yi GE y2)) ) return(1);
   else return(0);
}

void
main(int argc, char *argv[])
{
    Int16 lDummy;
    Int32 XDelta,YDelta,lLength;
    Uns16 Phase,i,j,lCount;
    Uns16 X1Line,Y1Line,X2Line,Y2Line;
    Uns16 X1Ship,Y1Ship,X2Ship,Y2Ship;
    float SlopeLine,IntLine;
    float XInt,YInt,SlopeShip,IntShip;
    Uns16 XShip[501],YShip[501];
    Boolean CaseCheck,sVertical,lVertical;

    Phase = processOptions(argc, argv);

    /* Initialize the PDK */
    InitPHOSTLib();

    if (Phase EQ 1) {
       gLogFile = OpenOutputFile("border.log", GAME_DIR_ONLY | TEXT_MODE);

       Info("\nBorder Line Version %u.%u\n", BORDER_VERSION_MAJOR, BORDER_VERSION_MINOR);

       Info("Backing up current version of ship files.\n");

       if (!CopyGameFile("shipxy.hst","shipxy.brd")) exit(1);
       if (!CopyGameFile("ship.hst","ship.brd")) exit(1);

       Info("Phase One Complete.\n");
    }
    
    if (Phase EQ 2) {
       gLogFile = OpenOutputFile("border.log", GAME_DIR_ONLY | TEXT_MODE | APPEND_MODE);

       if (gConfig.LineLength LT 0 OR gConfig.LineLength GT 2000) {
          Warning("Maximum Line Length must be between 0 and 2000.");
          Info("Setting Maximum Line Length to default value.\n");
          gConfig.LineLength=500;
       }
       if (gConfig.Detection LT 0 OR gConfig.Detection GT 100) {
          Warning("Chance of Detection must be between 0 and 100.");
          Info("Setting Chance of Detection to default value.\n");
          gConfig.Detection=60;
       }
       Info("Max Line Length = %u\nChance of Detection = %u\n",gConfig.LineLength,gConfig.Detection);
       /* Manipulate shipxy file to older version to read initial ship positions */
       if (!CopyGameFile("shipxy.hst","shipxy.bak")) exit(1);
       if (!CopyGameFile("shipxy.brd","shipxy.hst")) exit(1);
       if (!CopyGameFile("ship.hst","ship.bak")) exit(1);
       if (!CopyGameFile("ship.brd","ship.hst")) exit(1);
       if (!RemoveGameFile("shipxy.brd")) exit(1);
       if (!RemoveGameFile("ship.brd")) exit(1);
       if (!Read_Ships_File(&lDummy)) exit(1); 

       /* Scan for all ships and record positions */
       Info("Recording ships and previous positions\n");
       for (i=1; i<= SHIP_NR; i++) {
          if (!IsShipExist(i)) continue;
          XShip[i]=ShipLocationX(i);
          YShip[i]=ShipLocationY(i);
       }

       /* Return new version of shipxy file */

       if (!CopyGameFile("shipxy.bak","shipxy.hst")) exit(1);
       if (!CopyGameFile("ship.bak","ship.hst")) exit(1);

       /* Free and Initialize Pdk for rereading of ship file later. */
       FreePHOSTLib();
       InitPHOSTLib();

       /* Compile all borderlines for all players. */

       memset(BorderP, 0, sizeof(BorderP));
      
       if (! Read_Planets_File(&lDummy)) exit(1);
       if (! Read_Xyplan_File()) exit(1);
       if (! Read_Planetname_File()) exit(1);
       if (! Read_HostGen_File()) exit(1);
       if (! Read_Ships_File(&lDummy)) exit(1); 

       /* Scan all planets for Border fcodes and assign to owner */
       Info("Checking for Border Lines\n");
       for (i=1; i <= PLANET_NR; i++) {
           if (IsPlanetExist(i)) CheckfCode(i);
       }

       /* Now go through all 11 players and initialize the Borderlines. */
       lCount=0;       
       for (i=1; i <= 11; i++) {
          if (!PlayerIsActive(i)) continue;

          sprintf(MessageLine[0],"(-0000) <Borderline Status Report>\r");
          sprintf(MessageLine[1],"\r");
          sprintf(MessageLine[2],"\r");
          sprintf(MessageLine[3],"  Line    Status    From    To\r");
          sprintf(MessageLine[4]," ------------------------------\r");
          sprintf(MessageLine[5],"\r");
          sprintf(MessageLine[7],"\r");
          sprintf(MessageLine[9],"\r");
          sprintf(MessageLine[11],"\r");
          sprintf(MessageLine[13],"\r");
          sprintf(MessageLine[14]," ------------------------------\r");
          sprintf(MessageLine[15],"\r");
          sprintf(MessageLine[16]," Borderline Version %u.%u\r",BORDER_VERSION_MAJOR, BORDER_VERSION_MINOR);
          sprintf(MessageLine[17],"\r");
          sprintf(MessageLine[18],"\r");
          sprintf(MessageLine[19],"\r");
          sprintf(MessageLine[20],"\r");

          /* Check for active B1-B5 planets */
          for (j=1; j <= 4; j++) {
             if (BorderP[j][i] NEQ 0 AND BorderP[j+1][i] NEQ 0) {
                /* Check Distance between planets */
                XDelta = PlanetLocationX(BorderP[j][i])-PlanetLocationX(BorderP[j+1][i]);
                YDelta = PlanetLocationY(BorderP[j][i])-PlanetLocationY(BorderP[j+1][i]);
                lLength=gConfig.LineLength;
                if ((XDelta*XDelta + YDelta*YDelta) LE lLength*lLength) {
                   lCount=lCount+1;
                   Info("LCount %u\n",lCount);
                   sprintf(MessageLine[2*j+4],"    %u     Active     %3u   %3u\r",j,BorderP[j][i],BorderP[j+1][i]);
                   lPlan1[lCount]=BorderP[j][i];
                   lPlan2[lCount]=BorderP[j+1][i];
                }
                else sprintf(MessageLine[2*j+4],"    %u    Inactive\r",j);
             }
             else sprintf(MessageLine[2*j+4],"    %u    Inactive\r",j);
          }

          /* Write Borderline status message to player i */
          strcpy(Message,MessageLine[0]);
          for (j=1; j<=20; j++) strcat(Message,MessageLine[j]);
          WriteAUXHOSTMessage(i, Message);

       } /* End of for i */

       /* Rerecord cloaked ship positions and check for a cross */

       Info("Scanning for cloaked ships, current positions");
       Info("and checking for a Border Line cross.\n");
       for (i=1 ; i<= SHIP_NR ; i++) {
          if (!IsShipExist(i)) continue;

          sOwner=ShipOwner(i);
          if ( (ShipMission(i) NEQ 10) AND (ShipMission(i) NEQ 9 AND
               sOwner NEQ 3 )) continue;

          X1Ship=XShip[i];
          Y1Ship=YShip[i];
          X2Ship=ShipLocationX(i);
          Y2Ship=ShipLocationY(i);
          if ( X2Ship NEQ X1Ship ) {
             SlopeShip=((float)Y2Ship-(float)Y1Ship)/((float)X2Ship-(float)X1Ship);
             IntShip=(float)Y2Ship-SlopeShip*(float)X2Ship;
             sVertical=0;
          }
          else {
             sVertical=1;
             IntShip=(float)X2Ship;
          }

          for (j=1; j<=lCount; j++) {
             lOwner=PlanetOwner(lPlan1[j]);
             if (lOwner EQ sOwner) continue;
             CaseCheck=0;
             X1Line=PlanetLocationX(lPlan1[j]);
             Y1Line=PlanetLocationY(lPlan1[j]);
             X2Line=PlanetLocationX(lPlan2[j]);
             Y2Line=PlanetLocationY(lPlan2[j]);
             if (X1Line NEQ X2Line) {
                SlopeLine=((float)Y2Line-(float)Y1Line)/((float)X2Line-(float)X1Line);
                IntLine=(float)Y2Line-SlopeLine*(float)X2Line;
                lVertical=0;
             }
             else {
                lVertical=1;
                IntLine=(float)X1Line;
             }

             /*Check for all cases*/

             /* Case 1: Ship doesn't move but may be on line */
             if ((X2Ship EQ X1Ship) AND (Y2Ship EQ Y1Ship)) {
                CaseCheck=1;
                if (lVertical) {
                   if ( ((float)X2Ship EQ IntLine) AND Between(X1Line,Y1Line,X2Line,Y2Line,X2Ship,Y2Ship) ) Border_Cross(j);
                }
                else if ( (fabs((float)Y2Ship-(SlopeLine*(float)X2Ship+IntLine)) LE 0.5) AND Between(X1Line,Y1Line,X2Line,Y2Line,X2Ship,Y2Ship)) Border_Cross(j);
             }
          
             if (!CaseCheck) {
                /* Case 2: Ship moving along line */
                if (lVertical AND sVertical) {
                   if ( ((float)X2Ship EQ IntLine) AND BetweenAndInc(X1Line,Y1Line,X2Line,Y2Line,X2Ship,Y2Ship) ) {
                      Border_Cross(j);
                      CaseCheck=1;
                   }
                }
                else if ( (fabs(SlopeShip-SlopeLine) LT 0.000001) AND (fabs(IntShip-IntLine) LT 0.000001) AND BetweenAndInc(X1Line,Y1Line,X2Line,Y2Line,X2Ship,Y2Ship)) {
                   Border_Cross(j);
                   CaseCheck=1;
                }
             }

             if(!CaseCheck) {
                /* Case 3: Ship crossed a line */
                if (lVertical AND !sVertical) {
                   YInt = SlopeShip*(float)X1Line + IntShip;
                   XInt = (float)X1Line;
                   if ( Between(X1Line,Y1Line,X2Line,Y2Line,(int)XInt,(int)YInt) AND BetweenAndInc(X1Ship,Y1Ship,X2Ship,Y2Ship,(int)XInt,(int)YInt)) Border_Cross(j);
                }
                if (!lVertical AND sVertical) {
                   YInt = SlopeLine*(float)X2Ship + IntLine;
                   XInt = (float)X2Ship;
                   if ( Between(X1Line,Y1Line,X2Line,Y2Line,(int)XInt,(int)YInt) AND BetweenAndInc(X1Ship,Y1Ship,X2Ship,Y2Ship,(int)XInt,(int)YInt)) Border_Cross(j);
                }
                if (!lVertical AND !sVertical) {
                   XInt = (IntShip-IntLine)/(SlopeLine-SlopeShip);
                   YInt = SlopeShip*XInt + IntShip;
                   if ( Between(X1Line,Y1Line,X2Line,Y2Line,(int)XInt,(int)YInt) AND BetweenAndInc(X1Ship,Y1Ship,X2Ship,Y2Ship,(int)XInt,(int)YInt)) Border_Cross(j);
                }
             }
          } /* End of for j */
       } /* End of for i */

       Info("Phase Two Complete.");
       AUXHOSTMessageCleanup();

    } /* End of if Phase=2 */

    FreePHOSTLib();
    exit(0);
}

/*
**  FRAND.C - Public domain by Larry Hudson
*/

#include <math.h>
#include <time.h>

#define TEN_PI 31.41592653589793
#define E      2.718281828459045

/*--------------------------------------------------------------+
|           Return random double between 0.0 and 1.0            |
|                                                               |
|    If n is negative it will randomize the seed, based on the  |
|        current MSDOS time.                                    |
|    If n is zero it will return the next random number.        |
|    If n is positive it will set the seed to a value based on  |
|        the value of n.                                        |
+--------------------------------------------------------------*/

double frandom(int n)
{
      static double seed = E;
      double dummy;
      time_t tim;

      if (n < 0)
      {
            time(&tim);
            seed = (double)tim;
      }
      else if (n > 0)
            seed = (double)n * E;

      seed = modf(seed * TEN_PI + E, &dummy);
      return seed;
}
