unit Data;
{ Game data for Jeu de Taquin. Version 1.1 }

interface

uses
  Windows, Messages, Utility;

const
  NUMTILES = 4;
  EMPTY    = 16;
  NUMSCRAMBLES = 100;

type
  TGameState = (gsIdling, gsScrambling, gsPlaying); { game states }

  TGameData = class(TObject)
  private
    BlankRow, BlankColumn: Integer;
    Tiles: array [1..NUMTILES, 1..NUMTILES] of Integer;
    FState: TGameState;
  public
    constructor Create; virtual;
    procedure Initialize;
    function GetTileAt(Row, Column: Integer): Integer;
    procedure SetTileAt(Row, Column: Integer; Value: Integer);
    property State: TGameState read FState write FState;
    function GetBlankRow: Integer;
    function GetBlankColumn: Integer;
    procedure PaintTileAt(Row, Column: Integer; PaintDC: HDC; LogFont: TLogFont);
    procedure Scramble(Window: HWND);
    function Solved: Boolean;
  end;

var
  TileWidth, TileHeight: Integer;

implementation

uses
  Classes; { a VCL unit that contains the useful Bounds function (see PaintTileAt) }

constructor TGameData.Create;
begin
  inherited Create;

  Initialize;
  State := gsIdling;
end;

procedure TGameData.Initialize;
var Row, Column: Integer;
begin
  for Row := 1 to NUMTILES do
    for Column := 1 to NUMTILES do
      Tiles[Row, Column] := NUMTILES * (Row - 1) + Column;
  BlankRow := 4;
  BlankColumn := 4;
end;

function TGameData.GetTileAt(Row, Column: Integer): Integer;
begin
  Result := Tiles[Row, Column];
end;

procedure TGameData.SetTileAt(Row, Column: Integer; Value: Integer);
begin
  Tiles[Row, Column] := Value;
  if Value = EMPTY then
  begin
    BlankRow := Row;
    BlankColumn := Column;
  end;
end;

function TGameData.GetBlankRow: Integer;
begin
  Result := BlankRow;
end;

function TGameData.GetBlankColumn: Integer;
begin
  Result := BlankColumn;
end;

{ Paint a tile at (Row,Column) using the passed device context and logical font }
procedure TGameData.PaintTileAt(Row, Column: Integer; PaintDC: HDC; LogFont: TLogFont);
var
  TileRect: TRect;
  Value: Integer;
  BrushColor: Integer;
  BackBrush, OldBrush: HBRUSH;
  FramePen, OldPen: HPEN;
begin
  Value := GetTileAt(Row, Column);
  if Value = EMPTY then
    BrushColor := COLOR_WINDOW
  else
    BrushColor := COLOR_BTNFACE;

  { Create the background brush and select it. }
  BackBrush := CreateSolidBrush(GetSysColor(BrushColor));
  OldBrush := HBRUSH(SelectObject(PaintDC, BackBrush));

  { Create the frame pen and select it. }
  FramePen := CreatePen(PS_SOLID, 1, GetSysColor(COLOR_WINDOWFRAME));
  OldPen := SelectObject(PaintDC, FramePen);

  { Calculate the rectangle for the tile at (Row, Column). }
  TileRect := Bounds(TileWidth * (Column - 1),
                     TileHeight * (Row - 1),
                     TileWidth, TileHeight);

  { Draw tile rectangle around the tile using the frame pen, }
  { and fill it using the background brush. }
  Rectangle(PaintDC, TileRect.Left, TileRect.Top, TileRect.Right, TileRect.Bottom);

  if Value <> EMPTY then
  begin
    { Draw highlight and shadow for tile. }
    Draw3DBorder(PaintDC, TileRect);

    { Draw the value inside the tile. }
    DrawValue(PaintDC, TileRect, Value, LogFont);
  end;

  SelectObject(PaintDC, OldPen);
  DeleteObject(FramePen);

  SelectObject(PaintDC, OldBrush);
  DeleteObject(BackBrush);
end;

{ Scramble the puzzle }
procedure TGameData.Scramble(Window: HWND);
var
  Count: Integer;
  WindowMessage: TMsg;
  XPos, YPos: Integer;
begin
  for Count := 1 to NUMSCRAMBLES do
  begin
    { let other applications execute }
    while PeekMessage(WindowMessage, 0, 0, 0, PM_REMOVE) do
    begin
      TranslateMessage(WindowMessage);
      DispatchMessage(WindowMessage);
    end;

    { To move a tile we simulate a WM_LBUTTONDOWN message. We calculate
      the (X,Y) position of our "mouse click" using random neighbors of
      the blank cell. }
    XPos := Random(NUMTILES) + 1;
    YPos := BlankRow;
    SendMessage(Window, WM_LBUTTONDOWN, 0,
      MakeLParam((XPos - 1) * TileWidth, (YPos - 1) * TileHeight));
    UpdateWindow(Window);

    XPos := BlankColumn;
    YPos := Random(NUMTILES) + 1;
    SendMessage(Window, WM_LBUTTONDOWN, 0, MakeLParam((XPos - 1) * TileWidth, (YPos - 1) * TileHeight));
    UpdateWindow(Window);
  end;
end;

{ Check for a solution }
function TGameData.Solved;
var
  Row, Column: Integer;
begin
  Result := True; { assume a solution is found unless proved otherwise }

  { Iterate through the tiles. If an "out of order" tile is found,
    correct the result and break out of the loop. }
  for Row := 1 to NUMTILES do
    for Column := 1 to NUMTILES do
      if GetTileAt(Row, Column) <> NUMTILES * (Row - 1) + Column then
      begin
        Result := False;
        Break;
      end;
end;

begin
  TileWidth := 48;   { use a default size for the tiles }
  TileHeight := 48;  { changing these also affects the window size }
end.

