unit Utility;

{ Utility procedures and functions for Jeu de Taquin.
  Version 1.1. }
  
interface

uses
  Windows, SysUtils;

procedure ShowMessage(Instance: HINST; ParentWindow: HWnd; MessageID, TitleID: Integer; Style: UINT; var RC: Integer);
procedure Draw3DBorder(DC: HDC; Rect: TRect);
procedure DrawValue(DC: HDC; Rect: TRect; Value: Integer; LogFont: TLogFont);
function  Min(a, b: Integer): Integer;
function  Max(a, b: Integer): Integer;
procedure CenterWindow(ChildWindow, ParentWindow: HWND; var OK: Boolean);
procedure AdjustForStatusBar(MainWindow: HWND; StatusBarWindow: HWND; Show: Boolean);


implementation

uses
  Classes; { pull in VCL's handy Rect function }
  
var
  MessageString: array [0..255] of Char;
  TitleString: array [0..31] of Char;

{ A generic message box which loads its caption and message from the stringtable }
procedure ShowMessage(Instance: HINST; ParentWindow: HWnd; MessageID, TitleID: Integer; Style: UINT; var RC: Integer);
begin
  RC := 0; { initialize the return code }

  { If both string loads are successful, display the message box. }
  if (LoadString(Instance, MessageID, MessageString, SizeOf(MessageString)) <> 0) and
     (LoadString(Instance, TitleID, TitleString, SizeOf(TitleString)) <> 0) then
  begin
    RC := MessageBox(ParentWindow, MessageString, TitleString, Style);
  end;
end;

{ Draw a 3D border inside the passed rectangle on the specified device context }
procedure Draw3DBorder(DC: HDC; Rect: TRect);
var
  HighlightPen, ShadowPen, OldPen: HPEN;
begin
  HighlightPen := CreatePen(PS_SOLID, 1, GetSysColor(COLOR_BTNHIGHLIGHT));
  ShadowPen := CreatePen(PS_SOLID, 1, GetSysColor(COLOR_BTNSHADOW));

  OldPen := SelectObject(DC, HighlightPen);

  with Rect do
  begin
    MoveToEx(DC, Left + 1, Top + 1, nil);
    LineTo(DC, Right - 1, Top + 1);
    MoveToEx(DC, Left + 1, Top + 1, nil);
    LineTo(DC, Left + 1, Bottom - 1);

    MoveToEx(DC, Left + 2, Top + 2, nil);
    LineTo(DC, Right - 2, Top + 2);
    MoveToEx(DC, Left + 2, Top + 2, nil);
    LineTo(DC, Left + 2, Bottom - 2);

    SelectObject(DC, ShadowPen);

    MoveToEx(DC, Right - 2, Bottom - 2, nil);
    LineTo(DC, Right - 2, Top + 1);
    MoveToEx(DC, Right - 2, Bottom - 2, nil);
    LineTo(DC, Left + 1, Bottom - 2);

    MoveToEx(DC, Right - 3, Bottom - 3, nil);
    LineTo(DC, Left + 2, Bottom - 3);
    MoveToEx(DC, Right - 3, Bottom - 3, nil);
    LineTo(DC, Right - 3, Top + 2);

    SelectObject(DC, OldPen);
    DeleteObject(ShadowPen);
    DeleteObject(HighlightPen);
  end; { with Rect }
end;

{ Draw a numeric value inside the given rectangle using the given font }
procedure DrawValue(DC: HDC; Rect: TRect; Value: Integer; LogFont: TLogFont);
var
  TempString: string;
  ValueString: array [0..3] of Char;
  NewFont, OldFont: HFONT;
begin
  { Convert the value into a string }
  TempString := IntToStr(Value);
  StrPCopy(ValueString, TempString);

  { Create a new font based on the logical font parameter }
  NewFont := CreateFontIndirect(LogFont);
  OldFont := SelectObject(DC, NewFont);

  { Use the button color to draw the text. Use the transparent drawing mode
    so that the background will show through from under the number }
  SetTextColor(DC, GetSysColor(COLOR_BTNTEXT));
  SetBkMode(DC, TRANSPARENT);
  DrawText(DC, ValueString, -1, Rect, DT_SINGLELINE or DT_CENTER or DT_VCENTER );
  SetBkMode(DC, OPAQUE);

  SelectObject(DC, OldFont);
  DeleteObject(NewFont);
end;


function Min(a, b: Integer): Integer;
begin
  if a <= b then
    Min := a
  else
    Min := b
end;

function Max(a, b: Integer): Integer;
begin
  if a > b then
    Max := a
  else
    Max := b
end;

{ Center a window over its parent. To center a main window, pass the
  desktop window as the parent. }
procedure CenterWindow(ChildWindow, ParentWindow: HWND; var OK: Boolean);
var
  ChildRect, ParentRect, WorkArea: TRect;
  ChildWidth, ChildHeight, ParentWidth, ParentHeight: Integer;
  NewX, NewY: Integer;
begin
  { Get the height and width of the child window: }
  GetWindowRect(ChildWindow, ChildRect);
  ChildWidth  := ChildRect.Right - ChildRect.Left;
  ChildHeight := ChildRect.Bottom - ChildRect.Top;

  { Get the height and width of the parent window: }
  GetWindowRect(ParentWindow, ParentRect);
  ParentWidth  := ParentRect.Right - ParentRect.Left;
  ParentHeight := ParentRect.Bottom - ParentRect.Top;

  { Account for the Windows 95 Taskbar: }
  OK := SystemParametersInfo(SPI_GETWORKAREA, SizeOf(TRect), @WorkArea, 0);
  if not OK then
    WorkArea := Rect(0, 0, GetSystemMetrics(SM_CXSCREEN), GetSystemMetrics(SM_CYSCREEN));

  NewX := ParentRect.Left + ((ParentWidth - ChildWidth) div 2);
  NewY := ParentRect.Top  + ((ParentHeight - ChildHeight) div 2);

  if NewX < WorkArea.Left then
    NewX := WorkArea.Left
  else if NewX + ChildWidth > WorkArea.Right then
    NewX := WorkArea.Right - ChildWidth;

  if NewY < WorkArea.Top then
    NewY := WorkArea.Top
  else if NewY + ChildHeight > WorkArea.Bottom then
    NewY := WorkArea.Bottom - ChildHeight;

  { Move the window to the new position. Retain the same size and Z-order }
  OK := SetWindowPos(ChildWindow, 0, NewX, NewY, 0, 0,
                     SWP_NOSIZE or SWP_NOZORDER);
end;

{ Adjust the window size to accommodate the status bar. }
procedure AdjustForStatusBar(MainWindow: HWND; StatusBarWindow: HWND; Show: Boolean);
var
  WindowRect, StatusRect: TRect;
  NewHeight: Integer;
begin
  { Adjust the window size to accommodate the status bar. }
  GetWindowRect(MainWindow, WindowRect);
  GetWindowRect(StatusBarWindow, StatusRect);
  if Show then
    NewHeight := WindowRect.Bottom - WindowRect.Top +
                 StatusRect.Bottom - StatusRect.Top
  else
    NewHeight := (WindowRect.Bottom - WindowRect.Top) -
                 (StatusRect.Bottom - StatusRect.Top);
  MoveWindow(MainWindow, WindowRect.Left, WindowRect.Top,
             WindowRect.Right - WindowRect.Left, NewHeight, True);
end;

end.
