/******************************************************************************
	MODULE:		NODES.C
	WRITTEN BY:	Robert Fenske, Jr. (rfenske@swri.edu)
				Southwest Research Institute
				Electromagnetics Division
				6220 Culebra
				San Antonio, Texas 78238-5166
	CREATED:	Mar. 1994
	DESCRIPTION:	This module contains routines to generate the SEGS,
			SSECTORS, and NODES sections.  It needs only the
			LINEDEFS and VERTEXES as input.  These three sections
			combine to form a binary space partition tree.  This
			tree is built by recursively dividing the space into
			sections that balance (or at least try to) the number
			of segments and produce the least number of segment
			splits.  The nodes are kept in a binary tree.  The
			segments are kept in an ordered doubly-linked list.
			The created vertices are added to the end of the
			vertex list.  Once the divisions are complete, the
			SEGS, SSECTORS, and NODES structures are created from
			the binary tree and the segment and vertex lists.  Any
			memory allocated for the binary tree or the linked
			list is freed when no longer needed.

			This module does not do any error checking on any
			memory allocation.  If any allocation ever fails, this
			module will bomb.

			This module used to do some error checking while
			building the node tree, but now the tree is generated
			regardless of whether the input describes a correct,
			playable map (though it certainly may bomb if the
			input is not a valid map).

			I wrote the code from the description of the binary
			space partition in the Unofficial DOOM Specs written
			by Matt Fell.  I found these references after I did
			the coding:

			1) Computer Graphics Principles and Practice,
			   2nd ed 1990
			   Foley J.D., van Dam A., Feiner S.K., Hughes J.F.
			   ISBN 0-201-12110-7

			2) "On Visible Surface Generation by a Priori Tree
			   Structures"
			   Fuchs H., Kedem Z.M., Naylor B.F.
			   Computer Graphics (SIGGRAPH '80 Proceedings)
			   v14 #3 July 1980 pp 124-133

			3) "Near Real-Time Shaded Display of Rigid Objects"
			   Fuchs H., Abram G.D., Grant E.D.
			   Computer Graphics (SIGGRAPH '83 Proceesings)
			   v17 #3 July 1983 pp 65-72

			DOOM is a trademark of id Software, Inc.
******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "dmglobal.i"

#if !defined(M_SQRT2)				/* not all systems have this */
#define M_SQRT2		1.41421356237309504880	/* defined (why?)            */
#endif

#define tree_branch(c)	((c)=blockmem(NODE_TREE,1), \
			 (c)->left=NULL, (c)->right=NULL, (c))
#define two_sided(l)	(lines[l].lsidndx != -1)
#define vdist2(v1,v2)	(((long)(v1).x-(v2).x)*((long)(v1).x-(v2).x)+\
			 ((long)(v1).y-(v2).y)*((long)(v1).y-(v2).y))

typedef struct SEGS_INFO {
	WAD_SEGS seg;				/* a SEGment */
	short sector;				/* associated sector number */
	struct SEGS_INFO *prev, *next;		/* to previous, next SEGment */
} SEGS_INFO;

typedef struct NODE_TREE {
	short ymin, ymax, xmin, xmax;		/* node rectangle limits */
	SEGS_INFO *pseg;			/* partition line SEG */
	SEGS_INFO *segs;			/* node's SEGS */
	short nsegs;				/* # initial SEGS in node */
	struct NODE_TREE *left, *right;		/* left, right children */
} NODE_TREE;

typedef struct NODE_INFO {
	WAD_NODE *nodes;			/* all nodes */
	int nnodes;				/* total # NODES */
	WAD_VERT *sverts;			/* all SEGS vertices */
	int nsverts;				/* total # SEGS vertices */
	WAD_SEGS *segs;				/* all nodes' SEGS */
	int nsegs;				/* total # segs */
	WAD_SSECTOR *ssecs;			/* all nodes' SSECTORs */
	int nssecs;				/* total # SSECTORs */
	SEGS_INFO *sinfo;			/* all SEGS lists */
	int nverts;				/* # original vertices */
	int nlines;				/* # lines */
	boolean unique_sector;			/* SSECTORs w/uniq sects flg */
	boolean use_all_segs;			/* use all SEGS flag */
	boolean limit_io;			/* limit screen I/O flag */
} NODE_INFO;

local NODE_INFO ninfo;				/* node information */


/******************************************************************************
	ROUTINE:	nodes_segs_init(lines,nlines,sides)
	WRITTEN BY:	Robert Fenske, Jr.
	CREATED:	Mar. 1994
	DESCRIPTION:	This routine initializes the SEGS linked list by
			scanning the LINEDEFS list and creating the
			appropriate SEGS entries.  A seg is created for each
			side a LINEDEF has.  It returns the number of SEGS
			created.
	MODIFIED:		Robert Fenske, Jr.	May  1995
			Added the option to ignore LINEDEFS that do not have
			any affect on the display seen by the player.  This
			idea came from a suggestion by Bernd Kreimeier.
******************************************************************************/
#if defined(ANSI_C)
local int nodes_segs_init(register WAD_LINE *lines, long nlines,
                          WAD_SIDE *sides)
#else
local int nodes_segs_init(lines,nlines,sides)
register WAD_LINE *lines;
long nlines;
WAD_SIDE *sides;
#endif
{
  WAD_VERT *vf, *vt;
  int i;
  register SEGS_INFO *sinfo;
  register int nsegs;
  register int l;

  nsegs = 0;
  ninfo.sinfo = sinfo = blockmem(SEGS_INFO,1);
  sinfo->prev = NULL;
  for (l = 0; l < nlines; l++) {		/* scan all lines */
    if (!ninfo.use_all_segs &&
        two_sided(l) &&				/* two-sided lines whose    */
        sides[lines[l].rsidndx].sectndx ==	/* sides reference the same */
        sides[lines[l].lsidndx].sectndx &&	/* sector and have no       */
        strcmp(sides[lines[l].rsidndx].nwall,"-") == 0 &&/* textures can be */
        strcmp(sides[lines[l].lsidndx].nwall,"-") == 0)/* ignored           */
      continue;
    for (i = 0; i <= two_sided(l); i++) {
      sinfo->seg.fndx = i==0 ? lines[l].fndx : lines[l].tndx;
      sinfo->seg.tndx = i==0 ? lines[l].tndx : lines[l].fndx;
      vf = &ninfo.sverts[sinfo->seg.fndx], vt = &ninfo.sverts[sinfo->seg.tndx];
      if (vf->x == vt->x && vf->y == vt->y)	/* ignore zero-length lines */
        continue;
      sinfo->seg.angle = (bams)(vt->y==vf->y && vt->x<vf->x ? BAMS180 :
                                atan2((double)(vt->y-vf->y),
                                      (double)(vt->x-vf->x))*rad_to_bams+
                                0.5*sgn(vt->y-vf->y));
      sinfo->seg.sndx = i;			/* 0=right side,1=left side */
      sinfo->seg.lndx = l;
      sinfo->seg.loffset = 0;
      if (i == 0) sinfo->sector = sides[lines[l].rsidndx].sectndx;
      else        sinfo->sector = sides[lines[l].lsidndx].sectndx;
      nsegs++;
      sinfo->next = blockmem(SEGS_INFO,1);	/* set up for next one */
      sinfo->next->prev = sinfo;
      sinfo = sinfo->next;
    }
  }
  sinfo->prev->next = NULL;
  blockfree(sinfo);				/* don't need this one */
  return nsegs;					/* return # created SEGS */
}


/******************************************************************************
	ROUTINE:	nodes_split_seg(pseg,seg,right,left)
	WRITTEN BY:	Robert Fenske, Jr.
	CREATED:	Mar. 1994
	DESCRIPTION:	This routine splits the input segment into left and
			right segments based on the input partition line.  The
			intersection of the partition line (based on the input
			"from" and "to" coordinates) with the segment is found
			so that a new vertex can be added to the vertex list.
			The offset field of the left and right segments are
			computed from the distance to the new vertex along the
			segment.
******************************************************************************/
#if defined(ANSI_C)
local void nodes_split_seg(SEGS_INFO *pseg, SEGS_INFO *seg,
                           register SEGS_INFO **right,
                           register SEGS_INFO **left)
#else
local void nodes_split_seg(pseg,seg,right,left)
SEGS_INFO *pseg, *seg;
register SEGS_INFO **right, **left;
#endif
{
  WAD_VERT *pf = &ninfo.sverts[pseg->seg.fndx],
            *pt = &ninfo.sverts[pseg->seg.tndx],
            *vf = &ninfo.sverts[seg->seg.fndx],
            *vt = &ninfo.sverts[seg->seg.tndx];
  long Ap = -((long)pt->y - pf->y),		/* partition line is */
       Bp = (long)pt->x - pf->x,		/* Ax + By + C = 0   */
       Cp = (long)pt->y*pf->x - (long)pt->x*pf->y;
  long As = -((long)vt->y - vf->y),		/* SEG line is     */
       Bs = (long)vt->x - vf->x,		/* Ax + By + C = 0 */
       Cs = (long)vt->y*vf->x - (long)vt->x*vf->y;
  double x, y;					/* intersection coords */
  WAD_VERT iv;					/* intersection vertex */
  register int v;				/* intersection vertex index */

  *right = blockmem(SEGS_INFO,1);		/* new right side SEG */
  (*right)->seg = seg->seg;
  (*right)->sector = seg->sector;
  (*right)->next = NULL;
  *left = blockmem(SEGS_INFO,1);		/* new left side SEG */
  (*left)->seg = seg->seg;
  (*left)->sector = seg->sector;
  (*left)->next = NULL;
  x =  ((double)Bs*Cp - (double)Bp*Cs)/((double)Bp*As - (double)Bs*Ap);
  y = -((double)As*Cp - (double)Ap*Cs)/((double)Bp*As - (double)Bs*Ap);
  iv.x = x + sgn(x)*0.5;
  iv.y = y + sgn(y)*0.5;
  for (v = ninfo.nsverts-1; v >= ninfo.nverts; v--)/* check if vert already */
    if (ninfo.sverts[v].x == iv.x &&		/* used (will happen if 2-  */
        ninfo.sverts[v].y == iv.y) break;	/* sided LINEDEF split)     */
  if (v < ninfo.nverts)
    ninfo.sverts[v = ninfo.nsverts++] = iv;	/* add new vertex to list */
  if (seg->seg.sndx == 0) {			/* this is a right side SEG */
    if (Ap*vf->x + Bp*vf->y + Cp < 0) {
      (*right)->seg.tndx = v;
      (*left)->seg.fndx = v;
      (*left)->seg.loffset = seg->seg.loffset + sqrt((double)vdist2(*vf,iv));
    }else {
      (*right)->seg.fndx = v;
      (*right)->seg.loffset = seg->seg.loffset + sqrt((double)vdist2(*vf,iv));
      (*left)->seg.tndx = v;
    }
  }else {					/* this is a left side SEG */
    if (Ap*vt->x + Bp*vt->y + Cp < 0) {
      (*right)->seg.fndx = v;
      (*right)->seg.loffset = seg->seg.loffset + sqrt((double)vdist2(*vt,iv));
      (*left)->seg.tndx = v;
    }else {
      (*right)->seg.tndx = v;
      (*left)->seg.fndx = v;
      (*left)->seg.loffset = seg->seg.loffset + sqrt((double)vdist2(*vt,iv));
    }
  }
}


/******************************************************************************
	ROUTINE:	nodes_insert_seg(seglist,seg,preinsert)
	WRITTEN BY:	Robert Fenske, Jr.
	CREATED:	Mar. 1994
	DESCRIPTION:	This routine inserts the input SEG into the SEGS list
			either before or after the SEG that seglist references,
			depending on the preinsert flag.
******************************************************************************/
#if defined(ANSI_C)
local void nodes_insert_seg(register SEGS_INFO *seglist,
                            register SEGS_INFO *seg,
                            int preinsert)
#else
local void nodes_insert_seg(seglist,seg,preinsert)
register SEGS_INFO *seglist, *seg;
int preinsert;
#endif
{
  if (preinsert) {				/* insert before */
    seg->prev = seglist->prev;
    seg->next = seglist;
  }else {					/* insert after */
    seg->prev = seglist;
    seg->next = seglist->next;
  }
  if (seg->prev != NULL) seg->prev->next = seg;
  if (seg->next != NULL) seg->next->prev = seg;
}


/******************************************************************************
	ROUTINE:	nodes_segs_bounds(node)
	WRITTEN BY:	Robert Fenske, Jr.
	CREATED:	Mar. 1994
	DESCRIPTION:	This routine scans all the SEGS contained in the input
			node to find the minimum and maximum coordinates for
			the node boundaries.
******************************************************************************/
#if defined(ANSI_C)
local void nodes_segs_bounds(register NODE_TREE *node)
#else
local void nodes_segs_bounds(node)
register NODE_TREE *node;
#endif
{
  register SEGS_INFO *sinfo;
  register WAD_VERT *vf, *vt;
  register int s;

  node->xmin = node->ymin = ~(-1<<(8*sizeof(node->xmin)-1));
  node->xmax = node->ymax = -node->xmin;
  for (sinfo = node->segs, s = 0; s < node->nsegs; s++) {
    vf = &ninfo.sverts[sinfo->seg.fndx], vt = &ninfo.sverts[sinfo->seg.tndx];
    if (vf->x < node->xmin) node->xmin = vf->x;
    if (vf->y < node->ymin) node->ymin = vf->y;
    if (vt->x < node->xmin) node->xmin = vt->x;
    if (vt->y < node->ymin) node->ymin = vt->y;
    if (node->xmax < vf->x) node->xmax = vf->x;
    if (node->ymax < vf->y) node->ymax = vf->y;
    if (node->xmax < vt->x) node->xmax = vt->x;
    if (node->ymax < vt->y) node->ymax = vt->y;
    sinfo = sinfo->next;
  }
}


/******************************************************************************
	ROUTINE:	nodes_decide_side(pseg,seg)
	WRITTEN BY:	Robert Fenske, Jr.
	CREATED:	Mar. 1994
	DESCRIPTION:	This routine returns whether the input segment is
			on the right side, left side, or both sides of the
			partition line of the input node.  This is done by
			determining on which side of the partition line each
			vertex of the seg lies.  Given that the partition line
			is Ax + By + C = 0 and a vertex is (Vx,Vy), the
			intersection of the partition line and the shortest-
			distance line from the vertex to the partition line
			is given by
				Xi = Vx - b * d
				Yi = Vy - a * d
			where a = B/(A^2+B^2)^.5, b = A/(A^2+B^2)^.5,
			c = C/(A^2+B^2)^.5, and d = a*Vx + b*Vy + c.  Since
			the directional information can be obtained from
			Xi - Vx = Vx - b*d - Vx = -b*d and
			Yi - Vx = Vy - a*d - Vy = -a*d, only d is needed to
			determine which side the vertex lies on.  Thus only
			the sign (-1, 0, or +1) of d = (A*Vx + B*Vy + C) /
			(A^2 + B^2)^.5 needs to be considered.  This simple
			matrix can be used to determine the side:
				"from"       "to" vertex
				vertex	left	on	right

				left	left	left	both
				on	left	?	right
				right	both	right	right
			For the ? case, the side information in the seg must
			be used to determine the "sidedness".  Right is denoted
			with 0, left denoted by 1, and both denoted by -1.
			A, B, C, and d are calculated only when required to
			enhance the speed of the routine.
******************************************************************************/
#if defined(ANSI_C)
local int nodes_decide_side(WAD_SEGS *pseg, WAD_SEGS *seg)
#else
local int nodes_decide_side(pseg,seg)
WAD_SEGS *pseg, *seg;
#endif
{
  static int rightleft[3][3] = { {  1,  1, -1 },
                                 {  1, -2,  0 },
                                 { -1,  0,  0 } };
  static WAD_VERT *lpf = NULL, *lpt = NULL;	/* last partition line verts */
  static long A, B, C;				/* describes partition line */
  static long pd;
  WAD_VERT *pf = &ninfo.sverts[pseg->fndx],	/* partition line vertices */
           *pt = &ninfo.sverts[pseg->tndx],
           *vf = &ninfo.sverts[seg->fndx],	/* segment vertices */
           *vt = &ninfo.sverts[seg->tndx];
  long pfd, ptd;
  int sideflag;
  register int fside, tside;			/* "from"/"to" vertex side */

  if (lpf != pf || lpt != pt) {			/* update A,B,C if have to */
    A = -((long)pt->y - pf->y);			/* partition line is */
    B = (long)pt->x - pf->x;			/* Ax + By + C = 0   */
    C = (long)pt->y*pf->x - (long)pt->x*pf->y;
    pd = (long)(sqrt((double)A*A+(double)B*B)/M_SQRT2+0.5);
    lpf = pf;					/* save new vertices */
    lpt = pt;
  }
  pfd = A*vf->x + B*vf->y + C;
  fside = (pfd>=0)-(pfd<=-0);			/* "from" vertex side */
  ptd = A*vt->x + B*vt->y + C;
  tside = (ptd>=0)-(ptd<=-0);			/* "to" vertex side */
  sideflag = rightleft[1-fside][1-tside];
  if (sideflag == -1) {				/* test again for "both" */
    fside = (pfd>=pd)-(pfd<=-pd);		/* "from" vertex side */
    tside = (ptd>=pd)-(ptd<=-pd);		/* "to" vertex side */
    sideflag = rightleft[1-fside][1-tside];
  }
  if (sideflag == -2)				/* need to determine based */
    sideflag = pseg->angle != seg->angle;	/* on direction            */
  return sideflag;
}


/******************************************************************************
	ROUTINE:	nodes_divide_node(node,left,right)
	WRITTEN BY:	Robert Fenske, Jr.
	CREATED:	Mar. 1994
	DESCRIPTION:	This routine divides the input node into left and
			right nodes based on the partition line of the input
			node.  This essentially means that the list of SEGS
			associated with the input node must be divided into
			left and right collections of SEGS.  This division is
			done by moving all the left side SEGS to the end of
			the SEGS list, leaving all right side SEGS at the front
			of the list.  Those SEGS that need to be split have
			their new right side SEG take the position of the old
			SEG and their new left side SEG put at the end of the
			list.  Once the list is divided, the right and left
			nodes are set the appropriate section of the list and
			their bounds are computed.
******************************************************************************/
#if defined(ANSI_C)
local void nodes_divide_node(register NODE_TREE *node, NODE_TREE *right,
                             NODE_TREE *left)
#else
local void nodes_divide_node(node,right,left)
register NODE_TREE *node;
NODE_TREE *right, *left;
#endif
{
  int sideflag;
  int i;
  SEGS_INFO *next, *end;
  SEGS_INFO *lseg, *rseg;			/* for splitting seg in two */
  register SEGS_INFO *sinfo;

  sinfo = node->segs;
  right->nsegs = left->nsegs = 0;
  for (end = sinfo, i = 0; i < node->nsegs-1; i++) end = end->next;
  for (i = 0; i < node->nsegs; i++) {		/* scan all node's SEGS */
    if (node->pseg == sinfo)			/* either always on right */
      sideflag = node->pseg->sector != -1 ? 0 : 1;/* side or forced to be */
    else					/* on left side           */
      sideflag = nodes_decide_side(&node->pseg->seg,&sinfo->seg);
    next = sinfo->next;
    switch (sideflag) {
      case   0: right->nsegs++;			/* just add to right's total */
      bcase  1: if (sinfo->prev != NULL)	/* move to end of list */
                  sinfo->prev->next = sinfo->next;
                if (sinfo->next != NULL)
                  sinfo->next->prev = sinfo->prev;
                if (end == sinfo) end = sinfo->prev;
                if (node->segs == sinfo) node->segs = sinfo->next;
                nodes_insert_seg(end,sinfo,FALSE);
                end = sinfo;
                left->nsegs++;
      bcase -1: nodes_split_seg(node->pseg,sinfo,&rseg,&lseg);
                sinfo->seg = rseg->seg;		/* make this the right SEG */
                right->nsegs++;
                blockfree(rseg);		/* don't need this now */
                nodes_insert_seg(end,lseg,FALSE);/* add left SEG to end */
                end = lseg;
                left->nsegs++;
                ninfo.nsegs++;			/* one more for total */
    }
    sinfo = next;
  }
  for (sinfo = node->segs, i = 0; i < right->nsegs; i++)
    sinfo = sinfo->next;
  right->segs = node->segs;			/* SEGS on right side of   */
  nodes_segs_bounds(right);			/* partition line are here */
  left->segs = sinfo;				/* SEGS on left side of    */
  nodes_segs_bounds(left);			/* partition line are here */
}


/******************************************************************************
	ROUTINE:	nodes_partition_line(node)
	WRITTEN BY:	Robert Fenske, Jr.
	CREATED:	Mar. 1994
	DESCRIPTION:	This routine searches for a suitable SEG to use as a
			partition line (which will divide the input node).
			Suitable is taken to mean the SEG that most equalizes
			the number of SEGS on each side of the partition line
			and mimimizes the number of SEG splits that need to be
			done.  Ideally, the partition line should also do
			this for all the node's children as well, but the
			calculations would be far too time consuming; therefore
			only the input node is considered.  The most suitable
			partition is found by selecting the SEG that maximizes
			the (geometric mean)^2 of the right and left side
			counts and the number of splits.  The geometric mean
			is computed via A*Rc*Lc/(B*Sc+1) where Rc, Lc, Sc
			are the right side, left side, and split counts
			respectively and A, B are empirical constants (the
			+1 is to avoid division by zero).  The geometric mean
			variable is initialized to -1 so that at least one
			SEG will qualify (even if the maximum mean is zero).
			This routine sets the node's partition line to be the
			SEG that creates the best (maximum) geometric mean.
			This process is an N^2 operation, so a number of
			short cuts are taken to reduce the number of SEGS
			actually checked.  This effectively means reducing
			the number of times nodes_decide_side() is called.
			These short cuts are as follows:
				o After 400 SEGS have been considered as
				  partition lines, only those that are
				  horizontal or vertical are considered
				o If more than 400 SEGS need to be checked
				  for a partition line, every third one is
				  skipped simply for speed--the idea being
				  that there will still be enough SEGS to
				  make for a reasonable right/left SEGS count
				o Only one of the two SEGS spawned (which
				  are always consecutive in the SEGS list)
				  by a LINEDEF is checked
				o If the current # splits is > 15 or is more
				  than two higher than the # splits associated
				  with the best partition line so far, there
				  is no need to consider the current partition
				  any further since it is not a good candidate
				o The current partition line candidate is
				  considered the best one to choose if it
				  results in fewer than two splits, so
				  processing is stopped
			If bgm is still -1 after all SEGS are processed
			(meaning no suitable partition line was found), then
			a partition line choice is forced by choosing the
			first SEG as the partition line.
******************************************************************************/
#if defined(ANSI_C)
local void nodes_partition_line(register NODE_TREE *node)
#else
local void nodes_partition_line(node)
register NODE_TREE *node;
#endif
{
  long bgm = -1L;				/* max geometric mean count */
  int bsplits = -1L;				/* associated best # splits */
  long gm;					/* geometric mean count */
  int rcnt, lcnt, splits;			/* right, left, split count */
  register int sideflag;
  register SEGS_INFO *sinfo, *iseg;
  register int s, i;

  bsplits = 2*node->nsegs;
  sinfo = node->segs;
  for (s = 0; s < node->nsegs; s++) {		/* scan SEGS in node */
    if ((s < 400 || (sinfo->seg.angle&0x3FFF) == 0) &&
        (s == 0 || sinfo->seg.lndx != sinfo->prev->seg.lndx)) {
      sideflag = rcnt = lcnt = splits = 0;
      iseg = node->segs;
      for (i = 0; i < node->nsegs; i++) {	/* get SEGS pos w.r.t. pseg */
        if (i == s)				/* already know this will */
          sideflag = 0;				/* be on right side       */
        else if (sinfo->seg.lndx == iseg->seg.lndx)/* and this will be on */
          sideflag = 1;				/* left side              */
        else if (i == 0 ||			/* must compute which side */
                 iseg->seg.lndx != iseg->prev->seg.lndx) {
          if (node->nsegs > 400)
            if (i % 3 == 0) { iseg = iseg->next; continue; }
          sideflag = nodes_decide_side(&sinfo->seg,&iseg->seg);
        }
        if (sideflag == 0)       rcnt++;	/* count SEGS on both sides */
        else if (sideflag == 1)  lcnt++;	/* of the partition line    */
        else if (sideflag == -1) splits++;	/* count # splits */
        if (splits > 15 ||			/* 15 empirically determined */
            splits > bsplits+2) break;		/* already too many splits */
        iseg = iseg->next;
      }
      if (rcnt != node->nsegs) {
        gm = rcnt*lcnt/(17*splits/3+1);		/* 1, 17/3 are empirical */
        if (bgm < gm) {
          bgm = gm, bsplits = splits;
          node->pseg = sinfo;			/* best partition so far */
          if (splits < 2 && rcnt > 4 && lcnt > 4)
            break;				/* good enough: done */
        }
      }
    }
    sinfo = sinfo->next;
  }
  if (bgm == -1L) {				/* need to force a partition */
    node->pseg = node->segs;			/* line: use first SEG and   */
    node->pseg->sector = -1;			/* flag as artificial        */
  }
}


/******************************************************************************
	ROUTINE:	nodes_subsector_test(node)
	WRITTEN BY:	Robert Fenske, Jr.
	CREATED:	Mar. 1994
	DESCRIPTION:	This routine checks whether the input node can be
			an SSECTOR or not.  To be a subsector, the SEGS in
			the node must describe a convex polygon (no interior
			angles > 180 degrees).  This is equivalent to having
			all the SEGS on the right side of each other.  If the
			unique sector flag is set, then all the SEGS in the
			node must refer to the same SECTOR before it will be
			considered an SSECTOR.
******************************************************************************/
#if defined(ANSI_C)
local int nodes_subsector_test(register NODE_TREE *node)
#else
local int nodes_subsector_test(node)
register NODE_TREE *node;
#endif
{
  int subsector = TRUE;
  register SEGS_INFO *sinfo, *iseg;
  register int s, i;

  sinfo = node->segs;
  for (s = 0; s < node->nsegs; s++) {		/* scan all SEGS */
    for (iseg = node->segs, i = 0; i < node->nsegs; i++) {
      if (i != s) {
        if (ninfo.unique_sector && sinfo->sector != iseg->sector ||
            nodes_decide_side(&sinfo->seg,&iseg->seg) != 0) {
          subsector = FALSE;			/* interior angle > 180 degs */
          goto done;				/* so not an SSECTOR         */
        }
      }
      iseg = iseg->next;
    }
    sinfo = sinfo->next;
  }
done:
  return subsector;
}


/******************************************************************************
	ROUTINE:	nodes_partition_node(node)
	WRITTEN BY:	Robert Fenske, Jr.
	CREATED:	Mar. 1994
	DESCRIPTION:	This routine performs the binary space partitioning.
			It recursively divides (partitions) the input node
			until each leaf of the subtree defined by the input
			node is an SSECTOR.  A partition line is obtained and
			the left and right subtrees are allocated.  The left
			subtree is always checked first.  If it is not an
			SSECTOR, a recursive call is made to further divide
			the left subtree.  The same procedure is then performed
			on the right subtree.  The number of left and right
			children plus one for the current node is returned.
******************************************************************************/
#if defined(ANSI_C)
local int nodes_partition_node(register NODE_TREE *node)
#else
local int nodes_partition_node(node)
register NODE_TREE *node;
#endif
{
  int nl, nr;					/* # left, right nodes */
  register NODE_TREE *left, *right;		/* left, right children */

  nodes_partition_line(node);			/* obtain partition line */
  node->right = tree_branch(right);
  node->left = tree_branch(left);
  nodes_divide_node(node,right,left);
  if (nodes_subsector_test(left)) {		/* found an SSECTOR */
    if (!ninfo.limit_io) printf("*\010\010");
    nl = 1;
  }else {					/* need further subdivision */
    if (!ninfo.limit_io) printf("L");
    nl = nodes_partition_node(left);
  }
  if (nodes_subsector_test(right)) {		/* found an SSECTOR */
    if (!ninfo.limit_io) printf("*\010\010");
    nr = 1;
  }else {					/* need further subdivision */
    if (!ninfo.limit_io) printf("R");
    nr = nodes_partition_node(right);
  }
  return nl + nr + 1;				/* # left + # right + this 1 */
}


/******************************************************************************
	ROUTINE:	nodes_place_node(nodes,nndx,sndx,nodetree)
	WRITTEN BY:	Robert Fenske, Jr.
	CREATED:	Mar. 1994
	DESCRIPTION:	This routine builds the NODES structure from the
			input node tree.  It traverses the node tree in a
			post-order fashion, left side first.  As the tree is
			scanned, the NODES, SSECTORS, and SEGS lists are filled
			in as appropriate.  The SSECTORS and SEGS lists are
			referenced through the ninfo block.  The node tree
			entries are deleted after they are used.  The node
			number (or index) is returned, unless an SSECTOR is
			found, then a -(SSECTOR number) is returned.
******************************************************************************/
#if defined(ANSI_C)
local int nodes_place_node(register WAD_NODE *nodes, int *nndx, int *sndx,
                           register NODE_TREE *nodetree)
#else
local int nodes_place_node(nodes,nndx,sndx,nodetree)
register WAD_NODE *nodes;
int *nndx, *sndx;
register NODE_TREE *nodetree;
#endif

{
  int nnum;					/* node number to return */
  int lnum, rnum;
  SEGS_INFO *sinfo, *next;
  register WAD_NODE *node;
  register int s;

  if (nodetree->left != NULL) {			/* traverse node subtree */
    int fndx = nodetree->pseg->seg.fndx;
    int tndx = nodetree->pseg->seg.tndx;
    lnum = nodes_place_node(nodes,nndx,sndx,nodetree->left);
    rnum = nodes_place_node(nodes,nndx,sndx,nodetree->right);
    node = &nodes[nnum = (*nndx)++];
    node->x = ninfo.sverts[fndx].x;		/* these 4 fields describe */
    node->y = ninfo.sverts[fndx].y;		/* the partition line      */
    node->xdel = ninfo.sverts[tndx].x - node->x;
    node->ydel = ninfo.sverts[tndx].y - node->y;
    node->lymax = nodetree->left->ymax;
    node->lymin = nodetree->left->ymin;
    node->lxmin = nodetree->left->xmin;
    node->lxmax = nodetree->left->xmax;
    if (lnum < 0) node->nndx[1] = 0x8000 | (-lnum-1);/* mark as SSECTOR */
    else          node->nndx[1] = lnum;		/* mark as NODE */
    blockfree(nodetree->left);			/* done with left subtree */
    node->rymax = nodetree->right->ymax;
    node->rymin = nodetree->right->ymin;
    node->rxmin = nodetree->right->xmin;
    node->rxmax = nodetree->right->xmax;
    if (rnum < 0) node->nndx[0] = 0x8000 | (-rnum-1);/* mark as SSECTOR */
    else          node->nndx[0] = rnum;		/* mark as NODE */
    blockfree(nodetree->right);			/* done with right subtree */
  }else {					/* SSECTOR (tree leaf) */
    ninfo.ssecs[*sndx].count = nodetree->nsegs;
    if (*sndx == 0) ninfo.ssecs[*sndx].sndx = 0;
    else            ninfo.ssecs[*sndx].sndx = ninfo.ssecs[*sndx-1].sndx+
                                              ninfo.ssecs[*sndx-1].count;
    sinfo = nodetree->segs;
    for (s = 0; s < nodetree->nsegs; s++) {	/* copy node's SEGS to full */
      ninfo.segs[ninfo.nsegs++] = sinfo->seg;	/* SEGS array               */
      next = sinfo->next;
      blockfree(sinfo);
      sinfo = next;
    }
    nnum = -++(*sndx);				/* mark as leaf */
  }
  return nnum;					/* ret node # or <0 if leaf */
}


/******************************************************************************
	ROUTINE:	nodes_make(nodes,nnodes,ssecs,nssecs,segs,nsegs,
			           verts,nverts,lines,nlines,sides,genflags)
	WRITTEN BY:	Robert Fenske, Jr.
	CREATED:	Mar. 1994
	DESCRIPTION:	This routine generates the NODES, SSECTORS, and SEGS
			sections.  It first finds the minimum and maximum x,y
			coordinates of the map to use in the root of the node
			tree.  Then the node tree root is created.  The
			necessary counters in the ninfo block are zeroed and
			the SEGS vertices list is allocated.  Then
			nodes_partition_node() is called to build the node
			tree.  Next, the NODES, SSECTORS, and SEGS lists are
			allocated based on the values calculated during the
			construction of the node tree.  The necessary counters
			are zeroed and nodes_place_node() is called to fill
			the NODES, SSECTORS, and SEGS lists with the correct
			information.  All the appropriate values are placed
			into the return variables and the number of computed
			node entries is returned.
******************************************************************************/
#if defined(ANSI_C)
long nodes_make(WAD_NODE **nodes, long *nnodes, WAD_SSECTOR **ssecs,
               long *nssecs, WAD_SEGS **segs, long *nsegs,
               WAD_VERT **verts, long *nverts, WAD_LINE **lines,
               long *nlines, WAD_SIDE **sides, int genflags)
#else
long nodes_make(nodes,nnodes,ssecs,nssecs,segs,nsegs,verts,nverts,
                lines,nlines,sides,genflags)
WAD_NODE **nodes;
long *nnodes;
WAD_SSECTOR **ssecs;
long *nssecs;
WAD_SEGS **segs;
long *nsegs;
WAD_VERT **verts;
long *nverts;
WAD_LINE **lines;
long *nlines;
WAD_SIDE **sides;
int genflags;
#endif
{
  NODE_TREE *nodetree;				/* BSP node tree */
  register int i;

  ninfo.unique_sector = !!(genflags & RG_UNIQSECT);/* assign node generation */
  ninfo.use_all_segs = !!(genflags & RG_USEALL);/* flags                    */
  ninfo.limit_io = !!(genflags & RG_LIMIT_IO);
  ninfo.nsverts = -1;
  for (i = 0; i < *nlines; i++) {		/* find maximum used vertex */
    if (ninfo.nsverts < (*lines)[i].fndx) ninfo.nsverts = (*lines)[i].fndx;
    if (ninfo.nsverts < (*lines)[i].tndx) ninfo.nsverts = (*lines)[i].tndx;
  }
  ninfo.nverts = ++ninfo.nsverts;		/* this many lines' vertices */
  ninfo.sverts = blockmem(WAD_VERT,2*ninfo.nsverts);/* assume no more than  */
  for (i = 0; i < ninfo.nsverts; i++)		/* nsverts new verts created */
    ninfo.sverts[i] = (*verts)[i];
  ninfo.nlines = *nlines;			/* this many lines */
  ninfo.nsegs = nodes_segs_init(*lines,*nlines,*sides);/* init SEGS list */
  nodetree = tree_branch(nodetree);		/* node tree root */
  nodetree->nsegs = ninfo.nsegs;
  nodetree->segs = ninfo.sinfo;
  nodes_segs_bounds(nodetree);
  printf("%d used segs ==>  ",ninfo.nsegs);
  ninfo.nnodes = nodes_partition_node(nodetree)/2;/* BSP it */
  ninfo.nodes = blockmem(WAD_NODE,ninfo.nnodes);
  ninfo.nssecs = ninfo.nnodes+1;		/* always 1 more SSECTOR */
  ninfo.ssecs = blockmem(WAD_SSECTOR,ninfo.nssecs);
  ninfo.segs = blockmem(WAD_SEGS,ninfo.nsegs);
  ninfo.nsegs = ninfo.nssecs = ninfo.nnodes = 0;
  (void)nodes_place_node(ninfo.nodes,&ninfo.nnodes,&ninfo.nssecs,nodetree);
  if (nodes != NULL)  *nodes = ninfo.nodes;
  if (nnodes != NULL) *nnodes = ninfo.nnodes;
  if (ssecs != NULL)  *ssecs = ninfo.ssecs;
  if (nssecs != NULL) *nssecs = ninfo.nssecs;
  if (segs != NULL)   *segs = ninfo.segs;
  if (nsegs != NULL)  *nsegs = ninfo.nsegs;
  if (verts != NULL)  *verts = ninfo.sverts;
  if (nverts != NULL) *nverts = ninfo.nsverts;
  blockfree(nodetree);				/* done with the node tree */
  return *nnodes;				/* return number of nodes */
}
