unit Main; { Plomakkeen koodi. }

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, FileCtrl, ExtCtrls, ComCtrls, Grids;

const
  M = 1048576;  // megatavu, kytetn tulosten muotoilussa

type
  { TDriveInfo: levyosion tiedot }
  TDriveInfo = record
    ClusterSize: Integer;    { osion varausyksikn koko }
    TotalCapacity: Integer;  { osion koko }
    TotalFree: Integer;      { osion vapaa levytila }
    TotalNeeded: Integer;    { tiedostojen tarvitsema levytila }
    TotalAllocated: Integer; { tiedostoille varattu levytila }
    TotalSlack: Integer;     { osion hukkatila }
    FileCount: Integer;      { osion tiedostojen lukumr }
    DirCount: Integer;       { osion hakemistojen lukumr }
  end;

{
  Kytetyt etuliitteet ja niit vastaavat komponentit:
  FrmXxxx = TForm tai sen perillinen
  BtnXxxx = TButton tai TRadioButton
  LstXxxx = TListBox tai muu visuaalinen listakomponentti
  LblXxxx = TLabel
  GrdXxxx = Grid eli ruudukko, kuten TStringGrid tai TDrawGrid

  Muita etuliitteit:
  TxtXxxx = TEdit tai muu tekstinmuokkauskentt
  MnuXxxx = TMenu tai TMenuItem
  ChkXxxx = TCheckbox
  GrpXxxx = TRadioGroup tai muu ryhmkomponentti
  PnlXxxx = TPanel tai muu paneeli
}

  TMainForm = class(TForm)
    DriveList: TDriveComboBox;
    AnalyzeButton: TButton;
    DriveLabel: TLabel;
    StatusBar: TStatusBar;
    ResultsLabel: TLabel;
    ResultGrid: TStringGrid;
    procedure FormCreate(Sender: TObject);
    procedure DriveListChange(Sender: TObject);
    procedure AnalyzeButtonClick(Sender: TObject);
  private
    Info: TDriveInfo;  { ksiteltvn levyosion tiedot }
    procedure ProcessDirectory(FileMask: string);
    procedure ProcessFile(SearchRecord: TSearchRec);
    procedure DisplayResults;
  end;

var
  MainForm: TMainForm;

implementation

{$R *.DFM}

procedure TMainForm.FormCreate(Sender: TObject);
begin
  // est tai salli Analysoi-painike valitun aseman mukaan
  AnalyzeButton.Enabled := (Windows.GetDriveType(PChar(DriveList.Drive + ':\')) = DRIVE_FIXED);

  // alusta tulosruudukon otsakkeet
  with ResultGrid do
  begin
    Cells[0, 0] := 'Levyosion koko:';
    Cells[0, 1] := 'Varausyksikn koko:';
    Cells[0, 2] := 'Vapaa levytila:';
    Cells[0, 3] := 'Hakemistoja:';
    Cells[0, 4] := 'Tiedostoja:';
    Cells[0, 5] := 'Tiedostojen tarvitsema tila:';
    Cells[0, 6] := 'Tiedostojen yhteiskoko:';
    Cells[0, 7] := 'Hukkatila:';
  end;
end;


procedure TMainForm.DriveListChange(Sender: TObject);
begin
  // Tutki onko valittu osio kiintolevy vai ei
  // ja est tai salli Analysoi-painikkeen kytt sen mukaisesti:
  if Windows.GetDriveType(PChar(DriveList.Drive + ':\')) <> DRIVE_FIXED then
  begin
    AnalyzeButton.Enabled := False;
    MessageDlg('Valitse listalta jokin kiintolevyn osio!', mtWarning, [mbOK], 0);
  end
  else
  begin
    AnalyzeButton.Enabled := True;
  end;
end;


procedure TMainForm.AnalyzeButtonClick(Sender: TObject);
var
  SectorsPerCluster, BytesPerSector,
  NumberOfFreeClusters, TotalNumberOfClusters: DWORD; // API-kutsua varten
  FreePercentage: Real; // vapaa levytila prosentteina
  I: Integer;  // ruudukon merkkijonojen alustamiseen
  Mask: string;  // haettavien tiedostojen maski
begin
  // Totta kai ohjelmassa pit ainakin yksi Windows-API:n kutsu olla:
  // varausyksikn koon saa selville GetDiskFreeSpace-kutsulla (kts. WIN32.HLP)
  // Pelkn levyn koon saa Delphin DiskSize-funktiolla ja vapaan tilan
  // DiskFree-funktiolla.
  if not Windows.GetDiskFreeSpace(PChar(DriveList.Drive + ':\'),
      SectorsPerCluster, BytesPerSector,
      NumberOfFreeClusters, TotalNumberOfClusters) then
  begin
    Dialogs.MessageDlg('Ei voida lukea aseman ' + DriveList.Drive + ': tietoja',
      mtError, [mbOK], 0);
    Exit;
  end;

  AnalyzeButton.Enabled := False;  // analyysin aikana ei voi kynnist uutta

  with Info do
  begin
    ClusterSize := SectorsPerCluster * BytesPerSector;

    TotalCapacity := TotalNumberOfClusters * ClusterSize;  // see DiskSize of the RTL
    TotalFree := NumberOfFreeClusters * ClusterSize;       // see DiskFree of the RTL
    TotalAllocated := TotalCapacity - TotalFree;
    FreePercentage := 100.0 * (TotalFree / TotalCapacity);

    with ResultGrid do
    begin
      for I := 0 to RowCount - 1 do
      begin
        Cells[1, I] := '';
      end;

      Cells[1, 0] := IntToStr(TotalCapacity div M) + ' Mt';
      Cells[1, 1] := IntToStr(ClusterSize) + ' tavua';
      Cells[1, 2] := Format('%d Mt (%.1f %%)', [TotalFree div M, FreePercentage]);
      Cells[1, 5] := IntToStr(TotalAllocated div M) + ' Mt';

      Refresh;
    end; { with ResultGrid }

    TotalNeeded := 0;
    TotalSlack := 0;
    FileCount := 0;
    DirCount := 0;
  end; { with Info }

  // tee maski joka kattaa kaikki valitun aseman juurihakemiston tiedostot
  Mask := DriveList.Drive + ':\*.*';

  ProcessDirectory(Mask); // aloita ksittely

  // Valmista tuli. Soita kelloa ja nyt tulokset.
  SysUtils.Beep;
  StatusBar.SimpleText := 'Analyysi valmis';
  DisplayResults;

  AnalyzeButton.Enabled := True;
end;


procedure TMainForm.ProcessDirectory(FileMask: string);
var
  SearchRecord: SysUtils.TSearchRec;
  SearchResult: Integer;
  NewMask: string;
begin
  Inc(Info.DirCount);  // kasvata hakemistojen lukumr yhdell

  // Etsi tiedostoja kytten SysUtilsin FindFirst/FindNext/FindClose-kutsuja.
  SearchResult := FindFirst(FileMask, faAnyFile, SearchRecord);
  while SearchResult = 0 do
  begin
    if (SearchRecord.Attr and faDirectory) <> 0 then  // jos oli hakemisto
    begin
      // l ksittele tt hakemistoa itsen (.) tai sen iti (..)
      if (SearchRecord.Name <> '.') and (SearchRecord.Name <> '..') then
      begin
        NewMask := SysUtils.ExtractFilePath(FileMask) + SearchRecord.Name;
        StatusBar.SimpleText := 'Ksitelln hakemistoa ' + NewMask;
        NewMask := NewMask + '\*.*';
        ProcessDirectory(NewMask); // ksittele uusi hakemisto rekursiivisesti
      end;
    end
    else  // se olikin tiedosto
    begin
      ProcessFile(SearchRecord);
    end;

    SearchResult := FindNext(SearchRecord);  // etsi seuraava
  end;

  FindClose(SearchRecord); // pt etsint
end;

procedure TMainForm.ProcessFile(SearchRecord: TSearchRec);
var
  SlackAmount: Integer;
begin
  Inc(Info.FileCount);  // kasvata tiedostojen mr yhdell
  Inc(Info.TotalNeeded, SearchRecord.Size); // lis koko yhteissummaan

  // Laske tiedoston hukkatila, eli paljonko j yli tiedoston
  // koosta kun se pyristetn varausyksikn kokoon.
  SlackAmount := SearchRecord.Size mod Info.ClusterSize;
  if SlackAmount > 0 then  // jos hukkatilaa on, lis se yhteissummaan
  begin
    Inc(Info.TotalSlack, Info.ClusterSize - SlackAmount);
  end;
end;

{ Muotoilee tulokset ja nytt ne ruudukossa. }
procedure TMainForm.DisplayResults;
var
  SlackPercentage: Real;  // osion hukkaprosentti
begin
  SlackPercentage := 100.0 * Info.TotalSlack / Info.TotalCapacity;

  // Muotoile tulokset merkkijonoiksi ja sijoita ne ruudukkoon.
  // with-lauseet sstvt naputtelulta; esimerkiksi DirCount on itse asiassa
  // Info.DirCount ja Cells[i, j] on ResultGrid.Cells[i, j].
  with Info do
  begin
    with ResultGrid do
    begin
      Cells[1, 3] := IntToStr(DirCount) + ' kpl';
      Cells[1, 4] := IntToStr(FileCount) + ' kpl';
      Cells[1, 6] := IntToStr(TotalNeeded div M) + ' Mt';
      Cells[1, 7] := Format('%d Mt (%.1f %%)', [TotalSlack div M, SlackPercentage]);
      Refresh; // tss viittausalueessa Refresh tarkoittaa ruudukon metodia
    end;
  end;
end;

end.

{
Kuvake:
Tee Image Editorilla (sek 32x32 ett 16x16). Hukka.ico
Aseta: Project | Options / Application, Load icon...
Mys lomakkeelle: Object Inspectorissa valitse TFrmMain, sitten
listalta Icon ja Load...
}

{ HUOM! Ohjelma ei ymmrr yli 2 Gt osioita, koska se menee Integer-tyypin
  arvorajojen yli. }

