unit Main;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  RxGrdCpt, ComCtrls, ToolWin, ExtCtrls, LCD_Lab, Menus, MPlayer, Placemnt,
  AppEvent, RXShell;

resourcestring
  SUnknownArtist = '(tuntematon esittj)';
  SUnknownTitle = '(tuntematon levy)';
  SNoDisc = 'Asemassa ei ole levy';
  SInsertDisc = 'Syt ni-CD asemaan';
  SNoTrack = '';

  SRetrievingData = 'Haetaan levyn tietoja...';
  SStopped = 'Pysytetty';
  SPlaying = 'Toistaa';
  SPaused = 'Tauko toistossa';

type
  TTimeMode = (tmNone, tmDiscTotal, tmTrackElapsed, tmTrackRemaining,
               tmDiscElapsed, tmDiscRemaining);

const
  TimeModes: array [TTimeMode] of string =
    ('No disc', 'Disc length', 'Track elapsed', 'Track remaining',
     'Disc elapsed', 'Disc remaining');

type
  TFrmMain = class(TForm)
    RxGradientCaption1: TRxGradientCaption;
    StatusBar: TStatusBar;
    TbrControls: TToolBar;
    ImlControls: TImageList;
    BtnPlay: TToolButton;
    BtnPause: TToolButton;
    BtnStop: TToolButton;
    BtnPrevious: TToolButton;
    BtnBack: TToolButton;
    BtnForward: TToolButton;
    BtnNext: TToolButton;
    BtnEject: TToolButton;
    PnlDisplay: TPanel;
    LblArtist: TLCDLabel;
    LblTitle: TLCDLabel;
    LblTrackInfo: TLCDLabel;
    LblTrackName: TLCDLabel;
    MainMenu: TMainMenu;
    MnuDisc: TMenuItem;
    MnuDiscEdit: TMenuItem;
    MnuDiscSep1: TMenuItem;
    MnuDiscExit: TMenuItem;
    MnuHelp: TMenuItem;
    MnuHelpAbout: TMenuItem;
    Player: TMediaPlayer;
    TmrStatus: TTimer;
    FormPlacement1: TFormPlacement;
    ToolButton1: TToolButton;
    BtnTimeMode: TToolButton;
    RxTrayIcon1: TRxTrayIcon;
    AppEvents1: TAppEvents;
    procedure MnuDiscExitClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure BtnPlayClick(Sender: TObject);
    procedure BtnStopClick(Sender: TObject);
    procedure BtnPauseClick(Sender: TObject);
    procedure TmrStatusTimer(Sender: TObject);
    procedure BtnEjectClick(Sender: TObject);
    procedure MnuDiscEditClick(Sender: TObject);
    procedure BtnNextClick(Sender: TObject);
    procedure BtnPreviousClick(Sender: TObject);
    procedure BtnBackClick(Sender: TObject);
    procedure BtnForwardClick(Sender: TObject);
    procedure MnuHelpAboutClick(Sender: TObject);
    procedure BtnTimeModeClick(Sender: TObject);
    procedure AppEvents1Minimize(Sender: TObject);
    procedure AppEvents1Restore(Sender: TObject);
    procedure RxTrayIcon1DblClick(Sender: TObject);
  private
    { Private declarations }
    CurrentTrack: Longint;
    Paused: Boolean;
    HaveDisc: Boolean;
    Changed: Boolean;
    TimeMode: TTimeMode;
    procedure UpdateStatus;
    procedure WriteDiscInfo;
  public
    { Public declarations }
  end;

var
  FrmMain: TFrmMain;

implementation

uses
  MMSystem, Disc, IniFiles, DiscEdit, About;

{$R *.DFM}

var
  DiscInfo: TDiscInfo;
  IniFileName: string;
  WinDir: array [0..256] of Char;

procedure TFrmMain.MnuDiscExitClick(Sender: TObject);
begin
  Close;
end;


procedure TFrmMain.FormCreate(Sender: TObject);
begin
  FillChar(DiscInfo, SizeOf(DiscInfo), 0);
  GetWindowsDirectory(@WinDir, SizeOf(WinDir));
  IniFileName := WinDir + '\cdplayer.ini';

  Player.TimeFormat := tfTMSF;
  Player.Wait := False;

  HaveDisc := False;
  Paused := False;
  Changed := False;
  CurrentTrack := 1;
  TimeMode := tmNone;
end;

procedure TFrmMain.BtnPlayClick(Sender: TObject);
begin
  if Paused then
  begin
    (* set no starting position when resuming paused play *)
    Player.Play;
    (* time display mode stays the same *)
  end
  else
  begin
    Player.TimeFormat := tfTMSF;
    Player.Position := MCI_MAKE_TMSF(CurrentTrack, 0, 0, 0);
    Player.Play;
    TimeMode := tmTrackElapsed;
  end;
end;

procedure TFrmMain.BtnStopClick(Sender: TObject);
begin
  if Paused then
  begin
    Player.Play;
    Paused := False;
  end;
  Player.Stop;
  CurrentTrack := 1;
  TimeMode := tmDiscTotal;
end;

procedure TFrmMain.BtnPauseClick(Sender: TObject);
begin
  if Player.Mode <> mpPlaying then Exit;
  Player.Pause;
  Paused := True;
end;

procedure TFrmMain.UpdateStatus;
var
  Track, Min, Sec: Longint;
  DisplayTime: Longint;
  TimeTemplate: string;
  TrackName: string;
  TrackLen: Longint;
begin
  DisplayTime := Player.Position;

  case TimeMode of
    tmNone:
    begin
      Min := 0;
      Sec := 0;
      Track := 0;
    end;

    tmDiscTotal:
    begin
      Player.TimeFormat := tfMSF;
      DisplayTime := Player.Length;
      Min := MCI_MSF_MINUTE(DisplayTime);
      Sec := MCI_MSF_SECOND(DisplayTime);
      Track := Player.Tracks;
    end;

    tmTrackElapsed:
    begin
      Player.TimeFormat := tfTMSF;
      DisplayTime := Player.Position;
      Min := MCI_TMSF_MINUTE(DisplayTime);
      Sec := MCI_TMSF_SECOND(DisplayTime);
      Track := GetCurrentTrack(Player.DeviceID);
      CurrentTrack := Track;
    end;

    tmTrackRemaining:
    begin
      Player.TimeFormat := tfMilliseconds;
      (* the player's position should be further than the track starting point *)
      Track := GetCurrentTrack(Player.DeviceID);
      DisplayTime := Player.TrackLength[Track]
          - (Player.Position - Player.TrackPosition[Track]);
      MillisecToMinSec(DisplayTime, Min, Sec);
      CurrentTrack := Track;
    end;

    tmDiscElapsed:
    begin
      Player.TimeFormat := tfMSF;
      DisplayTime := Player.Position;
      Min := MCI_MSF_MINUTE(DisplayTime);
      Sec := MCI_MSF_SECOND(DisplayTime);
      Track := GetCurrentTrack(Player.DeviceID);
      CurrentTrack := Track;
    end;

    tmDiscRemaining:
    begin
      Player.TimeFormat := tfMilliseconds;
      DisplayTime := Player.Length - Player.Position;
      MillisecToMinSec(DisplayTime, Min, Sec);
      Track := GetCurrentTrack(Player.DeviceID);
      CurrentTrack := Track;
    end;
  end;

  TimeTemplate := '%.2d  %.2d:%.2d';

  if (TimeMode in [tmTrackRemaining, tmDiscRemaining]) and
     (Player.Mode <> mpStopped) then
  begin
    TimeTemplate := '%.2d -%.2d:%.2d';
  end;

  LblTrackInfo.Caption := Format(TimeTemplate,
                                 [Track, Min, Sec]);

  if Player.Mode = mpStopped then
    TrackName := SNoTrack
  else
    TrackName := DiscInfo.TrackNames[CurrentTrack];

  (* comparing strings should be faster than updating the display *)
  if LblTrackName.Caption <> TrackName then
    LblTrackName.Caption := TrackName;
end;

procedure TFrmMain.TmrStatusTimer(Sender: TObject);
var
  Ini: TIniFile;
  TrackNumber: Longint;
begin
  if Player.Mode = mpNotReady then
  begin
    Exit;
  end;

  (* A CD was inserted since we last were here. *)
  if (not HaveDisc) and MediaPresent(Player.DeviceID) then
  begin
    StatusBar.Panels[0].Text := SRetrievingData;
    HaveDisc := True;
    Paused := False;
    CurrentTrack := 1;

    DiscInfo.DiscID := MakeDiscID(Player);

    (* Must read from INI file *)
    Ini := TIniFile.Create(IniFileName);
    try
      DiscInfo.Artist := Ini.ReadString(DiscInfo.DiscID, 'artist', SUnknownArtist);
      DiscInfo.Title := Ini.ReadString(DiscInfo.DiscID, 'title', SUnknownTitle);

      Player.TimeFormat := tfTMSF;

      (* CDPLAYER.INI stores track names with zero based offsets. *)
      for TrackNumber := 0 to Player.Tracks - 1 do
      begin
        DiscInfo.TrackNames[TrackNumber + 1] :=
          Ini.ReadString(DiscInfo.DiscID,
                         IntToStr(TrackNumber),
                         'Track ' + IntToStr(TrackNumber + 1));
      end;

      TimeMode := tmDiscTotal;

      MnuDiscEdit.Enabled := True;
    finally
      Ini.Free;
    end;

    LblArtist.Caption := DiscInfo.Artist;
    LblTitle.Caption := DiscInfo.Title;
    UpdateStatus;
    StatusBar.Panels[0].Text := SStopped;
  end
 (* CD was removed since we last were here *)
  else if HaveDisc and (not MediaPresent(Player.DeviceID)) then
  begin
    HaveDisc := False;
    Paused := False;
    FillChar(DiscInfo, SizeOf(DiscInfo), 0);
    MnuDiscEdit.Enabled := False;
    Exit;
  end
  (* no disc in drive *)
  else if (not HaveDisc) and
          (not MediaPresent(Player.DeviceID)) then
  begin
    FillChar(DiscInfo, SizeOf(DiscInfo), 0);
    LblArtist.Caption := SNoDisc;
    LblTitle.Caption := SInsertDisc;
    UpdateStatus;
    Paused := False;
    StatusBar.Panels[0].Text := SNoDisc;
    MnuDiscEdit.Enabled := False;
    Exit;
  end;

  if (Player.Mode = mpPlaying) or Paused then
  begin
    UpdateStatus;
    if Player.Mode = mpPlaying then
      StatusBar.Panels[0].Text := SPlaying
    else
      StatusBar.Panels[0].Text := SPaused;
  end
  else if Player.Mode = mpStopped then
  begin
    Player.TimeFormat := tfMSF;
    UpdateStatus;
    StatusBar.Panels[0].Text := SStopped;
  end
  else (* maybe skip pressed? *)
  begin
    Player.TimeFormat := tfMSF;
    UpdateStatus;
  end;
end;

procedure TFrmMain.BtnEjectClick(Sender: TObject);
begin
  if not HaveDisc then
  begin
    LoadDisc(Player.DeviceID);
    TimeMode := tmDiscTotal;
  end
  else
  begin
    Player.Eject;
  end;
  TimeMode := tmNone;
end;

procedure TFrmMain.MnuDiscEditClick(Sender: TObject);
var
  I: Longint;
begin
  FrmDiscEditor.TxtArtist.Text := LblArtist.Caption;
  FrmDiscEditor.TxtTitle.Text := LblTitle.Caption;

  FrmDiscEditor.LstTracks.Items.Clear;

  for I := 1 to Player.Tracks do
  begin
    FrmDiscEditor.LstTracks.Items.Add(
      DiscInfo.TrackNames[I]
    );
  end;

  if FrmDiscEditor.ShowModal = mrOK then
  begin
    Changed := FrmDiscEditor.HasChanged;

    (* Zap the track name list and fill it again with the
       information from the form. *)
    FillChar(DiscInfo.TrackNames, SizeOf(DiscInfo.TrackNames), 0);
    for I := 0 to FrmDiscEditor.LstTracks.Items.Count - 1 do
    begin
      DiscInfo.TrackNames[I + 1] :=
        FrmDiscEditor.LstTracks.Items[I];
    end;

    LblArtist.Caption := FrmDiscEditor.TxtArtist.Text;
    LblTitle.Caption := FrmDiscEditor.TxtTitle.Text;

    WriteDiscInfo;
  end;
end;

procedure TFrmMain.WriteDiscInfo;
var
  Ini: TIniFile;
  I: Longint;
  Order: string;
begin
  if not Changed then Exit;

  (* First erase the old section for this disc *)
  Ini := TIniFile.Create(IniFileName);
  try
    Ini.EraseSection(DiscInfo.DiscID);

    with Ini do
    begin
      WriteInteger(DiscInfo.DiscID, 'EntryType', 1);
      WriteString(DiscInfo.DiscID, 'artist', LblArtist.Caption);
      WriteString(DiscInfo.DiscID, 'title', LblTitle.Caption);
      WriteInteger(DiscInfo.DiscID, 'numtracks', Player.Tracks);

      Order := '';
      for I := 1 to Player.Tracks do
      begin
        Order := Order + IntToStr(I - 1) + ' ';
        WriteString(DiscInfo.DiscID, IntToStr(I - 1),
          DiscInfo.TrackNames[I]);
      end;
      WriteString(DiscInfo.DiscID, 'order', Trim(Order));
      WriteInteger(DiscInfo.DiscID, 'numplay', Player.Tracks);
    end;
  finally
    Ini.Free;
  end;

  Changed := False;
end;

procedure TFrmMain.BtnNextClick(Sender: TObject);
begin
  if CurrentTrack < Player.Tracks then
  begin
    Inc(CurrentTrack);
    if Player.Mode = mpPlaying then
    begin
      Player.TimeFormat := tfTMSF;
      Player.Next;
      if Player.Error <> 0 then
      begin
        StatusBar.Panels[0].Text := Player.ErrorMessage;
      end;
    end;
  end;
end;

procedure TFrmMain.BtnPreviousClick(Sender: TObject);
begin
  if CurrentTrack > 1 then
  begin
    Dec(CurrentTrack);
    if Player.Mode = mpPlaying then
    begin
      Player.TimeFormat := tfTMSF;
      Player.Previous;
    end;
  end;
end;

procedure TFrmMain.BtnBackClick(Sender: TObject);
begin
  Player.Frames := 1;
  Player.Back;
end;

procedure TFrmMain.BtnForwardClick(Sender: TObject);
begin
  Player.Frames := 1;
  Player.Step;
end;

procedure TFrmMain.MnuHelpAboutClick(Sender: TObject);
begin
  FrmAbout.ShowModal;
end;

procedure TFrmMain.BtnTimeModeClick(Sender: TObject);
begin
  if Player.Mode = mpStopped then
  begin
    Exit;
  end;

  if TimeMode = tmDiscRemaining then
  begin
    TimeMode := tmTrackElapsed;
  end
  else
  begin
    TimeMode := TTimeMode(Succ(Ord(TimeMode)));
  end;

  StatusBar.Panels[1].Text := TimeModes[TimeMode];
end;

procedure TFrmMain.AppEvents1Minimize(Sender: TObject);
begin
  RxTrayIcon1.Active := True;
  FrmMain.Hide;
end;

procedure TFrmMain.AppEvents1Restore(Sender: TObject);
begin
  FrmMain.Show;
  Application.BringToFront;
  RxTrayIcon1.Active := False;
end;

procedure TFrmMain.RxTrayIcon1DblClick(Sender: TObject);
begin
  Application.Restore;
end;

end.
