/*
 *  DXDZFIX -- Fixes the DirectX 5.0 joystick deadzone default
 *  (c) 1998  Chris Roehrig <croehrig@House.ORG>
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>                 /* ANSI */

#ifdef _MSC_VER
#include <io.h>
#include <fcntl.h>
#endif

#ifndef TRUE
#define TRUE 1
#endif
#ifndef FALSE
#define FALSE 0
#endif

static unsigned char SearchBytes[] = {
    0x89, 0x47, 0x30,           	/* mov dword ptr [edi+30h],eax */
    0x89, 0x47, 0x34,           	/* mov dword ptr [edi+34h],eax */
    0xb8, 0x05, 0x0, 0x0, 0x0,		/* mov eax,5    <- the culprit */
    0xc7, 0x03, 0x88, 0x13, 0x0, 0x0	/* mov dword ptr [ebx],1388h   */
};
#define DZOFF 7

static unsigned char ReplaceBytes[sizeof(SearchBytes)];



/************************************************************************
 *   Simple byte queue primitives
 */
struct bytequeue {
    unsigned char *data;
    int	size;
    int tail;
};

static void dump_bytes( unsigned char *bytes, int len )
{
    int i;
    for( i=0; i<len; i++ ){
	fprintf( stderr, "%02x ", bytes[i] );
    }
    fprintf( stderr, "\n" );
}


struct bytequeue *queue_alloc( int size )
{
    struct bytequeue *queue;

    queue = (struct bytequeue *) malloc( sizeof(struct bytequeue) );
    queue->data = (char *)malloc( size * sizeof(char) );
    queue->size = size;
    queue->tail = -1;
    return queue;
}

void queue_free( struct bytequeue *q )
{
    if( q ){
	if( q->data )
	    free( q->data );
	free(q);
    }
}

void queue_dump( struct bytequeue *q )
{
    if( !q ){
	fprintf( stderr, "NULL\n" );
	return;
    }
    if( q->tail < 0 ){
	fprintf( stderr, "EMPTY\n" );
	return;
    }
    dump_bytes( q->data, q->tail+1 );
}

void queue_insert_tail( struct bytequeue *q, unsigned char c )
{
    if( q->tail == (q->size-1) ){
	fprintf( stderr, "Error in queue_insert_tail: Queue is full\n" );
	return;
    }
    q->tail++;
    q->data[q->tail] = c;
}

unsigned char queue_remove_head( struct bytequeue *q )
{
    int i;
    char c;

    if( q->tail < 0 ){
	fprintf( stderr, "Error in queue_remove_head: Queue is empty\n" );
	return '\0';
    }

    c = q->data[0];
    for( i=0; i<q->tail; i++ ){
	q->data[i] = q->data[i+1];
    }
    q->tail--;

    return c;
}

int queue_subst( struct bytequeue *q, int len, 
		    unsigned char *search, unsigned char *replace )
{
    if( q->size < len ){
	return FALSE;
    }
    if( memcmp( q->data, search, len ) == 0 ){
	memcpy( q->data, replace, len );
	return TRUE;
    } else {
	return FALSE;
    }
}

int queue_size( struct bytequeue *q )
{
    return q->tail+1;
}


void main( int argc, char **argv )
{
    int i, c;
    int deadzone = 0;
    struct bytequeue *q;

    fprintf( stderr, 
"DXDZFIX: sets the default DirectX 5.0 joystick deadzone to 0 percent.\n"
"Usage: \n"
"        DXDZFIX 0 <C:\\WINDOWS\\SYSTEM\\DINPUT.DLL >DINPUT.NEW\n"
"        COPY DINPUT.NEW C:\\WINDOWS\\SYSTEM\\DINPUT.DLL\n"
);

    if( argc != 2 ){
	exit(1);
    }
    sscanf( argv[1], "%d", &deadzone );
    if( deadzone < 0 ) deadzone = 0;
    if( deadzone > 100 ) deadzone = 100;

    memcpy( ReplaceBytes, SearchBytes, sizeof(SearchBytes) );
    ReplaceBytes[DZOFF] = deadzone & 0xff;
    ReplaceBytes[DZOFF+1] = (deadzone>>8) & 0xff;
    ReplaceBytes[DZOFF+2] = (deadzone>>16) & 0xff;
    ReplaceBytes[DZOFF+3] = (deadzone>>24) & 0xff;

    fprintf( stderr, "searching for:\n" );
    dump_bytes( SearchBytes, sizeof(SearchBytes) );
    fprintf( stderr, "replacing with:\n" );
    dump_bytes( ReplaceBytes, sizeof(ReplaceBytes) );

#ifdef _MSC_VER
    _setmode( _fileno( stdin ), _O_BINARY );
    _setmode( _fileno( stdout ), _O_BINARY );
#endif

    q = queue_alloc( sizeof(SearchBytes) );
    i = 0;
    while( (c=getchar()) != EOF ){
	queue_insert_tail( q, (unsigned char)(c&0xff) );
/*
if( (i>0xd8b0) && (i<0xd980) )
queue_dump(q);
*/
	if( queue_size(q) >= sizeof(SearchBytes)) {
	    if( queue_subst( q, sizeof(SearchBytes), 
			SearchBytes, ReplaceBytes ) ){
		fprintf( stderr, "found substitution at 0x%x\n", 
			    i-(int)sizeof(SearchBytes)+1 );
	    }
	    putchar( queue_remove_head(q) );
	}
	i++;
    }
    while( queue_size(q) > 0 ){
	putchar( queue_remove_head(q) );
    }
    queue_free(q);

    fprintf( stderr, "%d bytes\n", i );
}
