unit Upt;

interface

uses
  Windows;

type
  { Represents the system uptime in conventional units }
  TUptime = class(TObject)
  public
    constructor Create;
    procedure Format(var S: string; const Verbose: Boolean);
  private
    Days, Hours, Minutes, Seconds: DWORD;
  end;

procedure WriteLogEntry(GoingUp: Boolean);

implementation

uses
  SysUtils;

resourcestring
  SLogFileName = 'Uptime Log.txt';
  SStartup = 'STARTUP';
  SShutdown = 'SHUTDOWN';

const
  SECONDS_PER_MINUTE = 60;
  MINUTES_PER_HOUR   = 60;
  SECONDS_PER_HOUR   = SECONDS_PER_MINUTE *
                       MINUTES_PER_HOUR;
  HOURS_PER_DAY      = 24;
  SECONDS_PER_DAY    = SECONDS_PER_MINUTE *
                       MINUTES_PER_HOUR *
                       HOURS_PER_DAY;

constructor TUptime.Create;
var
  Ticks: DWORD;
begin
  inherited Create;

  Ticks := GetTickCount;
  
  Seconds := Ticks div 1000;
  Minutes := Seconds div SECONDS_PER_MINUTE;
  Hours   := Seconds div SECONDS_PER_HOUR;
  Days    := Seconds div SECONDS_PER_DAY;

  Dec(Seconds, Minutes * SECONDS_PER_MINUTE);
  Dec(Minutes, Hours * MINUTES_PER_HOUR);
  Dec(Hours, Days * HOURS_PER_DAY);
end;

procedure TUptime.Format(var S: string; const Verbose: Boolean);
var
  I: Integer;
begin
  if Verbose then
    S := SysUtils.Format('%d d %d h %d m %d s',
              [Days, Hours, Minutes, Seconds])
  else
  begin
    S := SysUtils.Format('%2d:%2d:%2d:%2d', [Days, Hours, Minutes, Seconds]);
    // It appears that Format does NOT pad shorter strings with zeros
    // contrary to the documentation. We have to fix this. In this case,
    // though, it is easy: just replace the blanks in the string with zeros.
    for I := 0 to Length(S) do
    begin
      if S[I] = ' ' then
        S[I] := '0'
    end;
  end;
end;

procedure WriteLogEntry(GoingUp: Boolean);
var
  LogFile: TextFile;
  EventString, TimeStamp, UptimeString: string;
  Uptime: TUptime;
begin
  if GoingUp then
    EventString := SStartup
  else
    EventString := SShutdown;

  if FileExists(SLogFileName) then
  begin
    AssignFile(LogFile, SLogFileName);
    Append(LogFile);
  end
  else
  begin
    AssignFile(LogFile, SLogFileName);
    Rewrite(LogFile);
  end;

  { STARTUP,YYYY-MM-DD HH:MM:SS }
  { SHUTDOWN,YYYY-MM-DD HH:MM:SS,00:04:12:39 }

  TimeStamp := SysUtils.FormatDateTime('yyyy-mm-dd hh:mm:ss', SysUtils.Now);
  Write(LogFile, EventString + ',' + TimeStamp);
  if not GoingUp then
  begin
    Uptime := TUptime.Create;
    try
      Uptime.Format(UptimeString, False);  // use the terse format
      Write(LogFile, ',' + UptimeString);
    finally
      Uptime.Destroy;
    end;
  end;
  WriteLn(LogFile);

  CloseFile(LogFile);
end;

function FileExists(FileName: string): Boolean;
{ Boolean function that returns True if the file exists; otherwise,
  it returns False. Closes the file if it exists. }
var
  F: file;
begin
  {$I-}
  AssignFile(F, FileName);
  FileMode := 0;  // Set file access to read only
  Reset(F);
  CloseFile(F);
  {$I+}
  FileExists := (IOResult = 0) and (FileName <> '');
end;  { FileExists }

end.

