/********************************************************************
  MBnetClock.c -- viisarikello MBnet-logolla Windows 95:lle
  Copyright (C) 1997 Jere Kpyaho (jere@sci.fi)
*********************************************************************/

#include <windows.h>
#include <windowsx.h>
#include <math.h>
#include "resource.h"

BOOL InitApplication( HINSTANCE hinstCurrent );
BOOL InitInstance( HINSTANCE hinstCurrent );
LRESULT CALLBACK ClockWndProc( HWND hwnd, UINT message, WPARAM wParam, LPARAM lParam );
BOOL Clock_OnCreate( HWND hwnd, LPCREATESTRUCT lpcs );
void Clock_OnSize( HWND hwnd, UINT state, int cx, int cy );
void Clock_OnPaint( HWND hwnd );
void Clock_OnTimer( HWND hwnd, UINT id );
void Clock_OnRButtonUp( HWND hwnd, int x, int y, UINT flags );
void Clock_OnCommand( HWND hwnd, int id, HWND hwndCtl, UINT codeNotify );
void Clock_OnDestroy( HWND hwnd );
void SetClockMapMode( HDC hdc );
void DrawFace( HDC hdc );
void DrawHands( HDC hdc, LPSYSTEMTIME st );
void DrawSecondHand( HDC hdc, LPSYSTEMTIME st );
void RotatePoint( POINT pt[], int iNum, int iAngle );
BOOL CALLBACK AboutDlgProc( HWND hdlg, UINT message, WPARAM wParam, LPARAM lParam );

const int ID_TIMER = 1;
const char gszAppName[] = "MBnetClock";
const char gszWindowTitle[] = "MBnet-kello";
HINSTANCE ghinstApp;
int gcxClient, gcyClient;
SYSTEMTIME gstCurrent, gstPrevious;
HMENU ghmenuPopup;
BOOL gbSoundEnabled;

int WINAPI WinMain( HINSTANCE hinstCurrent, HINSTANCE hinstPrevious, LPSTR lpszCmdLine, int nCmdShow )
{
	HWND hwndPrevious;
    MSG msg;

	hwndPrevious = FindWindow( gszAppName, NULL );
	if ( hwndPrevious != NULL )
	{
		SetForegroundWindow( hwndPrevious );
		return 0;
	}

    if ( !InitApplication( hinstCurrent ) )
        return 0;
    if ( !InitInstance( hinstCurrent ) )
        return 0;

    while ( GetMessage( &msg, NULL, 0, 0 ) )
    {
        TranslateMessage( &msg );
        DispatchMessage( &msg );
    }
    return (int) msg.wParam;
}

BOOL InitApplication( HINSTANCE hinstCurrent )
{
    WNDCLASSEX wcex;

    wcex.cbSize			= sizeof(WNDCLASSEX);
    wcex.style			= CS_HREDRAW | CS_VREDRAW | CS_DBLCLKS;
    wcex.lpfnWndProc	= ClockWndProc;
    wcex.cbClsExtra		= 0;
    wcex.cbWndExtra		= 0;
    wcex.hInstance		= hinstCurrent;
    wcex.hIcon			= LoadIcon( hinstCurrent, "AppIcon" );
    wcex.hCursor		= LoadCursor( NULL, IDC_ARROW );
    wcex.hbrBackground	= (HBRUSH)(COLOR_WINDOW + 1);
    wcex.lpszMenuName	= NULL;
    wcex.lpszClassName	= gszAppName;
    wcex.hIconSm		= LoadImage( hinstCurrent, "AppIcon",
							         IMAGE_ICON,
									 GetSystemMetrics(SM_CXSMICON),
									 GetSystemMetrics(SM_CYSMICON),
									 LR_DEFAULTCOLOR );
    if ( !RegisterClassEx( &wcex ) )
        return FALSE;
    return TRUE;
}

BOOL InitInstance( HINSTANCE hinstCurrent )
{
    HWND hwnd;

    ghinstApp = hinstCurrent;

    hwnd = CreateWindowEx(
        WS_EX_OVERLAPPEDWINDOW,
        gszAppName, gszWindowTitle,
        WS_OVERLAPPEDWINDOW,
        CW_USEDEFAULT, CW_USEDEFAULT,
        CW_USEDEFAULT, CW_USEDEFAULT,
        NULL, NULL, ghinstApp, NULL );
    if ( !hwnd )
        return FALSE;

    ShowWindow( hwnd, SW_SHOWDEFAULT );
    UpdateWindow( hwnd );

    return TRUE;
}

LRESULT CALLBACK ClockWndProc( HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam )
{
    LRESULT lResult = 0L;

    switch ( msg )
    {
        case WM_CREATE:
            lResult = HANDLE_WM_CREATE( hwnd, wParam, lParam, Clock_OnCreate );
            break;
        case WM_SIZE:
            lResult = HANDLE_WM_SIZE( hwnd, wParam, lParam, Clock_OnSize );
            break;
        case WM_PAINT:
            lResult = HANDLE_WM_PAINT( hwnd, wParam, lParam, Clock_OnPaint );
            break;
        case WM_TIMER:
            lResult = HANDLE_WM_TIMER( hwnd, wParam, lParam, Clock_OnTimer );
            break;
		case WM_RBUTTONUP:
			lResult = HANDLE_WM_RBUTTONUP( hwnd, wParam, lParam, Clock_OnRButtonUp );
			break;
		case WM_COMMAND:
			lResult = HANDLE_WM_COMMAND( hwnd, wParam, lParam, Clock_OnCommand );
			break;
        case WM_DESTROY:
            lResult = HANDLE_WM_DESTROY( hwnd, wParam, lParam, Clock_OnDestroy );
            break;
        default:
            lResult = DefWindowProc( hwnd, msg, wParam, lParam );
            break;
    }
    return lResult;
}

BOOL Clock_OnCreate( HWND hwnd, LPCREATESTRUCT lpcs )
{
    if ( !SetTimer( hwnd, ID_TIMER, 1000, NULL ) )
        return FALSE;

    ghmenuPopup = LoadMenu( ghinstApp, gszAppName );
	ghmenuPopup = GetSubMenu( ghmenuPopup, 0 );
	GetLocalTime( &gstCurrent );
    gstPrevious = gstCurrent;

	gbSoundEnabled = TRUE;
    return TRUE;
}

void Clock_OnSize( HWND hwnd, UINT state, int cx, int cy )
{
    gcxClient = cx;
    gcyClient = cy;
}

void Clock_OnPaint( HWND hwnd )
{
    HDC hdc;
    PAINTSTRUCT ps;

    hdc = BeginPaint( hwnd, &ps );
    SetClockMapMode( hdc );
	DrawFace( hdc );
	DrawHands( hdc, &gstPrevious );
    EndPaint( hwnd, &ps );
}

void Clock_OnTimer( HWND hwnd, UINT id )
{
    HDC hdc;

    GetLocalTime( &gstCurrent );

	if ( (gstCurrent.wHour != gstPrevious.wHour) ||
	     (gstCurrent.wMinute != gstPrevious.wMinute) )
	{
	    gstPrevious = gstCurrent;
		InvalidateRect( hwnd, NULL, TRUE );
	}
	else
	{
		hdc = GetDC( hwnd );
		SetClockMapMode( hdc );
		DrawSecondHand( hdc, &gstPrevious );
		DrawSecondHand( hdc, &gstCurrent );
		ReleaseDC( hwnd, hdc );
	    gstPrevious = gstCurrent;
	}

	if ( gbSoundEnabled )
		PlaySound( "tick.wav", NULL, SND_FILENAME | SND_ASYNC );
}

void Clock_OnRButtonUp( HWND hwnd, int x, int y, UINT flags )
{
	POINT pt;

	pt.x = x;
	pt.y = y;
	ClientToScreen( hwnd, &pt );
	TrackPopupMenu( ghmenuPopup,
		TPM_CENTERALIGN | TPM_LEFTBUTTON,
		pt.x, pt.y, 0, hwnd, NULL );
}

void Clock_OnCommand( HWND hwnd, int id, HWND hwndCtl, UINT codeNotify )
{
	switch ( id )
	{
		case IDM_SOUND:
			if ( gbSoundEnabled )
			{
				CheckMenuItem( ghmenuPopup, id, MF_UNCHECKED );
				gbSoundEnabled = FALSE;
			}
			else
			{
				CheckMenuItem( ghmenuPopup, id, MF_CHECKED );
				gbSoundEnabled = TRUE;
			}
			break;

		case IDM_ABOUT:
			DialogBox( ghinstApp, "AboutBox", hwnd, AboutDlgProc );
			break;

		default:
			break;
	}
}

void Clock_OnDestroy( HWND hwnd )
{
	KillTimer( hwnd, ID_TIMER );
	DestroyMenu( ghmenuPopup );
    PostQuitMessage( 0 );
}

void SetClockMapMode( HDC hdc )
{
    SetMapMode( hdc, MM_ISOTROPIC );
    SetWindowExtEx( hdc, 1000, 1000, NULL );
    SetViewportExtEx( hdc, gcxClient / 2, -gcyClient / 2, NULL );
    SetViewportOrgEx( hdc, gcxClient / 2, gcyClient / 2, NULL );
}

void DrawFace( HDC hdc )
{
	HENHMETAFILE hemf;
	const RECT rect = { -1000, 1000, 1730, -1000 };
	HBRUSH hbrOld, hbrYellow;
    int iAngle;
    HPEN hpenOld, hpenThick, hpenThin;
    const POINT ptStart = { 0, 990 };
    const POINT ptEnd = { 0, 950 };
    POINT ptTemp;

	hbrYellow = CreateSolidBrush( RGB( 255, 255, 0 ) );
	hbrOld = SelectBrush( hdc, hbrYellow );
	Ellipse( hdc, -1000, 1000, 1000, -1000 );
	SelectBrush( hdc, hbrOld );
    DeleteBrush( hbrYellow );

    hemf = GetEnhMetaFile( "MBnetLogo.emf" );
    PlayEnhMetaFile( hdc, hemf, &rect );
    DeleteEnhMetaFile( hemf );

    hpenThick = CreatePen( PS_SOLID, 10, GetSysColor(RGB(255,255,255)) );
    hpenThin  = CreatePen( PS_SOLID,  5, GetSysColor(RGB(255,255,255)) );
    hpenOld = SelectPen( hdc, hpenThick );
    for ( iAngle = 0;  iAngle < 360;  iAngle += 6 )
    {
        ptTemp = ptStart;
        RotatePoint( &ptTemp, 1, iAngle );
        MoveToEx( hdc, ptTemp.x, ptTemp.y, NULL );

        SelectPen( hdc, (iAngle % 5) ? hpenThin : hpenThick );

        ptTemp = ptEnd;
        RotatePoint( &ptTemp, 1, iAngle );
        LineTo( hdc, ptTemp.x, ptTemp.y );
    }
    SelectPen( hdc, hpenOld );
    DeletePen( hpenThick );
    DeletePen( hpenThin );
}

void DrawHands( HDC hdc, LPSYSTEMTIME st )
{
	const POINT aptHourHand[5] = {
        {-40, -100}, {-40,  600}, {0, 650},
        { 40,  600}, { 40, -100}
    };
	const POINT aptMinuteHand[5] = {
        {-40, -100}, {-40,  900}, {0, 950}, 
        { 40,  900}, { 40, -100}
    };
	POINT aptTemp[5];
	int iHourAngle, iMinuteAngle;
	HBRUSH hbrOld;

    iHourAngle = (st->wHour % 12) * 30 + st->wMinute / 2;
	iMinuteAngle = st->wMinute * 6;

	hbrOld = SelectBrush( hdc, GetStockBrush(WHITE_BRUSH) );

	CopyMemory( aptTemp, aptHourHand, sizeof(aptHourHand) );
	RotatePoint( aptTemp, 5, iHourAngle );
	Polygon( hdc, aptTemp, 5 );

	CopyMemory( aptTemp, aptMinuteHand, sizeof(aptMinuteHand) );
	RotatePoint( aptTemp, 5, iMinuteAngle );
	Polygon( hdc, aptTemp, 5 );

	SelectBrush( hdc, hbrOld );

	DrawSecondHand( hdc, st );
}

void DrawSecondHand( HDC hdc, LPSYSTEMTIME st )
{
	const POINT ptSecondHand = { 0, 950 };
	const RECT rcCenter = { -10, 10, 10, -10 };
	POINT aptTemp[2];
	int iAngle;

	iAngle = st->wSecond * 6;
	SetROP2( hdc, R2_NOTXORPEN );
	Ellipse( hdc, rcCenter.left, rcCenter.top,
                  rcCenter.right, rcCenter.bottom );
	aptTemp[0] = ptSecondHand;
	RotatePoint( aptTemp, 1, iAngle );
	MoveToEx( hdc, 0, 0, NULL );
	LineTo( hdc, aptTemp[0].x, aptTemp[0].y );
}

void RotatePoint( POINT pt[], int iNum, int iAngle )
{
	const double PI = 3.141592654;
	double dRad, dCos, dSin;
    int i;
    POINT ptTemp;

	dRad = PI * iAngle / 180;
	dCos = cos( dRad );
	dSin = sin( dRad );

    for ( i = 0;  i < iNum;  i++ )
    {
        ptTemp.x = (int)(pt[i].x * dCos + pt[i].y * dSin);
        ptTemp.y = (int)(pt[i].y * dCos - pt[i].x * dSin);
        pt[i] = ptTemp;
    }
}

BOOL CALLBACK AboutDlgProc( HWND hdlg, UINT message, WPARAM wParam, LPARAM lParam )
{
	BOOL bResult = FALSE;

	switch ( message )
	{
		case WM_INITDIALOG:
			bResult = TRUE;
			break;

		case WM_COMMAND:
			switch ( LOWORD( wParam ) )
			{
				case IDOK:
				case IDCANCEL:
					EndDialog( hdlg, 0 );
					bResult = TRUE;
					break;
			}
			break;
		
		default:
			break;
	}

	return bResult;
}
