unit in4share;

// Copyright  1999 by Ziff-Davis, Inc.
// Written by Neil J. Rubenking


interface
USES Forms, Classes, Windows, SysUtils, Graphics;

function FinSlash(const S : String) : String;
function NoFinSlash(const S : String) : String;
function AddToHierList(TS : TStrings; const S : String) : Integer;
function ValidINI(const S : String) : Boolean;
function ValidReg(const S : String) : Boolean;
function ProcMsgTerminated : Boolean;
function ExeType(const TheName : String; Pltfrm : Integer) : Integer;
function ExeTypeName(typ : Integer) : String;
function ExeNameFromPID(PID : THandle) : String;

type
  String4 = String[4];
  TTrackMode = (tmNone, tmReal, tmTime, tmDisk, tmIgno);

  UpdateStatusFunc = procedure(const S : String) of object;

  RegObject = class(TObject)
  public
    fType, fProcId : Integer;
    fVal, fDat, fPrv : String;
    constructor Create(vType, vProcId : Integer;
      const vVal, vDat, vPrv : String);
  end;

  FilOpeObject = class(TObject)
  public
    fSizeH : DWORD;
    fSizeL : DWORD;
    fDateH : DWORD;
    fDateL : DWORD;
    fProcID : Integer;
    constructor Create(vhigh_size, vlow_size, vhigh_datetime,
      vlow_datetime : DWORD; vProcID : Integer);
  end;

const
  // Constants for AddToHierList
  ADL_OK      = 0;
  ADL_BELOW   = 1;
  ADL_ABOVE   = 2;
  ADL_DUPE    = 3;
  // Constants for ExeType
  ET_UNKNOWN  = -3;
  ET_NOTEXE   = -2;
  ET_NOEXIST  = -1;
  ET_BLANK    = 0;
  ET_DOSEXE   = 1;
  ET_COM      = 2;
  ET_BAT      = 3;
  ET_WINEXE   = 4;
  ET_PEXE     = 5;
  ET_INFFILE  = 6;
  // Constants for windows platform/version
  WV_95       = 1;
  WV_98       = 2;
  WV_9x       = -1;
  WV_NT4      = 3;
  WV_NT5      = 4;
  WV_NT3      = -2;
  WV_NTx      = -3;
  WV_UNK      = -4;
  // constants for temporary files
  TempExt1 = '.$$1';
  TempExt2 = '.$$2';
  TempName = 'INCTRL$$';
  RegName  = 'REG$$$$$';
  DevName  = 'DEV$$$$$';

VAR
  IniName    : String;
  windir     : String;
  OutPath    : String;
  Platform   : Integer;
  PlatVer    : Integer;
  PlatVerStr : String;
  CurVerKey  : String;

implementation
uses IniFiles, Registry, ShellApi, CtrlTrap, tlhelp32;

constructor RegObject.Create(vType, vProcID : Integer;
  const vVal, vDat, vPrv : String);
begin
  inherited Create;
  fType     := vType;
  fProcId   := vProcId;
  IF vVal = '' THEN fVal := '@'
  ELSE fVal := vVal;
  fDat      := vDat;
  fPrv      := vPrv;
end;

constructor FilOpeObject.Create(vhigh_size, vlow_size, vhigh_datetime,
  vlow_datetime : DWORD; vProcID : Integer);
begin
  inherited Create;
  fSizeH := vhigh_size;
  fSizeL := vlow_size;
  fDateH := vhigh_datetime;
  fDateL := vlow_datetime;
  fProcID := vProcID;
end;

function FinSlash(const S : String) : String;
begin
  IF S = '' THEN Result := S
  ELSE IF S[length(S)] = '\' THEN Result := S
  ELSE Result := S + '\';
end;

function NoFinSlash(const S : String) : String;
begin
  Result := S;
  IF (Length(Result) > 0) AND (Result[Length(Result)] = '\') THEN
    SetLength(Result, Length(Result)-1);
end;

function ProcMsgTerminated : Boolean;
// call processMessages and return whether or not
// the app was terminated
BEGIN
  Application.ProcessMessages;
  Result := Application.Terminated OR (Application.Tag=-1);
  IF Result THEN Application.Tag := -1;
END;

function AddToHierList(TS : TStrings; const S : String) : Integer;
// TS is a sorted list of lowercase hierarchical paths (disk or reg).
// If S is a sub-item ofan item in the list, don't add it, and
// return ADL_BELOW. If an item in the list is a sub-item of S,
// replace it with S and return ADL_ABOVE. If S is present in the
// list, don't add it, and return ADL_DUPE. Otherwise, add S and
// return ADL_OK.
var idx : Integer;
begin
  Result := ADL_OK;
  idx    := TS.Add(lowercase(S));
  IF (idx > 0) THEN
    begin
      // If the item preceding S contains S, then
      //   don't add S
      IF Pos(FinSlash(TS[idx-1]), lowercase(S)) = 1 THEN
        begin
          TS.Delete(idx);
          Result := ADL_BELOW;
        end;
    end;
  IF (Result = ADL_OK) AND (idx < TS.Count-1) THEN
    begin
      // If the item following S equals S, don't add S;
      //   it's a duplicate.
      IF FinSlash(TS[idx+1]) = FinSlash(lowercase(S)) THEN
        begin
          TS.Delete(idx+1);
          Result := ADL_DUPE
        end
      // If S is a prefix of the item following S, remove
      //   the item following
      ELSE
        WHILE (idx < TS.Count-1) AND
              (Pos(FinSlash(lowercase(S)), TS[idx+1]) = 1) DO
          begin
            TS.Delete(idx+1);
            Result := ADL_ABOVE;
          end;
    end;
end;

function ValidINI(const S : String) : Boolean;
// Returns TRUE if the file exists and contains at
//   least one section heading
VAR temps : TStringList;
begin
  WITH TIniFile.Create(S) DO
  try
    temps := TStringList.Create;
    try
      ReadSections(temps);
      Result := temps.Count > 0;
    finally
      temps.Free;
    end;
  finally
    Free;
  end;
end;

function ValidReg(const S : String) : Boolean;
// Returns true only if the specified Registry key exists *and*
//   is among the area tracked by InCtrl4
VAR
  hRoot : hKey;
  Key : String;
  P : Integer;
begin
  Result := False;
  hRoot  := 0;
  P      := Pos('HKEY_USERS', S);
  IF P = 1 THEN
    begin
      hRoot := HKEY_USERS;
      Key   := S;
      Delete(Key, 1, Length('HKEY_USERS')+1);
    end
  ELSE
    begin
      P := Pos('HKEY_LOCAL_MACHINE', S);
      IF P = 1 THEN
        begin
          hRoot := HKEY_LOCAL_MACHINE;
          Key   := S;
          Delete(Key, 1, Length('HKEY_LOCAL_MACHINE')+1);
        end
    end;
  IF hRoot = 0 THEN Exit;
  WITH TRegistry.Create DO
  try
    RootKey := hRoot;
    Result  := OpenKey(Key, false);
  finally
    Free;
  end;
end;

FUNCTION ExeType(const TheName : String; Pltfrm : Integer) : Integer;
VAR
  TSFI : TShFileInfo;
  I    : DWORD;
  Ext  : String;
BEGIN
  Result := ET_BLANK;
  IF TheName = '' THEN Exit;
  Result := ET_UNKNOWN;
  IF Pos('.', theName) = 0 THEN Exit;
  IF Length(theName) - Pos('.', theName) < 3 THEN Exit;
  Result := ET_NOEXIST;
  IF NOT FileExists(theName) THEN Exit;
  IF Pltfrm = VER_PLATFORM_WIN32_WINDOWS THEN
    BEGIN
      I := SHGetFileInfo(PChar(theName), 0, TSFI, SizeOf(TSFI),
        SHGFI_EXETYPE);
      CASE LoWord(I) OF
        $4550 : Result := ET_PEXE;   //PE signature
        $454E : Result := ET_WINEXE; //NE signature
        $5A4D : BEGIN                //MZ signature
                  Ext := Uppercase(ExtractFileExt(theName));
                  IF Ext = '.EXE' THEN
                    Result := ET_DOSEXE
                  ELSE IF Ext = '.COM' THEN
                    Result := ET_COM
                  ELSE IF Ext = '.BAT' THEN
                    Result := ET_BAT
                  ELSE Result := ET_NOTEXE;
                END;
        ELSE
          BEGIN
            Ext := Uppercase(ExtractFileExt(theName));
            IF Ext = '.INF' THEN Result := ET_INFFILE
            ELSE Result := ET_NOTEXE;
          END;
      END;
    END
  ELSE
    BEGIN
      IF GetBinaryType(PChar(theName), I) THEN
        BEGIN
          CASE I OF
            SCS_32BIT_BINARY : Result := ET_PEXE;
            SCS_WOW_BINARY   : Result := ET_WINEXE;
            SCS_DOS_BINARY   : BEGIN
              Ext := Uppercase(ExtractFileExt(theName));
              IF Ext = '.EXE' THEN
                Result := ET_DOSEXE
              ELSE IF Ext = '.COM' THEN
                Result := ET_COM
              ELSE Result := ET_NOTEXE;
            END;
            ELSE
              BEGIN
                Ext := Uppercase(ExtractFileExt(theName));
                IF Ext = '.INF' THEN Result := ET_INFFILE
                ELSE IF Ext = '.BAT' THEN Result := ET_BAT
                ELSE Result := ET_NOTEXE;
              END;
          END;
        END
      ELSE Result := ET_NOTEXE;
    END;
END;

FUNCTION ExeTypeName(typ : Integer) : String;
BEGIN
  CASE typ OF
    ET_BLANK    : Result := '(no program)';
    ET_DOSEXE   : Result := 'DOS EXE file';
    ET_WINEXE   : Result := '16-bit Windows Executable';
    ET_PEXE     : Result := '32-bit Windows Executable';
    ET_COM      : Result := 'DOS COM file';
    ET_BAT      : Result := 'DOS batch file';
    ET_NOTEXE   : Result := 'Not an executable file';
    ET_NOEXIST  : Result := 'File does not exist';
    ET_INFFILE  : Result := 'Setup .INF file';
    ELSE          Result := '';
  END;
END;

function ExeNameFromPID(PID : THandle) : String;
const obsfucator : DWORD = 0;
VAR
  PI32  : TProcessentry32;
  hSnap : THandle;
  f     : Bool;
begin
  Result := '*error*';
  IF (PlatVer <> WV_95) AND (PlatVer <> WV_98) THEN Exit;
  IF obsfucator = 0 THEN
    obsfucator := getObsfucator;
  hSnap := CreateToolHelp32Snapshot(TH32CS_SNAPPROCESS, 0);
  IF hSnap = 0 THEN Exit;
  PI32.dwSize := SizeOf(pi32);
  f := Process32First(hSnap, PI32);
  WHILE f DO
    begin
      IF PI32.th32processid XOR obsfucator = PID THEN
        begin
          Result := StrPas(PI32.szExefile);
          Break;
        end;
      f := Process32Next(hSnap, PI32);
    end;
end;

VAR OVI : TOsVersionInfo;

initialization
  IniName := ChangeFileExt(Application.ExeName, '.INI');
  SetLength(windir, MAX_PATH);
  SetLength(windir, GetWindowsDirectory(PChar(windir), MAX_PATH));
  windir := LowerCase(FinSlash(windir));
  OutPath := ExtractFilePath(Application.ExeName);
  OVI.dwOSVersionInfoSize := SizeOf(OVI);
  GetVersionEx(OVI);
  Platform := OVI.dwPlatformID;
  CASE OVI.dwPlatformID OF
    VER_PLATFORM_WIN32_WINDOWS : begin
      CurVerKey := '\SOFTWARE\Microsoft\Windows\CurrentVersion';
      IF OVI.dwMajorVersion = 4 THEN
        begin
          CASE OVI.dwMinorVersion OF
            0    : PlatVer := WV_95;
            10   : PlatVer := WV_98;
            ELSE   PlatVer := WV_9x;
          END;
        end
      ELSE PlatVer := WV_9x;
    end;
    VER_PLATFORM_WIN32_NT : begin
      CurVerKey := '\SOFTWARE\Microsoft\Windows NT\CurrentVersion';
      CASE OVI.dwMajorVersion OF
        3    : PlatVer := WV_NT3;
        4    : PlatVer := WV_NT4;
        5    : PlatVer := WV_NT5;
        ELSE   PlatVer := WV_NTx;
      END;
    end;
    ELSE PlatVer := WV_UNK;
  END;
  CASE PlatVer OF
    WV_95  : PlatVerStr := 'Windows 95';
    WV_98  : PlatVerStr := 'Windows 98';
    WV_9x  : PlatVerStr := 'Windows 9x?';
    WV_NT4 : PlatVerStr := 'Windows NT4';
    WV_NT5 : PlatVerStr := 'Windows NT5';
    WV_NT3 : PlatVerStr := 'Windows NT3';
    WV_NTx : PlatVerStr := 'Windows NT?';
    WV_UNK : PlatVerStr := 'Windows ???';
  end;
end.
