unit in32INIu;

// Copyright  1999 by Ziff-Davis, Inc.
// Written by Neil J. Rubenking

interface
USES Windows, SysUtils, Classes, Forms, in4share;

procedure RememberINI(const InName, OutPath : String;
  Ext : String4);
procedure NormalizeINI(const InN, OutN : String);
procedure CompareINI(const OldN, NewN : TFilename;
  vAddK, vDelK, vChaK, vAddS, vDelS : TStringList);
procedure DoDevices(const InN, OutN : String);
procedure CompareDev(const OldN, NewN : TFilename;
  vAddK, vDelK, vChaK : TStringList);
function SafeName(const S : String) : String;

implementation
uses IniFiles;

function SafeName(const S : String) : String;
// Returns a safe name for temporary representation of
// this .INI file. The safe name is a full pathname with
// all backslash and colon characters changed to underscores.
VAR N : Integer;
begin
  Result := S;
  FOR N := 1 TO Length(Result) DO
    IF (Result[N] = ':') OR (Result[N] = '\') THEN
      Result[N] := '_';
end;

procedure RememberINI(const InName, OutPath : String;
  Ext : String4);
// Record a copy of an .INI file in a form that lends itself
// to before/after comparison
VAR
  iName, tName : String;
  buffer : ARRAY[0..255] OF Char;
BEGIN
  IF ProcMsgTerminated THEN Exit;
  iName := ExtractFilePath(InName);
  IF iName = '' THEN
    BEGIN
      GetWindowsDirectory(buffer, 255);
      iName := FinSlash(StrPas(Buffer));
      iName := iName + InName;
    END
  ELSE iName := InName;
  tName := SafeName(iname);
  tName := ChangeFileExt(OutPath + tName, Ext);
  IF FileExists(iName) THEN
    NormalizeIni(iName, tName);
END;

procedure CompareDev(const OldN, NewN : TFilename;
  vAddK, vDelK, vChaK : TStringList);
// Take two sorted lists of device= lines and compare them,
// reporting keys added or deleted}
VAR
  OldF, NewF : TextFile;
  OldL, NewL : String;
  N          : Integer;

  PROCEDURE ReadNext(VAR T : TextFile; VAR Line : String);
  // Read next LINE from file. If at end of file,
  // return a string of 255 Z characters
  BEGIN
    IF EoF(T) THEN
      BEGIN
        Line := StringOfChar('Z', 255);
        Exit;
      END;
    ReadLn(T, Line);
  END;

BEGIN
  AssignFile(OldF, OldN);
  AssignFile(NewF, NewN);
  FOR N := vAddK.Count-1 DOWNTO 0 DO
    IF Pos('[386ENH]DEVICE=', UpperCase(vAddK[N])) > 0 THEN
      vAddK.Delete(N);
  FOR N := vDelK.Count-1 DOWNTO 0 DO
    IF Pos('[386ENH]DEVICE=', UpperCase(vDelK[N])) > 0 THEN
      vDelK.Delete(N);
  // the only reason we pass vChaK is to delete any spurious changes
  FOR N := vChaK.Count-1 DOWNTO 0 DO
    IF Pos('[386ENH]DEVICE=', UpperCase(vChaK[N])) > 0 THEN
      vChaK.Delete(N);
  try
    FileMode := 0; // read only
    Reset(OldF);
    try
      Reset(NewF);
      ReadNext(OldF, OldL);
      ReadNext(NewF, NewL);
      WHILE NOT (EoF(OldF) AND EoF(NewF)) DO
        CASE AnsiCompareText(OldL, NewL) OF
          -1 : BEGIN
            vDelK.Add('[386Enh]'+OldL);
            ReadNext(OldF, OldL);
          END;
          0  : BEGIN
            ReadNext(OldF, OldL);
            ReadNext(NewF, NewL);
          END;
          1  : BEGIN
            vAddK.Add('[386Enh]'+NewL);
            ReadNext(NewF, NewL);
          END;
        END;
    IF OldL <> NewL THEN
      BEGIN
        IF Length(OldL) <> 255 THEN
          vDelK.Add('[386Enh]'+OldL);
        IF Length(NewL) <> 255 THEN
          vAddK.Add('[386Enh]'+NewL);
      END;
    finally
      CloseFile(NewF);
    end;
  finally
    CloseFile(OldF);
    FileMode := 2; // read/write
  end;
END;

procedure CompareINI(const OldN, NewN : TFilename;
      vAddK, vDelK, vChaK, vAddS, vDelS : TStringList);
// Take two normalized INI files and compare them, reporting
//  sections and keys added, deleted, or changed
VAR
  OldF, NewF : TextFile;
  OldL, NewL : String;
  OldC, NewC : String;
  OldK, NewK : String;
  OldV, NewV : String;

  PROCEDURE ReadNext(VAR T : TextFile; VAR Line, Sec, Key,
    Valu: String);
  //Read next item from file, extracting section and key.
  // If at end of file, return a string of 255
  // Z characters in the Sec parameter. (In the ANSI sorting
  // sequence, left-bracket comes last, Z next-to-last. Since
  // left-bracket is significant in INI files, we use Z)}
  VAR
    Posn : Integer;
  BEGIN
    IF EoF(T) THEN
      BEGIN
        Line := StringOfChar('Z', 255);
        Exit;
      END;
    ReadLn(T, Line);
    Posn := Pos(']', Line);
    Sec := Copy(Line, 2, Posn-2);
    Key := Copy(Line, Posn+1, 255);
    Posn := Pos('=', Key);
    IF Posn > 0 THEN
      BEGIN
        Valu := Copy(Key, Posn+1, 255);
        Key := Copy(Key, 1, Posn-1);
      END
    ELSE Valu := '';
  END;

//!!0.1  function CompareSecKeyVal(const s1, s2, k1, k2, v1, v2 : String) :
  function CompareSecKeyVal(const s1, s2, k1, k2, v1, v2 : String; VAR f1, f2 : Text) :
    Integer;
  begin
    IF EoF(f1) THEN Result := 1             //!!0.1
    ELSE IF EoF(f2) THEN Result := -1       //!!0.1
    ELSE Result := AnsiCompareText(s1, s2); //!!0.1
//!!0.1    Result := AnsiCompareText(s1, s2);
    IF Result = 0 THEN
      begin
        Result := AnsiCompareText(k1, k2);
        IF Result = 0 THEN
          Result := AnsiCompareStr(v1, v2);
      end;
  end;

BEGIN
  vAddK.Clear;
  vDelK.Clear;
  vChaK.Clear;
  vAddS.Clear;
  vDelS.Clear;
  IF NOT FileExists(OldN) THEN Exit; //!!0.1
  IF NOT FileExists(NewN) THEN Exit; //!!0.1
  AssignFile(OldF, OldN);
  AssignFile(NewF, NewN);
  try
    FileMode := 0; // read-only
    Reset(OldF);
    try
      Reset(NewF);
      ReadNext(OldF, OldL, OldC, OldK, OldV);
      ReadNext(NewF, NewL, NewC, NewK, NewV);
      // NOTE: Should compare section/section NO BRACES,
      //    then key/key, then valu by case
      WHILE NOT (EoF(OldF) AND EoF(NewF)) DO
//        CASE AnsiCompareText(OldL, NewL) OF
//!!0.1        CASE CompareSecKeyVal(OldC, NewC, OldK, NewK, OldV, NewV) OF
        CASE CompareSecKeyVal(OldC, NewC, OldK, NewK, OldV, NewV, OldF, NewF) OF
          -1 : BEGIN
            IF (OldK<>'') AND (AnsiCompareText(OldK,NewK)=0) THEN
              // just a change
              BEGIN
                vChaK.Add(OldL + '=to=' + NewV);
                ReadNext(OldF, OldL, OldC, OldK, OldV);
                ReadNext(NewF, NewL, NewC, NewK, NewV);
              END
            ELSE
              BEGIN  // old is less; was deleted
                IF OldK='' THEN vDelS.Add(OldL)
                ELSE vDelK.Add(OldL);
                ReadNext(OldF, OldL, OldC, OldK, OldV);
              END;
          END;
          0  : BEGIN
            ReadNext(OldF, OldL, OldC, OldK, OldV);
            ReadNext(NewF, NewL, NewC, NewK, NewV);
          END;
          1  : BEGIN
            IF (NewK<>'') AND (AnsiCompareText(OldK,NewK)=0) THEN
              // just a change
              BEGIN
                vChaK.Add(OldL + '=to=' + NewV);
                ReadNext(OldF, OldL, OldC, OldK, OldV);
                ReadNext(NewF, NewL, NewC, NewK, NewV);
              END
            ELSE
              BEGIN  // new is less; was added
                IF NewK = '' THEN vAddS.Add(NewL)
                ELSE vAddK.Add(NewL);
                ReadNext(NewF, NewL, NewC, NewK, NewV);
              END;
          END;
        END;
    IF OldL <> NewL THEN
      BEGIN
        IF Length(OldL) <> 255 THEN
          IF OldK='' THEN vDelS.Add(OldL)
          ELSE vDelK.Add(OldL);
        IF Length(NewL) <> 255 THEN
          IF NewK = '' THEN vAddS.Add(NewL)
          ELSE vAddK.Add(NewL);
      END;
    finally
      CloseFile(NewF);
    end;
  finally
    CloseFile(OldF);
    FileMode := 2; // read-write
  end;
END;

procedure NormalizeINI(const InN, OutN : String);
// Convert an INI file into "normalized form", sorted by
// section and then by key, with the section name prefixed
// to each key in that section, e.g.
// [Windows]NullPort=None
VAR
  OutF             : TextFile;
  SecList, KeyList : TStringList;
  SecNum, KeyNum   : Integer;
begin
  AssignFile(OutF, OutN);
  Rewrite(OutF);
  try
    WITH TIniFile.Create(InN) DO
      try
        SecList := TStringList.Create;
        try
          KeyList := TStringList.Create;
          try
            ReadSections(SecList);
            SecList.Sort;
            FOR SecNum := 0 TO SecList.Count-1 DO
              BEGIN
                WriteLn(OutF, '[', SecList[SecNum], ']');
                KeyList.Clear;
                ReadSectionValues(SecList[SecNum], KeyList);
                KeyList.Sort;
                FOR KeyNum := 0 TO KeyList.Count-1 DO
                  WriteLn(OutF, '[', SecList[SecNum], ']',
                    KeyList[KeyNum]);
              END;
          finally
            KeyList.Free;
          end;
        finally
          SecList.Free;
        end;
      finally
        Free;
      end;
  finally
    CloseFile(OutF);
  end;
end;

procedure DoDevices(const InN, OutN : String);
VAR
  InF     : TextFile;
  S       : String;
  Status  : Integer;
begin
  Assign(InF, InN);
  FileMode := 0; // read-only
  Reset(InF);
  FileMode := 2; // read-write
  try
    WITH TStringList.Create DO
      try
        Status := -1;
        WHILE (NOT EoF(InF)) AND (Status < 1) DO
          BEGIN
            ReadLn(InF, S);
            S := Trim(S);
            IF Status = 0 THEN
              BEGIN
                IF Length(S) > 0 THEN
                  BEGIN
                    IF S[1]='[' THEN Status := 1
                    ELSE IF Pos('DEVICE=', UpperCase(S)) > 0 THEN
                      Add(S);
                  END;
              END
            ELSE IF AnsiCompareText(S, '[386Enh]') = 0 THEN
              Status := 0;
          END;
        Sort;
        SaveToFile(OutN);
      finally
        Free;
      end;
  finally
    Close(InF);
  end;
end;

end.
