/*
 *  @(#) maplay.cc 1.20, last edit: 6/22/94 12:32:55
 *  @(#) Copyright (C) 1993, 1994 Tobias Bading (bading@cs.tu-berlin.de)
 *  @(#) Berlin University of Technology
 *
 *  Many thanks for ideas and implementations to:
 *  -> Jim Boucher (jboucher@flash.bu.edu)
 *     for his idea and first implementation of 8 kHz u-law output
 *  -> Louis P. Kruger (lpkruger@phoenix.princeton.edu)
 *     for his implementation of the LinuxObuffer class
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 *  Changes from version 1.1 to 1.2:
 *    - minor changes to create a LinuxObuffer object
 *    - minor changes for a u-law version, which creates 8 kHz u-law output
 *      on an amd device or in stdout mode, if compiled with ULAW defined
 *    - option -amd forces maplay to treat /dev/audio as an amd device
 *      in the u-law version. This is helpful on some SPARC clones.
 *    - iostreams manipulator calls like "cerr << setw (2) << ..." replaced by
 *      "cerr.width (2); ..." due to problems with older GNU C++ releases.
 */

/*
 * Playlists added and other minor changes by Timo Jantunen
 * (timo.jantunen@hut.fi).
 *
 * Added MPEG-1 Layer III routines from maplay 1.2+ 1.80 for
 * Windows by Jeff Tsay.
 *
 */

/*
 * Better randomizer implemented by Hendrik Pagenhardt
 * (pagenhar@sunpool.cs.uni-magdeburg.de)
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <iostream.h>
#include <iomanip.h>
#include "all.h"
#include "crc.h"
#include "header.h"
#include "ibitstream.h"
#include "obuffer.h"
#include "subband.h"
#include "subband_layer_1.h"
#include "subband_layer_2.h"
#include "synthesis_filter.h"

//L3
#include "bit_res.h"
#include "scalefactors.h"
#include "l3type.h"
#include "layer3.h"

#include <ctype.h>
#include <time.h>

#ifdef OS2
#define INCL_DOSPROCESS
#include <os2.h>
#endif


// space for storing file names & comments
#define MAXFILES 1024
#define BIGBSIZE 80*MAXFILES
static char *fname[MAXFILES],*fcomment[MAXFILES],fnbuffer[BIGBSIZE];
static char *nullcomment="";
int numfiles=0;

// data extracted from commandline arguments:
bool verbose_mode = False, filter_check = False;
bool quiet_mode=False, random_order=False;
static bool stdout_mode = False;
static enum e_channels which_channels = both, which_channels_save=both;
bool raise_priority = True, debug_mode = False;
#ifdef ULAW
static bool force_amd = False;
#endif
static bool use_own_scalefactor = False;
static real scalefactor;

// data extracted from header of first frame:
static uint32 layer;
static e_mode mode;
static e_sample_frequency sample_frequency;

// objects needed for playing a file:
static Ibitstream *stream;
static Crc16 *crc;
static SynthesisFilter *filter1 = NULL, *filter2 = NULL;
static Obuffer *buffer = NULL;
Header *header = NULL;
Subband *subbands[32] = {NULL, NULL, NULL, NULL,
			 NULL, NULL, NULL, NULL,
			 NULL, NULL, NULL, NULL,
			 NULL, NULL, NULL, NULL,
			 NULL, NULL, NULL, NULL,
			 NULL, NULL, NULL, NULL,
			 NULL, NULL, NULL, NULL,
			 NULL, NULL, NULL, NULL};

LayerIII_Decoder *l3decoder = NULL;

static void Exit (int returncode)
  // delete some objects and exit
{
  int i;

  if ((returncode == 1) && buffer) {
    if (header)
      for (i = 0; i < header->number_of_subbands (); ++i) {
      delete subbands[i];
      subbands[i] = NULL;
      }
  }
  delete buffer;	// delete on NULL-pointers are harmless
  delete filter1;
  delete filter2;
  delete header;
  delete stream;

#ifdef OS2
  // restore process priority
  ULONG rc;
  if(raise_priority) {
    rc=DosSetPriority(PRTYS_THREAD,PRTYC_REGULAR,0,0);
  }
#endif

  exit (returncode);
}


void addPlaylist(char *lname) {// add files (and comments) from a file
  static int pos=0;
  FILE *fp;
  char tmp[1024],*c,*c0;

  fp=fopen(lname,"ra");
  if(fp==NULL) {
    fprintf(stderr,"Error opening playlist file '%s'!\n",lname);
    exit(1);
  }

  while(!feof(fp)) {
    tmp[0]=0;
    fgets(tmp,1022,fp);
    c=tmp;while(isspace(*c)) c++; // skip leading blanks

    if(*c!='#' && *c!=';' && *c!=0) { // skip comments and empty lines
      if(strlen(c)+pos>BIGBSIZE) {
	fprintf(stderr,"Buffer overflow!\nDecrease number of files or size of comments.\n");
	exit(1);
      }
      fname[numfiles]=&fnbuffer[pos]; // start new filename
      while(!isspace(*c)) fnbuffer[pos++]=*c++; // copy name
      fnbuffer[pos++]=0;

      c0=c+1;// start of comment (if it exists) one char after last of filename
      while(isspace(*c)) c++; // skip blanks

      if(*c) { // comment exists
        c=c0; // skip only one space!
        fcomment[numfiles]=&fnbuffer[pos]; // start new comment
        while(*c&&*c!='\r'&&*c!='\n') fnbuffer[pos++]=*c++;// copy comment
        fnbuffer[pos++]=0;
      }
      else
	fcomment[numfiles]=nullcomment;

      numfiles++;
      if (numfiles==MAXFILES) {
        fprintf(stderr,"Too many files! Maximum is %d\n",MAXFILES);
        exit(0);
      }
    }
  }
  fclose(fp);
}


main (int argc, char *argv[])
{
  _wildcard(&argc,&argv); // expand command line file regexps

  uint32 layer;
  e_mode mode;
  e_sample_frequency sample_frequency, old_freq;
  bool stereo=False, old_stereo=True;
  char *huffdec_path="./Huffdec";

  int fnro;
  unsigned i;
  bool read_ready = False, write_ready = False;

  if (argc < 2 || !strncmp (argv[1], "-h", 2))
  {
usage:
    fprintf(stderr,"\nusage: maplay"  // << argv[0]
	    " [option(s)]"
	    " [filename(s)]\n"
	    "  filename   filename of a MPEG audio stream\n"// or - for stdin\n"
	    "  -v         verbose mode\n"
	    "  -q         quiet mode (don't print filenames, when playing multiple files)\n"
#ifdef ULAW
	    "  -s         write u-law samples at 8 kHz rate to 'testout.raw'\n"
#else
	    "  -s         write pcm samples to 'testout.raw'\n"
#endif
	    "  -l         decode only the left channel\n"
	    "  -r         decode only the right channel\n"
#ifdef OS2
	    "  -p         do NOT raise priority\n"
	    "  -d         print debug information\n"
#endif
	    "  -c         check for filter range violations\n"
	    "  -f ushort  use this scalefactor instead of the default value 32768\n"
	    "  -@ list    get files (and comments) from file 'list'\n"
	    "  -a         randomize order\n\n"
	    "@(#) MPEG Audio Player maplay 1.2 "
#ifdef ULAW
	    "(8 kHz u-law "
#else
	    "("
#endif
#ifdef OS2
	    "OS/2 version 0.97)\n"
#else
	    "unknown version)\n"
#endif
	    "@(#) Copyright (C) 1993, 1994 Tobias Bading (bading@cs.tu-berlin.de)\n"
	    "@(#) Berlin University of Technology\n"
	    "@(#) Created: 6/23/94 14:12:46\n"
	    "@(#) This program is free software. See the GNU General Public License\n"
	    "@(#) in the file COPYING for more details.\n"
	    "@(#) OS/2 version Copyright (C) 1996 by Timo Jantunen (timo.jantunen@hut.fi)\n"
	    "@(#) OS/2 version Optimizations and Changes\n"
	    "@(#)              by Hendrik Pagenhardt (pagenhar@sunpool.cs.uni-magdeburg.de)\n"
            "@(#) Compiled %s %s.\n",__TIME__,__DATE__);
    exit (0);
  }

  // parse arguments:
  for (i = 1; i < (unsigned)argc; ++i)
    if (argv[i][0] == '-' && argv[i][1])
      switch ((int)argv[i][1])
      {
	case 'v':
	  verbose_mode = True;
	  quiet_mode = False;
	  break;
	case 'q':
	  quiet_mode = True;
	  verbose_mode = False;
	  break;
	case 's':
	  stdout_mode = True;
	  break;
	case 'l':
	  which_channels = left;
	  break;
	case 'r':
	  which_channels = right;
	  break;
#ifdef OS2
	case 'p':
	  raise_priority = False;
	  break;
	case 'd':
	  debug_mode = True;
	  break;
#else
#ifdef ULAW 
	case 'a':
	  force_amd = True;
	  break;
#endif
#endif
	case 'c':
	  filter_check = True;
	  break;
	case 'f':
	  if (++i == (unsigned)argc)
	  {
	    fprintf(stderr,"please specify a new scalefactor after the -f option!\n");
	    exit (1);
	  }
	  use_own_scalefactor = True;
	  sscanf (argv[i], "%f", &scalefactor);
	  break;
	case '@':
	  if (++i == (unsigned)argc)
	  {
	    fprintf(stderr,"please specify a filename after the -@ option!\n");
	    exit (1);
	  }
	  addPlaylist(argv[i]);
	  break;
	case 'a':
	  random_order = True;
	  break;
	default:
	  goto usage;
      }
      else if (numfiles==MAXFILES) {
        fprintf(stderr,"Too many files! Maximum is %d\n",MAXFILES);
        exit(0);
      }        
      else {
        fcomment[numfiles]=nullcomment;
        fname[numfiles++]=argv[i];
      }

  if(numfiles==0)
    goto usage;

#ifdef OS2
  // raise process priority
  if(stdout_mode) raise_priority=False;

  ULONG rc;
  if(raise_priority) {
    rc=DosSetPriority(PRTYS_THREAD,PRTYC_TIMECRITICAL,0,0);
    //printf("DosSetPriority returned %lu\n",rc);
  }
#endif

  if(!quiet_mode && numfiles>1)
    fprintf(stderr,"Playing %d files...\n",numfiles);

// randomize order // #### I don't know how good this really is,
// so I wrote another randomizer. Hendrik

//  if(numfiles>1 && random_order) {
//    int t1,t2; char *t;
//    srand(time(NULL)*101);
//    for(fnro=0;fnro<numfiles*17;fnro++) {
//      t1=(rand()/13)%numfiles;t2=fnro%numfiles;
//      t=   fname[t1];   fname[t1]=   fname[t2];   fname[t2]=t;
//      t=fcomment[t1];fcomment[t1]=fcomment[t2];fcomment[t2]=t;
//    }
//  }

// #### randomizer, which generates a playing order equally distributed on
// all permutations of the files. Needs BSD-libs. Hendrik
  if(numfiles>1 && random_order) {
    int t1,t2; char *t;
    srandom(time(NULL)*1009); random();
    for(fnro = 0; fnro < numfiles - 1; fnro++) {
      t1=fnro+(random()%(numfiles-fnro));t2=fnro;
      t=   fname[t1];   fname[t1]=   fname[t2];   fname[t2]=t;
      t=fcomment[t1];fcomment[t1]=fcomment[t2];fcomment[t2]=t;
    }
  }
  for(fnro = 0; (fnro < numfiles) && (verbose_mode); fnro++)
	fprintf(stderr, "%s\n", fname[fnro]);

  which_channels_save=which_channels; // needed to remember original settings
  // if there is a mono stream in the middle of playlist...

// ******** BEGIN MULTIPLE FILE LOOP *******
// ugly bastard, but works... I think.
  for(fnro=0;fnro<numfiles;fnro++) {

//  if (!strcmp (fname[fnro], "-"))
//    stream = new Ibitstream (0);		// read from stdin
//  else
    stream = new Ibitstream (fname[fnro]);	// read from file

  header = new Header;
  if (!header->read_header (stream, &crc))
  {
    fprintf(stderr,"no header found! (in file %s)\n", fname[fnro]); // ####
    Exit (1);
  }

  // get info from header of first frame:
  layer = header->layer ();
  if ((mode = header->mode ()) == single_channel)
    which_channels = left;
  else
    which_channels=which_channels_save;

  sample_frequency = header->sample_frequency ();

  // create filter(s):
#ifdef ULAW
  if (use_own_scalefactor)
    filter1 = new SynthesisFilter (0, sample_frequency, scalefactor);
  else
    filter1 = new SynthesisFilter (0, sample_frequency);
  if (mode != single_channel && which_channels == both)
    if (use_own_scalefactor)
      filter2 = new SynthesisFilter (1, sample_frequency, scalefactor);
    else
      filter2 = new SynthesisFilter (1, sample_frequency);
#else
  if (use_own_scalefactor)
    filter1 = new SynthesisFilter (0, scalefactor);
  else
    filter1 = new SynthesisFilter (0);
  if (mode != single_channel && which_channels == both)
    if (use_own_scalefactor)
      filter2 = new SynthesisFilter (1, scalefactor);
    else
      filter2 = new SynthesisFilter (1);
#endif	// !ULAW

  // this mpeg stream has same sample frequency and number of channels as last one?
  stereo=(mode == single_channel || which_channels != both)? False:True;
  if(buffer!=NULL && (old_freq!=sample_frequency || old_stereo!=stereo)) {
    delete buffer;
    buffer=NULL;
    //fprintf(stderr,"new obuffer\n");
  }
  old_freq=sample_frequency;old_stereo=stereo;

  // create buffer:
  if(buffer==NULL) // if it don't already exists
  if (stdout_mode)
    if (mode == single_channel || which_channels != both)
      buffer = new FileObuffer (1);
    else
      buffer = new FileObuffer (2);
  else
#ifdef OS2
  {
    if (mode == single_channel || which_channels != both)
      buffer = new OS2Obuffer (1, header);
    else
      buffer = new OS2Obuffer (2, header);
  }
#else
  {
    fprintf(stderr,"Sorry, I don't know your audio device.\n"
	    "Please use the stdout mode.\n");
    Exit (0);
  }
#endif	// !OS2

  // filename without path either MS-DOS or UNIX style
  char *name = strrchr (fname[fnro], '/');
  if (name)
    ++name;
  else {
    name = strrchr (fname[fnro], '\\');
    if(name)
      ++name;
    else
      name = fname[fnro];
  }

  if(!quiet_mode && numfiles>1) {
    if(fcomment[fnro]!=nullcomment)
      fprintf(stderr,"%s %s\n",name,fcomment[fnro]);
    else
      if(!verbose_mode) fprintf(stderr,"%s\n",name);
  }

  if (verbose_mode)
  {
    // print informations about the stream
    fprintf(stderr,"%s is a layer %s %s MPEG audio stream with",
	name,header->layer_string(),header->mode_string());
    if (!header->checksums ())
      fprintf(stderr,"out");
    fprintf(stderr," checksums.\nThe sample frequency is %s"
	 " at a bitrate of %s.\n"
	    "This stream is ",header->sample_frequency_string(),
		header->bitrate_string());
    if (header->original ())
      fprintf(stderr,"an original");
    else
      fprintf(stderr,"a copy");
    fprintf(stderr," and is ");
    if (!header->copyright ())
      fprintf(stderr,"not ");
    fprintf(stderr,"copyright protected.\n");
  }

  // Layer III : initialize decoder

  if (layer == 3) {

    if (mode == single_channel)
      l3decoder = new LayerIII_Decoder(huffdec_path, stream, header, filter1,
                                       NULL, buffer);
    else
      l3decoder = new LayerIII_Decoder(huffdec_path, stream, header, filter1,
                                       filter2, buffer);
  }

  do
  {
    // is there a change in important parameters?
    // (bitrate switching is allowed)
    if (header->layer () != layer)
    {
      // layer switching is allowed

      if (header->layer() == 3) {

        if (mode == single_channel)
          l3decoder = new LayerIII_Decoder(huffdec_path, stream, header,
                                           filter1, NULL, buffer);
        else
          l3decoder = new LayerIII_Decoder(huffdec_path, stream, header,
                                           filter1, filter2, buffer);

      } else if (layer == 3) {
        delete l3decoder;
        l3decoder = NULL;
      }

      if (verbose_mode)
	fprintf(stderr,"switching to layer %s.\n",header->layer_string());
      layer = header->layer ();
    }
    if ((mode == single_channel && header->mode () != single_channel) ||
	(mode != single_channel && header->mode () == single_channel))
    {
      // switching from single channel to stereo or vice versa is not allowed
      fprintf(stderr,"illegal switch from single channel to stereo or vice versa!\n");
      Exit (1);
    }
    if (header->sample_frequency () != sample_frequency)
    {
      // switching the sample frequency is not allowed
      fprintf(stderr,"sorry, can't switch the sample frequency in the middle of the stream!\n");
      Exit (1);
    }

    if (header->layer() != 3) { // Layer III

      // create subband objects:
      if (header->layer () == 1)
	{
	  if (header->mode () == single_channel)
	    for (i = 0; i < header->number_of_subbands (); ++i)
	      subbands[i] = new SubbandLayer1 (i);
	  else if (header->mode () == joint_stereo)
	    {
	      for (i = 0; i < header->intensity_stereo_bound (); ++i)
		subbands[i] = new SubbandLayer1Stereo (i);
	      for (; i < header->number_of_subbands (); ++i)
		subbands[i] = new SubbandLayer1IntensityStereo (i);
	    }
	  else
	    for (i = 0; i < header->number_of_subbands (); ++i)
	      subbands[i] = new SubbandLayer1Stereo (i);
	}
      else if (header->layer () == 2)
	{
	  if (header->mode () == single_channel)
	    for (i = 0; i < header->number_of_subbands (); ++i)
	      subbands[i] = new SubbandLayer2 (i);
	  else if (header->mode () == joint_stereo)
	    {
	      for (i = 0; i < header->intensity_stereo_bound (); ++i)
		subbands[i] = new SubbandLayer2Stereo (i);
	      for (; i < header->number_of_subbands (); ++i)
		subbands[i] = new SubbandLayer2IntensityStereo (i);
	    }
	  else
	    for (i = 0; i < header->number_of_subbands (); ++i)
	      subbands[i] = new SubbandLayer2Stereo (i);
	}

      // start to read audio data:
      for (i = 0; i < header->number_of_subbands (); ++i)
	subbands[i]->read_allocation (stream, header, crc);

      if (header->layer () == 2)
	for (i = 0; i < header->number_of_subbands (); ++i)
	  ((SubbandLayer2 *)subbands[i])->read_scalefactor_selection (stream, crc);

      if (!crc || header->checksum_ok ())
	{
	  // no checksums or checksum ok, continue reading from stream:
	  for (i = 0; i < header->number_of_subbands (); ++i)
	    subbands[i]->read_scalefactor (stream, header);

	  do
	    {
	      for (i = 0; i < header->number_of_subbands (); ++i)
		read_ready = subbands[i]->read_sampledata (stream);

	      do
		{
		  for (i = 0; i < header->number_of_subbands (); ++i)
		    write_ready = subbands[i]->put_next_sample (which_channels, filter1, filter2);

		  filter1->calculate_pcm_samples (buffer);
		  if (which_channels == both && header->mode () != single_channel)
		    filter2->calculate_pcm_samples (buffer);
		}
	      while (!write_ready);
	    }
	  while (!read_ready);

	  buffer->write_buffer (1);		// write to stdout
	}
      else
	// Sh*t! Wrong crc checksum in frame!
	fprintf(stderr,"WARNING: frame contains wrong crc checksum! (throwing frame away)\n");

      for (i = 0; i < header->number_of_subbands (); ++i) {
	delete subbands[i];
	subbands[i] = NULL;
      }
    } else {  // Layer III
      l3decoder->decode();
    }
  }
  while (header->read_header (stream, &crc));

  uint32 range_violations = filter1->violations ();
  if (mode != single_channel && which_channels == both)
   range_violations += filter2->violations ();

  if (filter_check)
  {
    // check whether (one of) the filter(s) produced values not in [-1.0, 1.0]:
    if (range_violations)
    {
      fprintf(stderr,"%d range violations have occured!\n",range_violations);
      if (stdout_mode)
	fprintf(stderr,"If you notice these violations,\n");
      else
	fprintf(stderr,"If you have noticed these violations,\n");
      fprintf(stderr,"please use the -f option with the value ");
      if (mode != single_channel && which_channels == both &&
	  filter2->hardest_violation () > filter1->hardest_violation ())
	fprintf(stderr,"%g",(float)filter2->recommended_scalefactor ());
      else
	fprintf(stderr,"%g",(float)filter1->recommended_scalefactor ());
      fprintf(stderr,"\nor a greater value up to 32768 and try again.\n");
    }
  }

  if (verbose_mode)
  {
    // print playtime of stream:
    real playtime = filter1->seconds_played (Header::frequency (sample_frequency));
    uint32 minutes = (uint32)(playtime / 60.0);
    uint32 seconds = (uint32)playtime - minutes * 60;
    uint32 centiseconds = (uint32)((playtime - (real)(minutes * 60) - (real)seconds) * 100.0);
    fprintf(stderr,"end of stream, playtime: %02d:%02d:%02d\n",
	minutes,seconds,centiseconds);
  }

  //delete buffer;
  //buffer = NULL;

  delete filter1;
  filter1 = NULL;
  delete filter2;
  filter2 = NULL;
  delete l3decoder;
  l3decoder = NULL;

  delete stream;
  stream=NULL;
  delete header;
  header=NULL;

  } // for fnro (multifile loop)

#ifdef OS2
// restore process priority
  if(raise_priority) {
    rc=DosSetPriority(PRTYS_THREAD,PRTYC_REGULAR,0,0);
  }
#endif

  delete buffer;
  return 0;
}

