/*
 * Install.CMD
 *
 * Generic installation loader for Power User projects.
 * Power User is a monthly column by Jim Little published in OS/2 e-Zine!.
 * Find it on the World Wide Web at http://www.haligonian.com/os2/.
 *
 * This program depends on at least two installation files: INSTALL.LST and
 * at least one installation script for each installation option.
 * INSTALL.LST lists all the available project installation options.  It
 * must be in the same directory as this file.  INSTALL.LST uses the
 * following format ('`' is a back-quote, typically under the tilde ('~')
 * next to the 1 key):
 *    filename`install_script`ignore`name`description
 *
 *    filename - zip file the project is contained in; if dot ('.'), the
 *        project is assumed to have already been unzipped into the directory
 *        Install.CMD is in.  Note: zip files are not yet supported for this
 *        parameter in this version.
 *    install_script - name of installation script
 *    ignore - name of installation script that prevents this script from
 *        being an install option.  In other words, this parameter is used
 *        to prevent the user from being asked if they want to install a
 *        script whose program was already installed.  For example, if
 *        installation script A installed an entire project, and installation
 *        scripts B and C each installed half of the project, scripts B and C
 *        would put 'A' in this parameter.  When the user installed the
 *        project, if they installed A first, they would not be asked if
 *        they wanted to install B or C.
 *    name - name of installation script (displayed to user)
 *    description - description of project (displayed to user)
 *    
 * The installation script contains the specific information on how each
 * section of a project is installed.  Each section must have its own
 * zip file and installation script, unless the section is using the '.'
 * (local directory) instead of a zip file.  A future revision may allow a zip
 * file to have multiple scripts.  The installation script is a miniature
 * script language.  Each line in the file is interpreted in order.  The lines
 * have the following format:
 *    command options
 *
 * The following commands are supported in this version:
 * command (options) - description
 * -------------------------------
 * COPY (filename`target) - copies file; target can be directory name or folder object ID
 * MOVE (filename`target) - moves file; target can be directory name or folder object ID
 * OBJFOLDER (name`location`exists_flag`setup`icon_filename`icon_filename2`object_id) - creates folder object
 * OBJPROGRAM (name`location`exists_flag`setup`icon_filename`program_name`parameters`working_dir) - creates program object
 * OBJSHADOW (object`location) - creates a shadow of an object
 * UNZIP (filename`target) - unzips file; target can be directory name or folder object ID
 *
 * The 'exists_flag' option determines what to do if an object to be created
 * already exists.  The available parameters are:
 *    F - fail
 *    R - replace
 *    U - update
 * The 'setup' option allows you to enter additional setup strings.  Note
 * that, depending on the object being created, part of the setup string
 * is generated automatically from the other parameters.
 *
 * For the "location" parameters and for the OBJSHADOW "object" parameter, 
 * either a pathname or an object ID may be used.  If using a pathname, you 
 * may enter either a relative or full path.
 *
 * An example script:
 *    OBJFOLDER My Project`<jlPowerUser>`F``prjicon.ico`prjiconN.ico`<jlPowerUser_Project>
 *    OBJPROGRAM Project Program`<jlPowerUser_Project>`F``program.ico`program.exe
 *    MOVE program.exe`<jlPowerUser_Project>
 *
 * In this example, the program object was created first because there's no
 * way to tell the path of where PROGRAM.EXE will end up.  This way, the WPS
 * will automatically adjust the path when the program file is moved.
 *    
 * 
 * Version 1.0.
 * Last Modified 16 July 1996.
 * Copyright 1996 Jim Little.  Contact me via email at jiml@teleport.com
 * if you want permission to distribute a modified version of this script
 * with your own projects.
 *
 */

/* constants */
projlist = 'INSTALL.LST'		/* project list filename */
folderid = '<jlPowerUser>'		/* object ID of Power User folder */
ignflag = 'IGNORE'			/* flag used to mark installation options to be ignored */

/* Load necessary RexxUtil functions */
call RxFuncAdd 'SysLoadFuncs', 'RexxUtil', 'SysLoadFuncs' 
call SysLoadFuncs

call DisplayBanner
say 'Checking system:'

/* Determine directory this program is in */
say 'Where am I...'
parse source . . sourcename
sourcedir = filespec('D', sourcename)||filespec('P', sourcename)
sourcedir = left(sourcedir, length(sourcedir)-1)
call directory sourcedir

/* Check for UNZIP.EXE in path */
say 'Checking if UNZIP.EXE is in path...'
rc = SysSearchPath('PATH', 'UNZIP.EXE')
if rc = '' then
	do
	say
	say 'UNZIP.EXE must be in the path for installation to succeed.  You may download'
	say 'it via a link from the Power User column in OS/2 e-Zine!'
        say '(http://www.haligonian.com/os2/) or by ftp from hobbes.nmsu.edu.'
        call ExitProgram
	exit
	end

/* Create Power User folder if it doesn't already exist. */
say 'Preparing Power User folder...'
call SysCreateObject 'WPFolder', 'Power User', '<WP_DESKTOP>', 'OBJECTID='folderid';ICONFILE='sourcedir'\folder.ico;ICONNFILE=1,'sourcedir'\foldern.ico', 'F'

/* Check if project list file exists */
say 'Searching for project descriptions...'
rc = lines(sourcedir'\'projlist)
if rc = 0 then
	do
	say
	say 'The project list file ('projlist') could not be found or is empty.'
	call Corrupted
	call ExitProgram
	exit
	end

/* Load list of projects */
call DisplayBanner
say 'Installation Options'
say '===================='
curproj = 1
missing = 0	/* will change to 1 if can't find a project file */
do until lines(sourcedir'\'projlist) = 0
/* loop until all lines processed */
 	parse value linein(sourcedir'\'projlist) with projfile.curproj'`'projscr.curproj'`'projign.curproj'`'projname.curproj'`'projdesc.curproj
	if projfile.curproj = '' then
		iterate
	if projname.curproj = '' then
		projname.curproj = 'No name'
	if projfile.curproj \= '.' then
		do
		call SysFileTree sourcedir||projfile.curproj, file, 'FO'
		if file.0 = 0 then
			do
			say projname.curproj': Project file "'projfile.curproj'" not found!'
			missing = 1
			iterate
			end
		end
	if projscr.curproj = '' then
		do
		say projname.curproj': No installation script!'
		missing = 1
		iterate
		end
	if projdesc.curproj = '' then
		say projname.curproj
	else
		say projname.curproj': 'projdesc.curproj
	curproj = curproj + 1
	end
projfile.0 = curproj-1
say

if projfile.0 = 0 then
	do
	say 'There are no valid projects in this archive.'
	call Corrupted
	call ExitProgram
	exit
	end

if missing = 1 then
	do
	say 'There are some files missing from this archive.'
	call Corrupted
	say
	say 'Continue installation anyway? (y/N)'
	userkey = SysGetKey('NOECHO')
	if translate(userkey) \= 'Y' then
		call ExitProgram
		exit
	say
	end

/* Install projects */
do curproj=1 to projfile.0
	if projign.curproj \= ignflag then
		do
		say 'Install 'projname.curproj'? (y/N)'
		userkey = SysGetKey('NOECHO')
		if translate(userkey) \= 'Y' then
			iterate
		if InstallProject()=1 then
			/* Mark projects to be ignored */
			do i=1 to projfile.0
				if i=curproj then
					iterate
				if projign.i=projscr.curproj then
					projign.i=ignflag
				end
		end
	end

say 'Installation complete!'
call ExitProgram
call SysOpenObject folderid, 1, 1 
exit


/************/
DisplayBanner: procedure
call SysCls
say 'Power User - Project Installation'
say
return

/************/
Corrupted: procedure
say 'This archive has probably been corrupted.  You may download a new copy from'
say 'the Power User column in OS/2 e-Zine!, available on the World Wide Web'
say 'at http://www.haligonian.com/os2/.'
return

/************/
ExitProgram: procedure
say
say 'Press any key to exit...'
call SysGetKey('NOECHO')
return

/************/
Ephemeral: procedure expose ephtext
/* Displays the line of text in ephtext, then relocates the cursor so the
   next line will overwrite it.  If ephtext is empty, then the entire line
   is cleared and the cursor relocated as above. */

say left(ephtext, 80)
parse value SysCurPos() with row col
call SysCurPos row-2, 0
return

/************/
InstallProject: procedure expose sourcedir curproj projfile.curproj projscr.curproj projname.curproj
/* returns 1 if installation successful, 0 if not */

insttext = 'Installing 'projname.curproj': '
ephtext=insttext; call Ephemeral
if projfile.curproj \= '.' then
	do
	ephtext=''; call Ephemeral
	say insttext'Failed!'
	say '(Installation from zip file not yet supported.)'
	return 0
	end

/* Check if installation script exists */
script = sourcedir'\'projscr.curproj
rc = lines(script)
if rc = 0 then
	do
	ephtext=''; call Ephemeral
	say insttext'Failed!'
	say '(Installation script not found.)'
	return 0
	end

do until lines(script) = 0
/* loop until all lines processed */
 	parse value linein(script) with scrcmd scropt
	if ExecuteCommand()=0 then
		return 0
	end
	
ephtext=''; call Ephemeral
say insttext'Succeeded!'
return 1

/************/
ExecuteCommand: procedure expose sourcedir insttext scrcmd scropt
/* returns 1 if successful, 0 if not */

if translate(scrcmd) = 'COPY' then
	do
	parse var scropt source'`'dest
	ephtext=insttext'Copying 'source' to 'dest; call Ephemeral
	if left(dest, 1) = '<' then
		do
		file1.0 = 1
		file1.1 = dest
		end
	else
		call SysFileTree dest, file1, 'O'
	call SysFileTree source, file2, 'O'
	if file1.0 = 0 then
		do
		ephtext=''; call Ephemeral
		say insttext'Failed!'
		say '(Could not find destination 'dest'.)'
		return 0
		end
	if file2.0 = 0 then
		do
		ephtext=''; call Ephemeral
		say insttext'Failed!'
		say '(Could not find 'source'.)'
		return 0
		end
	rc = SysCopyObject(file2.1, file1.1)
	if rc = 0 then
		do
		ephtext=''; call Ephemeral
		say insttext'Failed!'
		say '(Could not copy 'source' to 'dest'.)'
		return 0
		end
	return 1
	end

if translate(scrcmd) = 'MOVE' then
	do
	parse var scropt source'`'dest
	ephtext=insttext'Moving 'source' to 'dest; call Ephemeral
	if left(dest, 1) = '<' then
		do
		file1.0 = 1
		file1.1 = dest
		end
	else
		call SysFileTree dest, file1, 'O'
	call SysFileTree source, file2, 'O'
	if file1.0 = 0 then
		do
		ephtext=''; call Ephemeral
		say insttext'Failed!'
		say '(Could not find destination 'dest'.)'
		return 0
		end
	if file2.0 = 0 then
		do
		ephtext=''; call Ephemeral
		say insttext'Failed!'
		say '(Could not find 'source'.)'
		return 0
		end
	rc = SysMoveObject(file2.1, file1.1)
	if rc = 0 then
		do
		ephtext=''; call Ephemeral
		say insttext'Failed!'
		say '(Could not move 'source' to 'dest'.)'
		return 0
		end
	return 1
	end

if translate(scrcmd) = 'OBJFOLDER' then
	do
	parse var scropt name'`'location'`'exists_flag'`'setup'`'icon_filename'`'icon_filename2'`'object_id
	ephtext=insttext'Creating folder 'name; call Ephemeral
	if left(location, 1) = '<' then
		do
		file1.0 = 1
		file1.1 = location
		end
	else
		call SysFileTree location, file1, 'O'
	if file1.0 = 0 then
		do
		ephtext=''; call Ephemeral
		say insttext'Failed!'
		say '(Could not find location 'location'.)'
		return 0
		end
	if icon_filename \= '' then
		do
		call SysFileTree icon_filename, file2, 'FO'
		if file2.0 = 0 then
			do
			ephtext=''; call Ephemeral
			say insttext'Failed!'
			say '(Could not find 'icon_filename'.)'
			return 0
			end
		end
	if icon_filename2 \= '' then
		do
		call SysFileTree icon_filename2, file3, 'FO'
		if file3.0 = 0 then
		do
			ephtext=''; call Ephemeral
			say insttext'Failed!'
			say '(Could not find 'icon_filename2'.)'
			return 0
			end
		end
	rc = SysCreateObject('WPFolder', name, file1.1, 'OBJECTID='object_id';ICONFILE='file2.1';ICONNFILE=1,'file3.1';'setup, exists_flag)
	return 1
	end

if translate(scrcmd) = 'OBJPROGRAM' then
	do
	parse var scropt name'`'location'`'exists_flag'`'setup'`'icon_filename'`'program_name'`'parameters'`'working_dir
	ephtext=insttext'Creating program object 'name; call Ephemeral
	if left(location, 1) = '<' then
		do
		file1.0 = 1
		file1.1 = location
		end
	else
		call SysFileTree location, file1, 'O'
	if file1.0 = 0 then
		do
		ephtext=''; call Ephemeral
		say insttext'Failed!'
		say '(Could not find location 'location'.)'
		return 0
		end
	if icon_filename \= '' then
		do
		call SysFileTree icon_filename, file2, 'FO'
		if file2.0 = 0 then
			do
			ephtext=''; call Ephemeral
			say insttext'Failed!'
			say '(Could not find 'icon_filename'.)'
			return 0
			end
		end
	call SysFileTree program_name, file3, 'FO'
	if file3.0 = 0 then
		do
		ephtext=''; call Ephemeral
		say insttext'Failed!'
		say '(Could not find 'program_name'.)'
		return 0
		end
	rc = SysCreateObject('WPProgram', name, file1.1, 'EXENAME='file3.1';PARAMETERS='parameters';STARTUPDIR='working_dir';ICONFILE='file2.1';'setup, exists_flag)
	return 1
	end

if translate(scrcmd) = 'OBJSHADOW' then
	do
	parse var scropt object'`'location
	ephtext=insttext'Creating shadow of 'object; call Ephemeral
	if left(object, 1) = '<' then
		do
		file1.0 = 1
		file1.1 = object
		end
	else
		call SysFileTree object, file1, 'O'
	if file1.0 = 0 then
		do
		ephtext=''; call Ephemeral
		say insttext'Failed!'
		say '(Could not find object 'object'.)'
		return 0
		end
	if left(location, 1) = '<' then
		do
		file2.0 = 1
		file2.1 = location
		end
	else
		call SysFileTree location, file2, 'O'
	if file2.0 = 0 then
		do
		ephtext=''; call Ephemeral
		say insttext'Failed!'
		say '(Could not find location 'location'.)'
		return 0
		end
	rc = SysCreateShadow(file1.1, file2.1)
	return 1
	end
	
if translate(scrcmd) = 'UNZIP' then
	do
	parse var scropt source'`'dest
	ephtext=insttext'Unzipping 'source' to 'dest; call Ephemeral
	'@unzip -o -qq "'source'" -d "'dest'"'
	return 1
	end
	
ephtext=''; call Ephemeral
say insttext'Failed!'
say '(Don''t know how to execute command "'scrcmd'".)'
return 0