/***************************************************************** CONFIG.CPP
 *                                                                          *
 *                        Clock Configuration Dialog                        *
 *                                                                          *
 ****************************************************************************/

#define INCL_BASE
#define INCL_PM
#define INCL_WINSTDSPIN
#define INCL_WINWORKPLACE
#include <os2.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "Debug.h"
#include "Support.h"

#include "Memsize.h"
#include "Config.h"
#include "Profile.h"

// #define DEBUG
 
/****************************************************************************
 *                                                                          *
 *                     Definitions & Declarations                           *
 *                                                                          *
 ****************************************************************************/

  // Function Prototypes

static FNWP InitDlg ;
static FNWP Control ;
static FNWP FormatFrame ;
static FNWP UpdateDrives ;

static FNWP Configure_Items_Processor ;
static FNWP Configure_Options1_Processor ;
static FNWP Configure_Options2_Processor ;
static FNWP Configure_Colors_Processor ;
static FNWP Configure_Anchor_Processor ;

class Slider {

   private:
      HWND Control, Entryfield ;
      ULONG MinLevel, MaxLevel ;
      int Divisor ;
      char *Suffix ;

   public:
      Slider ( HWND control, HWND entryfield, ULONG Level, ULONG minlevel, ULONG maxlevel, int divisor, char *suffix ) : 
         Control(control), Entryfield(entryfield), MinLevel(minlevel), MaxLevel(maxlevel), Divisor(divisor), Suffix(suffix) {

         // Set left label.
         char Text [20] ;
         sprintf ( Text, "%i%s", MinLevel/Divisor, Suffix ) ;
         WinSendMsg ( Control, SLM_SETSCALETEXT, MPFROMSHORT(0), MPFROMP(Text) ) ;

         // Set center label.
         sprintf ( Text, "%i%s", (MinLevel+(MaxLevel-MinLevel)/2)/Divisor, Suffix ) ;
         WinSendMsg ( Control, SLM_SETSCALETEXT, MPFROMSHORT(50), MPFROMP(Text) ) ;

         // Set right label.
         sprintf ( Text, "%i%s", MaxLevel/Divisor, Suffix ) ;
         WinSendMsg ( Control, SLM_SETSCALETEXT, MPFROMSHORT(100), MPFROMP(Text) ) ;

         // Position the slider.
         short Tick = short ( ( double ( Level - MinLevel ) / ( Divisor / 100.0 ) ) / ( double ( MaxLevel - MinLevel ) / Divisor ) ) ;
         WinSendMsg ( Control, SLM_SETSLIDERINFO, MPFROM2SHORT(SMA_SLIDERARMPOSITION,SMA_INCREMENTVALUE), MPFROMSHORT(Tick) ) ;

         // Set the entryfield text.
         sprintf ( Text, "%.1lf%s", double(Level)/Divisor, Suffix ) ;
         WinSetWindowText ( Entryfield, PSZ(Text) ) ;

      } /* endmethod */

      void Set ( ULONG Level, BOOL SetText=TRUE ) {

         // Position the slider.
         short Tick = short ( ( double ( Level - MinLevel ) / ( Divisor / 100.0 ) ) / ( double ( MaxLevel - MinLevel ) / Divisor ) ) ;
         WinSendMsg ( Control, SLM_SETSLIDERINFO, MPFROM2SHORT(SMA_SLIDERARMPOSITION,SMA_INCREMENTVALUE), MPFROMSHORT(Tick) ) ;

         // Set the entryfield text.
         if ( SetText ) {
            char Text [20] ;
            sprintf ( Text, "%.1lf%s", double(Level)/Divisor, Suffix ) ;
            WinSetWindowText ( Entryfield, PSZ(Text) ) ;
         } /* endif */

      } /* endmethod */

      ULONG Read ( ) {

         // Read the slider.
         ULONG Tick = LONGFROMMR ( WinSendMsg ( Control, SLM_QUERYSLIDERINFO, MPFROM2SHORT(SMA_SLIDERARMPOSITION,SMA_INCREMENTVALUE), 0 ) ) ;
         ULONG Level = ULONG ( MinLevel + ( Tick * ( double ( MaxLevel - MinLevel ) / ( Divisor * 100.0 ) ) * Divisor ) ) ;

         // Update the entry field.
         char Text [20] ;
         sprintf ( Text, "%.1lf%s", double(Level)/Divisor, Suffix ) ;
         WinSetWindowText ( Entryfield, PSZ(Text) ) ;

         // Return the result.
         return ( Level ) ;

      } /* endmethod */
} ;

typedef struct {
   char  DefaultLabel [80] ;
   char  CurrentLabel [80] ;
   ULONG DefaultLevels [2] ;
   ULONG WarningLevel ;
   ULONG ErrorLevel ;
   int   LevelSense ;
   ULONG MinLevel ;
   ULONG MaxLevel ;
   USHORT ShowTrueK ;
   Slider *Warning ;
   Slider *Error ;
} LABEL_PARMS, *PLABEL_PARMS ;

static FNWP Label_Processor ;
static FNWP Static_Processor ;
static PFNWP OldStaticProcessor = 0 ;
 
/****************************************************************************
 *                                                                          *
 *      "Configure" Dialog Processor                                        *
 *                                                                          *
 ****************************************************************************/

extern MRESULT EXPENTRY ConfigureProcessor ( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 ) {

  /**************************************************************************
   * Dispatch the message according to the method table and return the      *
   *   result.  Any messages not defined above get handled by the system    *
   *   default dialog processor.                                            *
   **************************************************************************/

   #ifdef DEBUG
      static int Indent = 0 ;
      Log ( "%*sCONFIG: Message %08X received.  Parm1=%08X, Parm2=%08X.", Indent, "", msg, mp1, mp2 ) ;
      Indent += 2 ;
   #endif

   static METHOD Methods [] = {
      { WM_INITDLG,      InitDlg      },
      { WM_CONTROL,      Control      },
      { WM_FORMATFRAME,  FormatFrame  },
      { WM_UPDATEDRIVES, UpdateDrives }
   } ;

   MRESULT Result = DispatchMessage ( hwnd, msg, mp1, mp2, Methods, sizeof(Methods)/sizeof(Methods[0]), WinDefDlgProc ) ;

   #ifdef DEBUG
      Indent -= 2 ;
      Log ( "%*sCONFIG: Message %08X done.  Result %08X.", Indent, "", msg, Result ) ;
   #endif

   return ( Result ) ;
}
 
/****************************************************************************
 *                                                                          *
 *      Initialize Dialog                                                   *
 *                                                                          *
 ****************************************************************************/

static MRESULT APIENTRY InitDlg ( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 ) {

  /**************************************************************************
   * Get initial parameters.                                                *
   **************************************************************************/

   PCONFIG_PARMS Parms = PCONFIG_PARMS ( PVOIDFROMMP ( mp2 ) ) ;

   WinSetWindowPtr ( hwnd, QWL_USER, Parms ) ;

  /**************************************************************************
   * Associate the help instance.                                           *
   **************************************************************************/

   WinAssociateHelpInstance ( WinQueryHelpInstance(OWNER(hwnd)), hwnd ) ;

  /**************************************************************************
   * Mark the dialog not ready so that Control messages are ignored.        *
   **************************************************************************/

   Parms->Ready = FALSE ;

  /**************************************************************************
   * Create the notebook pages.                                             *
   **************************************************************************/

   ResourceString   ItemsPageName ( LibraryHandle, IDS_CONFIG_PAGE_ITEMS ) ;
   ResourceString OptionsPageName ( LibraryHandle, IDS_CONFIG_PAGE_OPTIONS ) ;
   ResourceString  ColorsPageName ( LibraryHandle, IDS_CONFIG_PAGE_COLORS ) ;
   ResourceString  AnchorPageName ( LibraryHandle, IDS_CONFIG_PAGE_ANCHOR ) ;

   HWND hwndNotebook = WinWindowFromID ( hwnd, IDD_CONFIG_NOTEBOOK ) ;

   WinSendMsg ( hwndNotebook, BKM_SETNOTEBOOKCOLORS,
      MPFROMLONG(SYSCLR_FIELDBACKGROUND), MPFROMSHORT(BKA_BACKGROUNDPAGECOLORINDEX) ) ;

   struct {
      char *Name ;              FNWP *Processor ;             int DialogID ;
   } Pages [] = {
      { PCHAR(  ItemsPageName), Configure_Items_Processor,    IDD_CONFIG_PAGE_ITEMS    },
      { PCHAR(OptionsPageName), Configure_Options1_Processor, IDD_CONFIG_PAGE_OPTIONS1 },
      { 0,                      Configure_Options2_Processor, IDD_CONFIG_PAGE_OPTIONS2 },
      { PCHAR( ColorsPageName), Configure_Colors_Processor,   IDD_CONFIG_PAGE_COLORS   },
      { PCHAR( AnchorPageName), Configure_Anchor_Processor,   IDD_CONFIG_PAGE_ANCHOR   },
   } ;

   HPS hPS = WinGetPS ( hwnd ) ;
   RECTL Rectangle ;
   WinQueryWindowRect ( HWND_DESKTOP, &Rectangle ) ;
   char *Characters = "AaBbCcDdEeFfGgHhIiJjKkLlMmNnOoPpQqRrSsTtUuVvWwXxYyZz0123456789" ;
   WinDrawText ( hPS, strlen(Characters), PSZ(Characters), &Rectangle, 0, 0, DT_LEFT | DT_BOTTOM | DT_QUERYEXTENT ) ;
   long AvgWidth  = ( Rectangle.xRight - Rectangle.xLeft ) / strlen(Characters) ;
   long AvgHeight = Rectangle.yTop - Rectangle.yBottom ;

   long MajorTabWidth = 0 ;
   for ( int i=0; i<sizeof(Pages)/sizeof(Pages[0]); i++ ) {

      BOOL MajorTab ( ( i == 0 ) || Pages[i].Name ) ;
      int PageNumber(1), PageCount(1) ;  char *Name(0);
      for ( int j=i; j>=0; j-- ) 
         if ( Pages[j].Name ) 
            break ;
      Name = Pages[j].Name ;
      for ( j=j+1; j<sizeof(Pages)/sizeof(Pages[0]) && !Pages[j].Name; j++ ) {
         PageCount ++ ;
         if ( j <= i ) 
            PageNumber ++ ;
      } /* endfor */

      ULONG PageHandle = ULONG ( WinSendMsg ( hwndNotebook, BKM_INSERTPAGE,
         MPFROMLONG(0), MPFROM2SHORT( (MajorTab?BKA_MAJOR:0) | BKA_STATUSTEXTON, BKA_LAST ) ) ) ;

      char StatusText [80] ;
      if ( PageCount == 1 ) {
         strcpy ( StatusText, Pages[i].Name ) ;
      } else {
         ResourceString Page ( LibraryHandle, IDS_PAGE ) ;
         ResourceString Of ( LibraryHandle, IDS_OF ) ;
         sprintf ( StatusText, "%s, %s %i %s %i", Name, PCHAR(Page), PageNumber, PCHAR(Of), PageCount ) ;
      } /* endif */
      WinSendMsg ( hwndNotebook, BKM_SETSTATUSLINETEXT, MPFROMLONG(PageHandle), MPFROMP(StatusText) ) ;

      if ( MajorTab ) {
         WinSendMsg ( hwndNotebook, BKM_SETTABTEXT, MPFROMLONG(PageHandle), MPFROMP(Pages[i].Name) ) ;
         POINTL TextBox [TXTBOX_COUNT] ;
         GpiQueryTextBox ( hPS, strlen(Pages[i].Name), PCH(Pages[i].Name), TXTBOX_COUNT, TextBox ) ;
         MajorTabWidth = max ( MajorTabWidth, TextBox[TXTBOX_CONCAT].x ) ;
      } /* endif */

      HWND Dialog = WinLoadDlg ( hwndNotebook, hwnd, Pages[i].Processor, LibraryHandle, Pages[i].DialogID, Parms );
      WinSendMsg ( hwndNotebook, BKM_SETPAGEWINDOWHWND, MPFROMP(PageHandle), MPFROMLONG(Dialog) ) ;

      if ( i == 0 )
         WinSendMsg ( WinQueryHelpInstance(hwnd), HM_SET_ACTIVE_WINDOW, MPFROMHWND(Dialog), MPFROMHWND(Dialog) ) ;

   } /* endfor */

   WinSendMsg ( hwndNotebook, BKM_SETDIMENSIONS,
      MPFROM2SHORT( SHORT(MajorTabWidth+AvgWidth*2), SHORT((AvgHeight*3)/2) ),
      MPFROMSHORT( BKA_MAJORTAB ) ) ;

   WinReleasePS ( hPS ) ;

  /**************************************************************************
   * Initialize state.                                                      *
   **************************************************************************/

   Parms->Ready = TRUE ;
   Parms->MostRecentSelection = -1 ;

  /**************************************************************************
   * Return without error.                                                  *
   **************************************************************************/

   return ( 0 ) ;
}
 
/****************************************************************************
 *                                                                          *
 *      Process Control Messages                                            *
 *                                                                          *
 ****************************************************************************/

static MRESULT APIENTRY Control ( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 ) {

  /**************************************************************************
   * Find the instance data.                                                *
   **************************************************************************/

// PCONFIG_PARMS Parms = PCONFIG_PARMS ( Sys_GetWindowData ( hwnd ) ) ;

  /**************************************************************************
   * Decode the message.  Find out what control sent it, and what the       *
   *   control had to say.                                                  *
   **************************************************************************/
 
   SHORT Id = SHORT1FROMMP ( mp1 ) ;
   SHORT Message = SHORT2FROMMP ( mp1 ) ;

  /**************************************************************************
   * Process notification according to which control sent it.               *
   **************************************************************************/

   switch ( Id ) {

      case IDD_CONFIG_NOTEBOOK: {
         switch ( Message ) {
            case BKN_PAGESELECTED: {
               PAGESELECTNOTIFY *pInfo = (PAGESELECTNOTIFY*) PVOIDFROMMP(mp2) ;
               HWND Dialog = HWND ( WinSendMsg ( pInfo->hwndBook, BKM_QUERYPAGEWINDOWHWND, MPFROMLONG(pInfo->ulPageIdNew), 0 ) ) ;
               WinSendMsg ( WinQueryHelpInstance(hwnd), HM_SET_ACTIVE_WINDOW, MPFROMHWND(Dialog), MPFROMHWND(Dialog) ) ;
               break ; } /* endcase */
         } /* endswitch */
         break ; } /* endcase */

   } /* endswitch */

  /**************************************************************************
   * Return no error.                                                       *
   **************************************************************************/

   return ( MRFROMSHORT ( FALSE ) ) ;
}
 
/****************************************************************************
 *                                                                          *
 *      Process 'Format Frame' Message                                      *
 *                                                                          *
 ****************************************************************************/

static MRESULT APIENTRY FormatFrame ( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 ) {

  /**************************************************************************
   * First let the default dialog processor do its stuff.                   *
   **************************************************************************/

   MRESULT Result = WinDefDlgProc ( hwnd, msg, mp1, mp2 ) ;

  /**************************************************************************
   * Set the new notebook size, preserving the space for the controls below.*
   **************************************************************************/

   SWP DialogPosition ;
   WinQueryWindowPos ( hwnd, &DialogPosition ) ;

   SWP NotebookPosition ;
   WinQueryWindowPos ( WinWindowFromID(hwnd,IDD_CONFIG_NOTEBOOK), &NotebookPosition ) ;

   LONG cyBorder = WinQuerySysValue ( HWND_DESKTOP, SV_CYSIZEBORDER ) ;
   LONG cyTitle = WinQuerySysValue ( HWND_DESKTOP, SV_CYTITLEBAR ) ;

   NotebookPosition.cx = DialogPosition.cx - NotebookPosition.x*2 ;
   NotebookPosition.cy = DialogPosition.cy - NotebookPosition.y - cyBorder*2 - cyTitle ;

   WinSetWindowPos ( WinWindowFromID(hwnd,IDD_CONFIG_NOTEBOOK), 
      0, 0, 0, NotebookPosition.cx, NotebookPosition.cy, SWP_SIZE ) ;

  /**************************************************************************
   * Return whatever the default said.                                      *
   **************************************************************************/

   return ( Result ) ;
}
 
/****************************************************************************
 *                                                                          *
 *      Process 'Update Drives' Message                                     *
 *                                                                          *
 ****************************************************************************/

static MRESULT APIENTRY UpdateDrives ( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 ) {

  /**************************************************************************
   * Update the item list.                                                  *
   **************************************************************************/

   HWND Notebook = WinWindowFromID ( hwnd, IDD_CONFIG_NOTEBOOK ) ;
   ULONG ItemsID = LONGFROMMR ( WinSendMsg ( Notebook, BKM_QUERYPAGEID, 0, MPFROM2SHORT(BKA_FIRST,BKA_MAJOR) ) ) ;
   HWND ItemsPage = HWND ( WinSendMsg ( Notebook, BKM_QUERYPAGEWINDOWHWND, MPFROMLONG(ItemsID), 0 ) ) ;

   WinSendMsg ( ItemsPage, msg, mp1, mp2 ) ;

  /**************************************************************************
   * Return success.                                                        *
   **************************************************************************/

   return ( 0 ) ;
}
 
/****************************************************************************
 *                                                                          *
 *  Configuration Items Page message processor                              *
 *                                                                          *
 ****************************************************************************/

static MRESULT EXPENTRY Configure_Items_Processor ( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 ) {

   #ifdef DEBUG
      static int Indent = 0 ;
      Log ( "%*sItemsPage: Message %08X received.  Parm1=%08X, Parm2=%08X.", Indent, "", msg, mp1, mp2 ) ;
      Indent += 2 ;
   #endif

   switch ( msg ) {

      case WM_INITDLG: {
         PCONFIG_PARMS Parms = PCONFIG_PARMS ( PVOIDFROMMP ( mp2 ) ) ;
         WinSetWindowPtr ( hwnd, QWL_USER, Parms ) ;
         WinAssociateHelpInstance ( WinQueryHelpInstance(Parms->MainWindow), hwnd ) ;
         for ( int i=0; i<Parms->ItemCount; i++ ) {
            char Label [164] ;
            strcpy ( Label, Parms->CurrentLabels[i] ) ;
            if ( strcmp ( Parms->DefaultLabels[i], Parms->CurrentLabels[i] ) ) {
               strcat ( Label, " (" ) ;
               strcat ( Label, Parms->DefaultLabels[i] ) ;
               strcat ( Label, ")" ) ;
            } /* endif */
            WinSendDlgItemMsg ( hwnd, IDD_CONFIG_ITEMS, LM_INSERTITEM,
               MPFROMSHORT(LIT_END), MPFROMP(Label) ) ;
            if ( Parms->ItemFlags[i] )
               WinSendDlgItemMsg ( hwnd, IDD_CONFIG_ITEMS, LM_SELECTITEM, MPFROMSHORT(i), MPFROMSHORT(TRUE) ) ;
         } /* endfor */
         break; }

      case WM_CONTROL: {
         PCONFIG_PARMS Parms = PCONFIG_PARMS ( WinQueryWindowPtr ( hwnd, QWL_USER ) ) ;
         SHORT Id = SHORT1FROMMP ( mp1 ) ;
         SHORT Message = SHORT2FROMMP ( mp1 ) ;
         switch ( Id ) {
            case IDD_CONFIG_ITEMS: {
               switch ( Message ) {
                  case LN_SELECT: {
                     if ( NOT Parms->Ready )
                        break ;
                     SHORT Selection = LIT_FIRST ;
                     SHORT Last = -1 ;
                     do {
                        Selection = short ( SHORT1FROMMR ( WinSendDlgItemMsg ( hwnd,
                           IDD_CONFIG_ITEMS, LM_QUERYSELECTION,
                           MPFROMSHORT(SHORT(Selection)), 0 ) ) ) ;
                        if ( Selection != LIT_NONE ) {
                           for ( Last++; Last<Selection; Last++ ) {
                              if ( Parms->ItemFlags[Last] ) {
                                 Parms->ItemFlags[Last] = FALSE ;
                                 Parms->MostRecentSelection = Last ;
                              } /* endif */
                           } /* endfor */
                           if ( Parms->ItemFlags[Selection] == FALSE ) {
                              Parms->ItemFlags[Selection] = TRUE ;
                              Parms->MostRecentSelection = Selection ;
                           } /* endif */
                        } else {
                           for ( Last++; Last<Parms->ItemCount; Last++ ) {
                              if ( Parms->ItemFlags[Last] ) {
                                 Parms->ItemFlags[Last] = FALSE ;
                                 Parms->MostRecentSelection = Last ;
                              } /* endif */
                           } /* endfor */
                        } /* endif */
                     } while ( Selection != LIT_NONE ) ;
                     WinSendMsg ( Parms->MainWindow, WM_UPDATEPARMS, MPFROMP(Parms), 0 ) ;
                     break ;
                  } /* endcase */
                  case LN_ENTER: {
                     if ( Parms->MostRecentSelection == -1 ) 
                        break ;
                     LABEL_PARMS LabelParms ;
                     strcpy ( LabelParms.DefaultLabel, Parms->DefaultLabels[Parms->MostRecentSelection] ) ;
                     strcpy ( LabelParms.CurrentLabel, Parms->CurrentLabels[Parms->MostRecentSelection] ) ;
                     LabelParms.DefaultLevels [0] = Parms->DefaultLevels[Parms->MostRecentSelection] [0] ;
                     LabelParms.DefaultLevels [1] = Parms->DefaultLevels[Parms->MostRecentSelection] [1] ;
                     LabelParms.WarningLevel = Parms->WarningLevel[Parms->MostRecentSelection] ;
                     LabelParms.ErrorLevel = Parms->ErrorLevel[Parms->MostRecentSelection] ;
                     LabelParms.LevelSense = Parms->LevelSense[Parms->MostRecentSelection] ;
                     LabelParms.MinLevel = Parms->MinLevel[Parms->MostRecentSelection] ;
                     LabelParms.MaxLevel = Parms->MaxLevel[Parms->MostRecentSelection] ;
                     LabelParms.ShowTrueK = Parms->ShowTrueK ;
                     LabelParms.Warning = 0 ;
                     LabelParms.Error = 0 ;
                     if ( WinDlgBox ( HWND_DESKTOP, hwnd, Label_Processor, LibraryHandle, IDD_LABEL, &LabelParms ) ) {
                        strcpy ( Parms->CurrentLabels[Parms->MostRecentSelection], LabelParms.CurrentLabel ) ;
                        char Label [164] ;
                        strcpy ( Label, Parms->CurrentLabels[Parms->MostRecentSelection] ) ;
                        if ( strcmp ( Parms->DefaultLabels[Parms->MostRecentSelection], Parms->CurrentLabels[Parms->MostRecentSelection] ) ) {
                           strcat ( Label, " (" ) ;
                           strcat ( Label, Parms->DefaultLabels[Parms->MostRecentSelection] ) ;
                           strcat ( Label, ")" ) ;
                        } /* endif */
                        Parms->WarningLevel[Parms->MostRecentSelection] = LabelParms.WarningLevel ;
                        Parms->ErrorLevel[Parms->MostRecentSelection] = LabelParms.ErrorLevel ;
                        WinSendDlgItemMsg ( hwnd, IDD_CONFIG_ITEMS, LM_DELETEITEM,
                           MPFROMSHORT(Parms->MostRecentSelection), 0 ) ;
                        WinSendDlgItemMsg ( hwnd, IDD_CONFIG_ITEMS, LM_INSERTITEM,
                           MPFROMSHORT(Parms->MostRecentSelection), MPFROMP(Label) ) ;
                        WinSendDlgItemMsg ( hwnd, IDD_CONFIG_ITEMS, LM_SELECTITEM,
                           MPFROMSHORT(Parms->MostRecentSelection), MPFROMSHORT(TRUE) ) ;
                        WinSendMsg ( Parms->MainWindow, WM_UPDATEPARMS, MPFROMP(Parms), 0 ) ;
                     } /* endif */
                     break ;
                  } /* endcase */
               } /* endswitch */
               break; }
         } /* endswitch */
         break; }

      case WM_COMMAND: {
         #ifdef DEBUG
            Indent -= 2 ;
            Log ( "%*sItemsPage: Message %08X done.  Result %08X.", Indent, "", msg, 0 ) ;
         #endif
         return ( 0 ) ; }

      case WM_UPDATEDRIVES: {
         PCONFIG_PARMS Parms = PCONFIG_PARMS ( WinQueryWindowPtr ( hwnd, QWL_USER ) ) ;
         Parms->Ready = FALSE ;
         PINIDATA IniData = PINIDATA ( PVOIDFROMMP ( mp1 ) ) ;
         WinSendDlgItemMsg ( hwnd, IDD_CONFIG_ITEMS, LM_DELETEALL, 0, 0 ) ;
         Parms->ItemCount = IniData->ItemCount ;
         for ( int i=0; i<IniData->ItemCount; i++ ) {
            Item *pItem = IniData->Items[i] ;
            Parms->ItemFlags[i] = pItem->QueryFlag () ;
            strcpy ( Parms->CurrentLabels[i], PCHAR(pItem->QueryCurrentLabel()) ) ;
            strcpy ( Parms->DefaultLabels[i], PCHAR(pItem->QueryDefaultLabel()) ) ;
            pItem->QueryDefaultLevels ( Parms->DefaultLevels[i][0], Parms->DefaultLevels[i][1] ) ;
            Parms->WarningLevel[i] = pItem->QueryWarningLevel() ;
            Parms->ErrorLevel[i] = pItem->QueryErrorLevel() ;
            Parms->LevelSense[i] = pItem->QueryLevelSense() ;
            Parms->MinLevel[i] = pItem->QueryMinLevel() ;
            Parms->MaxLevel[i] = pItem->QueryMaxLevel() ;
         } /* endfor */
         for ( i=0; i<Parms->ItemCount; i++ ) {
            char Label [164] ;
            strcpy ( Label, Parms->CurrentLabels[i] ) ;
            if ( strcmp ( Parms->DefaultLabels[i], Parms->CurrentLabels[i] ) ) {
               strcat ( Label, " (" ) ;
               strcat ( Label, Parms->DefaultLabels[i] ) ;
               strcat ( Label, ")" ) ;
            } /* endif */
            WinSendDlgItemMsg ( hwnd, IDD_CONFIG_ITEMS, LM_INSERTITEM,
               MPFROMSHORT(LIT_END), MPFROMP(Label) ) ;
            if ( Parms->ItemFlags[i] )
               WinSendDlgItemMsg ( hwnd, IDD_CONFIG_ITEMS, LM_SELECTITEM, MPFROMSHORT(i), MPFROMSHORT(TRUE) ) ;
         } /* endfor */
         Parms->Ready = TRUE ;
         #ifdef DEBUG
            Indent -= 2 ;
            Log ( "%*sItemsPage: Message %08X done.  Result %08X.", Indent, "", msg, 0 ) ;
         #endif
         return ( 0 ) ; }

   } /* endswitch */

   MRESULT Result = WinDefDlgProc ( hwnd, msg, mp1, mp2 ) ;

   #ifdef DEBUG
      Indent -= 2 ;
      Log ( "%*sItemsPage: Message %08X done.  Result %08X.", Indent, "", msg, Result ) ;
   #endif

   return ( Result ) ;
}

/****************************************************************************
 *                                                                          *
 *  Configuration Options Page 1 message processor                          *
 *                                                                          *
 ****************************************************************************/

static MRESULT EXPENTRY Configure_Options1_Processor ( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 ) {

   #ifdef DEBUG
      static int Indent = 0 ;
      Log ( "%*sOptions1Page: Message %08X received.  Parm1=%08X, Parm2=%08X.", Indent, "", msg, mp1, mp2 ) ;
      Indent += 2 ;
   #endif

   switch ( msg ) {

      case WM_INITDLG: {
         PCONFIG_PARMS Parms = PCONFIG_PARMS ( PVOIDFROMMP ( mp2 ) ) ;
         WinSetWindowPtr ( hwnd, QWL_USER, Parms ) ;
         WinAssociateHelpInstance ( WinQueryHelpInstance(Parms->MainWindow), hwnd ) ;
         WinSendDlgItemMsg ( hwnd, IDD_CONFIG_HIDECONTROLS,
            BM_SETCHECK, MPFROMSHORT(Parms->HideControls), 0 ) ;
         WinSendDlgItemMsg ( hwnd, IDD_CONFIG_FLOAT,
            BM_SETCHECK, MPFROMSHORT(Parms->Float), 0 ) ;
         WinSendDlgItemMsg ( hwnd, IDD_CONFIG_ANIMATE,
            BM_SETCHECK, MPFROMSHORT(Parms->Animate), 0 ) ;
         WinSendDlgItemMsg ( hwnd, IDD_CONFIG_TABLE,
            BM_SETCHECK, MPFROMSHORT(Parms->TableFormat), 0 ) ;
         WinSendDlgItemMsg ( hwnd, IDD_CONFIG_CHIME,
            BM_SETCHECK, MPFROMSHORT(Parms->Chime), 0 ) ;
         WinSendDlgItemMsg ( hwnd, IDD_CONFIG_SHOWREMOTES,
            BM_SETCHECK, MPFROMSHORT(Parms->ShowRemoteDrives), 0 ) ;
         WinSendDlgItemMsg ( hwnd, IDD_CONFIG_FSNAME,
            BM_SETCHECK, MPFROMSHORT(Parms->ShowFileSystemNames), 0 ) ;
         WinSendDlgItemMsg ( hwnd, IDD_CONFIG_DLABEL,
            BM_SETCHECK, MPFROMSHORT(Parms->ShowDiskLabels), 0 ) ;
         WinSendDlgItemMsg ( hwnd, IDD_CONFIG_SECONDS,
            BM_SETCHECK, MPFROMSHORT(Parms->ShowSeconds), 0 ) ;
         WinSendDlgItemMsg ( hwnd, IDD_CONFIG_HOUR24,
            BM_SETCHECK, MPFROMSHORT(Parms->Hour24), 0 ) ;
         break; }

      case WM_CONTROL: {
         PCONFIG_PARMS Parms = PCONFIG_PARMS ( WinQueryWindowPtr ( hwnd, QWL_USER ) ) ;
         SHORT Id = SHORT1FROMMP ( mp1 ) ;
         SHORT Message = SHORT2FROMMP ( mp1 ) ;
         switch ( Id ) {
            case IDD_CONFIG_HIDECONTROLS: {
               if ( Message == BN_CLICKED ) {
                  Parms->HideControls = (BOOL) SHORT1FROMMR ( WinSendDlgItemMsg ( hwnd, Id, BM_QUERYCHECK, 0, 0 ) ) ;
                  WinSendMsg ( Parms->MainWindow, WM_UPDATEPARMS, MPFROMP(Parms), 0 ) ;
               } /* endif */
               break; }
            case IDD_CONFIG_FLOAT: {
               if ( Message == BN_CLICKED ) {
                  Parms->Float = (BOOL) SHORT1FROMMR ( WinSendDlgItemMsg ( hwnd, Id, BM_QUERYCHECK, 0, 0 ) ) ;
                  WinSendMsg ( Parms->MainWindow, WM_UPDATEPARMS, MPFROMP(Parms), 0 ) ;
               } /* endif */
               break; }
            case IDD_CONFIG_ANIMATE: {
               if ( Message == BN_CLICKED ) {
                  Parms->Animate = (BOOL) SHORT1FROMMR ( WinSendDlgItemMsg ( hwnd, Id, BM_QUERYCHECK, 0, 0 ) ) ;
                  WinSendMsg ( Parms->MainWindow, WM_UPDATEPARMS, MPFROMP(Parms), 0 ) ;
               } /* endif */
               break; }
            case IDD_CONFIG_TABLE: {
               if ( Message == BN_CLICKED ) {
                  Parms->TableFormat = (BOOL) SHORT1FROMMR ( WinSendDlgItemMsg ( hwnd, Id, BM_QUERYCHECK, 0, 0 ) ) ;
                  WinSendMsg ( Parms->MainWindow, WM_UPDATEPARMS, MPFROMP(Parms), 0 ) ;
               } /* endif */
               break; }
            case IDD_CONFIG_CHIME: {
               if ( Message == BN_CLICKED ) {
                  Parms->Chime = (BOOL) SHORT1FROMMR ( WinSendDlgItemMsg ( hwnd, Id, BM_QUERYCHECK, 0, 0 ) ) ;
                  WinSendMsg ( Parms->MainWindow, WM_UPDATEPARMS, MPFROMP(Parms), 0 ) ;
               } /* endif */
               break; }
            case IDD_CONFIG_SHOWREMOTES: {
               if ( Message == BN_CLICKED ) {
                  Parms->ShowRemoteDrives = (BOOL) SHORT1FROMMR ( WinSendDlgItemMsg ( hwnd, Id, BM_QUERYCHECK, 0, 0 ) ) ;
                  WinSendMsg ( Parms->MainWindow, WM_UPDATEPARMS, MPFROMP(Parms), 0 ) ;
               } /* endif */
               break; }
            case IDD_CONFIG_FSNAME: {
               if ( Message == BN_CLICKED ) {
                  Parms->ShowFileSystemNames = (BOOL) SHORT1FROMMR ( WinSendDlgItemMsg ( hwnd, Id, BM_QUERYCHECK, 0, 0 ) ) ;
                  WinSendMsg ( Parms->MainWindow, WM_UPDATEPARMS, MPFROMP(Parms), 0 ) ;
               } /* endif */
               break; }
            case IDD_CONFIG_DLABEL: {
               if ( Message == BN_CLICKED ) {
                  Parms->ShowDiskLabels = (BOOL) SHORT1FROMMR ( WinSendDlgItemMsg ( hwnd,
                     IDD_CONFIG_DLABEL, BM_QUERYCHECK, 0, 0 ) ) ;
                  WinSendMsg ( Parms->MainWindow, WM_UPDATEPARMS, MPFROMP(Parms), 0 ) ;
               } /* endif */
               break; }
            case IDD_CONFIG_SECONDS: {
               if ( Message == BN_CLICKED ) {
                  Parms->ShowSeconds = (BOOL) SHORT1FROMMR ( WinSendDlgItemMsg ( hwnd, Id, BM_QUERYCHECK, 0, 0 ) ) ;
                  WinSendMsg ( Parms->MainWindow, WM_UPDATEPARMS, MPFROMP(Parms), 0 ) ;
               } /* endif */
               break; }
            case IDD_CONFIG_HOUR24: {
               if ( Message == BN_CLICKED ) {
                  Parms->Hour24 = (BOOL) SHORT1FROMMR ( WinSendDlgItemMsg ( hwnd, Id, BM_QUERYCHECK, 0, 0 ) ) ;
                  WinSendMsg ( Parms->MainWindow, WM_UPDATEPARMS, MPFROMP(Parms), 0 ) ;
               } /* endif */
               break; }
         } /* endswitch */
         break; }

      case WM_COMMAND: {
         #ifdef DEBUG
            Indent -= 2 ;
            Log ( "%*sOptions1Page: Message %08X done.  Result %08X.", Indent, "", msg, 0 ) ;
         #endif
         return ( 0 ) ; }

   } /* endswitch */

   MRESULT Result = WinDefDlgProc ( hwnd, msg, mp1, mp2 ) ;

   #ifdef DEBUG
      Indent -= 2 ;
      Log ( "%*sOptions1Page: Message %08X done.  Result %08X.", Indent, "", msg, Result ) ;
   #endif

   return ( Result ) ;
}

/****************************************************************************
 *                                                                          *
 *  Configuration Options Page 2 message processor                          *
 *                                                                          *
 ****************************************************************************/

static MRESULT EXPENTRY Configure_Options2_Processor ( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 ) {

   #ifdef DEBUG
      static int Indent = 0 ;
      Log ( "%*sOptions2Page: Message %08X received.  Parm1=%08X, Parm2=%08X.", Indent, "", msg, mp1, mp2 ) ;
      Indent += 2 ;
   #endif

   switch ( msg ) {

      case WM_INITDLG: {
         PCONFIG_PARMS Parms = PCONFIG_PARMS ( PVOIDFROMMP ( mp2 ) ) ;
         WinSetWindowPtr ( hwnd, QWL_USER, Parms ) ;
         WinAssociateHelpInstance ( WinQueryHelpInstance(Parms->MainWindow), hwnd ) ;
         switch ( Parms->ShowK ) {
            case SHOWK_NEVER:
               WinSendDlgItemMsg ( hwnd, IDD_CONFIG_SHOWK_NEVER, BM_SETCHECK, MPFROMSHORT(TRUE), 0 ) ;
               break;
            case SHOWK_ALWAYS:
               WinSendDlgItemMsg ( hwnd, IDD_CONFIG_SHOWK_ALWAYS, BM_SETCHECK, MPFROMSHORT(TRUE), 0 ) ;
               break;
            case SHOWK_ABOVE512:
            default:
               WinSendDlgItemMsg ( hwnd, IDD_CONFIG_SHOWK_ABOVE512, BM_SETCHECK, MPFROMSHORT(TRUE), 0 ) ;
               break;
         } /* endswitch */
         WinSendDlgItemMsg ( hwnd, IDD_CONFIG_SHOW_TRUEK,
            BM_SETCHECK, MPFROMSHORT(Parms->ShowTrueK), 0 ) ;
         WinSendDlgItemMsg ( hwnd, IDD_CONFIG_SHOW_MEGABYTES,
            BM_SETCHECK, MPFROMSHORT(Parms->ShowM), 0 ) ;
         WinSendDlgItemMsg ( hwnd, IDD_CONFIG_PRIORITY,
            SPBM_SETLIMITS, (MPARAM)PRTYD_MAXIMUM, (MPARAM)0 ) ;
         WinSendDlgItemMsg ( hwnd, IDD_CONFIG_PRIORITY,
            SPBM_SETCURRENTVALUE, (MPARAM)(Parms->MonitorPriority), NULL ) ;
         WinSendDlgItemMsg ( hwnd, IDD_CONFIG_TIMER,
            SPBM_SETLIMITS, (MPARAM)300, (MPARAM)10 ) ;
         WinSendDlgItemMsg ( hwnd, IDD_CONFIG_TIMER,
            SPBM_SETCURRENTVALUE, (MPARAM)(Parms->TimerInterval/100), NULL ) ;
         break; }

      case WM_CONTROL: {
         PCONFIG_PARMS Parms = PCONFIG_PARMS ( WinQueryWindowPtr ( hwnd, QWL_USER ) ) ;
         SHORT Id = SHORT1FROMMP ( mp1 ) ;
         SHORT Message = SHORT2FROMMP ( mp1 ) ;
         switch ( Id ) {
            case IDD_CONFIG_SHOWK_NEVER: {
               if ( Message == BN_CLICKED ) {
                  Parms->ShowK = SHOWK_NEVER ;
                  WinSendMsg ( Parms->MainWindow, WM_UPDATEPARMS, MPFROMP(Parms), 0 ) ;
               } /* endif */
               break; }
            case IDD_CONFIG_SHOWK_ABOVE512: {
               if ( Message == BN_CLICKED ) {
                  Parms->ShowK = SHOWK_ABOVE512 ;
                  WinSendMsg ( Parms->MainWindow, WM_UPDATEPARMS, MPFROMP(Parms), 0 ) ;
               } /* endif */
               break; }
            case IDD_CONFIG_SHOWK_ALWAYS: {
               if ( Message == BN_CLICKED ) {
                  Parms->ShowK = SHOWK_ALWAYS ;
                  WinSendMsg ( Parms->MainWindow, WM_UPDATEPARMS, MPFROMP(Parms), 0 ) ;
               } /* endif */
               break; }
            case IDD_CONFIG_SHOW_TRUEK: {
               if ( Message == BN_CLICKED ) {
                  Parms->ShowTrueK = (USHORT) SHORT1FROMMR ( WinSendDlgItemMsg ( hwnd, Id, BM_QUERYCHECK, 0, 0 ) ) ;
                  WinSendMsg ( Parms->MainWindow, WM_UPDATEPARMS, MPFROMP(Parms), 0 ) ;
               } /* endif */
               break; }
            case IDD_CONFIG_SHOW_MEGABYTES: {
               if ( Message == BN_CLICKED ) {
                  Parms->ShowM = (USHORT) SHORT1FROMMR ( WinSendDlgItemMsg ( hwnd, Id, BM_QUERYCHECK, 0, 0 ) ) ;
                  WinSendMsg ( Parms->MainWindow, WM_UPDATEPARMS, MPFROMP(Parms), 0 ) ;
               } /* endif */
               break; }
            case IDD_CONFIG_PRIORITY: {
               if ( Message == SPBN_CHANGE ) {
                  WinSendDlgItemMsg ( hwnd, Id, SPBM_QUERYVALUE, &Parms->MonitorPriority, MPFROM2SHORT(0,SPBQ_UPDATEIFVALID) ) ;
                  WinSendMsg ( Parms->MainWindow, WM_UPDATEPARMS, MPFROMP(Parms), 0 ) ;
               } /* endif */
               break; }
            case IDD_CONFIG_TIMER: {
               if ( Message == SPBN_CHANGE ) {
                  WinSendDlgItemMsg ( hwnd, Id, SPBM_QUERYVALUE, &Parms->TimerInterval, MPFROM2SHORT(0,SPBQ_UPDATEIFVALID) ) ;
                  Parms->TimerInterval *= 100 ;
                  WinSendMsg ( Parms->MainWindow, WM_UPDATEPARMS, MPFROMP(Parms), 0 ) ;
               } /* endif */
               break; }
         } /* endswitch */
         break; }

      case WM_COMMAND: {
         #ifdef DEBUG
            Indent -= 2 ;
            Log ( "%*sOptions2Page: Message %08X done.  Result %08X.", Indent, "", msg, 0 ) ;
         #endif
         return ( 0 ) ; }

   } /* endswitch */

   MRESULT Result = WinDefDlgProc ( hwnd, msg, mp1, mp2 ) ;

   #ifdef DEBUG
      Indent -= 2 ;
      Log ( "%*sOptions2Page: Message %08X done.  Result %08X.", Indent, "", msg, Result ) ;
   #endif

   return ( Result ) ;
}

/****************************************************************************
 *                                                                          *
 *  Configuration Colors Page message processor                             *
 *                                                                          *
 ****************************************************************************/

static MRESULT EXPENTRY Configure_Colors_Processor ( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 ) {

   #ifdef DEBUG
      static int Indent = 0 ;
      Log ( "%*sColorsPage: Message %08X received.  Parm1=%08X, Parm2=%08X.", Indent, "", msg, mp1, mp2 ) ;
      Indent += 2 ;
   #endif

   switch ( msg ) {

      case WM_INITDLG: {
         PCONFIG_PARMS Parms = PCONFIG_PARMS ( PVOIDFROMMP ( mp2 ) ) ;
         WinSetWindowPtr ( hwnd, QWL_USER, Parms ) ;
         WinAssociateHelpInstance ( WinQueryHelpInstance(Parms->MainWindow), hwnd ) ;
         WinSetPresParam ( WinWindowFromID(hwnd,IDD_CONFIG_NORMAL), 
            PP_BACKGROUNDCOLOR, sizeof(Parms->NormalBackground), MPFROMP(&Parms->NormalBackground) ) ;
         WinSetPresParam ( WinWindowFromID(hwnd,IDD_CONFIG_NORMAL), 
            PP_FOREGROUNDCOLOR, sizeof(Parms->NormalForeground), MPFROMP(&Parms->NormalForeground) ) ;
         OldStaticProcessor = WinSubclassWindow ( WinWindowFromID(hwnd,IDD_CONFIG_NORMAL), Static_Processor ) ;
         WinSetPresParam ( WinWindowFromID(hwnd,IDD_CONFIG_WARNING), 
            PP_BACKGROUNDCOLOR, sizeof(Parms->WarningBackground), MPFROMP(&Parms->WarningBackground) ) ;
         WinSetPresParam ( WinWindowFromID(hwnd,IDD_CONFIG_WARNING), 
            PP_FOREGROUNDCOLOR, sizeof(Parms->WarningForeground), MPFROMP(&Parms->WarningForeground) ) ;
         WinSubclassWindow ( WinWindowFromID(hwnd,IDD_CONFIG_WARNING), Static_Processor ) ;
         WinSetPresParam ( WinWindowFromID(hwnd,IDD_CONFIG_ERROR), 
            PP_BACKGROUNDCOLOR, sizeof(Parms->ErrorBackground), MPFROMP(&Parms->ErrorBackground) ) ;
         WinSetPresParam ( WinWindowFromID(hwnd,IDD_CONFIG_ERROR), 
            PP_FOREGROUNDCOLOR, sizeof(Parms->ErrorForeground), MPFROMP(&Parms->ErrorForeground) ) ;
         WinSubclassWindow ( WinWindowFromID(hwnd,IDD_CONFIG_ERROR), Static_Processor ) ;
         break; }

      case WM_CONTROL: {
         PCONFIG_PARMS Parms = PCONFIG_PARMS ( WinQueryWindowPtr ( hwnd, QWL_USER ) ) ;
         SHORT Id = SHORT1FROMMP ( mp1 ) ;
         switch ( Id ) {
            case IDD_CONFIG_NORMAL: {
               ULONG ppid ;
               WinQueryPresParam ( WinWindowFromID(hwnd,IDD_CONFIG_NORMAL), PP_BACKGROUNDCOLOR, 0, &ppid,
                  sizeof(Parms->NormalBackground), &Parms->NormalBackground, 0 ) ;
               WinQueryPresParam ( WinWindowFromID(hwnd,IDD_CONFIG_NORMAL), PP_FOREGROUNDCOLOR, 0, &ppid,
                  sizeof(Parms->NormalForeground), &Parms->NormalForeground, 0 ) ;
               WinSendMsg ( Parms->MainWindow, WM_UPDATEPARMS, MPFROMP(Parms), 0 ) ;
               break; }
            case IDD_CONFIG_WARNING: {
               ULONG ppid ;
               WinQueryPresParam ( WinWindowFromID(hwnd,IDD_CONFIG_WARNING), PP_BACKGROUNDCOLOR, 0, &ppid,
                  sizeof(Parms->NormalBackground), &Parms->WarningBackground, 0 ) ;
               WinQueryPresParam ( WinWindowFromID(hwnd,IDD_CONFIG_WARNING), PP_FOREGROUNDCOLOR, 0, &ppid,
                  sizeof(Parms->NormalForeground), &Parms->WarningForeground, 0 ) ;
               WinSendMsg ( Parms->MainWindow, WM_UPDATEPARMS, MPFROMP(Parms), 0 ) ;
               break; }
            case IDD_CONFIG_ERROR: {
               ULONG ppid ;
               WinQueryPresParam ( WinWindowFromID(hwnd,IDD_CONFIG_ERROR), PP_BACKGROUNDCOLOR, 0, &ppid,
                  sizeof(Parms->NormalBackground), &Parms->ErrorBackground, 0 ) ;
               WinQueryPresParam ( WinWindowFromID(hwnd,IDD_CONFIG_ERROR), PP_FOREGROUNDCOLOR, 0, &ppid,
                  sizeof(Parms->NormalForeground), &Parms->ErrorForeground, 0 ) ;
               WinSendMsg ( Parms->MainWindow, WM_UPDATEPARMS, MPFROMP(Parms), 0 ) ;
               break; }
         } /* endswitch */
         break; }

      case WM_COMMAND: {
         #ifdef DEBUG
            Indent -= 2 ;
            Log ( "%*sColorsPage: Message %08X done.  Result %08X.", Indent, "", msg, 0 ) ;
         #endif
         return ( 0 ) ; }

   } /* endswitch */

   MRESULT Result = WinDefDlgProc ( hwnd, msg, mp1, mp2 ) ;

   #ifdef DEBUG
      Indent -= 2 ;
      Log ( "%*sColorsPage: Message %08X done.  Result %08X.", Indent, "", msg, Result ) ;
   #endif

   return ( Result ) ;
}

/****************************************************************************
 *                                                                          *
 *  Configuration Anchor Page message processor                             *
 *                                                                          *
 ****************************************************************************/

static MRESULT EXPENTRY Configure_Anchor_Processor ( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 ) {

   #ifdef DEBUG
      static int Indent = 0 ;
      Log ( "%*sAnchorPage: Message %08X received.  Parm1=%08X, Parm2=%08X.", Indent, "", msg, mp1, mp2 ) ;
      Indent += 2 ;
   #endif

   switch ( msg ) {

      case WM_INITDLG: {
         PCONFIG_PARMS Parms = PCONFIG_PARMS ( PVOIDFROMMP ( mp2 ) ) ;
         WinSetWindowPtr ( hwnd, QWL_USER, Parms ) ;
         WinAssociateHelpInstance ( WinQueryHelpInstance(Parms->MainWindow), hwnd ) ;
         switch ( Parms->AnchorCorner ) {
            case CORNER_BL:
            default:
               WinSendDlgItemMsg ( hwnd, IDD_CONFIG_BL, BM_SETCHECK, MPFROMSHORT(TRUE), 0 ) ;
               break;
            case CORNER_BR:
               WinSendDlgItemMsg ( hwnd, IDD_CONFIG_BR, BM_SETCHECK, MPFROMSHORT(TRUE), 0 ) ;
               break;
            case CORNER_TL:
               WinSendDlgItemMsg ( hwnd, IDD_CONFIG_TL, BM_SETCHECK, MPFROMSHORT(TRUE), 0 ) ;
               break;
            case CORNER_TR:
               WinSendDlgItemMsg ( hwnd, IDD_CONFIG_TR, BM_SETCHECK, MPFROMSHORT(TRUE), 0 ) ;
               break;
         } /* endswitch */
         break; }

      case WM_CONTROL: {
         PCONFIG_PARMS Parms = PCONFIG_PARMS ( WinQueryWindowPtr ( hwnd, QWL_USER ) ) ;
         SHORT Id = SHORT1FROMMP ( mp1 ) ;
         SHORT Message = SHORT2FROMMP ( mp1 ) ;
         switch ( Id ) {
            case IDD_CONFIG_BL: {
               if ( Message == BN_CLICKED ) {
                  Parms->AnchorCorner = CORNER_BL ;
                  WinSendMsg ( Parms->MainWindow, WM_UPDATEPARMS, MPFROMP(Parms), 0 ) ;
               } /* endif */
               break ; } /* endcase */
            case IDD_CONFIG_BR: {
               if ( Message == BN_CLICKED ) {
                  Parms->AnchorCorner = CORNER_BR ;
                  WinSendMsg ( Parms->MainWindow, WM_UPDATEPARMS, MPFROMP(Parms), 0 ) ;
               } /* endif */
               break ; } /* endcase */
            case IDD_CONFIG_TL: {
               if ( Message == BN_CLICKED ) {
                  Parms->AnchorCorner = CORNER_TL ;
                  WinSendMsg ( Parms->MainWindow, WM_UPDATEPARMS, MPFROMP(Parms), 0 ) ;
               } /* endif */
               break ; } /* endcase */
            case IDD_CONFIG_TR: {
               if ( Message == BN_CLICKED ) {
                  Parms->AnchorCorner = CORNER_TR ;
                  WinSendMsg ( Parms->MainWindow, WM_UPDATEPARMS, MPFROMP(Parms), 0 ) ;
               } /* endif */
               break ; } /* endcase */
         } /* endswitch */
         break; }

      case WM_COMMAND: {
         #ifdef DEBUG
            Indent -= 2 ;
            Log ( "%*sAnchorPage: Message %08X done.  Result %08X.", Indent, "", msg, 0 ) ;
         #endif
         return ( 0 ) ; }

   } /* endswitch */

   MRESULT Result = WinDefDlgProc ( hwnd, msg, mp1, mp2 ) ;

   #ifdef DEBUG
      Indent -= 2 ;
      Log ( "%*sAnchorPage: Message %08X done.  Result %08X.", Indent, "", msg, Result ) ;
   #endif

   return ( Result ) ;
}

/****************************************************************************
 *                                                                          *
 *      Item Label dialog processor.                                        *
 *                                                                          *
 ****************************************************************************/

inline void GetDivisor ( ULONG MaxLevel, USHORT ShowTrueK, int &Divisor, char* &Suffix ) {

   Divisor = 1 ;  
   Suffix = "" ;

   if ( ShowTrueK ) {
      if ( MaxLevel > 1024*1024 ) {
         Divisor = 1024*1024 ;
         Suffix = "M" ;
      } else if ( MaxLevel > 1024 ) {
         Divisor = 1024 ;
         Suffix = "K" ;
      } /* endif */

   } else {
      if ( MaxLevel > 1000*1000 ) {
         Divisor = 1000*1000 ;
         Suffix = "m" ;
      } else if ( MaxLevel > 1000 ) {
         Divisor = 1000 ;
         Suffix = "k" ;
      } /* endif */

   } /* endif */
}

inline char *TrimRight ( char *Text ) {
   while ( strlen(Text) && Text[strlen(Text)-1] == ' ' ) Text[strlen(Text)-1] = 0 ;
   return ( Text ) ;
}

static MRESULT APIENTRY Label_Processor ( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 ) {

   switch ( msg ) {

      case WM_INITDLG: {
         PLABEL_PARMS Parms = PLABEL_PARMS ( PVOIDFROMMP ( mp2 ) ) ;
         WinSetWindowPtr ( hwnd, QWL_USER, Parms ) ;
         WinAssociateHelpInstance ( WinQueryHelpInstance(OWNER(hwnd)), hwnd ) ;
         WinSetDlgItemText ( hwnd, IDD_LABEL_DEFLABEL, PSZ(Parms->DefaultLabel) ) ;
         WinSetDlgItemText ( hwnd, IDD_LABEL_NEWLABEL, PSZ(Parms->CurrentLabel) ) ;
         WinSendMsg ( WinWindowFromID(hwnd,IDD_LABEL_NEWLABEL), EM_SETTEXTLIMIT, MPFROM2SHORT(sizeof(Parms->CurrentLabel),0), 0 ) ;
         if ( Parms->LevelSense ) {
            int Divisor ;  char *Suffix ;
            GetDivisor ( Parms->MaxLevel, Parms->ShowTrueK, Divisor, Suffix ) ;
            Parms->Warning = new Slider ( WinWindowFromID(hwnd,IDD_LABEL_WARNING_SLIDER), WinWindowFromID(hwnd,IDD_LABEL_WARNING_VALUE),
               Parms->WarningLevel, Parms->MinLevel, Parms->MaxLevel, Divisor, Suffix ) ;
            Parms->Error = new Slider ( WinWindowFromID(hwnd,IDD_LABEL_ERROR_SLIDER), WinWindowFromID(hwnd,IDD_LABEL_ERROR_VALUE),
               Parms->ErrorLevel, Parms->MinLevel, Parms->MaxLevel, Divisor, Suffix ) ;
         } else {
            WinEnableWindow ( WinWindowFromID(hwnd,IDD_LABEL_WARNING_SLIDER), FALSE ) ;
            WinEnableWindow ( WinWindowFromID(hwnd,IDD_LABEL_WARNING_VALUE), FALSE ) ;
            WinEnableWindow ( WinWindowFromID(hwnd,IDD_LABEL_ERROR_SLIDER), FALSE ) ;
            WinEnableWindow ( WinWindowFromID(hwnd,IDD_LABEL_ERROR_VALUE), FALSE ) ;
         } /* endif */
         return ( 0 ) ; }

      case WM_DESTROY: {
         PLABEL_PARMS Parms = PLABEL_PARMS ( WinQueryWindowPtr ( hwnd, QWL_USER ) ) ;
         if ( Parms->Warning ) delete Parms->Warning, Parms->Warning = 0 ;
         if ( Parms->Error ) delete Parms->Error, Parms->Error = 0 ;
         return ( 0 ) ; }

      case WM_CONTROL: {
         PLABEL_PARMS Parms = PLABEL_PARMS ( WinQueryWindowPtr ( hwnd, QWL_USER ) ) ;
         SHORT Id = SHORT1FROMMP ( mp1 ) ;
         SHORT Message = SHORT2FROMMP ( mp1 ) ;
         int Divisor ;  char *Suffix ;
         GetDivisor ( Parms->MaxLevel, Parms->ShowTrueK, Divisor, Suffix ) ;
         switch ( Id ) {
            case IDD_LABEL_WARNING_SLIDER: {
               if ( WinQueryFocus(HWND_DESKTOP) == WinWindowFromID(hwnd,IDD_LABEL_WARNING_SLIDER) ) {
                  if ( ( Message == SLN_CHANGE ) || ( Message == SLN_SLIDERTRACK ) ) {
                     if ( Parms->Warning ) 
                        Parms->WarningLevel = Parms->Warning->Read ( ) ;
                  } /* endif */
               } /* endif */
               break; }
            case IDD_LABEL_ERROR_SLIDER: {
               if ( WinQueryFocus(HWND_DESKTOP) == WinWindowFromID(hwnd,IDD_LABEL_ERROR_SLIDER) ) {
                  if ( ( Message == SLN_CHANGE ) || ( Message == SLN_SLIDERTRACK ) ) {
                     if ( Parms->Error ) 
                        Parms->ErrorLevel = Parms->Error->Read ( ) ;
                  } /* endif */
               } /* endif */
               break; }
            case IDD_LABEL_WARNING_VALUE: {
               if ( WinQueryFocus(HWND_DESKTOP) == WinWindowFromID(hwnd,IDD_LABEL_WARNING_VALUE) ) {
                  if ( Message == EN_CHANGE ) {
                     char Text [20] ;
                     WinQueryDlgItemText ( hwnd, IDD_LABEL_WARNING_VALUE, sizeof(Text), PSZ(Text) ) ;
                     TrimRight ( Text ) ;
                     if ( Text[strlen(Text)-1] == 'M' ) {
                        Parms->WarningLevel = ULONG ( atof ( Text ) * 1024 * 1024 ) ;
                     } else if ( Text[strlen(Text)-1] == 'K' ) {
                        Parms->WarningLevel = ULONG ( atof ( Text ) * 1024 ) ;
                     } else if ( Text[strlen(Text)-1] == 'm' ) {
                        Parms->WarningLevel = ULONG ( atof ( Text ) * 1000 * 1000 ) ;
                     } else if ( Text[strlen(Text)-1] == 'k' ) {
                        Parms->WarningLevel = ULONG ( atof ( Text ) * 1000 ) ;
                     } else {
                        Parms->WarningLevel = atol ( Text ) ;
                     } /* endif */
                     if ( Parms->Warning ) 
                        Parms->Warning->Set ( Parms->WarningLevel, FALSE ) ;
                  } /* endif */
               } /* endif */
               break; }
            case IDD_LABEL_ERROR_VALUE: {
               if ( WinQueryFocus(HWND_DESKTOP) == WinWindowFromID(hwnd,IDD_LABEL_ERROR_VALUE) ) {
                  if ( Message == EN_CHANGE ) {
                     char Text [20] ;
                     WinQueryDlgItemText ( hwnd, IDD_LABEL_ERROR_VALUE, sizeof(Text), PSZ(Text) ) ;
                     TrimRight ( Text ) ;
                     if ( Text[strlen(Text)-1] == 'M' ) {
                        Parms->ErrorLevel = ULONG ( atof ( Text ) * 1024 * 1024 ) ;
                     } else if ( Text[strlen(Text)-1] == 'K' ) {
                        Parms->ErrorLevel = ULONG ( atof ( Text ) * 1024 ) ;
                     } else if ( Text[strlen(Text)-1] == 'm' ) {
                        Parms->ErrorLevel = ULONG ( atof ( Text ) * 1000 * 1000 ) ;
                     } else if ( Text[strlen(Text)-1] == 'k' ) {
                        Parms->ErrorLevel = ULONG ( atof ( Text ) * 1000 ) ;
                     } else {
                        Parms->ErrorLevel = atol ( Text ) ;
                     } /* endif */
                     if ( Parms->Error ) 
                        Parms->Error->Set ( Parms->ErrorLevel, FALSE ) ;
                  } /* endif */
               } /* endif */
               break; }
         } /* endswitch */
         return ( 0 ) ; }

      case WM_COMMAND: {
         PLABEL_PARMS Parms = PLABEL_PARMS ( WinQueryWindowPtr ( hwnd, QWL_USER ) ) ;
         switch ( SHORT1FROMMP(mp1) ) {
            case DID_OK: {
               WinQueryDlgItemText ( hwnd, IDD_LABEL_NEWLABEL, sizeof(Parms->CurrentLabel), PSZ(Parms->CurrentLabel) ) ;
               if ( Parms->CurrentLabel[0] == 0 ) {
                  WinAlarm ( HWND_DESKTOP, WA_NOTE ) ;
                  WinSetFocus ( HWND_DESKTOP, WinWindowFromID ( hwnd, IDD_LABEL_NEWLABEL ) ) ;
                  break ;
               } /* endif */
               char Text [20] ;
               WinQueryDlgItemText ( hwnd, IDD_LABEL_WARNING_VALUE, sizeof(Text), PSZ(Text) ) ;
               TrimRight ( Text ) ;
               if ( Text[strlen(Text)-1] == 'M' ) {
                  Parms->WarningLevel = ULONG ( atof ( Text ) * 1024 * 1024 ) ;
               } else if ( Text[strlen(Text)-1] == 'K' ) {
                  Parms->WarningLevel = ULONG ( atof ( Text ) * 1024 ) ;
               } else if ( Text[strlen(Text)-1] == 'm' ) {
                  Parms->WarningLevel = ULONG ( atof ( Text ) * 1000 * 1000 ) ;
               } else if ( Text[strlen(Text)-1] == 'k' ) {
                  Parms->WarningLevel = ULONG ( atof ( Text ) * 1000 ) ;
               } else {
                  Parms->WarningLevel = atol ( Text ) ;
               } /* endif */
               WinQueryDlgItemText ( hwnd, IDD_LABEL_ERROR_VALUE, sizeof(Text), PSZ(Text) ) ;
               TrimRight ( Text ) ;
               if ( Text[strlen(Text)-1] == 'M' ) {
                  Parms->ErrorLevel = ULONG ( atof ( Text ) * 1024 * 1024 ) ;
               } else if ( Text[strlen(Text)-1] == 'K' ) {
                  Parms->ErrorLevel = ULONG ( atof ( Text ) * 1024 ) ;
               } else if ( Text[strlen(Text)-1] == 'm' ) {
                  Parms->ErrorLevel = ULONG ( atof ( Text ) * 1000 * 1000 ) ;
               } else if ( Text[strlen(Text)-1] == 'k' ) {
                  Parms->ErrorLevel = ULONG ( atof ( Text ) * 1000 ) ;
               } else {
                  Parms->ErrorLevel = atol ( Text ) ;
               } /* endif */
               WinDismissDlg ( hwnd, TRUE ) ;
               break ; }
            case DID_CANCEL:
               WinDismissDlg ( hwnd, FALSE ) ;
               break ;
            case IDD_LABEL_DEFAULT:
               strcpy ( Parms->CurrentLabel, Parms->DefaultLabel ) ;
               WinSetDlgItemText ( hwnd, IDD_LABEL_NEWLABEL, PSZ(Parms->CurrentLabel) ) ;
               Parms->WarningLevel = Parms->DefaultLevels [0] ;
               Parms->ErrorLevel = Parms->DefaultLevels [1] ;
               if ( Parms->LevelSense ) {
                  int Divisor ;  char *Suffix ;
                  GetDivisor ( Parms->MaxLevel, Parms->ShowTrueK, Divisor, Suffix ) ;
                  if ( Parms->Warning ) 
                     Parms->Warning->Set ( Parms->WarningLevel ) ;
                  if ( Parms->Error ) 
                     Parms->Error->Set ( Parms->ErrorLevel ) ;
               } /* endif */
               WinSetFocus ( HWND_DESKTOP, WinWindowFromID ( hwnd, IDD_LABEL_NEWLABEL ) ) ;
               break ;
         } /* endswitch */
         return ( 0 ) ; }

   } /* endswitch */

   return ( WinDefDlgProc ( hwnd, msg, mp1, mp2 ) ) ;
}

/****************************************************************************
 *                                                                          *
 *      Subclasses STATIC control message processor                         *
 *                                                                          *
 ****************************************************************************/

typedef struct _STATICDATA {
   HMODULE hPMWP ;
   HOBJECT ( APIENTRY *pWinQueryObject ) ( PCSZ pszObjectID ) ;
   BOOL ( APIENTRY *pWinOpenObject ) ( HOBJECT hObject, ULONG ulView, BOOL Flag ) ;
} STATICDATA, *PSTATICDATA ;

static MRESULT APIENTRY Static_Processor ( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 ) {

   switch ( msg ) {

      case WM_INITDLG: 
      case WM_CREATE: {
         PSTATICDATA Data = PSTATICDATA ( malloc ( sizeof(STATICDATA) ) ) ;
         memset ( Data, 0, sizeof(STATICDATA) ) ;
         WinSetWindowPtr ( hwnd, QWL_USER, Data ) ;
         if ( DosLoadModule ( 0, 0, "PMWP", &Data->hPMWP ) ) {
            Data->hPMWP = 0;
            return ( 0 ) ;
         } /* endif */
         if ( DosQueryProcAddr ( Data->hPMWP, 252, 0, (PFN*) & Data->pWinQueryObject ) ) {
            DosFreeModule ( Data->hPMWP ) ;
            Data->hPMWP = 0;
            return ( 0 ) ;
         } /* endif */
         if ( DosQueryProcAddr ( Data->hPMWP, 286, 0, (PFN*) & Data->pWinOpenObject ) ) {
            DosFreeModule ( Data->hPMWP ) ;
            Data->pWinQueryObject = 0 ;
            Data->hPMWP = 0;
            return ( 0 ) ;
         } /* endif */
         return ( 0 ) ; }

      case WM_DESTROY: {
         PSTATICDATA Data = PSTATICDATA ( WinQueryWindowPtr ( hwnd, QWL_USER ) ) ;
         if ( Data && Data->hPMWP ) 
            DosFreeModule ( Data->hPMWP ) ;
         if ( Data ) free ( Data ) ;
         break; }

      case WM_SINGLESELECT:
      case WM_CONTEXTMENU: {
         PSTATICDATA Data = PSTATICDATA ( WinQueryWindowPtr ( hwnd, QWL_USER ) ) ;
         if ( !Data ) {
            WinSendMsg ( hwnd, WM_CREATE, 0, 0 ) ;
            Data = PSTATICDATA ( WinQueryWindowPtr ( hwnd, QWL_USER ) ) ;
         } /* endif */
         if ( Data && Data->pWinQueryObject && Data->pWinOpenObject ) {
            HOBJECT hObject = Data->pWinQueryObject ( "<WP_LORESCLRPAL>" ) ;
            if ( hObject ) {
               if ( !Data->pWinOpenObject ( hObject, 0, TRUE ) ) {
                  Data->pWinOpenObject ( hObject, 0, FALSE ) ;
               } /* endif */
            } /* endif */
         } /* endif */
         break ; }

      case WM_PRESPARAMCHANGED: {
         USHORT Id = WinQueryWindowUShort ( hwnd, QWS_ID ) ;
         switch ( LONGFROMMP(mp1) ) {
            case PP_BACKGROUNDCOLOR: {
               WinSendMsg ( OWNER(hwnd), WM_CONTROL, MPFROM2SHORT(Id,0), 0 ) ;
               break; }
            case PP_FOREGROUNDCOLOR: {
               WinSendMsg ( OWNER(hwnd), WM_CONTROL, MPFROM2SHORT(Id,0), 0 ) ;
               break; }
         } /* endswitch */
         break ; }

   } /* endswitch */

   return ( OldStaticProcessor ( hwnd, msg, mp1, mp2 ) ) ;
}


