{-------------------------- Dedication ---------------------------------

    Dedicated to our daughter Brianna Grieve stillborn 6th July 1996

 -----------------------------------------------------------------------}

 { TxyGraph version 2.20
  see TxyGraph.hlp for full documentation including
                  - current known buglist
                  - how to use TxyGraph
                  - my snail mail address for the postcard.}

 { copyright  Grahame Grieve, Kim Kirkpatrick }


{ definitions:}
{ win32 - delphi16 or delphi32.}

{ the first 2 defines are supplied to reduce code size. I'm sorry about the
  code bloat - but we all want the functionality.
  Where code sizes are specified, they are based on the effect on executable
  size for a D1 program containing 1 instance of TxyGraph using default
  settings. Base code size is 78kB}
{ STATISTICS - included to eliminate the statistics routines.
  remember that you should also set it in xydata and xyeditor as well.
  the stats code size is 19kB }
{ DESIGNER - include / exclude the code for the designer.
  you should always have this defined when you compile the library.
  you can turn it off if your programs don't use it.
  the designer code size is 143kB}

{ BACKWARD - backward compatibility: obsolete TxyGraph properties are now
  moved to Dimensions, Appearance, Axes.  Removes obsolete methods also.
  Experimentally, this doesn't save you any code size}

{$DEFINE BACKWARD }
{$DEFINE STATISTICS }
{$DEFINE DESIGNER }

{--------------- implementation section heading labels ---------------

    #1a. TSeries - administration
    #1b. TSeries - property servers
    #1c. TSeries - drawing routines
    #2a. TAxis  - scaling routines
    #2b. TAxis - property servers
    #3.  TDimensions
    #4.  TAppearance
    #5a. TxyGraph  - administration
    #5b. TxyGraph  - mouse routines
    #5c. TxyGraph  - metrics routines
    #5d. TxyGraph  - painting routines
    #5e. TxyGraph  - property servers
    #5f. TxyGraph  - series routines
    #5g. TxyGraph  - data manipulation
    #5h. TxyGraph  - Marks routines
    #5i. TxyGraph  - debug routine
    #6.  TxyGraph  - register routine
 ----------------------------------------------------------------------}

unit xyGraph;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics,
  Controls, Forms, Dialogs, ExtCtrls, DsgnIntf, Printers,
  xyData;

const
  versionstamp = '2.20';
  defMinYScale = 0.0;
  defMinXScale = 0.0;
  drawBefore = false;
  drawAfter = true;
  AllowedLogTickCounts = [0,1,2,5,7,9];
    {InitLogTicks and AdjustLogTickCount must correspond.
     AdjustLogLabelDecs must correspond with the result of InitLogTicks.}
  MaxLogTickCount = 9; {Must equal largest value in AllowedLogTickCounts}

  MouseToleranceMove = 4; { pixels mouse must move before it counts }
  MouseToleranceHit = 5; { how far from target mouse is allowed to be }

type
  et_PointShape = (ps_Square, ps_Circle, ps_Diamond, ps_Cross, ps_Wide);
  et_BoundType = (bt_None, bt_AsSet, bt_1SD, bt_2SD, bt_3SD, bt_allSD);
  et_MarkType = (mtXMark,mtYMark,mtPoint,mtLine);

  pMark = ^Tmark;
  tMark = record
          id:integer;
          x1,y1,x2,y2:double;
          color:Tcolor;
          caption:ShortString;
          next:pmark;
          marktype:et_Marktype;
          status:boolean;
          end;

  PlotFunction = function(x: Double; var parms): Double;

  Dateticks = (dt_minute, dt_5minute, dt_halfhourly, dt_hourly,
         dt_daily, dt_weekly, dt_monthly, dt_2monthly, dt_quarterly,
         dt_annually, dt_decade, dt_century, dt_custom);

  TLogTickInfo = record
         LogTickArray: array[0..MaxLogTickCount+1] of Double;
         LogTickCount: Word;
         LogTickIndex: Word;
         LogStepping: Boolean; {signals linear steps}
         DrawMinorLabels: Boolean;
         end;

  TSeriesEvent =  procedure (sender:TObject; index:integer) of object;
  TPaintEvent = procedure (sender:TObject; Canvas:TCanvas) of object;

  TxyGraph = class;
  TAxis = class;

{------------------------ TSeries --------------------------------*}
  TSeries = class (TDataSeries)
  private
    next: Tseries;
    MyGraph: TxyGraph;
    FWhichYAxis: TAxis;
    FActive, FDrawLine, FDrawPoints, FFillPoints,
    FHoldUpdates, UpdatesWaiting: Boolean;
    FLineColor, FPointColor, FBoundsColor: TColor;
    FPointShape: et_PointShape;
    FLineStyle,FBoundsLineStyle: TPenStyle;
    FSeriesIndex:longint;
    FPointSize: Word;
    FUpperBound,FLowerBound:Double;
    FBoundsType: et_BoundType;
   {$IFDEF STATISTICS}
    FRegrColor: TColor;
    FRegrLineStyle: TPenStyle;
   {$ENDIF}

    procedure setHoldUpdates(v: Boolean);
    procedure setActive(v: Boolean);
    procedure setDrawLine(v: Boolean);
    procedure SetDrawPoints(v: Boolean);
    procedure setFillPoints(v: Boolean);
    procedure SetLineColor(v: TColor);
    procedure SetPointColor(v: TColor);
    procedure SetBoundsColor(v: TColor);
    procedure SetPointShape(v: et_pointshape);
    procedure setLinestyle(v: TPenStyle);
    procedure setPointSize(v: Word);
    procedure SetWhichYAxis(v: TAxis);
    procedure SetUpperBound(v:double);
    procedure SetLowerBound(v:Double);
    procedure setBoundsType(v:et_BoundType);
    procedure SetBoundsLineStyle(v: TPenstyle);

   {$IFDEF STATISTICS}
    procedure SetRegrColor(v: TColor);
    procedure SetRegrLineStyle(v: TPenstyle);
   {$ENDIF}
  protected
    procedure RequestPaint(TheMessage:TDSChangeType); override;
    procedure WarnTooManyPoints; override;
    procedure DrawMyLine;
    procedure DrawMyPoints;
    procedure DrawBounds;
   {$IFDEF STATISTICS}
    procedure DrawMyRegression;
   {$ENDIF}
  public
    constructor Create(aGraph:TxyGraph; i:integer);
    destructor Destroy;  override;
    procedure Draw;

    property SeriesIndex: Longint read FSeriesIndex;
    property NextSeries: TSeries read Next;

    property HoldUpdates: Boolean read FHoldUpdates write setHoldUpdates;
    property Active: Boolean read FActive write setActive;
    Property DrawLine: Boolean read FDrawLine write setDrawLine;
    property DrawPoints: Boolean read FDrawPoints write SetDrawPoints;
    property FillPoints: Boolean read FFillPoints write setFillPoints;
    property LineColor: TColor read FLineColor write SetLineColor;
    property PointColor: TColor read FPointColor write SetPointColor;
    property BoundsColor: TColor read FBoundsColor write SetBoundsColor;
    property PointShape: et_PointShape read FPointShape write SetPointShape;
    property LineStyle: TPenStyle read FLineStyle write setLinestyle;
    property PointSize: Word read FPointSize write setPointsize;
    property WhichYAxis: TAxis read FWhichYAxis write SetWhichYAxis;
    property UpperBound:double read FUpperBound write SetUpperBound;
    property LowerBound:double read FLowerBound write SetLowerBound;
    property BoundsType: et_BoundType read FBoundstype write SetBoundstype;
    property BoundsLineStyle: TPenStyle read FBoundsLineStyle write SetBoundsLineStyle;
   {$IFDEF STATISTICS}
    property RegressionLineColor: TColor read FRegrColor write SetRegrColor;
    property RegressionLineStyle: TPenStyle read FRegrLineStyle write SetRegrLineStyle;
   {$ENDIF}
  end {class TSeries};

  THookType = (ht_PlotData {$IFDEF STATISTICS}, ht_PlotResiduals {$ENDIF});
  pHookedSeries = ^THookedSeries;
  THookedSeries = record
      id:TSeries;
      hooktype:THookType;
      Series:integer;
      next: pHookedSeries;
    end;


{------------ TAxis, TDimensions, TAppearance ----------------}

  TAxisOffsetType = (ao_Minimum, ao_Maximum, ao_percentage, ao_absolute);

  TAxis = class(TPersistent)
  private
    FGraph: TxyGraph;
    SecondAxis: boolean;
    FM: Double;
    oMin, oMax: Double;
    FMin, FMax, FStep: Double;
    FAutoSizing: Boolean;
    FAutoStepping: Boolean;
    FLogging: Boolean;
    FLogTickInfo: TLogTickInfo;
    FAxisTitle: ShortString;
    FLabelDec: Integer;
    FMinScale, FOffset: Double;
    FShowAsTime, FShowAxis, FReversed, FGridlines: Boolean;
    FDateFormat: string;
    FDateTickType: dateticks;
    FOffsetType:TAxisOffsetType;
    procedure SetLabelDec(v: Integer);
    function  GetLogTickCount: Word;
    procedure SetLogTickCount(v: Word);
    procedure SetLogging(v: Boolean);
    procedure SetAutoSizing(v: Boolean);
    procedure SetAutoStepping(v: Boolean);
    procedure SetMax(v: Double);
    procedure SetMin(v: Double);
    procedure SetStep(v: Double);
    procedure SetMinDiff(v: Double);
    procedure SetAxisTitle(v: ShortString);
    procedure setShowAsTime(v: Boolean);
    procedure setDateFormat(v: string);
    procedure setDateTickType(v:dateticks);
    procedure setshowAxis(v:boolean);
    procedure setreversed(v:boolean);
    procedure SetOffsetType(v:TAxisOffsetType);
    procedure setoffset(v:double);
    procedure SetGridLines(v: Boolean);
  protected
    procedure InitLogTicks;
    procedure AdjustLogTickCount;
    procedure CheckDrawMinorLabels;
    function  GetStep: Double;
    function  getDateStep: double;
    procedure SetMinMax;
    procedure SetDateMinMax;
    procedure SetLogMinMax;
    function  DoResize: Boolean;
    function  GetFirstTick(var logTickInfo: TLogTickInfo): Double;
    function  GetFirstDateTick:double;
    function  GetNextTick(tick: Double; var logTickInfo: TLogTickInfo;
                          var drawThisLabel: Boolean): Double;
    function  GetNextDateTick(tick: Double):double;
    procedure AdjustLabelDecs;
    procedure AdjustLogLabelDecs(v: Double);
    function  LabelString(tick: Double): ShortString;
  public
    function  CheckScale: Boolean;
  published
    property Title: ShortString read FAxisTitle write SetAxisTitle;
    property LabelDecimals: Integer read FLabelDec write SetLabelDec default 1;
    property LogCycleDivisions: Word read GetLogTickCount write SetLogTickCount default 2;
    property LogScale: Boolean read FLogging write SetLogging;
    property Max: Double read FMax write SetMax;
    property Min: Double read FMin write SetMin;
    property StepSize: Double read FStep write SetStep;
    property MinScaleLength: Double read FMinScale write SetMinDiff;
    property ShowAsTime: Boolean read FShowASTime write setShowAsTime default false;
    property DateTimeFormat: string read FDateFormat write SetDateFormat;
    property DateTickType: dateticks read FDateTickType write setDateTickType;
    property ShowAxis: boolean read FShowAxis write SetShowAxis;
    property Reversed: boolean read FReversed write setreversed;
    property OffsetType:TAxisOffsetType read FOffsettype write SetOffsetType;
    property Offset:double read FOffset write setoffset;
    property Gridlines: boolean read FGridlines write SetGridlines;

   { these 2 properties must come last to override the other properties'
      effects on the values at load time: }
    property AutoSizing: Boolean read FAutoSizing write SetAutoSizing;
    property AutoStepping: Boolean read FAutoStepping write SetAutoStepping;
  end {class TAxis};

  TDimensions = class(TPersistent)
  private
    FGraph: TxyGraph;
    FLeft, FRight, FTop, FBottom: Word;
    FTMLength: Word;
    FXAxisTitleDistance: Integer;
    FYAxisTitleDistance: Integer;
    FXAxisLabelDistance: Integer;
    FYAxisLabelDistance: Integer;
    FGraphTitleDistance: Integer;
    FScalePct: Word;
    FXOffsetPct, FYOffsetPct: Word;
    procedure SetMargBottom(v: Word);
    procedure SetMargTop(v: Word);
    procedure SetMargLeft(v: Word);
    procedure SetMargRight(v: Word);
    procedure SetTickLength(v: Word);
    procedure SetXAxisTitleDistance(v: Integer);
    procedure SetYAxisTitleDistance(v: Integer);
    procedure SetXAxisLabelDistance(v: Integer);
    procedure SetYAxisLabelDistance(v: Integer);
    procedure SetGraphTitleDistance(v: Integer);
    procedure SetScale(v: Word);
    procedure SetXOffset(v: Word);
    procedure SetYOffset(v: Word);
  published
    property BottomMargin: Word read FBottom write SetMargBottom default 40;
    property LeftMargin: Word read FLeft write SetMargLeft default 40;
    property RightMargin: Word read FRight write SetMargRight default 15;
    property TopMargin: Word read FTop write SetMargTop default 30;
    property TickLength: Word read FTMLength write SetTickLength default 4;
    property XAxisTitleOffset: Integer read FXAxisTitleDistance write SetXAxisTitleDistance default 4;
    property XAxisLabelOffset: Integer read FXAxisLabelDistance write SetXAxisLabelDistance default 2;
    property YAxisTitleOffset: Integer read FYAxisTitleDistance write SetYAxisTitleDistance default 4;
    property YAxisLabelOffset: Integer read FYAxisLabelDistance write SetYAxisLabelDistance default 2;
    property GraphTitleOffset: Integer read FGraphTitleDistance write SetGraphTitleDistance default 7;
   {set print offsets as integer percent (<=100) of page width, height}
    property PrintXOffsetPct: Word read FXOffsetPct write SetXOffset default 5;
    property PrintYOffsetPct: Word read FYOffsetPct write SetYOffset default 20;
   {set print scale as integer percent (<=100) of full page}
    property PrintScalePct: Word read FScalePct write SetScale default 90;
  end {TDimensions};

  TAppearance = class(TPersistent)
  private
    FGraph: TxyGraph;
    FTickMarks, FShowMarks: Boolean;
    FAllowedOut: Boolean;
    FLabelGraph: Boolean;
    FGridColor, FAxesColor, FBkgdCOlor: TColor;
    FGridStyle: TPenStyle;
    FBkgdWhenPrint:Boolean;
    FCaption: ShortString;
    FErrorCaption: ShortString;
    FGraphTitle: ShortString;
    FMinSteps,FMaxSteps: Word;
    FTitleFont, FLabelFont: TFont;
    procedure SetAxesColor(v: TColor);
    procedure SetBkgdColor(v: TColor);
    procedure SetErrorCaption(v: ShortString);
    procedure SetGridColor(v: TColor);
    procedure SetGridStyle(v: TPenStyle);
    procedure SetLabelGraph(v: Boolean);
    procedure SetAllowedOut(v: Boolean);
    procedure SetShowMarks(v: Boolean);
    procedure SetTickMarks(v: Boolean);
    procedure SetGraphTitle(v: ShortString);
    procedure SetMinSteps(v: Word);
    procedure SetMaxSteps(v: Word);
    procedure setTitleFont(v:TFont);
    procedure setLabelFont(v:TFont);
  published
    property AxesColor: TColor read FAxesColor write SetAxescolor default clBlack;
    property BackgroundColor: TColor read FBkgdColor write SetBkgdColor;
    property PrintBkgndColor: Boolean read FBkgdWhenPrint write FBkgdWhenPrint;
    property ErrorCaption: ShortString read FErrorCaption write SetErrorCaption;
    property GridColor: TColor read FGridColor write SetGridcolor default clSilver;
    property GridStyle: TPenStyle read FGridStyle write SetGridStyle default psDot;
    property ShowGraphLabels: Boolean read FLabelGraph write SetLabelGraph default True;
    property PlotOffGraph: Boolean read FAllowedOut write SetAllowedOut default False;
    property ShowMarks: Boolean read FShowMarks write SetShowMarks default false;
    property ShowTicks: Boolean read FTickMarks write SetTickmarks default True;
    property GraphTitle: ShortString read FGraphTitle write SetGraphTitle;
    property MinSteps: Word read FMinSteps write SetMinSteps default 5;
    property MaxSteps: Word read FMaxSteps write SetMaxSteps default 50;
    property TitleFont: TFont read FTitleFont write setTitleFont;
    property LabelFont: TFont read FLabelFont write setLabelFont;
  end {TAppearance};

 (* TLegendWin = class (TCustomPanel)
   private
   protected
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
   public
   published
  end;
                           }
  TLegend = class{ (TPersistent)} (TCustomPanel)
   private
{     FGraph: TxyGraph;
     FLegWin:TLegendWin;
     procedure setcolor(v:TColor);
     Function  GetColor:Tcolor;
     function  getvisible:boolean;
     procedure setvisible(v:boolean);}
   protected
     procedure MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
  public
   published
 {    property Color:Tcolor read getColor write SetColor;
     property Visible:boolean read getvisible write setvisible;}
     property color;
     property visible;
     property top;
     property left;
     property width;
     property height;
     property font;
  end;  *)

{-------------------------- TxyGraph ------------------------------*}

  TxyGraph = class(TCustomPanel)
  private
    { Private declarations }
    FCanvas: TCanvas;
    FSeries: TSeries;
    FDimensions: TDimensions;
    FAppearance: TAppearance;
    FXAxis, FYAxis, FYAxis2: TAxis;
{    FLegend:TLegend;}
{    FLegWin:TLegendWin;}
    FLastSeries:integer;
    FMarks: pMark;
    FFirstTime, FNoGridlines: Boolean;
    FHookedSeries: pHookedSeries;

  {Graph property data:}
    FPlotting: Boolean;
    FAllowDuplicates: Boolean;
    FScale: Double;
    FXOffset, FYOffset: Word;


   {dragging: }
    dRect: tRect;
    dx, dy: Integer;
    FDragging, FHasDragged: Boolean;
    FDraggable: Boolean;
    FHoldXAS, FHoldYAS, FHoldY2AS: boolean;
    FHoldXmin, FHoldXmax, FHoldYmin, FHoldYmax, FHoldY2min, FHoldY2max: double;

   {Events:}
    FOnNewSeries, FOnLogError, FOnTooManyPoints: TSeriesEvent;
    FOnRescale, FOnScaleError: TNotifyEvent;
    FOnPaintEnd: TPaintEvent;

    procedure SetPlotting(v: Boolean);
    procedure SetAllowDuplicates(v: Boolean);
    procedure SetHasDragged(v:boolean);

   {Legacy property methods:}
   {$IFDEF BACKWARD}
    procedure SetXAutosizing(v: Boolean);
    procedure SetAutoStepping(v: Boolean);
    procedure SetXLogging(v: Boolean);
    procedure SetXMin(v: Double);
    procedure SetXMax(v: Double);
    procedure SetXStep(v: Double);
    procedure SetMinXScale(v: Double);
    procedure SetXAxisTitle(v: ShortString);
    procedure SetXLabelDec(v: Integer);
    function  GetXAutosizing: Boolean;
    function  GetAutoStepping: Boolean;
    function  GetXLogging: Boolean;
    function  GetXMin: Double;
    function  GetXMax: Double;
    function  GetXStep: Double;
    function  GetMinXScale: Double;
    function  GetXAxisTitle: ShortString;
    function  GetXLabelDec: Integer;
    procedure SetYAutosizing(v: Boolean);
    procedure SetYLogging(v: Boolean);
    procedure SetYMin(v: Double);
    procedure SetYMax(v: Double);
    procedure SetYStep(v: Double);
    procedure SetMinYScale(v: Double);
    procedure SetYAxisTitle(v: ShortString);
    procedure SetYLabelDec(v: Integer);
    function  GetYAutosizing: Boolean;
    function  GetYLogging: Boolean;
    function  GetYMin: Double;
    function  GetYMax: Double;
    function  GetYStep: Double;
    function  GetMinYScale: Double;
    function  GetYAxisTitle: ShortString;
    function  GetYLabelDec: Integer;
    procedure SetMargLeft(v: Word);
    procedure SetMargRight(v: Word);
    procedure SetMargTop(v: Word);
    procedure SetMargBottom(v: Word);
    function  GetMargLeft: Word;
    function  GetMargRight: Word;
    function  GetMargTop: Word;
    function  GetMargBottom: Word;
    procedure SetTMLength(v: Word);
    function  GetTMLength: Word;
    procedure SetXAxisLabelDistance(v: Integer);
    function  GetXAxisLabelDistance: Integer;
    procedure SetXAxisTitleDistance(v: Integer);
    function  GetXAxisTitleDistance: Integer;
    procedure SetYAxisLabelDistance(v: Integer);
    function  GetYAxisLabelDistance: Integer;
    procedure SetYAxisTitleDistance(v: Integer);
    function  GetYAxisTitleDistance: Integer;
    procedure SetAllowedOut(v: Boolean);
    procedure SetLabelGraph(v: Boolean);
    procedure SetGraphTitle(v: ShortString);
    procedure SetGridlines(v: Boolean);
    procedure SetGridcolor(v: TColor);
    procedure SetGridStyle(v: TPenStyle);
    procedure SetTickmarks(v: Boolean);
    procedure SetAxescolor(v: TColor);
    function  GetAllowedOut: Boolean;
    function  GetLabelGraph: Boolean;
    function  GetGraphTitle: ShortString;
    function  GetGridlines: Boolean;
    function  GetGridcolor: TColor;
    function  GetGridStyle: TPenStyle;
    function  GetTickmarks: Boolean;
    function  GetAxescolor: TColor;
   {$ENDIF}{BACKWARD}

    procedure TitleFontChanged(Sender: TObject);
    procedure LabelFontChanged(Sender: TObject);

   {These two are needed for Series[] property:}
    function GetSeriesInt(i:integer):TSeries;
    function CreateNewSeries(i:integer):TSeries;
  protected
    procedure DoTightFit;
    procedure ClipGraph;
    procedure UnclipGraph;
    function  DoResizeX: Boolean;
    function  DoResizeY: Boolean;
    function  DoResizeY2: Boolean;
    function  DoResize: Boolean;
    procedure CalcXMetrics;
    procedure CalcYMetrics;
    procedure CalcY2Metrics;
    procedure CalcMetrics;
    function  fx(v: Double): Integer;
    function  fy(v: Double): Integer;
    function  fy2(v: Double): Integer;
    procedure DrawLineSegment(x1, y1, x2, y2: Integer);
    procedure DrawXGridlines;
    procedure DrawYGridlines(CYAxis: TAxis);
    procedure DrawXTickMarks(base:integer; reverse:boolean);
    procedure DrawYTickMarks(CYAxis: TAxis; base:integer; reverse:boolean);
    procedure DrawXLabels(base:integer; reverse:boolean);
    procedure DrawYLabels(CYAxis: TAxis; base:integer; reverse:boolean);
    procedure DrawGraphTitle;
    procedure DrawMarks(when: Boolean);
    procedure DrawXAxis;
    procedure DrawYAxis;
    procedure DrawY2Axis;
    procedure DrawAxes;
    procedure ColorBackground;
    procedure DrawLine(t: TSeries);
    procedure DrawPoints(t: TSeries);
   {$IFDEF STATISTICS}
    procedure DrawRegression(t:TSeries);
   {$ENDIF}
    procedure PaintGraph;
    procedure Paint; override;
    procedure DoRescaleEvent;
    function  DataFound(whichaxis:TAxis): Boolean;
   {$IFDEF BACKWARD}
    procedure DrawSeries(t: TSeries);
   {$ENDIF}{BACKWARD}

   {Implements dragging and resizing:}
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
    procedure SetUpDragging(X,Y: Integer);
    procedure MouseMove(Shift: TShiftState; X,Y: Integer); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState; X,Y: Integer); override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor  Destroy; override;

   { data routines }
    property serieshandle:TSeries read FSeries; {provided to iterate through
                                                 existing series}
    property LastSeriesIndex:integer read FLastSeries;
    property  Series[i:integer]: TSeries read GetSeriesInt; default;
    function  FindSeries(i:integer): Boolean;
    procedure ClearAll;

   { hooking other graph series }
    procedure Hookseries(Source:TSeries; Ahooktype:THooktype;
                     Dest:Integer; WantPersist:boolean);
    procedure unhookseries(source:TSeries; WantPersist:boolean);
    procedure HookedSeriesEvent(Sender:TObject; TheMessage:TDSChangeType);

   {$IFDEF BACKWARD} {obsolete series routines}
    procedure Clear(sN: Integer);
    function  Add(sN: Integer; x, y: Double): longint;
    function  GetValue(sN: Integer; curr: longint;
                   var index: longint;
                   var x, y, r: Double): Boolean;
    function  GetPoint(sN: Integer;
                   var index: longint;
                   var x, y, r: Double): Boolean;
    function  DeleteValue(sN: Integer; x: Double): Boolean;
    function  DeletePoint(sN: Integer; index: longint): Boolean;
    procedure SetSeries(sN: Integer;
                   Makeactive, Initialise, AutoZero: Boolean;
                   ZeroValue: Double);
    function  GetSeries(sN: Integer;
                   var active: Boolean;
                   var ZeroValue: Double): Boolean;
    procedure SetSeriesLine(sN: Integer;
                   status: Boolean;
                   Color: TColor;
                   Style: TPenStyle);
    function  GetSeriesLine(sN: Integer;
                   var Color: TColor;
                   var Style: TPenStyle): Boolean;
    procedure SetSeriesPoints(sN: Integer;
                   status: Boolean;
                   Size: Integer;
                   Shape: et_PointShape;
                   Filled: Boolean;
                   Color: TColor);
    function  GetSeriesPoints(sN: Integer;
                   var Size: Integer;
                   var Shape: et_PointShape;
                   var Filled: Boolean;
                   var Color: TColor): Boolean;
   {$ENDIF}{BACKWARD} {obsolete series routines}

   { Marks }
    procedure ClearMarks;
    procedure Addmark(id: Integer; x1,y1, x2,y2: Double;
                   c: Tcolor; name: ShortString;
                   marktype:et_marktype; status:boolean);
    function  DeleteMark(id: Integer): Boolean;

   { for mouse events }
    property HasDragged: boolean read FHasDragged write SetHasDragged;
    function  GetMouseXY(sN, x, y: Integer; var index: Integer;
                   var xScaled, yScaled,
                   xNearest, yNearest: Double): Boolean;
    function  GetMouseX(x: Integer): Double;
    function  GetMouseY(y: Integer): Double;
    function  GetMouseY2(y: Integer): Double;
    function  GetMouseVals(sN, x: Integer; var index: Integer;
                   var xNearest, yNearest: Double): Boolean;

   { PrintOnPage allows more than one graph to be printed on a page; calls
    to it must be preceded by BeginDoc amd followed by EndDoc.
    Print prints only the current graph on its own sheet.
    The graph is printed without change in aspect ratio.
    The size of the printed graph is controlled by PrintScalePct,
    which makes the dimension of the printed graph a percentage of the page
    dimension (on the axis with the larger Screen/Page ratio)
    The location of the upper left corner of the graph is controlled by
    PrintX(Y)OffsetPct, which locate the corner as percentages of the
    page height and width}
    procedure PrintOnPage;
    procedure Print;

  { A convenient way to graph a function; automatically fills graph; handles
    singularities in the function.
    F must be far; parms may be of any type (usu. array of Double) -- no error checks;
    the graph of the function will be drawn from x=x1 to x=x2;
    if x1=x2, will be set to XMin, XMax; if larger, will be clipped to XMin, XMax
    if steps=0, will choose "enough" to be smooth }
    procedure DrawFunction(F: PlotFunction; var parms; x1, x2: Double;
                   color: TColor; style: TPenStyle; steps: Word);

    function  CheckScales: Boolean;
    procedure Debug(i: Integer; fn: String);

   {$IFDEF BACKWARD}
   {properties moved to TAxis (FXAxis/FYAxis); retained here for compatibilty; not published}
    property XAxisTitle: ShortString read GetXAxisTitle write SetXAxisTitle;
    property XLabelDecimals: Integer read GetXLabelDec write SetXLabelDec;
    property XLogScale: Boolean read GetXLogging write SetXLogging;
    property XMax: Double read GetXMax write SetXMax;
    property XMin: Double read GetXMin write SetXMin;
    property XMinDiff: Double read GetMinXScale write SetMinXScale;
    property XStep: Double read GetXStep write SetXStep;
    property AutoXSizing: Boolean read GetXAutoSizing write SetXAutosizing;
    property AutoStepping: boolean read GetAutoStepping write SetAutostepping;
    property YAxisTitle: ShortString read GetYAxisTitle write SetYAxisTitle;
    property YLabelDecimals: Integer read GetYLabelDec write SetYLabelDec;
    property YLogScale: Boolean read GetYLogging write SetYLogging;
    property YMax: Double read GetYMax write SetYMax;
    property YMin: Double read GetYMin write SetYMin;
    property YMinDiff: Double read GetMinYScale write SetMinYScale;
    property YStep: Double read GetYStep write SetYStep;
    property AutoYSizing: Boolean read GetYAutoSizing write SetYAutosizing;
    property GridLines: Boolean read GetGridlines write SetGridlines;

   {properties moved to TDimensions; retained here for compatibilty; not published}
    property MarginBottom: Word read GetMargBottom write SetMargBottom;
    property MarginLeft: Word read GetMargLeft write SetMargLeft;
    property MarginRight: Word read GetMargRight write SetMargRight;
    property MarginTop: Word read GetMargTop write SetMargTop;
    property TMLength: Word read GetTMLength write SetTMLength;
    property XAxisTitleDist: Integer read GetXAxisTitleDistance write SetXAxisTitleDistance;
    property XLabelDistance: Integer read GetXAxisLabelDistance write SetXAxisLabelDistance;
    property XAxisLabelDist: Integer read GetXAxisLabelDistance write SetXAxisLabelDistance;
    property YLabelDistance: Integer read GetYAxisLabelDistance write SetYAxisLabelDistance;

   {properties moved to TAppearance; retained here for compatibilty; not published}
    property AxesColor: TColor read GetAxesColor write SetAxescolor;
    property GridColor: TColor read GetGridColor write SetGridcolor;
    property GridStyle: TPenStyle read GetGridStyle write SetGridStyle;
    property LabelGraph: Boolean read GetLabelGraph write SetLabelGraph;
    property PlotOutOfScale: Boolean read GetAllowedOut write SetAllowedOut;
    property GraphTitle: ShortString read GetGraphTitle write SetGraphTitle;
   {$ENDIF}{BACKWARD}

   {$IFDEF DESIGNER}
    procedure rundesigner;
   {$ENDIF}
  published
    property Plotting: Boolean read FPlotting write SetPlotting default true;
    property DragAllowed: Boolean read FDraggable write FDraggable default true;
    property AllowDuplicates: Boolean read FAllowDuplicates write SetAllowDuplicates default false;

    property XAxis: TAxis read FXAxis write FXAxis;
    property YAxis: TAxis read FYAxis write FYAxis;
    property YAxis_Second: TAxis read FYAxis2 write FYAxis2;
    property Dimensions: TDimensions read FDimensions write FDimensions;
    property Appearance: TAppearance read FAppearance write FAppearance;
{    property Legend: TLegend read FLegend write Flegend;}

    property OnLogError: TSeriesEvent read FOnLogError write FOnLogError;
    property OnNewSeries: TSeriesEvent read FOnNewSeries write FOnNewSeries;
    property OnTooManyPoints: TSeriesEvent read FOnTooManyPoints write FOnTooManyPoints;
    property OnRescale: TNotifyEvent read FOnRescale write FOnRescale;
    property OnScaleError: TNotifyEvent read FOnScaleError write FOnScaleError;
    property OnPaintEnd: TPaintEvent read FOnPaintEnd write FOnPaintEnd;

    property Align;
    property BevelInner;
    property BevelOuter;
    property BevelWidth;
    property BorderStyle default bsSingle;
    property BorderWidth;
    property Ctl3D;
    property Cursor;
    property Color;
    property Font;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property Visible;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnResize;
  end {class TxyGraph};

  TxyGraphCompEditor = class (TComponentEditor)
  private
    procedure editGraph;
  public
    function GetVerb(Index: Integer): string; override;
    function GetVerbCount: Integer; override;
    procedure ExecuteVerb(Index: Integer); override;
    procedure Edit; override;
  end;

procedure Register;

implementation

{$IFDEF DESIGNER}
uses xyeditor;
{$ENDIF}

const
  Loge = 0.4342944818;
 { time constants: Alert users will note that here and elsewhere where date/time
   values are used, small errors creep in due to assumptions implicit in the
   way TDateTime works. These have been tolerated in the interests of speed.
   Month and year are more variable: the only place where the length of these
   is approximated is for the guestimation of the number of ticks; I will
   be interested to hear if anybody has problems with this assumption }
  cminute1 = 1/(24 * 60);
  cminute5 = 5/(24 * 60);
  cminute30 = 30/(24 * 60);
  chour = (1/24);

procedure incmonth(var y,m:word; i:word);
{ add i months to date xx/m/y }
begin
 if m + i > 12 then
   begin
   inc(y);
   m := m + i - 12;
   end
 else inc(m,i);
end;

{----------------------------------------------------------------------------
    #1a. TSeries - administration
 ----------------------------------------------------------------------------}

 constructor TSeries.create(aGraph:TxyGraph; i:integer);
 begin
   inherited create;
   next := nil;
   MyGraph := aGraph;
   WhichYAxis := myGraph.FYAxis;
   FSeriesIndex := i;
   FseriesName := 'Series ' +inttostr(i);
   if MyGraph.FLastSeries < i then MyGraph.FLastSeries := i;
   FActive := true;
   FDrawLine := true;
   FDrawPoints := true;
   FHoldUpdates := false;
   FAllowDuplicates := mygraph.allowduplicates;
   case FSeriesIndex of
    1:FPointColor := clgreen;
    2:FPointColor := clblue;
    3:FPointColor := clred;
    4:FPointColor := clGray;
    5:FPointColor := clMaroon;
    else FPointColor := clgreen;
   end;
   FLineColor := FPointColor;
   case FSeriesIndex of
    1:FPointShape := ps_Square;
    2:FPointShape := ps_Circle;
    3:FPointShape := ps_Diamond;
    4:FPointShape := ps_Cross;
    5:FPointShape := ps_Square;
    else FPointShape := ps_Circle;
   end;
   FFillPoints := (FSeriesIndex <= 4);

   case FSeriesIndex of
    1:FLineStyle := psSolid;
    2:FLineStyle := psDash;
    3:FLineStyle := psDot;
    4:FLineStyle := PsDashDot;
    5:FLineStyle := PsDashDotDot;
    else FLineStyle := psSolid;
   end;

   FPointSize := 4;
   FUpperBound := 0;
   FLowerBound := 0;
   FBoundsType := bt_none;
   FBoundsLineStyle := psDashDot;
   {$IFDEF STATISTICS}
   FRegrColor := clgray;
   FRegrLineStyle := psDot;
   {$ENDIF}
end;

destructor TSeries.destroy;
var p1,p2:TSeries;
begin
  {splice self out of MyGraph's series list}
  p1 := MyGraph.Fseries;
  p2 := nil;
  while p1.Fseriesindex <> Fseriesindex do
    begin
    p2 := p1;
    p1 := p1.next;
    end;
  if p2 = nil then
   MyGraph.Fseries := p1.next
  else p2.next := p1.next;
  inherited destroy;
end;

{----------------------------------------------------------------------------
    #1b. TSeries - property servers
 ----------------------------------------------------------------------------}

procedure TSeries.setHoldUpdates(v:boolean);
begin
 if v <> FHoldUpdates then
   begin
   FHoldUpdates := v;
   if FHoldUpdates then
     updateswaiting := false
   else
     if updateswaiting then RequestPaint(dsClearUpdates);
   end;
end;

procedure TSeries.setActive(v:boolean);
begin
  if v <> FActive then
  begin
    FActive := v;
    RequestPaint(dsChangeLook);
  end;
end;

procedure TSeries.setDrawLine(v:boolean);
begin
  if v <> FDrawline then
  begin
    FDrawline := v;
    RequestPaint(dsChangeLook);
  end;
end;

procedure TSeries.SetDrawPoints(v:boolean);
begin
  if v <> FDrawpoints then
  begin
    FDrawpoints := v;
    RequestPaint(dsChangeLook);
  end;
end;

procedure TSeries.setFillPoints(v:boolean);
begin
  if v <> Ffillpoints then
  begin
    Ffillpoints := v;
    RequestPaint(dsChangeLook);
  end;
end;

procedure TSeries.SetLineColor(v:TColor);
begin
  if v <> FLineColor then
  begin
    FLineColor := v;
    RequestPaint(dsChangeLook);
  end;
end;

procedure TSeries.SetPointColor(v:TColor);
begin
  if v <> FPointColor then
  begin
    FPointColor := v;
    RequestPaint(dsChangeLook);
  end;
end;

procedure TSeries.SetBoundsColor(v:TColor);
begin
  if v <> FBoundsColor then
  begin
    FBoundsColor := v;
    RequestPaint(dsChangeLook);
  end;
end;

procedure TSeries.SetPointShape(v:et_pointshape);
begin
  if v <> FPointShape then
  begin
    FPointShape := v;
    if (FPointShape <> ps_Wide) then
    begin
     if fPointsize < 2 then fPointsize := 2;
     if odd(fPointsize) then Inc(fPointsize);
    end;
    RequestPaint(dsChangeLook);
  end;
end;

procedure TSeries.SetLinestyle(v:TPenStyle);
begin
  if v <> fLinestyle then
  begin
    fLinestyle := v;
    RequestPaint(dsChangeLook);
  end;
end;

procedure TSeries.SetBoundsLinestyle(v:TPenStyle);
begin
  if v <> fBoundsLinestyle then
  begin
    fBoundsLinestyle := v;
    RequestPaint(dsChangeLook);
  end;
end;

procedure TSeries.SetPointsize(v:word);
begin
  if v <> fPointsize then
  begin
    fPointsize := v;
    if (FPointShape <> ps_Wide) then
    begin
     if fPointsize < 2 then fPointsize := 2;
     if odd(fPointsize) then Inc(fPointsize);
    end;
    RequestPaint(dsChangeLook);
  end;
end;

procedure TSeries.SetUpperBound(v:double);
begin
 if v <> FUpperBound then
   begin
   FUpperBound := v;
   if FBoundsType = bt_AsSet then requestpaint(dsChangeLook);
   end;
end;

procedure TSeries.SetLowerBound(v:Double);
begin
 if v <> FLowerBound then
   begin
   FLowerBound := v;
   if FBoundsType = bt_AsSet then requestpaint(dsChangeLook);
   end;
end;

procedure TSeries.setBoundsType(v:et_BoundType);
begin
  if v <> FBoundsType then
    begin
    FBoundsType := v;
    requestpaint(dsChangeLook);
    end;
end;

{$IFDEF STATISTICS}
procedure TSeries.SetRegrColor(v:TColor);
begin
  if v <> FRegrColor then
  begin
    FRegrColor := v;
    if FRegType <> rg_none then RequestPaint(dsChangeLook);
  end;
end;

procedure TSeries.SetRegrLineStyle(v:TPenstyle);
begin
  if v <> FRegrLineStyle then
  begin
    FRegrLineStyle := v;
    if FRegType <> rg_none then RequestPaint(dsChangeLook);
  end;
end;
{$ENDIF}{STATISTICS}

procedure TSeries.SetWhichYAxis;
begin
  if v <> FWhichYAxis then
    begin
    FWhichYAxis := v;
    RequestPaint(dsChangeLook);
    end;
end;

procedure TSeries.WarnTooManyPoints;
begin
  if assigned(MyGraph.FOnTooManyPoints) then
    MyGraph.FOnTooManyPoints(MyGraph, FSeriesIndex);
  inherited WarnToomanyPoints;
     { I don't know whether this inherited bit is a good idea or not - Don't
       think it's very likely to be used. But when adding a lot of points,
       the overhead is insignificant, in that it's expected to be slow?? }
end;

{----------------------------------------------------------------------------
    #1c. TSeries - drawing routines
 ----------------------------------------------------------------------------}

procedure TSeries.RequestPaint;
begin
 if FHoldUpdates
  then updateswaiting := true
  else
   begin
   inherited requestpaint(TheMessage); 
   if (TheMessage <> dsUnLoad) and MyGraph.fplotting then MyGraph.paint;
   end;
end;


type
  pLinePoints = ^TLinePoints;
  TLinePoints = array[0..15000] of TPoint;

procedure TSeries.DrawMyLine;
var
  pp: pgPoint;
  i: Word;
  lp: pLinePoints;
  Zero: double;
  wyax : boolean;
begin
  wyax := (FWhichYAxis = MyGraph.FYAxis);
  if FData <> nil then
  with MyGraph do
  begin
    FCanvas.Pen.Color := FLineColor;
    FCanvas.Pen.Style := FLineStyle;
    New(lp);
    try
      pp := FData;
      zero := FZeroOffset;
      while pp <> nil do
        begin
        i := 0;
        while (pp <> nil) and (i < 15000) do
             { I routinely turn the i < 15,000 check off for speed. }
          begin
{Jeff Roberts possible bug fix - see the comments in the help under bugs:
          if ((pp^.xv >= fxaxis.fmin) and (pp^.xv <= FXaxis.fmax)) then
          begin}
          lp^[i].x := fx(pp^.xv);
          if wyax
            then lp^[i].y := fy(pp^.yv - Zero)
            else lp^[i].y := fy2(pp^.yv - Zero);
          Inc(i);
          {end;  paired from the begin in brackets above}
          pp := pp^.next;
          end;
        Polyline(FCanvas.Handle, lp^, i);
        end;
    finally
      Dispose(lp);
    end;
  end;
end {DrawMyLine};


procedure TSeries.DrawMyPoints;
var
  s1Size: Word;

  function Pt(AX, AY: Integer): TPoint;
  begin
    with Result do
    begin
      X := AX;
      Y := AY;
    end;
  end;

  procedure PlotSquare(x, y: Integer); far;
  begin
    MyGraph.FCanvas.Polygon([Pt(x - s1Size, y - s1Size), Pt(x + s1size, y - s1Size),
                             Pt(x + s1Size, y + s1size), Pt(x - s1Size, y + s1Size)]);
  end;

  procedure PlotCircle(x, y: Integer);
  begin
    MyGraph.FCanvas.Ellipse(x - s1Size, y - s1Size, x + s1size, y + s1size);
  end;

  procedure PlotDiamond(x, y: Integer);
  begin
    MyGraph.FCanvas.Polygon([Pt(x, y - s1Size), Pt(x + s1size, y),
                             Pt(x, y + s1size), Pt(x - s1Size, y)]);
  end;

  procedure PlotCross(x, y: Integer);
  begin
    MyGraph.DrawLineSegment(x - s1Size, y, x + s1size, y);
    MyGraph.DrawLineSegment(x, y - s1Size, x, y + s1size);
  end;

var
  pp: pgPoint;
  zero: Double;
  ty, OldPenWidth:integer;
  wyax:boolean;
begin {DrawMyPoints}
  wyax := (FWhichYAxis = MyGraph.FYAxis);
  if FData <> nil then
  with MyGraph do
  begin
    FCanvas.Pen.Color := FPointColor;
    FCanvas.Pen.Style := psSolid;
    FCanvas.Brush.Style := bsSolid;
    if FFillPoints then
      FCanvas.Brush.Color := FPointColor
    else
      FCanvas.Brush.Color := Color;
    if FPointShape = ps_Wide then
    begin
      OldPenWidth := FCanvas.pen.width;
      FCanvas.Pen.Width := Round(FScale*(FPointSize div 2));
      FCanvas.Pen.Color := FPointColor;
    end;
    s1Size := Round(FScale*(FPointSize div 2));
    pp := FData;
    zero := FZeroOffset;
   { we could fool around with assigned procedures here, but while
     it might tidy the code up a bit, it would be slower }
    if FAppearance.FAllowedOut then
    begin
      case FPointShape of           { only test this once }
        ps_Square: while pp <> nil do
                   begin
                     if wyax then
                       PlotSquare(fx(pp^.xv), fy(pp^.yv - zero))
                     else
                       PlotSquare(fx(pp^.xv), fy2(pp^.yv - zero));
                     pp := pp^.next;
                   end;
        ps_Circle: while pp <> nil do
                   begin
                     if wyax then
                       PlotCircle(fx(pp^.xv), fy(pp^.yv - zero))
                     else
                       PlotCircle(fx(pp^.xv), fy2(pp^.yv - zero));
                     pp := pp^.next;
                   end;
        ps_Diamond: while pp <> nil do
                   begin
                     if wyax then
                       PlotDiamond(fx(pp^.xv), fy(pp^.yv - zero))
                     else
                       PlotDiamond(fx(pp^.xv), fy2(pp^.yv - zero));
                      pp := pp^.next;
                    end;
        ps_Cross: while pp <> nil do
                  begin
                    if wyax then
                      PlotCross(fx(pp^.xv), fy(pp^.yv - zero))
                    else
                      PlotCross(fx(pp^.xv), fy2(pp^.yv - zero));
                    pp := pp^.next;
                  end;
        ps_Wide: begin
                 if pp = nil then exit;
                 while pp^.next <> nil do
                 begin
                   if wyax then ty := fy(pp^.yv) else ty := fy2(pp^.yv);
                   DrawLineSegment(fx(pp^.xv), ty, fx(pp^.next^.xv), ty);
                   pp := pp^.next;
                 end;
                 if pp^.xv < FXaxis.Max then
                 begin
                   if wyax then ty := fy(pp^.yv) else ty := fy2(pp^.yv);
                   DrawLineSegment(fx(pp^.xv), ty, fx(FXaxis.max), ty);
                 end;
                 end;
      end;
    end
    else { not allowed out }
    with MyGraph do
    begin
      while (pp <> nil) and (pp^.xv < FXAxis.Min) do pp := pp^.next;
        case FPointShape of  { only test this once }
        ps_Square: while (pp <> nil) and (pp^.xv <= FXAxis.Max) do
                   begin
                     if (pp^.yv <= WhichYAxis.Max) and (pp^.yv >= WhichYAxis.Min) then
                       if wyax then
                         PlotSquare(fx(pp^.xv), fy(pp^.yv - zero))
                       else
                         PlotSquare(fx(pp^.xv), fy2(pp^.yv - zero));
                     pp := pp^.next;
                   end;
        ps_Circle: while (pp <> nil) and (pp^.xv <= FXAxis.Max) do
                   begin
                     if (pp^.yv <= WhichYAxis.Max) and (pp^.yv >= WhichYAxis.Min) then
                       if wyax then
                         PlotCircle(fx(pp^.xv), fy(pp^.yv - zero))
                       else
                         PlotCircle(fx(pp^.xv), fy2(pp^.yv - zero));
                     pp := pp^.next;
                   end;
        ps_Diamond: while (pp <> nil) and (pp^.xv <= FXAxis.Max) do
                    begin
                      if (pp^.yv <= WhichYAxis.Max) and (pp^.yv >= WhichYAxis.Min) then
                        if wyax then
                          PlotDiamond(fx(pp^.xv), fy(pp^.yv - zero))
                        else
                          PlotDiamond(fx(pp^.xv), fy2(pp^.yv - zero));
                      pp := pp^.next;
                    end;
        ps_Cross: while (pp <> nil) and (pp^.xv <= FXAxis.Max) do
                  begin
                    if (pp^.yv <= WhichYAxis.Max) and (pp^.yv >= WhichYAxis.Min) then
                      if wyax then
                        PlotCross(fx(pp^.xv), fy(pp^.yv - zero))
                      else
                        PlotCross(fx(pp^.xv), fy2(pp^.yv - zero));
                    pp := pp^.next;
                  end;
        ps_Wide: begin
                 if pp = nil then exit;
                 while (pp^.next <> nil) and (pp^.next^.xv < FXAxis.max) do
                 begin
                   if wyax then ty := fy(pp^.yv) else ty := fy2(pp^.yv);
                   DrawLineSegment(fx(pp^.xv), ty, fx(pp^.next^.xv), ty);
                   pp := pp^.next;
                 end;
                 if pp^.xv < FXaxis.Max then
                 begin
                   if wyax then ty := fy(pp^.yv) else ty := fy2(pp^.yv);
                   DrawLineSegment(fx(pp^.xv), ty, fx(FXaxis.max), ty);
                 end;
                 end;
      end;
    end;
    FCanvas.Brush.Color := Color;
    FCanvas.Brush.Style := bsClear;
    if FPointShape = ps_Wide then FCanvas.Pen.Width := OldPenWidth;
  end;
end {DrawMyPoints};


{$IFDEF STATISTICS}
procedure TSeries.DrawMyRegression;
var pp: pgPoint;
    Zero: Double;
    i: Word;
    lp: pLinePoints;
    wyax : boolean;
begin
wyax := (FWhichYAxis = MyGraph.FYAxis);
if (FData <> nil) and (FData^.next <> nil) then { no regressions unless at least 2 data points }
 with MyGraph do
 begin
   if NeedRegreCalc then DoRegression;
   { can't store smoothing curve (spline, dwls) result with the point - this
     defeats the whole purpose of the spline. So they have to be drawn
     differently to other regression types. for the same reasons, residuals
     are not available like they are with the other regressions - in this
     case the figure returned in getpoint etc. is the second differential
     with spline or nothing with dwls
     The data is available through the RegressionData property if you want it }
   if (FRegType = rg_Spline) or (FRegType = rg_dwls)
     then pp := FRegrData
     else pp := FData;
   FCanvas.Pen.Color := FRegrColor;
   FCanvas.Pen.Style := FRegrLineStyle;
   New(lp);
   try
     Zero := FZeroOffset;
     while (pp <> nil) do
       begin
       i := 0;
       while (pp <> nil) and (i < 15000) do
         {I routinely have the i < 15,000 check turned off}
         begin
         lp^[i].x := fx(pp^.xv);
         if wyax
           then lp^[i].y := fy(pp^.rv - zero)
           else lp^[i].y := fy2(pp^.rv - zero);
         pp := pp^.next;
         Inc(i);
         end;
       Polyline(FCanvas.Handle, lp^, i);
       end;
   finally
     Dispose(lp);
   end;
 end;
end {DrawMyRegression};
{$ENDIF}{STATISTICS}

procedure TSeries.DrawBounds;
  procedure DrawBound(y:double);
  begin
  with MyGraph do
    begin
    FCanvas.pen.color := FBoundsColor;
    FCanvas.pen.style := FBoundsLineStyle;
    DrawLineSegment(fx(XMinVal), fy(y), fx(XMaxVal), fy(y));
    end;
  end;
var mean,SD:double;
begin
 if FBoundsType = bt_None then exit;
 if FBoundsType = bt_AsSet then
  begin
  DrawBound(FUpperBound);
  DrawBound(FLowerBound);
  exit;
  end;
 {$IFDEF STATISTICS}
 mean := XStats.Mean;
 SD := XStats.SD;
 case FBoundsType of
   bt_1SD  :begin
            drawbound(mean + SD);
            drawbound(mean - SD);
            end;
   bt_2SD  :begin
            drawbound(mean + 2 * SD);
            drawbound(mean - 2 * SD);
            end;
   bt_3SD  :begin
            drawbound(mean + 3 * SD);
            drawbound(mean - 3 * SD);
            end;
   bt_allSD:begin
            drawbound(mean + SD);
            drawbound(mean - SD);
            drawbound(mean + 2 * SD);
            drawbound(mean - 2 * SD);
            drawbound(mean + 3 * SD);
            drawbound(mean - 3 * SD);
            end;
  end;
 {$ENDIF}
end;

procedure TSeries.Draw;
begin
  if not FActive or (FData = nil) then Exit;
  with MyGraph do
  begin
    if FXAxis.LogScale or WhichYAxis.LogScale then
    begin
      if Changed then GetMinMax;
      if (FXAxis.LogScale and (xvMin <= 0)) or
         (WhichYAxis.LogScale and (yvMin <= 0)) then
      begin
        if Assigned(FOnLogError) then FOnLogError(self, FSeriesIndex);
        Exit;
      end;
    end;
    if not FAppearance.FAllowedOut then ClipGraph;
    DrawBounds;
   {$IFDEF STATISTICS}
    if (FRegType <> rg_None)
      {and possibly allow 1 of these 2: }
      { and not WhichYAxis.LogScale }
      { and not (WhichYAxis.LogScale and
         ((FRegType = rg_Spline) or (FRegType = rg_dwls) or (FRegType = rg_polint)
                or (FRegType = rg_quadratic)) )}
      then DrawMyRegression;

    { Q. What's all this about??
      A. Any of the regression types may cause an exception with ylogging
         turned on. You have 3 options:
        1. Try to draw them anyway, accepting that an exception may occur.
             For this option leave both checks commented out
        2. Be very careful and not allow any to be drawn.
             For this option, allow the second check
        3. Only allow the linear types to be drawn - these are less likely to
           cause an exception.
             For this option, allow the third check but not the second

     !! Note that if a series contains negative data while Ylogging is true,
        the paint will never make it this far   }
   {$ENDIF}
    if FDrawLine then DrawMyLine;
    if not FAppearance.FAllowedOut then UnclipGraph;
   { it'd be nice if we were able to clip the points, but the clipping
      applies to the centre of the point, not the plotted shape }
    if FDrawPoints then DrawPoints(self);
  end;
end {Draw};

{--------------------------------------------------------------------
    #2a. TAxis  - scaling routines
 --------------------------------------------------------------------}

procedure TAxis.InitLogTicks;
begin
  with FLogTickInfo do
  begin
    DrawMinorLabels := true;
    if LogTickCount > MaxLogTickCount then LogTickCount := MaxLogTickCount;
    LogTickArray[0] := 1;
   {CAREFUL! these LogTickCount values must correspond with AllowedLogTickCounts}
    case LogTickCount of
      0:
         begin
           LogTickCount := 0;
         end;
      1:
         begin
           LogTickCount := 1;
           LogTickArray[1] := 5;
         end;
      2, 3:
         begin
           LogTickCount := 2;
           LogTickArray[1] := 2;
           LogTickArray[2] := 5;
         end;
      4, 5, 6:
         begin
           LogTickCount := 5;
           LogTickArray[1] := 1.5;
           LogTickArray[2] := 2;
           LogTickArray[3] := 3;
           LogTickArray[4] := 5;
           LogTickArray[5] := 7;
         end;
      7, 8:
         begin
           LogTickCount := 7;
           LogTickArray[1] := 1.5;
           LogTickArray[2] := 2;
           LogTickArray[3] := 3;
           LogTickArray[4] := 4;
           LogTickArray[5] := 5;
           LogTickArray[6] := 6;
           LogTickArray[7] := 8;
         end;
      9: begin
           LogTickArray[1] := 1.2;
           LogTickArray[2] := 1.5;
           LogTickArray[3] := 2;
           LogTickArray[4] := 2.5;
           LogTickArray[5] := 3;
           LogTickArray[6] := 4;
           LogTickArray[7] := 5;
           LogTickArray[8] := 6;
           LogTickArray[9] := 8;
         end;
    end {case};
    LogTickArray[LogTickCount + 1] := 10;
      {This is a guard value for the linear searches in GetFirstTick and SetLogMinMax}
  end {with};
end {InitLogTicks};

procedure TAxis.AdjustLogTickCount;
{CAREFUL! these LogTickCount values must correspond with AllowedLogTickCounts}
var
  r: Double;
begin
  r := FGraph.FAppearance.MinSteps/(ln(FMax/FMin)*Loge);
  with FLogTickInfo do
    if      r >    10 then LogTickCount := 9
    else if r >     8 then LogTickCount := 7
    else if r >     6 then LogTickCount := 5
    else if r >     3 then LogTickCount := 2
    else if r >     2 then LogTickCount := 1
    else                   LogTickCount := 0;
  InitLogTicks;
end {AdjustLogTickCount};

procedure TAxis.CheckDrawMinorLabels;
begin
  with FLogTickInfo do
  if LogStepping then
    DrawMinorLabels := true
  else
    DrawMinorLabels := ( (ln(FMax/FMin)*Loge)*LogTickCount < 10)
end;

{ step size chosen in a 1,2,5,10 squence depending not only on the
  characteristic, but also the mantissa, of the range}
function TAxis.GetStep: Double;
var
  w, t, B: Double; minSteps: Word;
begin
  minSteps := FGraph.FAppearance.MinSteps;
  w := FMax - FMin;
if w <= 0 then raise Exception.Create('GetStep entered with bad range');
  t := ln(w)*Loge;
  if t < 0 then t := t - 1;
  B := exp( trunc(t * 1.001) / Loge );

  if         w/B >= minSteps then Result := B
  else if  2*w/B >= minSteps then Result := B/2
  else if  5*w/B >= minSteps then Result := B/5
  else if 10*w/B >= minSteps then Result := B/10
  else if 20*w/B >= minSteps then Result := B/20
  else if 50*w/B >= minSteps then Result := B/50
  else                            Result := B/100;
  {sufficient for maxSteps <= 125}
end {GetStep};

function TAxis.getDateStep: double;
begin
{ these are approximations - they are not used for actually
  plotting the ticks, but for checking the scale }
  case FDateTickType of
    dt_minute    :result := cminute1;
    dt_5minute   :result := cminute5;
    dt_halfhourly:result := cminute30;
    dt_hourly    :result := chour;
    dt_daily     :result := 1;
    dt_weekly    :result := 7;
    dt_monthly   :result := 30;
    dt_2monthly  :result := 61;
    dt_quarterly :result := 91;
    dt_annually  :result := 365;
    dt_decade    :result := 3654;
    dt_century   :result := 36540;
    dt_custom    :result := getStep;
  end;
end;

{--------- DoResize helper functions -------}
procedure TAxis.SetDateMinMax;
var nYear, nMonth, nDay, nHour, nMin, nSec, nMSec,
    xYear, xMonth, xDay, xHour, xMin, xSec, xMSec: Word;
begin
  if (oMax - oMin <= 0) or (oMin < 1) then exit;
  DecodeDate(oMin, nYear, nMonth, nDay);
  DecodeDate(oMax, xYear, xMonth, xDay);
  DecodeTime(oMin, nHour, nMin, nSec, nMSec);
  DecodeTime(oMax, xHour, xMin, xSec, xMSec);
  case FDateTickType of
   dt_custom:{ in this case no action is taken to change the scale of the axes }
      setMinMax;
   dt_minute:      {set begin to nearest whole minute before start and set end to}
                   {nearest whole minute after end}
      begin
      FMin := EncodeDate(nYear, nMonth, nDay) + EncodeTime(nHour, nMin, 0, 0);
      If xSec <> 0
       then FMax := EncodeDate(xYear, xMonth, xDay)
                      + EncodeTime(xHour, xMin, 0, 0) + encodetime(0,1,0,0)
       else FMax := EncodeDate(xYear, xMonth, xDay)
                      + EncodeTime(xHour, xMin, 0, 0);
      end;
   dt_5minute:begin
      FMin := EncodeDate(nYear, nMonth, nDay) + EncodeTime(nHour, 5 * (nMin div 5), 0, 0);
      If (xMin mod 5 <> 0) or (xSec <> 0)
        then FMax := EncodeDate(xYear, xMonth, xDay) + EncodeTime(xHour,
                        5 * (xMin div 5), 0, 0) + encodetime(0,5,0,0)
        else FMax := EncodeDate(xYear, xMonth, xDay) + EncodeTime(xHour, xMin, 0, 0);
      end;
   dt_halfhourly:begin
      if nMin >= 30
       then FMin := EncodeDate(nYear, nMonth, nDay) + EncodeTime(nHour, 30, 0, 0)
       else FMin := EncodeDate(nYear, nMonth, nDay) + EncodeTime(nHour, 0, 0, 0);
      If (xMin = 0) and (xSec = 0)
       then FMax := EncodeDate(xYear, xMonth, xDay) + EncodeTime(xHour, 0, 0, 0)
       else if (xMin >= 30)
        then FMax := EncodeDate(xYear, xMonth, xDay) + EncodeTime(xHour, 0, 0, 0)
                      + encodetime(1,0,0,0)
        else FMax := EncodeDate(xYear, xMonth, xDay) + EncodeTime(xHour, 30, 0, 0);
      end;
   dt_hourly:begin
      Fmin := EncodeDate(nYear, nMonth, nDay) + EncodeTime(nHour, 0, 0, 0);
      if (xSec <> 0) or (xMin <> 0)
       then FMax := EncodeDate(xYear, xMonth, xDay) + EncodeTime(xHour, 0, 0, 0)
                  + encodetime(1,0,0,0)
       else FMax := EncodeDate(xYear, xMonth, xDay) + EncodeTime(xHour, 0, 0, 0);
      end;
   dt_daily:begin
      FMin :=  EncodeDate(nYear, nMonth, nDay);
      if (xHour <> 0) or (xMin <>0) or (xSec <> 0)
       then FMax := EncodeDate(xYear, xMonth, xDay) + 1
       else FMax := EncodeDate(xYear, xMonth, xDay);
      end;
   dt_weekly:begin
      FMin := EncodeDate(nYear, nMonth, nDay) - (DayOfWeek(oMin) - 1);
      FMax := EncodeDate(xYear, xMonth, xDay) + 8 - DayOfWeek(oMin); {???}
      end;
   dt_monthly:begin
      FMin :=  EncodeDate(nYear, nMonth, 1);
      if (xday > 1) or (xHour <> 0) or (xMin <>0) or (xSec <> 0)
       then incmonth(xYear,xMonth,1); {month is not set time period}
      FMax := EnCodeDate(xYear,xMonth, 1);
      end;
   dt_2monthly:begin
      if nMonth mod 2 <> 1 then dec(nMonth);
      FMin :=  EncodeDate(nYear, nMonth, 1);
      if (xday > 1) or (xHour <> 0) or (xMin <>0) or (xSec <> 0)
       then incmonth(xYear,xMonth,1); {month is not set time period}
      if xMonth mod 2 <> 1 then inc(xMonth);
      FMax := EnCodeDate(xYear,xMonth, 1);
      end;
   dt_quarterly:begin
      while nMonth mod 3 <> 1 do dec(nMonth);
      FMin :=  EncodeDate(nYear, nMonth, 1);
      if (xday > 1) or (xHour <> 0) or (xMin <>0) or (xSec <> 0)
        then incmonth(xYear,xMonth,1); {month is not set time period}
      while xMonth mod 3 <> 1 do incmonth(xYear,xMonth,1);
      FMax := EnCodeDate(xYear,xMonth, 1);
      end;
   dt_annually:begin
      FMin := EncodeDate(nYear, 1, 1);
      if (xMonth > 1) or(xday > 1) or (xHour <> 0) or (xMin <>0) or (xSec <> 0)
       then FMax := EncodeDate(xYear+1,1,1)
                  { year is also not a set time,  but there can't be overflow here }
       else FMax := EnCodeDate(xYear,1,1);
      end;
   dt_decade:begin
      FMin := EncodeDate(10 * (nYear div 10), 1, 1);
      if (xMonth > 1) or(xday > 1) or (xHour <> 0) or (xMin <>0) or (xSec <> 0)
        then inc(xYear);
      FMax := EncodeDate(10 * (xYear div 10),1,1);
      end;
   dt_century:begin
      FMin := EncodeDate(100 * (nYear div 100), 1, 1);
      if (xMonth > 1) or(xday > 1) or (xHour <> 0) or (xMin <>0) or (xSec <> 0)
        then inc(xYear);
      FMax := EncodeDate(100 * (xYear div 100),1,1);
      end;
  end;
end;

procedure TAxis.SetMinMax;
begin
if oMin > oMax then raise Exception.Create('Impossible: oMin > oMax in SetMinMax!');
  if (oMin = oMax) then
  begin
    if (oMin <> 0) then
    begin
      if (oMin < 0) then
      begin
        FMin := oMin * 1.2;
        FMax := 0;
      end
      else
      begin
        FMin := 0;
        FMax := oMax * 1.2;
      end;
      FStep := GetStep;
      FMax := trunc(FMax / FStep) * FStep;
    end
    else
    begin
      FMin := -1;
      FMax := 1;
      FStep := GetStep;
    end;
    Exit;
  end
  else
  begin
    FMin := oMin;
    FMax := oMax;
  end;
  {assert: FMin<FMax}
  if FAutoStepping or (FStep <= 0) or ( FStep/(FMax-FMin) < 1/FGraph.appearance.FMaxSteps ) then
    FStep := GetStep;

  FMin := trunc( FMin / FStep) * FStep;
  if (oMin < 0) then FMin := FMin - FStep;
  if (FMin > oMin - 0.01*FStep) then FMin := FMin - FStep;

  FMax := trunc(FMax / FStep) * FStep + FStep;
  if (oMax < 0) then FMax := FMax - FStep;
  if (FMax < oMax + 0.01*FStep) then FMax := FMax + FStep;

  if (oMin > 0) and (oMin /(FMax - FMin) < 0.25) then
    FMin := 0
  else if (oMax < 0) and (-oMax / (FMax - FMin) < 0.25) then
    FMax := 0;
if Fmin >= FMax then raise Exception.Create('SetMinMax failed');
end {SetMinMax};


procedure TAxis.SetLogMinMax;
const
  above = true;
  below = not above;

  function GetLogTick(b: Boolean; v: Double): Double;
  var
    ix: Word;
  begin
    with FLogTickInfo do
    begin
      if b = above then
      begin
        ix := 1;
        while (ix < LogTickCount+1) and
              (LogTickArray[ix] < v) do Inc(ix);
      end
      else {below}
      begin
        ix := LogTickCount;
        while (ix > 0) and
              (LogTickArray[ix] > v) do Dec(ix);
      end;
      Result := LogTickArray[ix];
    end;
  end {GetLogTick};

var
  minChi, maxChi, minB, maxB, t: Double;
  minN, maxN : Integer;

begin {SetLogMinMax}
  if (oMin = oMax) then
  begin
    FMin := 0.5*oMax;
    FMax := oMax;
  end
  else
  begin
    FMin := oMin;
    FMax := oMax;
  end;

  t := ln(FMin) * Loge;
  minN := trunc(t);
  if (t < 0) and (t <> minN) then Dec(minN);
  minChi := exp( (t - minN) / Loge );
  minB := exp(minN/Loge);

  t := ln(FMax) * Loge;
  maxN := trunc(t);
  if ((t < 0) and (t <> maxN)) or (t = 0) then Dec(maxN);
  maxChi := exp( (t - maxN) / Loge );
  maxB := exp(maxN/Loge);

  if FAutoStepping then
    AdjustLogTickCount;

  if (FMax/FMin < 4) then
  begin
    {need to set up for constant ticks across FMin..FMax}
    FLogTickInfo.LogStepping := true;
    SetMinMax;
  end
  else
  begin
    FLogTickInfo.LogStepping := false;
    if (FMax/FMin < 10) then
    begin
      FMin := minB*GetLogTick(below, minChi);
      FMax := maxB*GetLogTick(above, maxChi);
    end
    else  if (FMax/FMin < 30) then
    begin
      FMin := GetLogTick(below, minChi);
      if FMin > 1 then FMin := GetLogTick(below, FMin);
      FMin := minB*FMin;
      FMax := GetLogTick(above, maxChi);
      if FMax < 10 then FMax := GetLogTick(above, FMax);
      FMax := maxB*FMax;
    end
    else
    begin
      FMin := minB;
      FMax := maxB*10;
    end;
  end;
  CheckDrawMinorLabels;
end {SetLogMinMax};


function TAxis.DoResize: Boolean;

  function ArbitraryMinMax: Boolean;
  var hold: Double;
  begin
    Result := true;
    if not FLogging then
      if FShowAsTime then
      begin
      FMax := now;
      Fmin := FMax - 5 * GetDateStep;
      end
      else
      begin
        if (FMin = FMax) then
        begin
          if (FMin <> 0) then
            FMin := 0
          else
          begin
            FMin := -1;
            FMax := 1;
          end;
        end
        else if (FMin > FMax) then
        begin
          hold := FMin;
          FMin := FMax;
          FMax := hold;
        end
        else
          Result := false;
        FStep := GetStep;
    end
    else {logging}
    begin
      Result := false;
      if (FMin > FMax) then
      begin
        hold := FMin;
        FMin := FMax;
        FMax := hold;
        Result := true;
      end;
      if (FMax <= 0) then
      begin
        FMax := 10;
        FMin := 1;
      end
      else if (FMin <= 0) then
        FMin := FMax/10
      else
        Result := false or Result;
    end
  end {ArbitraryMinMax};

begin {DoResize}
  if oMax - oMin < FMinScale then oMax := oMin + FMinScale;

  Result := false;
  if not FGraph.DataFound(self) then
  begin
    Result := ArbitraryMinMax;
    if FLogging then
      CheckDrawMinorLabels;
    Exit;
  end {not datafound};

 {data: pick intelligent min/max/tick values:}
  if FAutoSizing then
  begin
    if not FLogging then
      if FShowAsTime then
        SetDateMinMax
      else
      begin
        SetMinMax;
        if FAutoStepping and (oMin < oMax) then
        begin
          FStep := GetStep;
          SetMinMax;
        end;
      end
    else
    begin {logging}
      if (oMin <= 0) then Exit; {nothing can be done}
      SetLogMinMax;
      if FAutoStepping and (oMin < oMax) and (not FLogTickInfo.LogStepping) then
      begin
        AdjustLogTickCount;
        SetLogMinMax;
      end;
    end;
    Result := (FMin <> oMin) or (FMax <> oMax);
  end;
end {DoResize};

function OneSigFigDecs(v: Double): Integer;
{eg: OneSigFigDecs(0.1)->1; (100)-> -2}
var
  t: Double;
begin
  t := ln(v*1.01)*Loge;
  if t < 0 then
    Result  := -trunc(t) + 1
  else
    Result  := -trunc(t);
end;


procedure  TAxis.AdjustLabelDecs;
begin
  FLabelDec := OneSigFigDecs(FStep*1.01);
  if FLabelDec < 0 then
    FLabelDec := 0;
end;

procedure  TAxis.AdjustLogLabelDecs(v: Double);
{the >= 5 must correspond with InitLogTicks: where do fraction steps start?}
begin
  FLabelDec := OneSigFigDecs(v*1.01);
  if FLogTickInfo.LogTickCount >= 5 then Inc(FLabelDec);
  if FLabelDec < 0 then
    FLabelDec := 0;
end;

function TAxis.GetFirstDateTick:double;
var Year, Month, Day, Hour, mMin, Sec, MSec : Word;
begin
  DecodeDate(Fmin, Year, Month, Day);
  DecodeTime(FMin, Hour, mMin, Sec, MSec);
  case FDateTickType of
   dt_custom:result := FMin;
   dt_minute:If Sec <> 0 then
              result := EncodeDate(Year, Month, Day) + EncodeTime(Hour, mMin, 0, 0)
                  + encodetime(0,1,0,0) else
              result := EncodeDate(Year, Month, Day) + EncodeTime(Hour, mMin, 0, 0);
   dt_5minute:If (mMin mod 5 <> 0) or (Sec <> 0) then
              result := EncodeDate(Year, Month, Day) + EncodeTime(Hour,
                        5 * (mMin div 5), 0, 0) + encodetime(0,5,0,0)  else
              result := EncodeDate(Year, Month, Day) + EncodeTime(Hour, mMin, 0, 0);
   dt_halfhourly:If (mMin = 0) and (Sec = 0) then
              result := EncodeDate(Year, Month, Day) + EncodeTime(Hour, 0, 0, 0)
                 else if (mMin >= 30) then
              result := EncodeDate(Year, Month, Day) + EncodeTime(Hour, 0, 0, 0)
               + encodetime(1,0,0,0) else
              result := EncodeDate(Year, Month, Day) + EncodeTime(Hour, 30, 0, 0);
   dt_hourly:if (Sec <> 0) or (mMin <> 0) then
              result := EncodeDate(Year, Month, Day) + EncodeTime(Hour, 0, 0, 0)
                  + encodetime(1,0,0,0) else
              result := EncodeDate(Year, Month, Day) + EncodeTime(Hour, 0, 0, 0);
   dt_daily:if (Hour <> 0) or (mMin <>0) or (Sec <> 0) then
              result := EncodeDate(Year, Month, Day) + 1 else
              result := EncodeDate(Year, Month, Day);
   dt_weekly:result := EncodeDate(Year, Month, Day) + 8 - DayOfWeek(oMin); {???}
   dt_monthly:begin
             if (Day > 1) or (Hour <> 0) or (mMin <>0) or (Sec <> 0) then
                incmonth(Year,Month,1); {month is not set time period}
             result := EnCodeDate(Year,Month, 1);
             end;
   dt_2monthly:begin
               if (Day > 1) or (Hour <> 0) or (mMin <>0) or (Sec <> 0) then
                 incmonth(Year,Month,1); {month is not set time period}
               if Month mod 2 <> 1 then inc(Month);
               result := EnCodeDate(Year,Month, 1);
               end;
   dt_quarterly:begin
               if (Day > 1) or (Hour <> 0) or (mMin <>0) or (Sec <> 0) then
                 incmonth(Year,Month,1); {month is not set time period}
               while Month mod 3 <> 1 do incmonth(Year,Month,1);
               result := EnCodeDate(Year,Month, 1);
               end;
   dt_annually:begin
               if (Month > 1) or(Day > 1) or (Hour <> 0) or
                     (mMin <>0) or (Sec <> 0) then
                result := EncodeDate(Year+1,1,1)
                  { year is also not a set time,  but there can't be overflow here }
               else result := EnCodeDate(Year,1,1);
               end;
   dt_decade:begin
               if (Month > 1) or(Day > 1) or (Hour <> 0) or
                     (mMin <>0) or (Sec <> 0) then  inc(Year);
               result := EncodeDate(10 * (Year div 10),1,1);
               end;
   dt_century:begin
               if (Month > 1) or(Day > 1) or (Hour <> 0) or
                     (mMin <>0) or (Sec <> 0) then  inc(Year);
               result := EncodeDate(100 * (Year div 100),1,1);
               end;
  end;
end;

function TAxis.GetFirstTick(var logTickInfo: TLogTickInfo): Double;
var
  t, B: Double;
  j: Word;
begin
  if not FLogging or logTickInfo.LogStepping then
  if FShowAsTime then result := getfirstdatetick else
  begin
    Result := FMin + 0.01*FStep;
    Result := trunc( Result / FStep ) * FStep;
    if (FMin < 0) then Result := Result - FStep;
    if (FMin > Result + 0.01*FStep) then Result := Result + FStep;
    AdjustLabelDecs;
  end
  else {logging}
  begin
    t := ln(FMin) * Loge;
    if t < 0 then t := t - 1;
    B := exp( trunc(1.001 * t) / Loge ); { OK for FMin < 10^1000 }
    if B > FMin then B := B/10;

    {pre-condition: the TLogTickInfo has been initialized with 1..10:}
    with logTickInfo do
    begin
      for j := 0 to LogTickCount+1 do
        LogTickArray[j] := B * LogTickArray[j];

      LogTickIndex := 0;
      t := FMin*0.999;
      while logTickInfo.LogTickArray[LogTickIndex] < t do
        Inc(LogTickIndex);
      Result := LogTickArray[LogTickIndex];
      AdjustLogLabelDecs(Result);
    end;
  end;
end {GetFirstTick};


function TAxis.GetNextDateTick(tick: Double):double;
var year,month,day:word;
begin
 case FDateTickType of
  dt_minute: result := tick + cminute1;
  dt_5minute: result := tick + cminute5;
  dt_halfhourly: result := tick + cminute30;
  dt_hourly: result := tick + chour;
  dt_daily: result := tick + 1;
  dt_weekly: result := tick + 7;
  dt_monthly:begin
             decodedate(tick,year,month,day);
             incmonth(year,month,1);
             result := encodedate(year, month, 1);
             end;
  dt_2monthly:begin
             decodedate(tick,year,month,day);
             incmonth(year,month,2);
             result := encodedate(year, month, 1);
             end;
  dt_quarterly:begin
             decodedate(tick,year,month,day);
             incmonth(year,month,3);
             result := encodedate(year, month, 1);
             end;
  dt_annually:begin
             decodedate(tick,year,month,day);
             result := encodedate(year+1, 1, 1);
             end;
  dt_decade:begin
             decodedate(tick,year,month,day);
             result := encodedate(year+10, 1, 1);
             end;
  dt_century:begin
             decodedate(tick,year,month,day);
             result := encodedate(year+100, 1, 1);
             end;
  dt_custom:Result := tick + FStep;
 end;
end;

function TAxis.GetNextTick(tick: Double; var logTickInfo: TLogTickInfo; var drawThisLabel: Boolean): Double;
var j: Word;
begin
  if not FLogging or logTickInfo.LogStepping then
    if FShowAsTime then
    begin
      result := getnextdateTick(tick);
      drawthislabel := true;
    end
    else
    begin
      Result := tick + FStep;
      drawThisLabel := true;
    end
  else {logging}
    with logTickInfo do
    begin
      Inc(LogTickIndex);
      if LogTickIndex >= LogTickCount + 1 then
      begin
        for j := 0 to LogTickCount + 1 do
          LogTickArray[j] := 10 * LogTickArray[j];
        LogTickIndex := 0;
      end;
      Result := LogTickArray[LogTickIndex + 1];
      if (LogTickIndex = 0) then
        AdjustLogLabelDecs(Result);
      drawThisLabel := DrawMinorLabels or
                       (LogTickIndex = 0) or
                       (LogTickIndex = LogTickCount+1);
    end {with, else};
end {GetNextTick};


function TAxis.CheckScale: Boolean;
begin
  Result := (FMin < FMax) and ((not FLogging) or (FMin > 0)) and
              not (FShowAsTime and (Fmin < 1))
end;


{--------------------------------------------------------------------
    #2b. TAxis - property servers
 --------------------------------------------------------------------}

procedure TAxis.SetAxisTitle(v: ShortString);
begin
  FAxisTitle := v;
  if FGraph.fplotting then FGraph.Paint;
end;

procedure TAxis.SetLabelDec(v: Integer);
begin
  FLabelDec := v;
  if FGraph.fplotting then FGraph.Paint;
end;

procedure TAxis.SetLogTickCount(v: Word);
begin
  FLogTickInfo.LogTickCount := v;
  InitLogTicks;
  if FGraph.fplotting then FGraph.Paint;
end;

procedure TAxis.setdateticktype;
begin
  if v <> FDateTickType then
    begin
    FDateTickType := v;
    if FGraph.fplotting then FGraph.Paint;
    end;
end;

function TAxis.GetLogTickCount: Word;
begin
  Result := FLogTickInfo.LogTickCount;
end;

procedure TAxis.SetLogging(v: Boolean);
begin
  FLogging := v;
  if FGraph.fplotting then FGraph.Paint;
end;

procedure TAxis.SetAutoSizing(v: Boolean);
begin
  FAutoSizing := v;
  if FGraph.fplotting then FGraph.Paint;
end;

procedure TAxis.SetAutoStepping(v: Boolean);
begin
  FAutoStepping := v;
  if FGraph.fplotting then FGraph.Paint;
end;

procedure TAxis.SetMax(v: Double);
begin
  FAutoSizing := False;
  if FMax <> v then
  begin
    FMax := v;
    if FShowASTime and not FGraph.datafound(self) then oMax := FMax;
    FGraph.DoRescaleEvent;
    if FGraph.fplotting then FGraph.Paint;
  end;
end;

procedure TAxis.SetMin(v: Double);
begin
  FAutoSizing := False;
  if FMin <> v then
  begin
    FMin := v;
    if FShowASTime and not FGraph.datafound(self) then oMin := FMin;
    FGraph.DoRescaleEvent;
    if FGraph.fplotting then FGraph.Paint;
  end;
end;

procedure TAxis.SetStep(v: Double);
begin
  FAutoStepping := False;
  if FStep <> v then
  begin
    FStep := v;
    FGraph.DoRescaleEvent;
    if FGraph.fplotting then FGraph.Paint;
  end;
end;

procedure TAxis.SetMinDiff(v: Double);
var holdmin: Double;
begin
  holdmin := FMinScale;
  FMinScale := v;
  if FGraph.Plotting and ((FMax - FMin < FMinScale) or (FMax - FMin = holdmin))
  then FGraph.Paint;
end;

procedure TAxis.setShowASTime;
begin
  if v <> FShowASTime then
  begin
    FShowAsTime := v;
    if FShowASTime and not FGraph.datafound(self) then
      begin
      oMin := FMin;
      oMax := FMax;
      end;
    if FGraph.Plotting then FGraph.paint;
  end;
end;

procedure TAxis.SetDateFormat;
begin
  if v <> FDateFormat then
    begin
    FDateFormat := v;
    if FGraph.Plotting then FGraph.Paint;
    end;
end;

procedure TAxis.SetShowAxis;
begin
  if v <> FShowAxis then
    begin
    FShowAxis := v;
    if FGraph.Plotting then FGraph.Paint;
    end;
end;

procedure TAxis.SetReversed;
begin
  if v <> FReversed then
    begin
    FReversed := v;
    if FGraph.Plotting then FGraph.Paint;
    end;
end;

procedure TAxis.setOffsetType;
begin
  if v <> FOffsetType then
    begin
    FOffsetType := v;
    if FGraph.Plotting then FGraph.Paint;
    end;
end;

procedure TAxis.setoffset;
begin
  if v <> FOffset then
    begin
    FOffset := v;
    if FGraph.Plotting then FGraph.Paint;
    end;
end;

procedure TAxis.SetGridlines(v: Boolean);
begin
  FGridlines := v;
  if FGraph.fplotting then FGraph.Paint;
end;

{-------------------------------------------------------------------
    #3. TDimensions
--------------------------------------------------------------------}

procedure TDimensions.SetMargBottom(v: Word);
begin
  FBottom := v;
  if FGraph.fplotting then FGraph.Paint;
end;

procedure TDimensions.SetMargTop(v: Word);
begin
  FTop := v;
  if FGraph.fplotting then FGraph.Paint;
end;

procedure TDimensions.SetMargLeft(v: Word);
begin
  FLeft := v;
  if FGraph.fplotting then FGraph.Paint;
end;

procedure TDimensions.SetMargRight(v: Word);
begin
  FRight := v;
  if FGraph.fplotting then FGraph.Paint;
end;

procedure TDimensions.SetTickLength(v: Word);
begin
  FTMLength := v;
  if v = 0 then FGraph.FAppearance.FTickMarks := False;
  if FGraph.fplotting then FGraph.Paint;
end;

procedure TDimensions.SetGraphTitleDistance(v: Integer);
begin
  FGraphTitleDistance := v;
  if FGraph.fplotting then FGraph.Paint;
end;

procedure TDimensions.SetXAxisTitleDistance(v: Integer);
begin
  FXAxisTitleDistance := v;
  if FGraph.fplotting then FGraph.Paint;
end;

procedure TDimensions.SetYAxisTitleDistance(v: Integer);
begin
  FYAxisTitleDistance := v;
  if FGraph.fplotting then FGraph.Paint;
end;

procedure TDimensions.SetXAxisLabelDistance(v: Integer);
begin
  FXAxisLabelDistance := v;
  if FGraph.fplotting then FGraph.Paint;
end;

procedure TDimensions.SetYAxisLabelDistance(v: Integer);
begin
  FYAxisLabelDistance := v;
  if FGraph.fplotting then FGraph.Paint;
end;

procedure TDimensions.SetScale(v: Word);
begin
  if v < 20 then v := 20 else if v > 100 then v := 100;
  FScalePct := v;
end;

procedure TDimensions.SetXOffset(v: Word);
begin
  if v < 0 then v := 0 else if v > 100 then v := 100;
  FXOffsetPct := v;
end;

procedure TDimensions.SetYOffset(v: Word);
begin
  if v < 0 then v := 0 else if v > 100 then v := 100;
  FYOffsetPct := v;
end;


{-------------------------------------------------------------------
    #4. TAppearance
---------------------------------------------------------------------}
procedure TAppearance.SetAllowedOut;
begin
  FAllowedOut := v;
  if FGraph.fplotting then FGraph.Paint;
end;

procedure TAppearance.SetLabelGraph(v: Boolean);
begin
  FLabelGraph := v;
  if FGraph.fplotting then FGraph.Paint;
end;

procedure TAppearance.SetTickmarks(v: Boolean);
begin
  FTickMarks := v;
  if FGraph.fplotting then FGraph.Paint;
end;

procedure TAppearance.SetShowMarks;
begin
  if v <> FShowMarks then
  begin
    FShowMarks := v;
    if FGraph.fplotting then FGraph.Paint;
  end;
end;

procedure TAppearance.SetGridcolor(v: TColor);
begin
  FGridColor := v;
  if FGraph.fplotting then FGraph.Paint;
end;

procedure TAppearance.SetAxescolor(v: TColor);
begin
  FAxesColor := v;
  if FGraph.fplotting then FGraph.Paint;
end;

procedure TAppearance.SetBkgdcolor(v: TColor);
begin
  FBkgdColor := v;
  if FGraph.fplotting then FGraph.Paint;
end;

procedure TAppearance.SetGridStyle(v: TPenStyle);
begin
  FGridStyle := v;
  if FGraph.fplotting then FGraph.Paint;
end;

procedure TAppearance.SetErrorCaption;
begin
  FErrorCaption := v;
  if not FGraph.CheckScales then if FGraph.fplotting then FGraph.Paint;
end;

procedure TAppearance.SetGraphTitle;
begin
  FGraphTitle := v;
  if FGraph.fplotting then FGraph.Paint;
end;

procedure TAppearance.SetMinSteps;
begin
  if (v > 0) and (v < FMaxSteps/2.5) then
    FMinSteps := v;
  if FGraph.fplotting then FGraph.Paint;
end;

procedure TAppearance.SetMaxSteps;
begin
  if (v > FMinSteps*2.5) and (v <= 90) then
    FMaxSteps := v;
  if FGraph.fplotting then FGraph.Paint;
end;

procedure TAppearance.setTitleFont(v:TFont);
begin
  FTitleFont.assign(v);
  if FGraph.fplotting then FGraph.Paint;
end;

procedure TAppearance.setLabelFont(v:TFont);
begin
  FLabelFont.assign(v);
  if FGraph.fplotting then FGraph.Paint;
end;

{procedure TLegend.setcolor(v:TColor);
begin
  FLegWin.Color := v;
end;

Function TLegend.GetColor:Tcolor;
begin
  result := FLegWin.Color;
end;

function TLegend.getvisible:boolean;
begin
  result := FLegWin.visible;
end;

procedure TLegend.setvisible;
begin
 if v <> FLegWin.visible then
   begin
   FLegWin.visible := v;
   FGraph.paint;
   end;
end;}

(*procedure TLegend{Win}.MouseDown;
const
  SC_DragMove = $f012;
  SC_SizeLR   = $f008;
begin
  if (Button = mbLeft) then
  begin
    if (width - x > 10) or (height - y > 10) then
    begin
      Screen.Cursor := crSize;
      ReleaseCapture;
      Perform(WM_SysCommand, SC_DragMove, 0);
      Screen.Cursor := crDefault;
    end
  else
    begin
      Screen.Cursor := crSizeNWSE;
      ReleaseCapture;
      Perform(WM_SysCommand, SC_SizeLR, 0);
      Screen.Cursor := crDefault;
    end;
  end;
  inherited MouseDown(Button, Shift, X, Y);
end {MouseDown};

  *)
{---------------------------------------------------------------------
    #5a. TxyGraph  - administration
 ---------------------------------------------------------------------}

constructor TxyGraph.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FCanvas := Canvas;
  FSeries := nil;
  FLastSeries := 0;
  FHookedSeries := nil;
  FMarks := nil;
  FFirstTime := True;
  FScale := 1.0;
  FXOffset := 0;
  FYOffset := 0;
  FDragging := false;
  FDraggable := true;
  FNoGridlines := false;

{property defaults}
  Width := 200;
  Height := 200;
  Font.Color := clBlack;
  Font.Height := - 9;
  Font.Name := 'Arial';
  Font.Style := [];
  BevelInner := bvNone;
  BevelOuter := bvNone;
  BorderStyle := bsSingle;
  Color := clWhite;

  FAllowDuplicates := False;
  FPlotting := True;

  FXAxis := TAxis.Create;
  FXAxis.FGraph := self;
  with FXAxis do
  begin
    SecondAxis := false;
    FShowAxis := true;
    Freversed := false;
    FMinScale := defMinXScale;
    FMin := 0;
    FMax := 1.0;
    FStep := FMax / 5;
    FLogTickInfo.LogTickCount := 2;
    InitLogTicks;
    FAutoSizing := False;
    FAutoStepping := False;
    FLogging := False;
    FAxisTitle := 'X axis';
    FGridlines := True;
    FLabelDec := 1;
    FShowASTime := false;
    FDateFormat := 'dd-mmm';
    FOffsetType := ao_Minimum;
    FOffset := 0;
  end;

  FYAxis := TAxis.Create;
  FYAxis.FGraph := self;
  with FYAxis do
  begin
    SecondAxis := false;
    FShowAxis := true;
    FReversed := false;
    FMinScale := defMinYScale;
    FMin := 0;
    FMax := 1.0;
    FStep := FMax / 5;
    FLogTickInfo.LogTickCount := 2;
    InitLogTicks;
    FAutoSizing := False;
    FAutoStepping := False;
    FLogging := False;
    FGridlines := True;
    FAxisTitle := 'Y axis';
    FLabelDec := 1;
    FShowASTime := false;
    FDateFormat := 'dd-mmm';
    FOffsetType := ao_Minimum;
    FOffset := 0;
  end;

  FYAxis2 := TAxis.Create;
  FYAxis2.FGraph := self;
  with FYAxis2 do
  begin
    SecondAxis := true;
    FShowAxis := false;
    FReversed := true;
    FMinScale := defMinYScale;
    FMin := 0;
    FMax := 1.0;
    FStep := FMax / 5;
    FLogTickInfo.LogTickCount := 2;
    InitLogTicks;
    FAutoSizing := False;
    FAutoStepping := False;
    FLogging := False;
    FGridlines := False;
    FAxisTitle := 'Second Y axis';
    FLabelDec := 1;
    FShowASTime := false;
    FDateFormat := 'dd-mmm';
    FOffsetType := ao_Maximum;
    FOffset := 0;
  end;

  FDimensions := TDimensions.Create;
  FDimensions.FGraph := self;
  with FDimensions do
  begin
    FBottom := 40;
    FLeft := 40;
    FRight := 15;
    FTop := 30;
    FTMLength := 4;
    FGraphTitleDistance := 7;
    FXAxisTitleDistance := 4;
    FXAxisLabelDistance := 2;
    FYAxisTitleDistance := 4;
    FYAxisLabelDistance := 2;
    FScalePct := 90;
    FXOffsetPct := 5;
    FYOffsetPct := 20;
  end;

  FAppearance := TAppearance.Create;
  FAppearance.FGraph := self;
  with FAppearance do
  begin
    FLabelGraph := True;
    FShowMarks := False;
    FTickMarks := True;
    FAllowedOut := False;
    FErrorCaption := 'MAX/MIN VALUES INCONSISTENT';
    FAxesColor := clBlack;
    FGridColor := clSilver;
    FGridStyle := psdot;
    FBkgdColor := color;
    FGraphTitle := 'xyGraph ' + versionstamp;
    FMinSteps := 5;
    FMaxSteps := 50;
    FTitleFont := TFont.Create;
    FTitleFont.assign(font);
    FTitleFont.OnChange := TitleFontChanged;
    FLabelFont := TFont.Create;
    FLabelFont.assign(font);
    FLabelFont.OnChange := LabelFontChanged;
  end;

 (*FLegend := TLegend.create(self);
 with FLegend do
   begin
{   FGraph := self;
   FLegWin := TLegendWin.Create(self);}
{   FLegWin.}Parent := self;
 {  FlegWin.ctl3d := false;
   FLegWin.BevelOuter := bvNone;
   FLegWin.Borderstyle := bsSingle;}
   end;*)

  {dragging: }
 FHasDragged := false;
 drect := rect(0,0,0,0);
end {Create};


destructor TxyGraph.Destroy;
var p, p1: TSeries;
    p2, p3: pHookedSeries;
    om: pMark;
begin
  FXAxis.Free;
  FYAxis.Free;
  FYAxis2.Free;
  FDimensions.free;
  FAppearance.free;
{  FLegend.free;}
{  FLegWin.free;}

  p := FSeries;
  while p <> nil do
    begin
    p1 := p;
    p := p.next;
    p1.free;
    end;

  p2 := FHookedSeries;
  while p2 <> nil do
    begin
    p3 := p2;
    p2 := p2^.next;
    p3^.id.hooks[self] := nil;
    dispose(p3);
    end;

  while FMarks <> nil do
  begin
    om := FMarks;
    FMarks := FMarks^.next;
    Dispose(om);
  end;
  inherited Destroy;
end;

{---------------------------------------------------------------------
    #5b. TxyGraph  - mouse routines
 ---------------------------------------------------------------------}

{general outline of responding to mouse events:
  if it's not the left mouse button, just call the inherited handler

  if the shift button is down, the user wants to change the plot area
  if the ctrl button is down, the user wants to run the designer
  if Draggable is set to true:
     if the mouse is in the top left corner, the user wants to move the graph
     if the mouse is in the bottom right corner, the user wants to resize the graph}

procedure TxyGraph.MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
const
  SC_DragMove = $f012;
  SC_SizeLR   = $f008;
begin
  if (Button = mbLeft) then
  begin
   {$IFDEF DESIGNER}
    if (ssCtrl in Shift) then rundesigner else
   {$ENDIF}
    if (ssShift in Shift) then setupdragging(x,y) else
    with FDimensions do
     {3. is mouse in bottom right corner? }
      if FDraggable and (Width - X < FRight) and (Height - Y < FBottom) then
      begin
        Screen.Cursor := crSizeNWSE;
        ReleaseCapture;
        Perform(WM_SysCommand, SC_SizeLR, 0);
        Screen.Cursor := crDefault;
      end
     {is mouse in top left corner? }
      else if FDraggable and (X < FLeft) and (Y < FTop) then
      begin
        Screen.Cursor := crSize;
        ReleaseCapture;
        Perform(WM_SysCommand, SC_DragMove, 0);
        Screen.Cursor := crDefault;
      end
  end;
  inherited MouseDown(Button, Shift, X, Y);
end {MouseDown};

procedure TxyGraph.SetupDragging;
begin
 fdragging := true;
 dx := x;
 dy := y;
 screen.cursor := crCross;
 if not FHasDragged then
   begin
   FHoldXAS := FXAxis.Autosizing;
   FHoldXmin := FXAxis.Min;
   FHoldXmax := FXAxis.Max;
   FHoldYAS := FYAxis.Autosizing;
   FHoldYmin := FYAxis.Min;
   FHoldYmax := FYAxis.Max;
   FHoldY2AS := FYAxis2.Autosizing;
   FHoldY2min := FYAxis2.Min;
   FHoldY2max := FYAxis2.Max;
   end;
 FHasDragged := true;
end;

procedure TxyGraph.MouseMove(Shift: TShiftState; X, Y: Integer);
begin
 if fdragging then
 begin
   FCanvas.DrawFocusRect(drect);
   if x > dx then
   begin
     drect.left := dx;
     drect.right := x;
   end
   else
   begin
     drect.left := x;
     drect.right := dx;
   end;
   if y > dy then
   begin
     drect.top := dy;
     drect.bottom := y;
   end
   else
   begin
     drect.top := y;
     drect.bottom := dy;
   end;
   FCanvas.DrawFocusRect(drect);
 end;
 inherited mousemove(shift, x, y);
end;

procedure TxyGraph.MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
var
  x1,x2,y1,y2,y3,y4: Double;
begin
  if Fdragging then
  begin
    drect := rect(0,0,0,0);
    releaseCapture;
    FDragging := False;
    screen.cursor := crDefault;
    if (abs (x - dx) > MouseToleranceMove) and (abs (y - dy) > MouseToleranceMove) then
    begin
      if x > dx then
      begin
        x1 := getmousex(dx);
        x2 := getmousex(x);
      end
      else
      begin
        x1 := getmousex(x);
        x2 := getmousex(dx);
      end;
      if y < dy then
      begin
        y1 := getmousey(dy);
        y2 := getmousey(y);
        y3 := getmousey2(dy);
        y4 := getmousey2(y);
      end
      else
      begin
        y1 := getmousey(y);
        y2 := getmousey(dy);
        y3 := getmousey2(y);
        y4 := getmousey2(dy);
      end;
      plotting := false;
      Fxaxis.min := x1;
      Fxaxis.max := x2;
      Fyaxis.min := y1;
      Fyaxis.max := y2;
      Fyaxis2.min := y3;
      FYAxis2.max := y4;
      plotting := true;
    end
    else
    begin
      FHasDragged := false;
      plotting := false; 
      FXaxis.autosizing := FHoldXAS;
      if not FHoldXAS then
      begin
        FXAxis.min := FHoldXMin;
        FXAxis.max := FHoldXMax;
      end;
      FYaxis.autosizing := FHoldYAS;
      if not FHoldYAS then
      begin
        FYAxis.min := FHoldYMin;
        FYAxis.max := FHoldYMax;
      end;
      FYaxis2.autosizing := FHoldY2AS;
      if not FHoldY2AS then
      begin
        FYAxis2.min := FHoldY2Min;
        FYAxis2.max := FHoldY2Max;
      end;
      plotting := true;
    end;
  end;
  inherited mouseup(button, shift, x, y);
end;

{---------------------- Reverse Metric Routines -------------------------------}

function TxyGraph.GetMouseXY(sN, x, y: Integer;
           var index: Integer;
           var xScaled, yScaled, xNearest, yNearest: Double): Boolean;
begin
  xScaled := GetMouseX(x);
  yScaled := GetMouseY(y);
  Result := GetMouseVals(sN, x, index, xNearest, yNearest);
end {GetMouseXY};

{ getmousex/y return the value as determined by the scales }
function TxyGraph.GetMouseX(x: Integer): Double;
begin
  with FXAxis do
    if FLogging then
      Result := FMin*Exp((x - FDimensions.FLeft) / FM)
    else
      Result := FMin + ((x - FDimensions.FLeft) / FM);
end;

function TxyGraph.GetMouseY(y: Integer): Double;
begin
with FYAxis do
  if FLogging then
    Result := FMax*Exp(-(y - FDimensions.FTop) / FM)
  else
    Result := FMax - ((y - FDimensions.FTop) / FM);
end;


function TxyGraph.GetMouseY2(y: Integer): Double;
begin
with FYAxis2 do
  if FLogging then
    Result := FMax*Exp(-(y - FDimensions.FTop) / FM)
  else
    Result := FMax - ((y - FDimensions.FTop) / FM);
end;


{ GetMouseVals returns the value of the data point matching the
mouse point best, or false if the mouse is outside the plot area }
function TxyGraph.GetMouseVals(sN, x: Integer; var index: Integer;
                               var xNearest, yNearest: Double): Boolean;
var
  tx: Double;
  i: Integer;
  p: TSeries;
  p1, p2: pgPoint;
begin
  tx := GetMouseX(x);
  p := Series[sN];
  Result := (p <> nil) and (p.FData <> nil) and
                       ((tx <= FXAxis.Max) and (tx >= FXAxis.Min));
  if Result then
  begin
    p1 := p.FData;
    i := 1;
    if tx <= p1^.xv then
      p2 := p1
    else
    begin
      while (p1^.next <> nil) and (tx > p1^.xv) do
      begin
        p2 := p1;
        p1 := p1^.next;
        Inc(i);
      end;
      if p1^.next = nil then
        p2 := p1
      else if tx = p1^.xv then
        p2 := p1
      else if tx > (p1^.xv + p2^.xv) / 2 then
        p2 := p1
      else Dec(i);
    end;
    index := p2^.i;
    xNearest := p2^.xv;
    yNearest := p2^.yv;
  end;
end {GetMouseVals};


{---------------------------------------------------------------------
    #5c. TxyGraph  - metrics routines
 ---------------------------------------------------------------------}

procedure TxyGraph.DoRescaleEvent;
begin
  if FPlotting and CheckScales and Assigned(FOnRescale) then
    FOnRescale(self);
end;

procedure TxyGraph.CalcXMetrics;
begin
  with FXAxis do
  begin
    if (FMin>=FMax) then raise Exception.Create('CalcXMetrics: XMin>=XMax');
    if FLogging then
      FM := (Width - FDimensions.FLeft - FDimensions.FRight) / (Ln(FMax) - Ln(FMin))
    else
      FM := (Width - FDimensions.FLeft - FDimensions.FRight) / (FMax - FMin);
  end;
end;

procedure TxyGraph.CalcYMetrics;
begin
  with FYAxis do
  begin
    if (FMin>=FMax) then raise Exception.Create('CalcYMetrics: YMin>=YMax');
    if FLogging then
      FM := (Height - FDimensions.FTop - FDimensions.FBottom) / (Ln(FMax) - Ln(FMin))
    else
      FM := (Height - FDimensions.FTop - FDimensions.FBottom) / (FMax - FMin);
  end;
end;

procedure TxyGraph.CalcY2Metrics;
begin
  with FYAxis2 do
  begin
    if (FMin>=FMax) then raise Exception.Create('CalcY2Metrics: YMin>=YMax');
    if FLogging then
      FM := (Height - FDimensions.FTop - FDimensions.FBottom) / (Ln(FMax) - Ln(FMin))
    else
      FM := (Height - FDimensions.FTop - FDimensions.FBottom) / (FMax - FMin);
  end;
end;

procedure TxyGraph.CalcMetrics;
begin
  CalcXMetrics;
  CalcYMetrics;
  CalcY2Metrics;
end;

function TxyGraph.fx(v: Double): Integer;
var w: Double;
begin
  with FXAxis do
    if FLogging then
      w := (Ln(v) - Ln(FMin)) * FM
    else
      w :=(v - FMin) * FM;
  if abs(w) > 20000 then
    Result := 20000
  else
    Result := Round(w) + FDimensions.FLeft + FXOffset;
{20000  is large compared with device pixel size; this avoids range
 error if some points are far outside the axis range}
end;

function TxyGraph.fy(v: Double): Integer;
var w: Double;
begin
  with FYAxis do
    if FLogging then
      w := (Ln(FMax) - Ln(v)) * FM
    else
      w := (FMax - v) * FM;
  if abs(w) > 20000 then
    Result := 20000
  else
    Result := Round(w) + FDimensions.FTop + FYOffset;
end;

function TxyGraph.fy2(v: Double): Integer;
var w: Double;
begin
  with FYAxis2 do
    if FLogging then
      w := (Ln(FMax) - Ln(v)) * FM
    else
      w := (FMax - v) * FM;
  if abs(w) > 20000 then
    Result := 20000
  else
    Result := Round(w) + FDimensions.FTop + FYOffset;
end;

{---------------------- graph scaling ------------------------------------}
procedure TxyGraph.DoTightFit;
var
  initx,inity,inity2: Boolean;
  p: TSeries;
begin
  p := FSeries;
  while p <> nil do
  begin
    if p.Changed then p.GetMinMax;
    p := p.next;
  end;

  initx := true;
  inity := true;
  inity2 := true;
  p := FSeries;
  while p <> nil do
  begin
    if p.FActive then
    begin
      if initx then
        begin
        FXAxis.oMin := p.xvMin;
        FXAxis.oMax := p.xvMax;
        initx := false;
        end
      else
        begin
        if p.xvMin < FXAxis.oMin then FXAxis.oMin := p.xvMin;
        if p.xvMax > FXAxis.oMax then FXAxis.oMax := p.xvMax;
        end;
      if p.FWhichYAxis = FYAxis then
        begin
        if inity then
          begin
          FYAxis.oMin := p.yvMin;
          FYAxis.oMax := p.yvMax;
          inity := false
          end
        else
          begin
          if p.yvMin < FYAxis.oMin then FYAxis.oMin := p.yvMin;
          if p.yvMax > FYAxis.oMax then FYAxis.oMax := p.yvMax;
          end;
        end
      else
        begin
        if inity2 then
          begin
          FYAxis2.oMin := p.yvMin;
          FYAxis2.oMax := p.yvMax;
          inity2 := false
          end
        else
          begin
          if p.yvMin < FYAxis2.oMin then FYAxis2.oMin := p.yvMin;
          if p.yvMax > FYAxis2.oMax then FYAxis2.oMax := p.yvMax;
          end;
        end
    end;
    p := p.next;
  end;
end {DoTightFit};

function TxyGraph.DoResizeX: Boolean;
begin
  Result := FXAxis.DoResize;
end;

function TxyGraph.DoResizeY: Boolean;
begin
  Result := FYAxis.DoResize;
end;

function TxyGraph.DoResizeY2: Boolean;
begin
  Result := FYAxis2.DoResize;
end;

function TxyGraph.DoResize: Boolean;
var b: Boolean;
begin
  b := DoResizeY;    {force evaluation of both functions - need side effects!}
  b := DoResizeY2 or b;
  Result := DoResizeX or b;
end;

{---------------------------------------------------------------------
    #5d. TxyGraph  - painting routines
 ---------------------------------------------------------------------}

procedure TxyGraph.ClipGraph;
var
  ClipRgn: HRgn;
begin
   ClipRgn := CreateRectRgn(FDimensions.FLeft + FXOffset,
                            FDimensions.FTop + FYOffset,
                            Width - FDimensions.FRight + 1 + FXOffset,
                            Height - FDimensions.FBottom + 1 + FYOffset);
   SelectClipRgn(FCanvas.Handle, ClipRgn);
   DeleteObject(ClipRgn);
end;

procedure TxyGraph.UnclipGraph;
var
  ClipRgn: HRgn;
begin
  { note for confused readers: the shortcut to unclip that is valid for
       screen metrics is not valid for printers :) }
   ClipRgn := CreateRectRgn(FXOffset,
                            FYOffset,
                            Width + 1 + FXOffset,
                            Height + 1 + FYOffset);
   SelectClipRgn(FCanvas.Handle, ClipRgn);
   DeleteObject(ClipRgn);
end;

procedure TxyGraph.DrawLineSegment(x1, y1, x2, y2: Integer);
begin
  FCanvas.MoveTo(x1, y1);
  FCanvas.LineTo(x2, y2);
end;

procedure TxyGraph.DrawXGridlines;
var
  tick, maxTick: Double;
  tx1, ty1, ty2: Word;
  b: Boolean;
  tempLogTickInfo : TLogTickInfo;
begin
  FCanvas.Pen.Color := FAppearance.FGridColor;
  FCanvas.Pen.Style := FAppearance.FGridStyle;
  FCanvas.Brush.Color := Color;
  with FXAxis do
    maxTick := FMax + 0.001*(FMax-FMin); { rounding errors might exclude last point }
  with FYAxis do
  begin
    ty1 := fy(FMin);
    ty2 := fy(FMax);
  end;
  with FXAxis do
  begin
    if FLogging then tempLogTickInfo := FLogTickInfo;
    tick := GetFirstTick(tempLogTickInfo);
   {don't draw the first grid line on top of axis:}
    if tick < FMin + 0.01*FStep then
      tick := GetNextTick(tick, tempLogTickInfo, b);
    while tick < maxTick do
    begin
      tx1 := fx(tick);
      DrawLineSegment(tx1, ty1, tx1, ty2);
      tick := GetNextTick(tick, tempLogTickInfo, b);
    end;
  end;
end {DrawXGridlines};

procedure TxyGraph.DrawYGridlines;
var
  tick, maxTick: Double;
  tx1, tx2, ty1: Word;
  b: Boolean;
  tempLogTickInfo : TLogTickInfo;
  obc:TColor;
  wyax:boolean;
begin
  wyax := (cYAxis = FYAxis);
  FCanvas.Pen.Color := FAppearance.FGridColor;
  FCanvas.Pen.Style := FAppearance.FGridStyle;
  obc := SetBkColor(FCanvas.handle, FAppearance.FBkgdColor);
  with cYAxis do
    maxTick := FMax + 0.001*(FMax-FMin); { rounding errors might exclude last point }
  with FXAxis do
  begin
    tx1 := fx(FMin);
    tx2 := fx(FMax);
  end;
  with cYAxis do
  begin
    if FLogging then tempLogTickInfo := FLogTickInfo;
    tick := GetFirstTick(tempLogTickInfo);
   {don't draw the first grid line on top of axis:}
    if tick < FMin + 0.01*FStep then
      tick := GetNextTick(tick, tempLogTickInfo, b);
    while tick < maxTick do
    begin
      if wyax then ty1 := fy(tick) else ty1 := fy2(tick);
      DrawLineSegment(tx1, ty1, tx2, ty1);
      tick := GetNextTick(tick, tempLogTickInfo, b);
    end;
  end;
  SetBkColor(FCanvas.handle, obc);
end {DrawYGridlines};

procedure TxyGraph.DrawXTickMarks;
var
  tick, maxTick: Double;
  tx, ty1, ty2: {Word} longint;
  b: Boolean;
  tempLogTickInfo : TLogTickInfo;
  obc:TColor;
begin
  FCanvas.Pen.Color := FAppearance.FAxesColor;
  FCanvas.Pen.Style := psSolid;
  obc := SetBkColor(FCanvas.handle, FAppearance.FBkgdColor);
  with FXAxis do
    maxTick := FMax + 0.001*(FMax-FMin);
  ty1 := base;
  if reverse then
    ty2 := ty1 - FDimensions.FTMLength
  else
    ty2 := ty1 + FDimensions.FTMLength;
  with FXAxis do
  begin
    if FLogging then tempLogTickInfo := FLogTickInfo;
    tick := GetFirstTick(tempLogTickInfo);
    while tick < maxTick do
    begin
      tx := fx(tick);
      DrawLineSegment(tx, ty1, tx, ty2);
      tick := GetNextTick(tick, tempLogTickInfo, b);
    end;
  end;
  SetBkColor(FCanvas.handle, obc);
end {DrawXTickMarks};

procedure TxyGraph.DrawYTickMarks;
var
  tick, maxTick: Double;
  tx1, tx2, ty: {Word} longint;
  b, wyax: Boolean;
  tempLogTickInfo : TLogTickInfo;
begin
  wyax := (cYAxis = FYAxis);
  FCanvas.Pen.Color := FAppearance.FAxesColor;
  FCanvas.Pen.Style := psSolid;
  with cYAxis do
    maxTick := FMax + 0.001*(FMax-FMin);
  tx1 := base;
  if reverse then
    tx2 := tx1 + FDimensions.FTMLength
  else
    tx2 := tx1 - FDimensions.FTMLength;
  with cYAxis do
  begin
    if FLogging then tempLogTickInfo := FLogTickInfo;
    tick := GetFirstTick(tempLogTickInfo);
    while tick < maxTick do
    begin
      if wyax then ty := fy(tick) else ty := fy2(tick);
      DrawLineSegment(tx1, ty, tx2, ty);
      tick := GetNextTick(tick, tempLogTickInfo, b);
    end;
  end;
end {DrawYTickMarks};

function  TAxis.LabelString(tick: Double): ShortString;
begin
  if not FShowAsTime then
    if (abs(tick) < (FMax - FMin)*0.001 {zero}) or
        ((abs(tick) > 0.000999) and (abs(tick) < 9999)) then
      Result := FloatToStrF(tick, ffFixed, 5, FLabelDec)
    else {very small or very large}
      Result := FloatToStrF(tick, ffExponent, 2, 0)
  else
   result := FormatDateTime(FDateFormat,tick);
end;

procedure TxyGraph.DrawXLabels;
var
  tick, maxTick: Double;
  tx, ty: {Word} longint;
  lblStr: ShortString;
  drawIt: Boolean;
  tempLogTickInfo : TLogTickInfo;
begin
{ X-axis labels }
  FCanvas.Font := FAppearance.FLabelFont;
  with FXAxis do
    maxTick := FMax + 0.001*(FMax-FMin);   { rounding errors might exclude last point }
  if reverse then
    ty := base - (FDimensions.FTMLength + FDimensions.FXAxisLabelDistance)
  else
    ty := base + FDimensions.FTMLength + FDimensions.FXAxisLabelDistance;
  with FXAxis do
  begin
    if FLogging then tempLogTickInfo := FLogTickInfo;
    tick := GetFirstTick(tempLogTickInfo);
    drawIt := true;
    while tick < maxTick do
    begin
      lblStr := LabelString(tick);
      if drawIt or (maxTick - tick < FStep) {ie, last one} then
        if reversed then
          FCanvas.TextOut(fx(tick) - FCanvas.TextWidth(lblStr) div 2,
                        ty - FCanvas.TextHeight(lblstr),
                        lblStr)
        else
          FCanvas.TextOut(fx(tick) - FCanvas.TextWidth(lblStr) div 2,
                        ty,
                        lblStr);
      tick := GetNextTick(tick, tempLogTickInfo, drawIt);
    end;
  end;
{X-axis title}
  FCanvas.Font := FAppearance.FTitleFont;
  with FDimensions do
  begin
    tx := FLeft + (Width - FLeft - FRight) div 2 + FXOffset;
    if reverse then
      ty := base - (FTMLength + FXAxisTitleDistance)
               - FCanvas.TextHeight(FXAxis.Title) * 2
    else
      ty := base + FTMLength + FXAxisTitleDistance + FCanvas.TextHeight(FXAxis.Title);
    end;
  FCanvas.TextOut(tx - FCanvas.TextWidth(FXAxis.Title) div 2,
                  ty, FXAxis.Title);
end {DrawXLabels};

procedure TxyGraph.DrawYLabels;
var
  tick, maxTick: Double;
  tx, ty: Integer;
  lblStr: ShortString;
  Angle:integer;
  LogRec: TLOGFONT;              { Storage area for font information }
  OldFont,
  NewFont: HFONT;
  drawIt, wyax: Boolean;
  tempLogTickInfo : TLogTickInfo;
begin
{ Y-axis Labels }
  FCanvas.Font := FAppearance.FLabelFont;
  wyax := (cYAxis = FYAxis);
  with cYAxis do
    maxTick := FMax + 0.001*(FMax-FMin);   { rounding errors might exclude last point }
  if reverse then
    tx := base + (FDimensions.FTMLength + FDimensions.FYAxisLabelDistance)
  else
    tx := base - (FDimensions.FTMLength + FDimensions.FYAxisLabelDistance);
  with cYAxis do
  begin
    if FLogging then tempLogTickInfo := FLogTickInfo;
    tick := GetFirstTick(tempLogTickInfo);
    drawIt := true;
    while tick < maxTick do
    begin
      lblStr := LabelString(tick);
      if wyax then ty := fy(tick) else ty := fy2(tick);
      if drawIt or (maxTick - tick < FStep) then
        if reverse then
          FCanvas.TextOut(tx,
                        ty - FCanvas.TextHeight(lblStr) div 2,
                        lblStr)
        else
          FCanvas.TextOut(tx - FCanvas.TextWidth(lblStr),
                        ty - FCanvas.TextHeight(lblStr) div 2,
                        lblStr);
      tick := GetNextTick(tick, tempLogTickInfo, drawIt);
    end;
  end;
{Y-axis title}
  FCanvas.Font := FAppearance.FTitleFont;
  Angle := 90;
  { Get the current font information. We only want to modify the angle }
  GetObject(FCanvas.Font.Handle, SizeOf(LogRec), @LogRec);
  LogRec.lfEscapement := Angle * 10;
  LogRec.lfOrientation := Angle * 10; {see win32 api?}
  LogRec.lfOutPrecision := OUT_TT_ONLY_PRECIS;
       {doesn't seem to work in win95?? not tested in NT. Use a truetype font for
        the Titlefont property instead for vertical text}
  NewFont := CreateFontIndirect(LogRec);
  OldFont := SelectObject(FCanvas.Handle, NewFont); {Save old font}
  with FDimensions do
  begin
    if reverse then
      tx := base + (FTMLength + FYAxisTitleDistance)
    else
      tx := base - (FTMLength + FYAxisTitleDistance);
    ty := FTop + (Height - FTop - FBottom) div 2 + FYOffset;
  end;
  with FCanvas do
  begin
    if reverse then
      tx := tx + TextWidth(lblStr)
    else
      tx := tx - TextWidth(lblStr) - TextHeight(cYAxis.Title);
    if tx < 0 then tx := 0;
    TextOut(tx, ty + (TextWidth(cYAxis.Title) div 2), cYAxis.Title);
  end;
  NewFont := SelectObject(FCanvas.Handle, OldFont); {Restore oldfont}
  DeleteObject(NewFont);
end {DrawYLabels};

procedure TxyGraph.DrawGraphTitle;
var
  tx: Word;
begin
  FCanvas.Font := Font;
  with FDimensions, FAppearance do
  begin
    tx := FLeft + (Width - FLeft - FRight) div 2 + FXOffset;
    with FCanvas do
      TextOut(tx - TextWidth(FGraphTitle) div 2,
              FTop - FGraphTitleDistance - TextHeight(FGraphTitle) + FYOffset,
              FGraphTitle);
  end;
end {DrawGraphTitle};

procedure TXyGraph.DrawMarks;

  function inrange(v,min,max:double):boolean;
  begin
   result := (v >= min) and (v <= max);
  end;

  procedure MarkFontOut(x,y:integer;angle:double;
               s:ShortString; offset:integer; color:TColor);
  var LogRec  : TLOGFONT;     {* Storage area for font information *}
      OldFont,
      NewFont: HFONT;
      xoffst,yoffst:integer;
      c:Tcolor;
  begin
    { Get the current font information. We only want to modify the angle }
    c := FCanvas.Font.Color;
    FCanvas.Font.Color := color;
    GetObject(Font.Handle,SizeOf(LogRec),@LogRec);
    LogRec.lfEscapement := trunc(angle *10 * 180 / pi);
    LogRec.lfOrientation := trunc(angle *10 * 180 / pi);    {see win32 api?}
    LogRec.lfOutPrecision := OUT_TT_ONLY_PRECIS;  {doesn't work under win95??}
    NewFont := CreateFontIndirect(LogRec);
    OldFont := SelectObject(Canvas.Handle,NewFont);  {Save old font}
  { offsets }
    xoffst := Round(FScale*(sin(angle) * offset));
    yoffst := Round(FScale*(cos(angle) * offset));
    FCanvas.TextOut(x - xoffst, y - yoffst, s);
    NewFont := SelectObject(Canvas.Handle,OldFont); {Restore oldfont}
    DeleteObject(NewFont);
    FCanvas.font.color := c;
  end {markFontOut};

  procedure drawxmark(p:Pmark);
  begin
    Fcanvas.pen.color := p^.color;
    DrawLineSegment(fx(p^.x1), fy(FYAxis.FMin), fx(p^.x1), fy(FYAxis.FMax));
    MarkFontOut(fx(p^.x1),fy(FYAxis.FMin) - 3, pi/2, p^.caption, FCanvas.font.size + 5,
                p^.color);
  end;

  procedure drawymark(p:Pmark);
  var c:Tcolor;
  begin
    Fcanvas.pen.color := p^.color;
    DrawLineSegment(fx(FXAxis.Fmin), fy(P^.y1), fx(FXAxis.Fmax), fy(p^.y1));
    c := FCanvas.font.color;
    FCanvas.font.color := p^.color;
    FCanvas.TextOut(FDimensions.FLeft + Round(FScale*10) + FXOffset,
                    fy(p^.y1) - Round(FScale*2) - FCanvas.TextHeight(p^.caption),
                    p^.caption);
    FCanvas.font.color := c;
  end;

  procedure drawboxmark(p:Pmark);
  var d:integer;
      c:Tcolor;
  begin
    d := trunc(FScale*p^.x2);
    FCanvas.pen.color := p^.color;
    FCanvas.brush.style := bsclear;
    FCanvas.rectangle(fx(p^.x1) - d, fy(p^.y1) - d,
                      fx(p^.x1) + d, fy(p^.y1) + d);
    c := FCanvas.font.color;
    FCanvas.font.color := p^.color;
    FCanvas.TextOut(fx(p^.x1) - d, fy(p^.y1) + d, p^.caption);
    FCanvas.font.color := c;
  end;

  procedure drawLineMark(p:Pmark);

    procedure swap(var n1,n2:integer);
    var t:integer;
    begin
      t := n1;
      n1 := n2;
      n2 := t;
    end;

  var minx,miny,maxy,maxx:integer;
      angle:double;
  begin {drawLineMark}
    FCanvas.pen.color := p^.color;
    minx := fx(p^.x1);
    miny := fy(p^.y1);
    maxx := fx(p^.x2);
    maxy := fy(p^.y2);
    if maxx < minx then
    begin
      swap(minx,maxx);
      swap(miny,maxy);
    end;
    DrawLineSegment(minx, miny, maxx, maxy);
    if maxx - minx = 0 then
      angle := pi/2
    else
      angle := - arctan((maxy-miny)/(maxx-minx));
      MarkFontOut(minx,miny,angle,p^.caption, FCanvas.font.size + 5, p^.color);
  end {drawLineMark};

var p: pmark;
begin {DrawMarks}
  if not FAppearance.FAllowedOut then
    ClipGraph;
  p := fmarks;
  while p <> nil do
  begin
    if p^.status = when then
      case p^.marktype of
      mtXMark: drawxmark(p);
      mtYMark: drawymark(p);
      mtPoint: drawboxmark(p);
      mtLine : drawLineMark(p);
      end;
    p := p^.next;
  end;
 if not FAppearance.FAllowedOut then
   UnclipGraph;
end {DrawMarks};


procedure TxyGraph.ColorBackground;
begin
  if (FAppearance.FBkgdColor <> Color) then
  begin
    FCanvas.brush.color := FAppearance.FBkgdColor;
    FCanvas.brush.style := bsSolid;
    FCanvas.pen.color := FAppearance.FBkgdColor;
    FCanvas.pen.style := psSolid;
    FCanvas.rectangle(fx(FXAxis.FMin), fy(FYAxis.FMax),
                     fx(FXAxis.FMax), fy(FYAxis.FMin));
    FCanvas.brush.color := Color;
  end;
end {ColorBackground};


procedure TxyGraph.DrawXAxis;
var
  holdXStep: Double;
  ypoint:integer;
begin
  with FAppearance, FXAxis do
  begin
    holdXStep := FStep;
    {if the number of steps is ridiculously large, do something reasonable:}
    if FShowAsTime then
      begin
       FStep := GetDatestep;
       while ((FMax - FMin) / FStep > FMaxSteps) do
        begin
        inc(FDateTickType); {this can't finish up in an infinite loop
           because the last datetick type will always terminate the loop}
        FStep := GetDatestep;
        SetDateMinMax;
        FGraph.Calcmetrics;
        end;
      end else
    if not FLogging and ((FStep = 0) or ((FMax - FMin) / FStep > FMaxSteps)) then
      FStep := GetStep;
    if FLogging then InitLogTicks;
    if not FNoGridlines and FGridlines then DrawXGridlines;
    if not FShowAxis then exit;
    case FOffsetType of
       ao_Minimum:ypoint := fy(FYAxis.FMin);
       ao_Maximum:ypoint := fy(FYAxis.FMax);
       ao_percentage:ypoint := fy(FYAxis.FMin +
                  (FOffset / 100 * (FYAxis.FMax - FYAxis.FMin)));
       ao_absolute:ypoint := fy(FOffset);
    end;
    if FTickMarks then DrawXTickMarks(ypoint,Freversed);
    if FLabelGraph then DrawXLabels(ypoint,Freversed);
    FStep := holdXStep;
    FCanvas.Pen.Color := FAxesColor;
    FCanvas.Pen.Style := psSolid;
    DrawLineSegment(fx(FXAxis.FMin), ypoint, fx(FXAxis.FMax), ypoint);
  end;
end {DrawXAxis};

procedure TxyGraph.DrawYAxis;
var holdYStep: Double;
    xpoint:integer;
begin
  with FAppearance, FYAxis do
  begin
    holdYStep := FStep;
    if FShowAsTime then
      begin
       FStep := GetDatestep;
       while ((FMax - FMin) / FStep > FMaxSteps) do
        begin
        inc(FDateTickType);  {this can't finsh up in an infinite loop
           because the last datetick type will always terminate the loop}
        FStep := GetDatestep;
        SetDateMinMax;
        FGraph.Calcmetrics;
        end;
      end else
    if not FLogging and ((FStep = 0) or ((FMax - FMin) / FStep > FMaxSteps)) then
      FStep := GetStep;
    if FLogging then FYAxis.InitLogTicks;
    if not FNoGridlines and FGridlines then DrawYGridlines(FYAxis);
    if not FShowAxis then exit;
    case FOffsetType of
       ao_Minimum:xpoint := fx(FXAxis.FMin);
       ao_Maximum:xpoint := fx(FXAxis.FMax);
       ao_percentage:xpoint := fx(FXAxis.FMin +
                  (FOffset / 100 * (FXAxis.FMax - FXAxis.FMin)));
       ao_absolute:xpoint := fx(FOffset);
    end;
    if FTickMarks then DrawYTickMarks(FYAxis, xpoint,Freversed);
    if FLabelGraph then DrawYLabels(FYAxis, xpoint,Freversed);
    FStep := holdYStep;
    FCanvas.Pen.Color := FAxesColor;
    FCanvas.Pen.Style := psSolid;
    DrawLineSegment(xpoint, fy(FYAxis.FMax), xpoint, fy(FYAxis.FMin));
  end;
end {DrawYAxis};

procedure TxyGraph.DrawY2Axis;
var holdYStep: Double;
    xpoint:integer;
begin
  with FAppearance, FYAxis2 do
  begin
    holdYStep := FStep;
    if FShowAsTime then
      begin
       FStep := GetDatestep;
       while ((FMax - FMin) / FStep > FMaxSteps) do
        begin
        inc(FDateTickType);  {this can't finsh up in an infinite loop
           because the last datetick type will always terminate the loop}
        FStep := GetDatestep;
        SetDateMinMax;
        FGraph.Calcmetrics;
        end;
      end else
    if not FLogging and ((FStep = 0) or ((FMax - FMin) / FStep > FMaxSteps)) then
      FStep := GetStep;
    if FLogging then FYAxis2.InitLogTicks;
    if not FNoGridlines and FGridlines then
       DrawYGridlines(FYAxis2);
    if not FShowAxis then exit;
    case FOffsetType of
       ao_Minimum:xpoint := fx(FXAxis.FMin);
       ao_Maximum:xpoint := fx(FXAxis.FMax);
       ao_percentage:xpoint := fx(FXAxis.FMin +
                  (FOffset / 100 * (FXAxis.FMax - FXAxis.FMin)));
       ao_absolute:xpoint := fx(FOffset);
    end;
    if FTickMarks then DrawYTickMarks(FYAxis2, xpoint,Freversed);
    if FLabelGraph then DrawYLabels(FYAxis2, xpoint,Freversed);
    FStep := holdYStep;
    FCanvas.Pen.Color := FAxesColor;
    FCanvas.Pen.Style := psSolid;
    DrawLineSegment(xpoint, fy(FYAxis.FMax), xpoint, fy(FYAxis.FMin));
  end;
end {DrawYAxis};

procedure TxyGraph.DrawAxes;
begin
  if (FCanvas <> Printer.Canvas) or FAppearance.FBkgdWhenPrint then
    ColorBackground;
  DrawXAxis;
  DrawYAxis;
  DrawY2Axis;
  if FAppearance.FLabelGraph then DrawGraphTitle;
end;

{$IFDEF STATISTICS}
procedure TxyGraph.DrawRegression;
begin
 t.drawmyregression;
end;
{$ENDIF}{STATISTICS}

procedure TxyGraph.DrawLine;
begin
  t.DrawMyLine;
end;

procedure TxyGraph.DrawPoints(t: TSeries);
begin
  t.DrawMyPoints;
end;

{$IFDEF BACKWARD}
procedure TxyGraph.DrawSeries(t: TSeries);
begin
  t.Draw;
end;
{$ENDIF}{BACKWARD}

procedure TxyGraph.DrawFunction(F: PlotFunction; var parms;
                x1, x2: Double;
                color: TColor; style: TPenStyle; steps: Word);
var
  x, step: Double;
  j: Word;
  started: Boolean;
  holdColor: TColor;
  holdStyle: TPenStyle;
begin
  holdColor := FCanvas.Pen.Color;
  holdStyle := FCanvas.Pen.Style;
  FCanvas.Pen.Color := color;
  FCanvas.Pen.Style := style;
  with FXAxis do
    if x1 = x2 then
    begin
      x1 := FMin;
      x2 := FMax;
    end
    else
    begin
      if x1 < FMin then
        x1 := FMin;
      if x2 > FMax then
        x2 := FMax;
    end;
  if (steps <= 0) or (steps > 3500){best likely printer resolution?} then
  begin
  {this can be made more intelligent, but this is usually smooth enough:}
    step := (x2 - x1) / 100;
    steps := 100;
  end
  else
    step := (x2 - x1) / steps;

  x := x1;
  started := false;
  j := 0;
  try
    if not FAppearance.FAllowedOut then
      ClipGraph;
    while j < steps do begin
      while (not started) and (j < steps) do
        try
          FCanvas.MoveTo( fx(x), fy(F(x, parms)) );
          started := true;
        except
          x := x + step;
          Inc(j);
        end;
      while (started) and (j < steps) do
      try
        x := x + step;
        FCanvas.LineTo( fx(x), fy(F(x, parms)) );
        Inc(j);
      except
        Started := false; {skip out-of-support x's}
      end;
    end;
  finally
    FCanvas.Pen.Color := holdColor;
    FCanvas.Pen.Style := holdStyle;
    if not FAppearance.FAllowedOut then
      UnclipGraph;
  end;
end;

function TxyGraph.CheckScales: Boolean;
begin
  Result := FXAxis.CheckScale and FYAxis.CheckScale and FYAxis2.CheckScale;
end;

procedure TxyGraph.PaintGraph;
var
  p: TSeries;
begin
  CalcMetrics;
  DrawAxes;
  if not FPlotting then Exit;

  {Printing: linewidth for series, points thinner than for axes}
  if FCanvas.Pen.Width >= 3 then
    FCanvas.Pen.Width := (FCanvas.Pen.Width + 1) div 2;
  if FAppearance.FShowMarks then DrawMarks(drawbefore);
  p := FSeries;
  while p <> nil do
  begin
    p.Draw;
    p := p.next;
  end;
  if FAppearance.FShowMarks then DrawMarks(drawafter);
end;


procedure TxyGraph.Paint;
var
  XOK, YOK, Y2OK, xResized, yResized, y2Resized: Boolean;
  holdMin, holdMax: Double;
begin
  inherited Paint;

  if FFirstTime then
  begin
    FFirstTime := False;
    Caption := Name;
    FAppearance.FCaption := Caption;
  end;

  if DataFound(nil) then
    DoTightFit;

  xResized := false;
  yResized := false;
  y2Resized := false;
  XOK := FXAxis.CheckScale;
  YOK := FYAxis.CheckScale;
  Y2OK := FYAxis2.Checkscale;
  with FAppearance do
  begin
    with FXAxis do
    begin
      if FAutoSizing then
      begin
        xResized := DoResizeX;
        XOK := CheckScale;
      end
      else if XOK and FAutoStepping then
      begin
        if not FLogging then
          FStep := GetStep
        else
        begin
          AdjustLogTickCount;
          CheckDrawMinorLabels;
        end;
      end;
    end;
    with FYAxis do
    begin
      if FAutoSizing then
      begin
        yResized := DoResizeY;
        YOK := CheckScale;
      end
      else if YOK and FAutoStepping then
      begin
        if not FLogging then
          FStep := GetStep
        else
        begin
          AdjustLogTickCount;
          CheckDrawMinorLabels;
        end;
      end;
    end;
    with FYAxis2 do
    begin
      if FAutoSizing then
      begin
        y2Resized := DoResizeY2;
        Y2OK := CheckScale;
      end
      else if Y2OK and FAutoStepping then
      begin
        if not FLogging then
          FStep := GetStep
        else
        begin
          AdjustLogTickCount;
          CheckDrawMinorLabels;
        end;
      end;
    end;

    YOK := YOK and Y2OK;
    if XOK and YOK then
    begin
      if xResized or yResized or y2resized then
        DoRescaleEvent;
      if (csDesigning in ComponentState) then
        Caption := FCaption
      else
        Caption := '';
      PaintGraph;
      If (csDesigning in ComponentState) and Assigned(FOnPaintEnd) then
            FOnPaintEnd(self, FCanvas);
    end
    else  {at least one axis is bad:}
    begin
      FNoGridLines := true;
      DrawGraphTitle;
      if not XOK and not YOK then
        Caption := 'BOTH AXES: ' + FErrorCaption
      else if XOK then
      begin
        CalcXMetrics;
       { set arbitrary FMin,FMax values to allow CalcYMetrics
         and fy(FMin,FMax) in x-axis ticks, labels:}
        with FYAxis do
        begin
          holdMin := FMin;
          holdMax := FMax;
          FMin := 1;
          FMax := 10;
          CalcYMetrics; {needed to place axis at correct "x=0" - not at pixel 0}
          DrawXAxis;
          FMin := holdMin;
          FMax := holdMax;
        end;
        Caption := 'Y-AXES: ' + FErrorCaption;
      end
      else {YOK}
      begin
        CalcYMetrics;
       { set arbitrary FXMin,Max values to allow fx(FXMin,Max)
         in y-axis ticks, labels:}
        with FXAxis do
        begin
          holdMin := FMin;
          holdMax := FMax;
          FMin := 1;
          FMax := 10; {don't need CalcXMetrics, because y-axis is at pixel 0}
          DrawYAxis;
          FMin := holdMin;
          FMax := holdMax;
        end;
        Caption := 'X-AXIS: ' + FErrorCaption;
      end;
      FNoGridLines := false;
    end;
  end;
end {Paint};


procedure TxyGraph.Print;
begin
  if not CheckScales then Exit;
  Printer.BeginDoc;
  PrintOnPage;
  Printer.EndDoc;
end;

procedure TxyGraph.PrintOnPage;
var
  scaleX, scaleY: Double;
  saveWidth, saveHeight,
  saveMargLeft, saveMargRight, saveMargTop, saveMargBottom,
  saveTMLength, saveXLabelDistance, saveYLabelDistance,
  saveXAxisDistance, saveYAxisDistance, saveGraphTitleDistance: Integer;
  videoFontColor: TColor;
  videoFontSize: Integer;
  videoFontName: ShortString;
  videoFontStyles: TFontStyles;
  videoPenWidth: Integer;
  VertPixPerInch: Integer;
begin
  if not CheckScales then Exit;
  saveWidth := Width;
  saveHeight := Height;
  with FDimensions do
  begin
    saveMargLeft := FLeft;
    saveMargRight := FRight;
    saveMargTop := FTop;
    saveMargBottom := FBottom;
    saveTMLength := FTMLength;
    saveXLabelDistance := FXAxisLabelDistance;
    saveYLabelDistance := FYAxisLabelDistance;
    saveXAxisDistance := FXAxisTitleDistance;
    saveYAxisDistance := FYAxisTitleDistance;
    saveGraphTitleDistance := FGraphTitleDistance;
  end;
  videoFontName := Font.Name;
  videoFontColor := Font.Color;
  videoFontStyles := Font.Style;
  videoFontSize := Font.Size;
  videoPenWidth := FCanvas.Pen.Width;
 {scale & offsets}
  scaleX := Printer.PageWidth / ClientWidth;
  scaleY := Printer.PageHeight / ClientHeight;
  with FDimensions do
  begin
    if scaleX < scaleY then
    begin
      FScale := FScalePct/100 * scaleX;
      if FScalePct + FXOffsetPct > 100 then
        FXOffsetPct := (100 - FScalePct) div 2;
      if FScalePct + FYOffsetPct > 100*(scaleY/scaleX) then
        FYOffsetPct := (Round(100*(scaleY/scaleX)) - FScalePct) div 2;
    end
    else
    begin
      FScale := FScalePct * scaleY / 100;
      if FScalePct + FYOffsetPct > 100 then
        FYOffsetPct := (100 - FScalePct) div 2;
      if FScalePct + FXOffsetPct > 100*(scaleX/scaleY) then
        FXOffsetPct := (Round(100*(scaleX/scaleY)) - FScalePct) div 2;
    end;
    FXOffset := Round(FXOffsetPct * (Printer.PageWidth / 100));
    FYOffset := Round(FYOffsetPct * (Printer.PageHeight / 100));
  end;
  Width := Round(FScale*Width);
  Height := Round(FScale*Height);
  with FDimensions do
  begin
    FLeft := Round(FScale*FLeft);
    FRight := Round(FScale*FRight);
    FTop := Round(FScale*FTop);
    FBottom := Round(FScale*FBottom);
    FTMLength := Round(FScale*FTMLength);
    FXAxisLabelDistance := Round(FScale*FXAxisLabelDistance);
    FYAxisLabelDistance := Round(FScale*FYAxisLabelDistance);
    FXAxisTitleDistance := Round(FScale*FXAxisTitleDistance);
    FYAxisTitleDistance := Round(FScale*FYAxisTitleDistance);
    FGraphTitleDistance := Round(FScale*FGraphTitleDistance);
  end;
 {set up printer canvas}
  FCanvas := Printer.Canvas;
  with FCanvas do   {for now there are no special printer font properties}
  begin
    Font.Name := videoFontName;
    Font.Color := videoFontColor;
    Font.Style := videoFontStyles;
       {magic of scaling fonts to printer while it's running???}
    VertPixPerInch := GetDeviceCaps(Printer.Handle, LogPixelsY);
    Font.Size := Round((FScale / VertPixPerInch) * videoFontSize * Canvas.Font.PixelsPerInch);
    Pen.Width := Round(videoPenWidth * FScale);
  end;

 {print and restore canvas}
  try
    PaintGraph;
    If assigned(FOnPaintEnd) then FOnPaintEnd(self, FCanvas);
  finally
    FCanvas := Canvas;
    FScale := 1.0;
    FXOffset := 0;
    FYOffset := 0;

    Width := saveWidth;
    Height := saveHeight;
    with FDimensions do
    begin
      FLeft := saveMargLeft;
      FRight := saveMargRight;
      FTop := saveMargTop;
      FBottom := saveMargBottom;
      FTMLength := saveTMLength;
      FXAxisLabelDistance := saveXLabelDistance;
      FYAxisLabelDistance := saveYLabelDistance;
      FXAxisTitleDistance := saveXAxisDistance;
      FYAxisTitleDistance := saveYAxisDistance;
      FGraphTitleDistance := saveGraphTitleDistance;
    end;
  end;
end {PrintOnPage};


{---------------------------------------------------------------------
    #5e. TxyGraph  - property servers
 ---------------------------------------------------------------------}

procedure TxyGraph.SetAllowDuplicates;
var p:TSeries;
begin
  FAllowDuplicates := v;
  p := FSeries;
  while p <> nil do
  begin
    p.Allowduplicates := FAllowDuplicates;
    p := p.next;
  end;
  Paint;
end;

procedure TxyGraph.SetPlotting(v: Boolean);
begin
  FPlotting := v;
  Paint;
  if v then DoRescaleEvent;
end;

procedure TxyGraph.SetHasDragged(v:boolean);
{ this procedure is primarily present to set FHasdragged to false
  and override any user changes }
begin
 if v <> FHasDragged then
  if v then
  begin
    FHasDragged := true;
    FHoldXAS := FXAxis.Autosizing;
    FHoldXmin := FXAxis.Min;
    FHoldXmax := FXAxis.Max;
    FHoldYAS := FYAxis.Autosizing;
    FHoldYmin := FYAxis.Min;
    FHoldYmax := FYAxis.Max;
    FHoldY2AS := FYAxis2.Autosizing;
    FHoldY2min := FYAxis2.Min;
    FHoldY2max := FYAxis2.Max;
  end
  else
  begin
    FHasDragged := false;
    plotting := false;
    FXaxis.autosizing := FHoldXAS;
    if not FHoldXAS then
    begin
      FXAxis.min := FHoldXMin;
      FXAxis.max := FHoldXMax;
    end;
    FYaxis.autosizing := FHoldYAS;
    if not FHoldYAS then
    begin
      FYAxis.min := FHoldYMin;
      FYAxis.max := FHoldYMax;
    end;
    FYaxis2.autosizing := FHoldY2AS;
    if not FHoldY2AS then
    begin
      FYAxis2.min := FHoldY2Min;
      FYAxis2.max := FHoldY2Max;
    end;
    plotting := true;
  end;
end;

{--------- TxyGraph obsolete property access methods--------------}
{$IFDEF BACKWARD}

procedure TxyGraph.SetXAutosizing(v: Boolean);
begin
  FXAxis.Autosizing := v;
end;

procedure TxyGraph.SetYAutosizing(v: Boolean);
begin
  FYAxis.Autosizing := v;
end;

procedure TxyGraph.SetAutoStepping(v: Boolean);
begin
  FXAxis.AutoStepping := v;
  FYAxis.AutoStepping := v;
end;

procedure TxyGraph.SetXLabelDec(v: Integer);
begin
  FXAxis.LabelDecimals := v;
end;

procedure TxyGraph.SetYLabelDec(v: Integer);
begin
  FYAxis.LabelDecimals := v;
end;

procedure TxyGraph.SetXLogging;
begin
  FXAxis.LogScale := v;
end;

procedure TxyGraph.SetYLogging;
begin
  FYAxis.LogScale := v;
end;

procedure TxyGraph.SetXAxisTitle;
begin
  FXAxis.Title := v;
end;

procedure TxyGraph.SetYAxisTitle;
begin
  FYAxis.Title := v;
end;

procedure TxyGraph.SetMinXScale(v: Double);
begin
  FXAxis.MinScaleLength := v;
end;

procedure TxyGraph.SetMinYScale(v: Double);
begin
  FYAxis.MinScaleLength := v;
end;

procedure TxyGraph.SetXMax(v: Double);
begin
  FXAxis.Max := v;
end;

procedure TxyGraph.SetXMin(v: Double);
begin
  FXAxis.Min := v;
end;

procedure TxyGraph.SetXStep(v: Double);
begin
  FXAxis.StepSize := v;
end;

procedure TxyGraph.SetYMax(v: Double);
begin
  FYAxis.Max := v;
end;

procedure TxyGraph.SetYMin(v: Double);
begin
  FYAxis.Min := v;
end;

procedure TxyGraph.SetYStep(v: Double);
begin
  FYAxis.StepSize := v;
end;

function  TxyGraph.GetXAutosizing: Boolean;
begin
  Result := FXAxis.Autosizing;
end;

function  TxyGraph.GetYAutosizing: Boolean;
begin
  Result := FYAxis.Autosizing;
end;

function  TxyGraph.GetAutoStepping: Boolean;
begin
  Result := FXAxis.AutoStepping or FYAxis.AutoStepping;
end;

function  TxyGraph.GetXLabelDec: Integer;
begin
  Result := FXAxis.LabelDecimals;
end;

function  TxyGraph.GetYLabelDec: Integer;
begin
  Result := FYAxis.LabelDecimals;
end;

function  TxyGraph.GetXLogging;
begin
  Result := FXAxis.LogScale;
end;

function  TxyGraph.GetYLogging;
begin
  Result := FYAxis.LogScale;
end;

function  TxyGraph.GetXAxisTitle;
begin
  Result := FXAxis.Title;
end;

function  TxyGraph.GetYAxisTitle;
begin
  Result := FYAxis.Title;
end;

function  TxyGraph.GetMinXScale: Double;
begin
  Result := FXAxis.MinScaleLength;
end;

function  TxyGraph.GetMinYScale: Double;
begin
  Result := FYAxis.MinScaleLength;
end;

function  TxyGraph.GetXMax: Double;
begin
  Result := FXAxis.Max;
end;

function  TxyGraph.GetXMin: Double;
begin
  Result := FXAxis.Min;
end;

function  TxyGraph.GetXStep: Double;
begin
  Result := FXAxis.StepSize;
end;

function  TxyGraph.GetYMax: Double;
begin
  Result := FYAxis.Max;
end;

function  TxyGraph.GetYMin: Double;
begin
  Result := FYAxis.Min;
end;

function  TxyGraph.GetYStep: Double;
begin
  Result := FYAxis.StepSize;
end;

procedure TxyGraph.SetAllowedOut;
begin
  FAppearance.PlotOffGraph := v;
end;

procedure TxyGraph.SetLabelGraph(v: Boolean);
begin
  FAppearance.ShowGraphLabels := v;
end;

procedure TxyGraph.SetGridlines(v: Boolean);
begin
  FXAxis.GridLines := v;
  FYAxis.Gridlines := v;
end;

procedure TxyGraph.SetTickmarks(v: Boolean);
begin
  FAppearance.ShowTicks := v;
end;

procedure TxyGraph.SetGridcolor(v: TColor);
begin
  FAppearance.GridColor := v;
end;

procedure TxyGraph.SetAxescolor(v: TColor);
begin
  FAppearance.AxesColor := v;
end;

procedure TxyGraph.SetGridStyle(v: TPenStyle);
begin
  FAppearance.GridStyle := v;
end;

procedure TxyGraph.SetGraphTitle;
begin
  FAppearance.GraphTitle := v;
end;

function  TxyGraph.GetAllowedOut: Boolean;
begin
  Result := FAppearance.PlotOffGraph;
end;

function  TxyGraph.GetLabelGraph: Boolean;
begin
  Result := FAppearance.ShowGraphLabels;
end;

function  TxyGraph.GetGraphTitle: ShortString;
begin
  Result := FAppearance.GraphTitle;
end;

function  TxyGraph.GetGridlines: Boolean;
begin
  Result := FYAxis.GridLines;
end;

function  TxyGraph.GetGridcolor: TColor;
begin
  Result := FAppearance.GridColor;
end;

function  TxyGraph.GetGridStyle: TPenStyle;
begin
  Result := FAppearance.GridStyle;
end;

function  TxyGraph.GetTickmarks: Boolean;
begin
  Result := FAppearance.ShowTicks;
end;

function  TxyGraph.GetAxescolor: TColor;
begin
  Result := FAppearance.AxesColor;
end;

procedure TxyGraph.SetMargLeft(v: Word);
begin
  FDimensions.LeftMargin := v;
end;

procedure TxyGraph.SetMargRight(v: Word);
begin
  FDimensions.RightMargin := v;
end;

procedure TxyGraph.SetMargTop(v: Word);
begin
  FDimensions.TopMargin := v;
end;

procedure TxyGraph.SetMargBottom(v: Word);
begin
  FDimensions.BottomMargin := v;
end;

function  TxyGraph.GetMargLeft: Word;
begin
  Result := FDimensions.LeftMargin;
end;

function  TxyGraph.GetMargRight: Word;
begin
  Result := FDimensions.RightMargin;
end;

function  TxyGraph.GetMargTop: Word;
begin
  Result := FDimensions.TopMargin;
end;

function  TxyGraph.GetMargBottom: Word;
begin
  Result := FDimensions.BottomMargin;
end;

procedure TxyGraph.SetTMLength(v: Word);
begin
  FDimensions.TickLength := v;
end;

function  TxyGraph.GetTMLength: Word;
begin
  Result := FDimensions.TickLength;
end;

procedure TxyGraph.SetXAxisTitleDistance;
begin
  FDimensions.XAxisTitleOffset := v;
end;

procedure TxyGraph.SetYAxisTitleDistance;
begin
  FDimensions.YAxisTitleOffset := v;
end;

function  TxyGraph.GetXAxisTitleDistance;
begin
  Result := FDimensions.XAxisTitleOffset;
end;

function  TxyGraph.GetYAxisTitleDistance;
begin
  Result := FDimensions.YAxisTitleOffset;
end;

procedure TxyGraph.SetXAxisLabelDistance;
begin
  FDimensions.XAxisLabelOffset := v;
end;

procedure TxyGraph.SetYAxisLabelDistance;
begin
  FDimensions.YAxisLabelOffset := v;
end;

function  TxyGraph.GetXAxisLabelDistance;
begin
  Result := FDimensions.XAxisLabelOffset;
end;

function  TxyGraph.GetYAxisLabelDistance;
begin
  Result := FDimensions.YAxisLabelOffset;
end;

{$ENDIF}{BACKWARD}

procedure TxyGraph.TitleFontChanged(Sender: TObject);
begin
  paint;
end;

procedure TxyGraph.LabelFontChanged(Sender: TObject);
begin
  paint;
end;

{---------------------------------------------------------------------
    #5f. TxyGraph  - series routines
 ---------------------------------------------------------------------}

function TxyGraph.GetSeriesInt;
var p: TSeries;
begin
  p := FSeries;
  while (p <> nil) and (p.FSeriesIndex <> i) do p := p.next;
  if p = nil then
    result := CreateNewSeries(i)
  else
    Result := p;
end;

function TxyGraph.CreateNewSeries;
var t,p,p1:TSeries;
begin
{note! the list of series is maintained on insertion here,
       but on deletion in the TSeries.destroy routine }
  if assigned(FOnNewSeries) then FOnNewSeries(self,i);
  t := TSeries.create(self,i);
  result := t;
  if Fseries = nil then
    Fseries := t
  else
  begin
    p := FSeries;
    p1 := nil;
    while (p <> nil) and (p.FSeriesIndex < i) do
      begin
      p1 := p;
      p := p.next;
      end;
    if p1 = nil then
      begin
      t.next := Fseries;
      Fseries := t;
      end
    else
      begin
      p1.next := t;
      t.next := p;
      end;
  end;
end;

{---------------------------------------------------------------------
    #5g. TxyGraph  - series hooking routines
 ---------------------------------------------------------------------}

procedure TxyGraph.Hookseries(Source:TSeries; Ahooktype:THooktype;
                     Dest:Integer; WantPersist:boolean);
var p:pHookedSeries;
begin
 p := FHookedSeries;
 while (p <> nil) and (p^.id <> Source) do p := p^.next;
 if p = nil then
   begin
   new(p);
   p^.next := FHookedSeries;
   FHookedSeries := p;
   end
 else
   if Not WantPersist then getseriesint(dest).clear;
   {clear series if we're reassigning to another one unless user says otherwise }
 p^.id := source;
 p^.Series := dest;
 p^.hooktype := AHooktype;
 p^.id.hooks[self] := HookedSeriesEvent;
 HookedSeriesEvent(p^.id, dsClearUpdates); {dsClearUpdates = anything might have changed}
end;

procedure TxyGraph.unhookseries(source:TSeries; WantPersist:boolean);
var p,p1:phookedseries;
begin
 p1 := nil;
 p := FHookedSeries;
 while (p <> nil) and (p^.id <> source) do
   begin
   p1 := p;
   p := p^.next;
   end;
 if p = nil then raise exception.create('trying to unhook a series that is not hooked');
 if p1 = nil then
   FhookedSeries := p^.next
 else
   p1^.next := p^.next;
 p^.id.hooks[self] := nil;
 if Not WantPersist then getseriesint(p^.series).clear;
 dispose(p);
 if not application.terminated then paint;
end;

procedure TxyGraph.HookedSeriesEvent(Sender:TObject; TheMessage:TDSChangeType);
var p:pHookedSeries;
  procedure readhookedseriesdata;
  var ps:TSeries;
      pp:pgpoint;
  begin
   ps := getseriesint(p^.series);
   ps.holdupdates := true;
   ps.clear;
   case p^.hooktype of
    ht_plotdata:begin
       pp := p^.id.Data;
       while pp <> nil do
         begin
         ps.add(pp^.xv,pp^.yv);
         pp := pp^.next;
         end;
       end;
   {$IFDEF STATISTICS}
    ht_plotresiduals:if p^.id.RegType in [rg_Linear, rg_passingBablok,
                                          rg_quadratic, rg_RunningAverage] then
       begin
       p^.id.DoRegression;
       pp := p^.id.Data;
       while pp <> nil do
         begin
         ps.add(pp^.xv,pp^.yv-pp^.rv);
         pp := pp^.next;
         end;
       end;
   {$ENDIF}
   end;
   ps.holdupdates := false;
  end;

begin
 p := FHookedSeries;
 while (p <> nil) and (p^.id <> Sender as TSeries) do p := p^.next;
 if p = nil then raise exception.create('HookedSeriesEvent but series not properly hooked');
 case TheMessage of
  dsUnLoad:unhookseries(p^.id, false);
  dsDataChange:readhookedseriesdata;
  {$IFDEF STATISTICS}
  dsRegressionChange:if p^.hooktype = ht_plotResiduals then readhookedseriesdata;
  {$ENDIF}
  dsZeroChange:{ignore};
  dsClearUpdates:readhookedseriesdata;
  dsChangeLook:{ignore};
 end;
end;

function TxyGraph.FindSeries;
var p: TSeries;
begin
  p := FSeries;
  while (p <> nil) and (p.FSeriesIndex <> i) do p := p.next;
  result := (p <> nil);
end;

{------------ Obsolete TxyGraph Series methods: ----------------}
{$IFDEF BACKWARD}

procedure TxyGraph.SetSeries(sN: Integer;
                             Makeactive, Initialise, AutoZero: Boolean;
                             ZeroValue: Double);
var p:TSeries;
begin
  p := series[sN];
  p.HoldUpdates := true;
  p.Active := MakeActive;
  if Initialise then p.clear;
  p.InternalZero := ZeroValue; {set this first - don't override AutoZero accidently}
  p.AutoZero := AutoZero;
  p.HoldUpdates := false;
end;

function TxyGraph.GetSeries(sN: Integer;
                        var active: Boolean;
                        var ZeroValue: Double): Boolean;
var p: TSeries;
begin
  result := findseries(sN);
  if result then
    begin
    p := series[sN];
    Result := true;
    ZeroValue := p.InternalZero;
    Active := p.Active;
    end;
end;

procedure TxyGraph.SetSeriesLine(sN: Integer;
                            status: Boolean;
                            Color: TColor;
                            Style: TPenStyle);
var p: TSeries;
begin
 p := series[sN];
 p.HoldUpdates := true;
 p.DrawLine := status;
 p.LineColor := Color;
 p.LineStyle := Style;
 p.HoldUpdates := false;
end;

function TxyGraph.GetSeriesLine(sN: Integer;
                                var Color: TColor;
                                var Style: TPenStyle): Boolean;
var p: TSeries;
begin
  p := Series[sN];
  Result := p.DrawLine;
  Color := p.LineColor;
  Style := p.LineStyle;
end;

procedure TxyGraph.SetSeriesPoints(sN: Integer;
                              status: Boolean;
                              Size: Integer;
                              Shape: et_PointShape;
                              Filled: Boolean;
                              Color: TColor);
var p: TSeries;
begin
  p := Series[sN];
  p.HoldUpdates := true;
  p.DrawPoints := status;
  p.PointSize := Size;
  p.PointColor := Color;
  p.PointShape := Shape;
  p.FillPoints := Filled;
  p.HoldUpdates := false;
end;

function TxyGraph.GetSeriesPoints(sN: Integer;
                              var Size: Integer;
                              var Shape: et_PointShape;
                              var Filled: Boolean;
                              var Color: TColor): Boolean;
var p: TSeries;
begin
  p := Series[sN];
  Result := p.DrawPoints;
  Color := p.PointColor;
  Size := p.PointSize;
  Shape := p.PointShape;
  Filled := p.FillPoints;
end;

{----- (obsolete) data manipulation ----------------------}

procedure TxyGraph.Clear;
var OFplotting:boolean;
begin
 OFPlotting := FPlotting;
 FPLotting := false;
 if sN = 0 then
   while Fseries <> nil do Fseries.free
 else
   series[sN].destroy;
 FPlotting := OFPlotting;
 If Fplotting then Paint;
end;

function TxyGraph.Add(sN: Integer; x, y: Double): longint;
begin
  result := series[sN].add(x,y);
end;

function TxyGraph.GetValue(sN: Integer;
                           curr: longint;
                           var index: longint;
                           var x, y, r: Double): Boolean;
begin
 result := series[sN].GetValue(curr,index,x,y,r);
end;

function TxyGraph.GetPoint(sN: Integer;
                           var index: longint;
                           var x, y, r: Double): Boolean;
begin
 result := series[sN].getpoint(index, x,y,r);
end;

function TxyGraph.DeletePoint(sN: Integer; index: longint): Boolean;
begin
 result := series[sN].deletepoint(index);
end;

function TxyGraph.DeleteValue(sN: Integer; x: Double): Boolean;
var p: TSeries;
    OFplotting:boolean;
begin
  OFPlotting := FPlotting;
  FPLotting := false;
  if sN = 0 then
  begin
    p := FSeries;
    Result := False;
    while p <> nil do
    begin
      Result := Result or p.DeleteValue(x);
      p := p.next;
    end;
  end
  else
   Result := series[sN].DeleteValue(x);
 FPlotting := OFPlotting;
end;

{$ENDIF}{BACKWARD}

{---------------------------------------------------------------------
    #5g. TxyGraph  - data manipulation
 ---------------------------------------------------------------------}

function TxyGraph.DataFound;
var p: TSeries;
begin
  Result := False;
  p := FSeries;
  while (p <> nil) and not Result do
  begin
    Result := Result or (p.FData <> nil) and ((whichaxis = FXAxis)
           or (whichaxis = nil) or (p.FWhichYAxis = whichaxis));
    p := p.next;
  end;
end;

procedure TxyGraph.ClearAll;
var holdPlotting: Boolean;
begin
  holdPlotting := FPlotting;
  FPlotting := false;
  while FSeries <> nil do FSeries.Free;
  FPlotting := holdPlotting;
  if FPlotting then Paint;
end;

{---------------------------------------------------------------------
    #5h. TxyGraph  - Marks routines
 ---------------------------------------------------------------------}

procedure TxyGraph.ClearMarks;
var om: pMark;
begin
  while FMarks <> nil do
  begin
    om := FMarks;
    FMarks := FMarks^.next;
    Dispose(om);
  end;
  if FPlotting then Paint;
end;

procedure TxyGraph.addmark;
var tm,p,p1:pmark;
begin
 p := fmarks;
 while (p <> nil) and (p^.id <> id) do p := p^.next;
 if p <> nil then
 begin
   p^.id := id;
   p^.color := c;
   p^.x1 := x1;
   p^.y1 := y1;
   p^.x2 := x2;
   p^.y2 := y2;
   p^.caption := name;
   p^.marktype := marktype;
   p^.status := status;
 end
 else
 begin
   new(tm);
   tm^.id := id;
   tm^.color := c;
   tm^.x1 := x1;
   tm^.y1 := y1;
   tm^.x2 := x2;
   tm^.y2 := y2;
   tm^.caption := name;
   tm^.marktype := marktype;
   tm^.status := status;
   if (fmarks = nil) or (fmarks^.id > id) then
   begin
     tm^.next := Fmarks;
     Fmarks := tm;
   end
   else
   begin
    p := fmarks^.next;
    p1 := fmarks;
    while (p <> nil) and (p^.id > id) do
    begin
      p1 := p;
      p := p^.next;
    end;
    tm^.next := p;
    p1^.next := tm;
   end;
 end;
 if fplotting then paint;
end;


function TxyGraph.DeleteMark;
var p, p1: pMark;
begin
  Result := False;
  p := FMarks;
  while p <> nil do
    if p^.id = id then
    begin
      Result := True;
      if p = FMarks then FMarks := p^.next else
      begin
        p1 := FMarks;
        while p1 <> p^.next do p1 := p1^.next;
        p1^.next := p^.next;
        Dispose(p);
      end;
    end
    else p := p^.next;
  if FPlotting and Result then Paint;
end;

{---------------------------------------------------------------------
    #5i. TxyGraph  - debug routine
 ---------------------------------------------------------------------}

procedure TxyGraph.Debug;
var f: system.Text;
  p: TSeries;
  p1: pgPoint;
begin
  if not findseries(i) then p := nil else p := series[i];
  AssignFile(f, fn);
  Rewrite(f);
  if p <> nil then
  with p do
  begin
    Write(f, 'Series ', p.FSeriesIndex);
    if next = nil then WriteLn(f, ',  last series!') else WriteLn(f);
    if FData = nil then WriteLn(f, '  no data') else WriteLn(f, '  has data');
    if active then WriteLn(f, '  active') else WriteLn(f, '  not active');
    if Changed then WriteLn(f, '  changed') else WriteLn(f, '  not changed');
    if DrawLine then WriteLn(f, '  has line') else WriteLn(f, '  no line');
    if DrawPoints then WriteLn(f, '  has points') else WriteLn(f, '  no points');
    if FillPoints then WriteLn(f, '  filled points') else WriteLn(f, '  empty points');
    WriteLn(f, '  min/max, (x/y), 0: ', xvMin:0:2, ',', xvMax:0:2, ',',
            yvMin:0:2, ',', yvMax:0:2, ',', InternalZero:0:2);
 {$IFDEF STATISTICS}
    if NeedRegreCalc then WriteLn(f, '  need regrcalc') else WriteLn(f, 'regr done');
    case RegType of
      rg_None: WriteLn(f, '  regression: None');
      rg_passingBablok: WriteLn(f, '  regression: Passing-Bablok');
      rg_Linear: WriteLn(f, '  regression: linear');
      rg_RunningAverage: WriteLn(f, '  regression: running average');
      rg_Spline: WriteLn(f, '  regresssion: spline,', RegControl1:0, ',', RegControl2:0:3);
      rg_DWLS: Writeln(f, '  regression: DWLS,', RegControl1:0, ',', RegControl2:0:3);
      rg_quadratic: writeln(f, '  regression: quadratic');
    end;
 {$ENDIF}
    p1 := FData;
    while p1 <> nil do
    begin
      WriteLn(f, p1^.i:0, ',', p1^.xv:0:4, ',', p1^.yv:0:4, ',', p1^.rv:0:4);
      p1 := p1^.next;
    end;
  end
  else WriteLn(f, 'No series ', i:0, ' exists');
  CloseFile(f);
end;

{$IFDEF DESIGNER}
procedure TxyGraph.rundesigner;
begin
  xyedit := txyedit.create(nil);
  try
   xyedit.Fgraph := self;
   xyedit.designtime := (csDesigning in componentstate);
   xyedit.showmodal;
  finally
   xyedit.free;
  end;
end;
{$ENDIF}

{---------------------------------------------------------------------
    #6. TxyGraph  - register routine and component editor
 ---------------------------------------------------------------------}

procedure TxyGraphCompEditor.editGraph;
begin
{$IFDEF DESIGNER}
  (component as txyGraph).rundesigner;
  designer.modified;
{$ELSE}
  messagedlg('The designer is not available - DESIGNER define not declared',
            mterror,[mbok],0);
{$ENDIF}
end;

function TxyGraphCompEditor.GetVerb(Index: Integer): string;
begin
  result := 'Designer...';
end;

function TxyGraphCompEditor.GetVerbCount: Integer;
begin
  result := 1;
end;

procedure TxyGraphCompEditor.ExecuteVerb(Index: Integer);
begin
  editgraph;
end;

procedure TxyGraphCompEditor.edit;
begin
  editgraph;
end;

procedure Register;
begin
  RegisterComponents('TxyGraph', [TxyGraph]);
  RegisterComponentEditor(TxyGraph, TxyGraphCompEditor);
end;

end.

