unit PolygonSprite;

(*********************************************
TPolygonSprite->TSprite

This descendant of TSprite renders itself by drawing
a set of lines.  This is an abstract class ... to
implement, derive a new class and override the
AddVertices method.  The the sample TSquareSprite that
is provided at the end of this unit as an example.

Properties

Angle-
  Controls the angle of rotation of the polygon.
ColorIndex-
  The color of the polygon's lines.
Spin-
  The direction of the polygon's spin, if any.
SpinSpeed-
  The speed at which the polygon spins.  The higher the
  number, the faster the spin.

Events

Methods

AddVertices-
  This method MUST be overriden in derived classes.
  It's purpose is to populate the Vertex list of
  the TPolygon object contained within this class.
  See the TSquareSprite at the end of this unit as
  an example.
CreatePolygonSprite-
  Pass the colorindex of the sprite as a constructor
  paramater.
*********************************************)

interface

uses
  Windows, SysUtils, Classes, Graphics, Controls, DIBDrawingSurface, Sprite,
  Grafix;

type

  TRotation = ( rotNone, rotClockwise, rotCounterClockwise );

  TPolygonSprite = class( TSprite )
  private
     r: TRectangle;
     FSpin: TRotation;
     FSpinSpeed: integer;
  protected
     poly: TPolygon;
     polyOriginal: TPolygon;
     FAngle: integer;
     FColorIndex: byte;
     procedure SetAngle( n: integer );
     procedure SetSize; virtual;
  public
     constructor CreatePolygonSprite( const nCol: byte );
     destructor Destroy; override;
     procedure AddVertices; virtual; abstract;
     procedure Render; override;
     property Angle: integer read FAngle write SetAngle;
     property ColorIndex: byte read FColorIndex write FColorIndex;
     property Spin: TRotation read FSpin write FSpin;
     property SpinSpeed: integer read FSpinSpeed write FSpinSpeed;
  end;

  TSquareSprite = class( TPolygonSprite )
  private
  protected
     procedure AddVertices; override;
  public
  end;

implementation

constructor TPolygonSprite.CreatePolygonSprite( const nCol: byte );
begin
  inherited Create;

  Spin := rotNone;
  SpinSpeed := 10;
  Angle := 0;
  FColorIndex := nCol;
  poly := TPolygon.Create;

  AddVertices;

  r := TRectangle.CreateRectangle( 0, 0, 0, 0 );

  SetSize;

  polyOriginal := TPolygon.Create;
  polyOriginal.Assign( poly );
end;

destructor TPolygonSprite.Destroy;
var
  i: integer;
begin
  polyOriginal.Free;
  poly.Free;
  r.Free;
  inherited Destroy;
end;

(*********************************************
Spinning of the polygon is not considered important,
so it is done within render so it will not take
up processing time if the sprite is off the edge
of the surface.
*********************************************)
procedure TPolygonSprite.Render;
var
  i: integer;
  ptFrom, ptTo: TPoint;
  OldAngle: integer;
begin
  inherited Render;

  OldAngle := Angle;
  case Spin of
     rotCounterClockwise:
        Angle := Angle + SpinSpeed;
     rotClockwise:
        Angle := Angle - SpinSpeed;
  end;

  if Angle <> OldAngle then
     begin
        poly.Assign( polyOriginal );
        poly.Rotate( TDegree( Angle ) );
        SetSize;
     end;

  ptFrom := poly.Vertex[0].Point;

  Inc( ptFrom.X, ptPhysical.X );
  Inc( ptFrom.Y, ptPhysical.Y );

  dds.DIBCanvas.MoveTo( ptFrom.X, ptFrom.Y );
  dds.DIBCanvas.PenColorIndex := FColorIndex;

  for i := 1 to poly.VertexCount - 1 do
     begin
        ptTo := poly.Vertex[i].Point;
        Inc( ptTo.X, ptPhysical.X );
        Inc( ptTo.Y, ptPhysical.Y );
        dds.DIBCanvas.LineTo( ptTo.X, ptTo.Y );
     end;

  dds.DIBCanvas.LineTo( ptFrom.X, ptFrom.Y );
end;

procedure TPolygonSprite.SetAngle( n: integer );
begin
  FAngle := IntToDegree( n );
end;

procedure TPolygonSprite.SetSize;
var
  i: integer;
begin
  r.Left := 0;
  r.Right := 0;
  r.Top := 0;
  r.Bottom := 0;
  for i := 0 to poly.VertexCount - 1 do
     with poly.Vertex[i] do
        begin
           if X < r.Left then
              r.Left := X;
           if Y < r.Top then
              r.Top := Y;
           if X > r.Right then
              r.Right := X;
           if Y > r.Bottom then
              r.Bottom := Y;
        end;
  Width := r.Width + 2;
  Height := r.Height + 2;
end;

(*********************************************
TSquareSprite
An example descendant of TPolygonSprite.
*********************************************)
procedure TSquareSprite.AddVertices;
begin
  with poly do
     begin
        AddVertex( TVertex.CreateVertex( -5, -5 ) );
        AddVertex( TVertex.CreateVertex( 5, -5 ) );
        AddVertex( TVertex.CreateVertex( 5, 5 ) );
        AddVertex( TVertex.CreateVertex( -5, 5 ) );
     end;
end;

end.
