{This is a component I wrote to read tiff files, as part of
a project that subsequently didn't go anywhere. I thought about
writing it up as a shareware component, but I didn't have time
to add writing to the tiff file, which I would have had to do
and also color images. So I have chosen to release the code
for general use. Warning: this isn't finished. There's
a bug or to, and some shortcomings you might consider
bugs. Where I am aware of these, they are clearly documented.

There's no native vcl to read tiff files. Something like imageknife
or other things are available, but dll's, ocx's and vbx's just not as
convenient as having source code. Hence this component, which will
compile either under delphi 1 or 2.

This component is based on the official tiff 6.0 specification.
I downloaded this from adobe in .pdf format.

This component will parse standard tiff files, and read mono
images - uncompressed and ccitt 3.0 compressed images.

Copyright  Grahame Grieve.

This code is public domain, but if you distribute components based
on this you must acknowledge the source.
If you develop any more sophisticated components from this source,
please send me a copy.

}

unit Tifffile;

{ public properties / routines

Filename:  use to open a file
ImageCount: how many images are in the file
FCurrImage: which image the following read only properties relate to:
Width,Length,InvertColour,ImageType

readbitmap - returns HBitmap containing current image }

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, Dsgnintf;

type

  et_compress = (tfc_none, tfc_ccitt, tfc_packbit, tfc_unknown);
  et_imagetype = (ti_mono, ti_grayscale, ti_palette, ti_rgb);

{ tbitstream is used in the reading and writing of ccitt compressed images }

type tbitstream = class
        private
          Fstart,Fcurr:pchar;
          Findx:byte;
          FMax:longint;
        public
          constructor create(p:pchar; size:longint);
          function atbyte:boolean;
          procedure addbit(s:boolean);
          procedure addbyte(s:boolean);
          function getbit:boolean;
          function getbyte:byte;
          procedure align;
     end;

  byte2 = array [1..2] of byte;
  byte4 = array [1..4] of byte;

  Ttifftag = record
        tagtype,ftype:word;
        vcount:longint;
        value:byte4;
       end;

  PTiffImageMap = ^TTiffImageMap;
  TTiffImageMap = record
     imageID:word;
     next:PTiffImageMap;
     imagetype:et_imagetype;
     IFDOffset:longint;
     width,length,bitspersample:word;
     compression:et_compress;
     invertcolour:boolean;
     resunits:string;
     Xres,Yres:real;
     numstrips,rowsperstrip:word;
     stripoffsetstart,stripbytecountstart:longint;
   end;

type
  TTiffFile = class(TComponent)
  private
    { Private declarations }
    FImageCount,FCurrImage:word;
    FFilename:string;
    FBigEndian:boolean;
    FImageMaps:PTiffImageMap;

    procedure setFilename(v:string);
    procedure SetCurrentImage(v:word);
    function getwidth:word;
    function getlength:word;
    function getinvertcolour:boolean;
    function getResUnits:string;
    function getXres:real;
    function getYres:real;
    function getImageType:et_imagetype;

    function read2byte(f:tfilestream):byte2;
    function read4byte(f:tfilestream):byte4;
    function longintfrom4byte(b:byte4):longint;
    function wordfrom2byte(b:byte2):word;
    function wordfrom4byte(b:byte4):word;
    function readtifftag(f:tfilestream; os:longint; i:word):TTiffTag;
    function checkheader(f:tfilestream):boolean;

    procedure resetdata;
    function parseTiff:word;
    function getcurrentimagemap:PTiffImageMap;
    function AddimageMap(i:word):PTiffImageMap;
    procedure parseIFD(f:tfilestream; var os:longint; ifdkey:word);

    function findoffset(i:word; f:tfilestream; p:PTiffImageMap):longint;
    function findbytes(i:word; f:tfilestream; p:PTiffImageMap):longint;
    procedure readstripinfo(i:word; f:tfilestream; q:PTiffImageMap);
    function decoderow(s,t:tbitstream; spacer:boolean; w:word):word;

    procedure readuncompressedmono(f:tfilestream; var p:PTiffImageMap;
                    mem:pointer; memsize:longint);
    procedure readcompressedmono(f:tfilestream; var p:PTiffImageMap;
                    mem:pointer; memsize:longint);
    procedure readtiff(p:PTiffImageMap; mem:pointer; memsize:longint);
  protected
    { Protected declarations }
    constructor create(owner:TComponent); override;
    destructor destroy; override;

  public
    { Public declarations }
    function readbitmap:HBitMap;
    property ImageCount:word read FImageCount;
    property Width:word read getwidth;
    property Length:word read getlength;
    property InvertColour:boolean read getinvertcolour;
    property ImageType:et_imagetype read getImagetype;
  published
    { Published declarations }
    property Filename:string read FFilename write setFilename;
    property CurrentImage:word read FCurrImage write SetCurrentImage;
  end;

TFileNameProperty = class (TStringProperty)
public
  function getattributes: TPropertyattributes; override;
  procedure Edit; override;
end;


procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('Additional', [TTiffFile]);
  RegisterPropertyEditor(Typeinfo(string),
       TTiffFile, 'Filename', TFileNameProperty);
end;

{---------------------------- tbitstream ------------------------------}
procedure pointerinc(var p:pchar; i:word);
begin
 if ofs(p^) < 65535 - i + 1 then inc(p,i) else
   begin
   Inc(PtrRec(P).Seg, SelectorInc);
   ptrRec(p).Ofs := ofs(p^) + i;
   end;
end;

function tbitstream.atbyte;
begin
 result := (Findx = 0);
end;

constructor tbitstream.create;
begin
  Fstart := p;
  FCurr := p;
  FMax := size;
  Findx := 0;
end;

procedure tbitstream.addbit;
begin
  if s
   then byte(FCurr^) := byte(Fcurr^) or (1 shl (7 - Findx))
   else byte(FCurr^) := byte(Fcurr^) and ((1 shl (7 - Findx)) xor $FF);
  inc(Findx);
  if Findx = 8 then
    begin
    Findx := 0;
{$IFDEF WIN32} inc(FCurr,1); {$ELSE}
    if ofs(FCurr^) < 65535 then inc(FCurr,1) else
      begin
      Inc(PtrRec(FCurr).Seg, SelectorInc);
      ptrRec(FCurr).Ofs := 0;
      end;                 {$ENDIF}
    end;
end;

function tbitstream.getbit;
begin
  result := (byte(FCurr^) and (1 shl (7 - Findx))) <> 0;
  inc(Findx);
  if Findx = 8 then
    begin
    Findx := 0;
{$IFDEF WIN32} inc(FCurr,1); {$ELSE}
    if ofs(FCurr^) < 65535 then inc(FCurr,1) else
      begin
      Inc(PtrRec(FCurr).Seg, SelectorInc);
      ptrRec(FCurr).Ofs := 0;
      end;                   {$ENDIF}
    end;
end;

procedure tbitstream.addbyte;
begin
{ if Findx <> 0 then raise exception.create('not on bit boundary');}
 if s then byte(FCurr^) := $FF else byte(FCurr^) := $00;
{$IFDEF WIN32} inc(FCurr,1); {$ELSE}
 if ofs(FCurr^) < 65535 then inc(FCurr,1) else
      begin
      Inc(PtrRec(FCurr).Seg, SelectorInc);
      ptrRec(FCurr).Ofs := 0;
      end;                          {$ENDIF}
end;

function tbitstream.getbyte;
begin
{ if Findx <> 0 then raise exception.create('not on bit boundary');}
 result := byte(FCurr^);
{$IFDEF WIN32} inc(FCurr,1); {$ELSE}
 if ofs(FCurr^) < 65535 then inc(FCurr,1) else
      begin
      Inc(PtrRec(FCurr).Seg, SelectorInc);
      ptrRec(FCurr).Ofs := 0;
      end;                    {$ENDIF}
end;

procedure tbitstream.align;
begin
 if Findx <> 0 then
   begin
   Findx := 0;
{$IFDEF WIN32} inc(FCurr,1); {$ELSE}
   if ofs(FCurr^) < 65535 then inc(FCurr,1) else
      begin
      Inc(PtrRec(FCurr).Seg, SelectorInc);
      ptrRec(FCurr).Ofs := 0;
      end;                             {$ENDIF}
   end;
end;

{----------------------------- admin crap -----------------------------}
constructor TTiffFile.create(owner:TComponent);
begin
  inherited create(owner);
  FImageMaps := nil;
  resetdata;
end;

destructor TTiffFile.destroy;
begin
  resetdata;
  inherited destroy;
end;


{----------------------- tiff tag values -----------------------------}
const
  tt_color = 262;
  tt_compression = 259;
  tt_imagelength = 257;
  tt_imagewidth = 256;
  tt_ResUnit = 296;
  tt_Xres = 282;
  tt_Yres = 283;
  tt_RowsPerStrip = 278;
  tt_StripOffsets = 273;
  tt_StripByteCounts = 279;
  tt_bitspersample = 258;
  tt_ColorMap = 320;
  tt_samplesPerPixel = 277;
  tt_artist = 315;
  tt_copyright = 33432;
  tt_datetime = 306;
  tt_hostcomp = 316;
  tt_imagename = 270;
  tt_scanner = 271;
  tt_ScannerMod = 272;
  tt_subfile = 254;
  tt_orientation = 274;
  tt_planar = 284;
  tt_software = 305;

  ft_short = 3;

{-------------------- tiff parsing procedures --------------------------}

{ 1. low level readers / converters }
function TTiffFile.read2byte;
begin
 f.read(result,2);
end;

function TTiffFile.read4byte;
begin
 f.read(result,4);
end;

{ problem: this should really return a 32bit cardinal unsigned
integer. This is not possible in 16bit. Raise an error in this case.
This should only occur when really big files are used. And it doesn't
bother me that I can only read files as big as 2Gb! }
function TTiffFile.longintfrom4byte;
var v:longint;
begin
 if FBigEndian then
   if b[1] > 127 then raise Exception.create('Tiff - Longint Conversion Error')
     else result := longint(b[4]) + 256*longint(b[3]) + 256*256*longint(b[2]) + 256*256*256*longint(b[1]) else
   if b[4] > 127 then raise Exception.create('Tiff - Longint Conversion Error')
     else result := longint(b[1]) + 256*longint(b[2]) + 256*256*longint(b[3]) + 256*256*256*longint(b[4]);
end;

function TTiffFile.wordfrom2byte;
begin
 if FBigEndian then result := b[2] + 256*b[1] else result := b[1] + 256*b[2];
end;

function TTiffFile.wordfrom4byte;
begin
 if FBigEndian then result := b[4] + 256*b[3] else result := b[1] + 256*b[2];
end;

function TTiffFile.readtifftag;
begin
 f.seek(os + 2 + (i * 12),0);
 result.tagtype := wordfrom2byte(read2byte(f));
 result.ftype := wordfrom2byte(read2byte(f));
 result.vcount := longintfrom4byte(read4byte(f));
 result.value := read4byte(f);
end;

procedure TTiffFile.parseIFD;
var c,tagcount:word;
    imap:PTiffImageMap;
    tt:TTiffTag;
begin
 { 1. create a ImageMap }
 imap := addimagemap(ifdkey);
 imap^.imageID := ifdKey;
 imap^.IFDOffset := os;
 f.seek(os,0);
 tagcount := wordfrom2byte(read2byte(f));
 for c := 0 to tagcount - 1 do
    begin
    tt := readtifftag(f,os,c);
    case tt.tagtype of
     tt_color:imap^.invertcolour := (wordfrom4byte(tt.value)=0); { xor when read }
     tt_compression:case wordfrom4byte(tt.value) of
                      1:imap^.compression  := tfc_none;
                      2:imap^.compression  := tfc_ccitt;
                      32773:imap^.compression  := tfc_packbit;
                     else imap^.compression  := tfc_unknown;
                    end;
     tt_imagelength:if tt.ftype = ft_short then
                     imap^.length := wordfrom4byte(tt.value) else
                     imap^.length := longintfrom4byte(tt.value);
     tt_imagewidth :if tt.ftype = ft_short then
                     imap^.width := wordfrom4byte(tt.value) else
                     imap^.width := longintfrom4byte(tt.value);
     tt_ResUnit    :case wordfrom4byte(tt.value) of
                      1:imap^.resunits := 'none';
                      2:imap^.resunits := 'inch';
                      3:imap^.resunits := 'cm';
                     end;
     tt_Xres:begin
              f.seek(longintfrom4byte(tt.value),0);
              imap^.Xres := longintfrom4byte(read4byte(f)) /
                               longintfrom4byte(read4byte(f));
             end;
     tt_Yres:begin
              f.seek(longintfrom4byte(tt.value),0);
              imap^.Yres := longintfrom4byte(read4byte(f)) /
                               longintfrom4byte(read4byte(f));
             end;
     tt_RowsPerStrip:imap^.rowsperstrip := longintfrom4byte(tt.value);
     tt_StripOffsets:begin
              imap^.numstrips := tt.vcount;
              imap^.stripoffsetstart := longintfrom4byte(tt.value);
             end;
     tt_StripByteCounts:imap^.stripbytecountstart := longintfrom4byte(tt.value);
     { image type identification }
     tt_BitsPerSample:begin
              imap^.bitspersample := wordfrom4byte(tt.value);
              if imap^.bitspersample > 1 then
               if imap^.bitspersample = 2 then imap^.imagetype := ti_rgb
                       else imap^.imagetype := ti_grayscale;
             end;
     tt_colorMap:begin
{              imap^.colormap := ;}
              imap^.imagetype := ti_palette
             end;
     { else tag will be deleted / ignored }
     end;
    end;
 f.seek(os + 2 + tagcount * 12,0);
 os := longintfrom4byte(read4byte(f));
end;

function TTiffFile.checkheader;
var b:byte2;
begin
 result := false;
 b := read2byte(f);
 FBigEndian := (b[1] = $4D);
 if ((b[1] = $49) or (b[1] = $4D)) and (b[1] = b[2])
  and  (wordfrom2byte(read2byte(f)) = 42) then
   result := true else raise exception.create('Not a valid tiff file');
end;

function TTiffFile.parsetiff;
var f:tfilestream;
    ifdcount:word;
    os:longint;
begin
 f := tfilestream.create(FFilename,fmOpenRead);
 try
  if checkheader(f) then
    begin
    ifdcount := 0;
    os := longintfrom4byte(read4byte(f));
    while os <> 0 do
      begin
      inc(ifdcount);
      parseifd(f,os,ifdcount);
      end;
    end;
  finally
    f.free;
  end;
  result := ifdcount;
end;

{--------------- internal datastructure maintainance -------------------}

procedure TTiffFile.resetdata;
procedure deleteimagemap(p:PTiffImageMap);
begin
 if p^.next <> nil then deleteimagemap(p^.next);
 dispose(p^.next);
end;

begin
 FImageCount := 0;
 FCurrImage := 0;
 FFilename := '';
 if FImageMaps <> nil then
   begin
   if FImageMaps^.next <> nil then deleteimagemap(FImageMaps^.next);
   dispose(FImageMaps);
   FImageMaps := nil;
   end;
end;

function TTiffFile.AddimageMap(i:word):PTiffImageMap;
var p:PTiffImageMap;
begin
 new(p);
 p^.next := FImageMaps;
 FImageMaps := p;
 result := p;
 p^.imageID := i;
 p^.IFDOffset := 0;
 p^.imagetype := ti_mono;
 p^.width := 0;
 p^.length := 0;
 p^.compression := tfc_none;
 p^.invertcolour := false;
 p^.resunits := 'not assigned';
 p^.Xres := 200;
 p^.Yres := 200;
 p^.numstrips := 0;
 p^.rowsperstrip := 0;
 p^.stripoffsetstart := 0;
 p^.stripbytecountstart := 0;
 p^.bitspersample := 1;
end;

function TTiffFile.getcurrentimagemap:PTiffImageMap;
var p:PTiffImageMap;
begin
 p := FImageMaps;
 while (p <> nil) and (p^.imageID <> FCurrImage) do
  p := p^.next;
 if (p <> nil) and (p^.imageID = FCurrImage)
    then result := p else result := nil;
end;

{------------------------- property servers ----------------------------}
procedure TTiffFile.setFilename(v:string);
begin
 if v <> FFilename then
   begin
   resetdata;
   FFilename := v;
   FImageCount := parseTiff;
   if FImageCount = 0 then raise exception.create('Internal Error in TIFF file')
    else FCurrImage := 1;
   end;
end;

procedure TTiffFile.SetCurrentImage;
begin
 if (FImageCount > 0) and (v > 0) and (v <= FImageCount) then
 FCurrImage := v;
end;

function TTiffFile.getwidth:word;
var p:PTiffImageMap;
begin
 p := getcurrentimagemap;
 if p = nil then result := 0 else result := p^.width;
end;

function TTiffFile.getlength:word;
var p:PTiffImageMap;
begin
 p := getcurrentimagemap;
 if p = nil then result := 0 else result :=  p^.length;
end;

function TTiffFile.getInvertcolour:boolean;
var p:PTiffImageMap;
begin
 p := getcurrentimagemap;
 if p = nil then result := false else result :=  p^.invertcolour;
end;

function TTiffFile.getResUnits:string;
var p:PTiffImageMap;
begin
 p := getcurrentimagemap;
 if p = nil then result := 'Not Found' else result :=  p^.ResUnits;
end;

function TTiffFile.getXres:real;
var p:PTiffImageMap;
begin
 p := getcurrentimagemap;
 if p = nil then result := 0 else result :=  p^.Xres;
end;

function TTiffFile.getYres:real;
var p:PTiffImageMap;
begin
 p := getcurrentimagemap;
 if p = nil then result := 0 else result := p^.Yres;
end;

function TTiffFile.getImageType;
var p:PTiffImageMap;
begin
 p := getcurrentimagemap;
 if p = nil then result := ti_mono else result := p^.imagetype;
end;

{--------------------- reading tiff files -----------------------------}
{--------------------- strip info data structures ---------------------}
type pstripinfo=^Tstripinfo;
     tstripinfo=record
       next:pstripinfo;
       offset,bytes:longint;
      end;
var firststrip,laststrip:Pstripinfo;
procedure deletestripinfo;
  procedure deletenode(p:PStripinfo);
  begin
   if p^.next <> nil then deletenode(p^.next);
   dispose(p);
  end;
begin
 if firststrip <> nil then deletenode(firststrip);
 laststrip := nil;
 firststrip := nil;
end;

function TTiffFile.findoffset(i:word; f:tfilestream; p:PTiffImageMap):longint;
var sos,sos2:longint;
begin
 if i > p^.numstrips then raise exception.create('read past end of tiff image offsets');
 f.seek(p^.stripoffsetstart + (i * 4),0);
 result := longintfrom4byte(read4byte(f));
end;

function TTiffFile.findbytes(i:word; f:tfilestream; p:PTiffImageMap):longint;
begin
 f.seek(p^.stripbytecountstart + (i * 2),0);
 result := wordfrom2byte(read2byte(f));
end;

procedure TTiffFile.readstripinfo(i:word; f:tfilestream; q:PTiffImageMap);
var p:PStripinfo;
begin
 new(p);
 p^.next := nil;
 p^.offset := findoffset(i,f,q);
 p^.bytes := findbytes(i,f,q);
 if firststrip = nil then
   begin
   firststrip := p;
   laststrip := p;
   end
 else
   begin
   laststrip^.next := p;
   laststrip := p;
   end;
end;

{--------------------- ccitt decoding ---------------------------------}
{ rough outline of decoding method: an array of possible stop bits
and their meanings is constructed. array [bitcount, bitcode].
each time a bit is read the corresponding code is checked.
This approach gives decoding times in the same order as real
tiff readers, although a little slower. }

type tccittdatanode = record
       isvalid,stop:boolean;
       length:word;
     end;
     pccittdata = ^tccittdata;
     tccittdata = array [2..13,0..255] of tccittdatanode;

var whitedata,blackdata:pccittdata;

procedure fillblackdata; forward;  { located at the end for coding conveniance }
procedure fillwhitedata; forward;

procedure initccittdata;
begin
 new(whitedata);
 new(blackdata);
 fillblackdata;
 fillwhitedata;
end;

procedure clearccittdata;
begin
 dispose(whitedata);
 dispose(blackdata);
end;

function getrunlength(s:tbitstream; white:boolean):word;
var i,numbits,bits,rlength:word;
    stopcode:boolean;
begin
 result := 0;
 stopcode := false;
 if white then
   begin
   bits := 0;
   bits := (bits shl 1) or word(s.getbit);
   bits := (bits shl 1) or word(s.getbit);
   bits := (bits shl 1) or word(s.getbit);
   bits := (bits shl 1) or word(s.getbit);
   numbits := 4;
   while not stopcode do
     begin
     if  whitedata^[numbits,bits].isvalid then
       begin
       inc(result,whitedata^[numbits,bits].length);
       stopcode := whitedata^[numbits,bits].stop;
       if not stopcode then
         begin
         bits := 0;
         bits := (bits shl 1) or word(s.getbit);
         bits := (bits shl 1) or word(s.getbit);
         bits := (bits shl 1) or word(s.getbit);
         bits := (bits shl 1) or word(s.getbit);
         numbits := 4;
         end;
       end
     else
       begin
       bits := bits shl 1 or word(s.getbit);
       inc(numbits);
       end;
     end;
   end
 else
   begin
   bits := 0;
   bits := bits shl 1 or word(s.getbit);
   bits := bits shl 1 or word(s.getbit);
   numbits := 2;
   while not stopcode do
     begin
     if blackdata^[numbits,bits].isvalid then
       begin
       inc(result,blackdata^[numbits,bits].length);
       stopcode := blackdata^[numbits,bits].stop;
       if not stopcode then
         begin
         bits := 0;
         bits := bits shl 1 or word(s.getbit);
         bits := bits shl 1 or word(s.getbit);
         numbits := 2;
         end;
       end
     else
       begin
       bits := bits shl 1 or word(s.getbit);
       inc(numbits);
       end;
     end;
   end;
end;

procedure putrun(var t:tbitstream; white:boolean; rl:word);
{ adds rl bits of white/black to t }
var j:word;
begin
 j := 0;
 while (j < rl) and not t.atbyte do
   begin
   t.addbit(white);
   inc(j);
   end;
 while (j + 8 < rl) do
   begin
   t.addbyte(white);
   inc(j,8);
   end;
 while (j < rl) do
   begin
   t.addbit(white);
   inc(j);
   end;
end;

function tTiffFile.decoderow(s,t:tbitstream; spacer:boolean; w:word):word;
{ returns the number of pixels in the line. the pointers are both
  advanced appropriately }
var pcount,rl:word;
    white:boolean;
begin
 pcount := 0;
 white := true;
 while pcount < w do
   begin
   rl := getrunlength(s,white);
   if rl <> 0 then
     begin
     putrun(t,white,rl);
     inc(pcount,rl);
     end;
   white := not white;
   end;
 s.align;
 t.align;
 if spacer then t.addbyte(false);
 result := pcount;
end;

{----------------------- read tiff file routines -----------------------}
procedure TTiffFile.readcompressedmono;
var i,j,k,return,w,rowbytewidth,h,count,maxstripsize:word;
    memt,bt:tbitstream;
    b:pchar;
    insertbyte:boolean;
    currentstrip:Pstripinfo;
begin
 firststrip := nil;
 for i := 0 to p^.numstrips - 1 do readstripinfo(i,f,p);
 currentstrip := firststrip;
 maxstripsize := 0;
 while currentstrip <> nil do
   begin
   if currentstrip^.bytes > maxstripsize then maxstripsize := currentstrip^.bytes;
   currentstrip := currentstrip^.next;
   end;
 b := memalloc(maxstripsize);
 currentstrip := firststrip;
 count := 0;
 memt := tbitstream.create(mem,memsize);
 w := p^.width;
 h := p^.rowsperstrip;
 rowbytewidth := ((w  div 8));
 if (w mod 8) <> 0 then inc(rowbytewidth);
 insertbyte := (rowbytewidth mod 2 = 1);
 initccittdata;
 while currentstrip <> nil do
   begin
   f.seek(currentstrip^.offset,0);
   return := f.read(b^,currentstrip^.bytes);
   bt :=tbitstream.create(b,maxstripsize);
   if count = p^.numstrips - 1
      then h := p^.length mod p^.rowsperstrip
      else h := p^.rowsperstrip;
   for j := 1 to h do
     { ok. now we have the compressed bits in b. we have to decompress them
      into mem. in the ccitt specs, rows are padded to end on a byte. And rows have an
      error check built in, so these are administered here }
      if decoderow(bt,memt,insertbyte,p^.width) <> p^.width then
                   raise exception.create('Error in compressed image');
   bt.free;
   inc(count);
   currentstrip := currentstrip^.next;
   end;
 deletestripinfo;
 clearccittdata;
 memt.free;
 freemem(b,maxstripsize);
end;


procedure TTiffFile.readuncompressedmono;
var i,j,k,return,w,rowbytewidth,h,count,maxstripsize:word;
    memt,b,bt:pchar;
    insertbyte:boolean;
    currentstrip:Pstripinfo;
begin
 firststrip := nil;
 for i := 0 to p^.numstrips - 1 do readstripinfo(i,f,p);
 currentstrip := firststrip;
 maxstripsize := 0;
 while currentstrip <> nil do
   begin
   if currentstrip^.bytes > maxstripsize then maxstripsize := currentstrip^.bytes;
   currentstrip := currentstrip^.next;
   end;
 b := memalloc(maxstripsize);
 currentstrip := firststrip;
 count := 0;
 memt := mem;
 w := p^.width;
 rowbytewidth := ((w div 8));
 if (w mod 8) <> 0 then inc(rowbytewidth);
 insertbyte := (rowbytewidth mod 2 = 1);
 while currentstrip <> nil do
   begin
   f.seek(currentstrip^.offset,0);
   return := f.read(b^,currentstrip^.bytes);
   bt := b;
   if count = p^.numstrips - 1
      then h := p^.length mod p^.rowsperstrip
      else h := p^.rowsperstrip;
   for j := 1 to h do
     begin
    { there is a bug here: if the pointerinc crosses the segment *next* time,
       a section of memory will go unread ! - 16 bit only}
     move(bt^,memt^,rowbytewidth);
     {$IFDEF WIN32} inc(bt,rowbytewidth) {$ELSE} pointerinc(bt,rowbytewidth) {$ENDIF};
     if insertbyte then {$IFDEF WIN32} inc(memt,rowbytewidth + 1)
                  {$ELSE} inc(memt,rowbytewidth + 1)  {$ENDIF}
      else {$IFDEF WIN32} inc(memt,rowbytewidth)
                 {$ELSE} inc(memt,rowbytewidth + 1);  {$ENDIF}
     end;
   inc(count);
   currentstrip := currentstrip^.next;
   end;
 deletestripinfo;
 freemem(b,maxstripsize);
end;

{----------------- read bitmap from tiff file routines -----------------}
function supportedformat(p:PTiffImageMap):boolean;
begin
 result := (((p^.imagetype = ti_mono) and (p^.compression = tfc_none))
           or ((p^.imagetype = ti_mono) and (p^.compression = tfc_ccitt)));
end;

procedure TTiffFile.readtiff(p:PTiffImageMap; mem:pointer; memsize:longint);
var f:tfilestream;
begin
 f := tfilestream.create(FFilename,fmOpenRead);
 try
   if (p^.imagetype = ti_mono) and (p^.compression = tfc_none) then
     readuncompressedmono(f,p,mem,memsize) else
   if (p^.imagetype = ti_mono) and (p^.compression = tfc_ccitt) then
     readcompressedmono(f,p,mem,memsize) else ;
 finally
   f.free;
 end;
end;

function TTiffFile.readbitmap:HBitMap;
var memsize,l,w,time:longint;
    membits:pointer;
    p:PTiffImageMap;

begin
 p := getcurrentimagemap;
 if (p = nil) then raise exception.create('No image selected');
 if not supportedformat(p) then raise exception.create('Not a supported image tpye');
 l := p^.length;
 w := p^.width;
 memsize := l * ((w div 8) + 2);
 membits := memalloc(memsize);
 try
   readtiff(p,membits,memsize);
   {create a bmp, rather than a dib. Works just fine for mono images
    in 16 bit, should work in 32 bit, but would be better to use palatte
    and createDIBitmap because sometimes the image is all black or black
    and transparent rather than b/w.}
   result := CreateBitmap(p^.Width, p^.length, 1, 1, membits);
   if result = 0 then raise exception.create('Failed to create Bitmap');
 finally
   freemem(membits,memsize);
 end;
end;


{------------------------ property editor ---------------------------------}
function TFileNameProperty.GetAttributes;
begin
  Result := [paDialog];
end;

Procedure TFilenameProperty.edit;
var
  MPFileOpen: TOpenDialog;
begin
  MPFileOpen := TOpenDialog.Create(Application);
  MPFileOpen.Filename := GetValue;
  MPFileOpen.Filter := 'tiff files|*.tif';
  MPFileOpen.Options := MPFileOpen.Options + [ofPathMustExist,ofFileMustExist];
  try
    if MPFileOpen.Execute then SetValue(MPFileOpen.Filename);
  finally
    MPFileOpen.Free;
  end;
end;

{------------------------ ccitt decoding codes ----------------------------}

{More compact code could be achieved by hardcoding the conversion of
the adddata routine in the source. I have refrained from doing this
in the interests of clarity.}

procedure adddata(p:Pccittdata; rl:word; s:string; stopcode:boolean);
var c,n,i:word;
begin
 n := length(s);
 c := 0;
 for i := 1 to n do c := (c shl 1) or word((s[i] = '1'));
 p^[n,c].length := rl;
 p^[n,c].stop := stopcode;
 p^[n,c].isvalid := true;
end;

procedure fillblackdata;
var i,j:word;
begin
 for i := 2 to 13 do
  for j := 0 to 255 do
   blackdata^[i,j].isvalid := false;
 adddata(blackdata,0, '0000110111',true);
 adddata(blackdata,1, '010',true);
 adddata(blackdata,2, '11',true);
 adddata(blackdata,3, '10',true);
 adddata(blackdata,4, '011',true);
 adddata(blackdata,5, '0011',true);
 adddata(blackdata,6, '0010',true);
 adddata(blackdata,7, '00011',true);
 adddata(blackdata,8, '000101',true);
 adddata(blackdata,9, '000100',true);
 adddata(blackdata,10,'0000100',true);
 adddata(blackdata,11,'0000101',true);
 adddata(blackdata,12,'0000111',true);
 adddata(blackdata,13,'00000100',true);
 adddata(blackdata,14,'00000111',true);
 adddata(blackdata,15,'000011000',true);
 adddata(blackdata,16,'0000010111',true);
 adddata(blackdata,17,'0000011000',true);
 adddata(blackdata,18,'0000001000',true);
 adddata(blackdata,19,'00001100111',true);
 adddata(blackdata,20,'00001101000',true);
 adddata(blackdata,21,'00001101100',true);
 adddata(blackdata,22,'00000110111',true);
 adddata(blackdata,23,'00000101000',true);
 adddata(blackdata,24,'00000010111',true);
 adddata(blackdata,25,'00000011000',true);
 adddata(blackdata,26,'000011001010',true);
 adddata(blackdata,27,'000011001011',true);
 adddata(blackdata,28,'000011001100',true);
 adddata(blackdata,29,'000011001101',true);
 adddata(blackdata,30,'000001101000',true);
 adddata(blackdata,31,'000001101001',true);
 adddata(blackdata,32,'000001101010',true);
 adddata(blackdata,33,'000001101011',true);
 adddata(blackdata,34,'000011010010',true);
 adddata(blackdata,35,'000011010011',true);
 adddata(blackdata,36,'000011010100',true);
 adddata(blackdata,37,'000011010101',true);
 adddata(blackdata,38,'000011010110',true);
 adddata(blackdata,39,'000011010111',true);
 adddata(blackdata,40,'000001101100',true);
 adddata(blackdata,41,'000001101101',true);
 adddata(blackdata,42,'000011011010',true);
 adddata(blackdata,43,'000011011011',true);
 adddata(blackdata,44,'000001010100',true);
 adddata(blackdata,45,'000001010101',true);
 adddata(blackdata,46,'000001010110',true);
 adddata(blackdata,47,'000001010111',true);
 adddata(blackdata,48,'000001100100',true);
 adddata(blackdata,49,'000001100101',true);
 adddata(blackdata,50,'000001010010',true);
 adddata(blackdata,51,'000001010011',true);
 adddata(blackdata,52,'000000100100',true);
 adddata(blackdata,53,'000000110111',true);
 adddata(blackdata,54,'000000111000',true);
 adddata(blackdata,55,'000000100111',true);
 adddata(blackdata,56,'000000101000',true);
 adddata(blackdata,57,'000001011000',true);
 adddata(blackdata,58,'000001011001',true);
 adddata(blackdata,59,'000000101011',true);
 adddata(blackdata,60,'000000101100',true);
 adddata(blackdata,61,'000001011010',true);
 adddata(blackdata,62,'000001100110',true);
 adddata(blackdata,63,'000001100111',true);
 adddata(blackdata,   64,'0000001111',false);
 adddata(blackdata,  128,'000011001000',false);
 adddata(blackdata,  192,'000011001001',false);
 adddata(blackdata,  256,'000001011011',false);
 adddata(blackdata,  320,'000000110011',false);
 adddata(blackdata,  384,'000000110100',false);
 adddata(blackdata,  448,'000000110101',false);
 adddata(blackdata,  512,'0000001101100',false);
 adddata(blackdata,  576,'0000001101101',false);
 adddata(blackdata,  640,'0000001001010',false);
 adddata(blackdata,  704,'0000001001011',false);
 adddata(blackdata,  768,'0000001001100',false);
 adddata(blackdata,  832,'0000001001101',false);
 adddata(blackdata,  896,'0000001110010',false);
 adddata(blackdata,  960,'0000001110011',false);
 adddata(blackdata, 1024,'0000001110100',false);
 adddata(blackdata, 1088,'0000001110101',false);
 adddata(blackdata, 1152,'0000001110110',false);
 adddata(blackdata, 1216,'0000001110111',false);
 adddata(blackdata, 1280,'0000001010010',false);
 adddata(blackdata, 1344,'0000001010011',false);
 adddata(blackdata, 1408,'0000001010100',false);
 adddata(blackdata, 1472,'0000001010101',false);
 adddata(blackdata, 1536,'0000001011010',false);
 adddata(blackdata, 1600,'0000001011011',false);
 adddata(blackdata, 1664,'0000001100100',false);
 adddata(blackdata, 1728,'0000001100101',false);
 adddata(blackdata, 1792,'00000001000',false);
 adddata(blackdata, 1856,'00000001100',false);
 adddata(blackdata, 1920,'00000001101',false);
 adddata(blackdata, 1984,'000000010010',false);
 adddata(blackdata, 2048,'000000010011',false);
 adddata(blackdata, 2112,'000000010100',false);
 adddata(blackdata, 2176,'000000010101',false);
 adddata(blackdata, 2240,'000000010110',false);
 adddata(blackdata, 2304,'000000010111',false);
 adddata(blackdata, 2368,'000000011100',false);
 adddata(blackdata, 2432,'000000011101',false);
 adddata(blackdata, 2496,'000000011110',false);
 adddata(blackdata, 2560,'000000011111',false);
end;

procedure fillwhitedata;
var i,j:word;
begin
 for i := 2 to 13 do
  for j := 0 to 255 do
   whitedata^[i,j].isvalid := false;
 adddata(whitedata, 0,'00110101', true);
 adddata(whitedata, 1,'000111', true);
 adddata(whitedata, 2,'0111', true);
 adddata(whitedata, 3,'1000', true);
 adddata(whitedata, 4,'1011', true);
 adddata(whitedata, 5,'1100', true);
 adddata(whitedata, 6,'1110', true);
 adddata(whitedata, 7,'1111', true);
 adddata(whitedata, 8,'10011', true);
 adddata(whitedata, 9,'10100', true);
 adddata(whitedata,10,'00111', true);
 adddata(whitedata,11,'01000', true);
 adddata(whitedata,12,'001000', true);
 adddata(whitedata,13,'000011', true);
 adddata(whitedata,14,'110100', true);
 adddata(whitedata,15,'110101', true);
 adddata(whitedata,16,'101010', true);
 adddata(whitedata,17,'101011', true);
 adddata(whitedata,18,'0100111', true);
 adddata(whitedata,19,'0001100', true);
 adddata(whitedata,20,'0001000', true);
 adddata(whitedata,21,'0010111', true);
 adddata(whitedata,22,'0000011', true);
 adddata(whitedata,23,'0000100', true);
 adddata(whitedata,24,'0101000', true);
 adddata(whitedata,25,'0101011', true);
 adddata(whitedata,26,'0010011', true);
 adddata(whitedata,27,'0100100', true);
 adddata(whitedata,28,'0011000', true);
 adddata(whitedata,29,'00000010', true);
 adddata(whitedata,30,'00000011', true);
 adddata(whitedata,31,'00011010', true);
 adddata(whitedata,32,'00011011', true);
 adddata(whitedata,33,'00010010', true);
 adddata(whitedata,34,'00010011', true);
 adddata(whitedata,35,'00010100', true);
 adddata(whitedata,36,'00010101', true);
 adddata(whitedata,37,'00010110', true);
 adddata(whitedata,38,'00010111', true);
 adddata(whitedata,39,'00101000', true);
 adddata(whitedata,40,'00101001', true);
 adddata(whitedata,41,'00101010', true);
 adddata(whitedata,42,'00101011', true);
 adddata(whitedata,43,'00101100', true);
 adddata(whitedata,44,'00101101', true);
 adddata(whitedata,45,'00000100', true);
 adddata(whitedata,46,'00000101', true);
 adddata(whitedata,47,'00001010', true);
 adddata(whitedata,48,'00001011', true);
 adddata(whitedata,49,'01010010', true);
 adddata(whitedata,50,'01010011', true);
 adddata(whitedata,51,'01010100', true);
 adddata(whitedata,52,'01010101', true);
 adddata(whitedata,53,'00100100', true);
 adddata(whitedata,54,'00100101', true);
 adddata(whitedata,55,'01011000', true);
 adddata(whitedata,56,'01011001', true);
 adddata(whitedata,57,'01011010', true);
 adddata(whitedata,58,'01011011', true);
 adddata(whitedata,59,'01001010', true);
 adddata(whitedata,60,'01001011', true);
 adddata(whitedata,61,'00110010', true);
 adddata(whitedata,62,'00110011', true);
 adddata(whitedata,63,'00110100', true);
 adddata(whitedata,  64,'11011', false);
 adddata(whitedata, 128,'10010', false);
 adddata(whitedata, 192,'010111', false);
 adddata(whitedata, 256,'0110111', false);
 adddata(whitedata, 320,'00110110', false);
 adddata(whitedata, 384,'00110111', false);
 adddata(whitedata, 448,'01100100', false);
 adddata(whitedata, 512,'01100101', false);
 adddata(whitedata, 576,'01101000', false);
 adddata(whitedata, 640,'01100111', false);
 adddata(whitedata, 704,'011001100', false);
 adddata(whitedata, 768,'011001101', false);
 adddata(whitedata, 832,'011010010', false);
 adddata(whitedata, 896,'011010011', false);
 adddata(whitedata, 960,'011010100', false);
 adddata(whitedata,1024,'011010101', false);
 adddata(whitedata,1088,'011010110', false);
 adddata(whitedata,1152,'011010111', false);
 adddata(whitedata,1216,'011011000', false);
 adddata(whitedata,1280,'011011001', false);
 adddata(whitedata,1344,'011011010', false);
 adddata(whitedata,1408,'011011011', false);
 adddata(whitedata,1472,'010011000', false);
 adddata(whitedata,1536,'010011001', false);
 adddata(whitedata,1600,'010011010', false);
 adddata(whitedata,1664,'011000', false);
 adddata(whitedata,1728,'010011011', false);
 adddata(whitedata,1792,'00000001000',false);
 adddata(whitedata,1856,'00000001100',false);
 adddata(whitedata,1920,'00000001101',false);
 adddata(whitedata,1984,'000000010010',false);
 adddata(whitedata,2048,'000000010011',false);
 adddata(whitedata,2112,'000000010100',false);
 adddata(whitedata,2176,'000000010101',false);
 adddata(whitedata,2240,'000000010110',false);
 adddata(whitedata,2304,'000000010111',false);
 adddata(whitedata,2368,'000000011100',false);
 adddata(whitedata,2432,'000000011101',false);
 adddata(whitedata,2496,'000000011110',false);
 adddata(whitedata,2560,'000000011111',false);
end;

end.

