{*******************************************************}
{                                                       }
{       Delphi Visual Component Library                 }
{                                                       }
{       Copyright (c) 1996 Anna B. Sotnichenko          }
{                                                       }
{       TLayoutPanel - panel that can                   }
{                      layout its children              }
{                                                       }
{*******************************************************}

unit LOPanel;

{$S-,W-,R-}
{$C PRELOAD}

interface

uses WinTypes, WinProcs, Messages, SysUtils, Classes, Controls, Forms, Menus,
     ExtCtrls, Dialogs;

const

  lmParent : TControl = nil; { Parent (myself) pointer }
  ParentNum = -1; { Parent's number }
  Format10 = -10; { format with names indication }

type
  { Specific layout exceptions }
  ELayoutErr = class(Exception);
  ELayoutIncomplete = class(ELayoutErr); { Too complicated layout metrics }
  ELayoutBadRelCtl = class(ELayoutErr);  { Relative control has no layout metrics }

  { Edge and size - had to merge them into one type }
  TEdge = (lmLeft, lmTop, lmRight, lmBottom, lmCenter, lmWidth, lmHeight);

  { Unit identifier }
  TMeasurementUnits = (lmPixels, lmLayoutUnits);

  { Relationship between controls }
  TRelationship = (lmAsIs, lmPercentOf, lmAbove, lmBelow, lmSameAs, lmAbsolute);

const

  lmLeftOf = lmAbove;  { Aliases }
  lmRightOf = lmBelow;

type

  {
  record TLayoutConstraint
  ------ -----------------

  layout constraints are specified as a relationship between an edge/size
  of one window and an edge/size of one of the window's siblings or parent

  it is acceptable for a control to have one of its sizes depend on the
  size of the opposite dimension (e.g. width is twice height)

  distances can be specified in either pixels or layout units

  a layout unit is defined by dividing the font "em" quad into 8 vertical
  and 8 horizontal segments. we get the font by self-sending WM_GETFONT
  (we use the system font if WM_GETFONT returns 0)

  "lmAbove", "lmBelow', "lmLeftOf", and "lmRightOf" are only used with edge
  constraints. they place the window 1 pixel to the indicated side(i.e.
  adjacent to the other window) and then adjust it by "Margin"(e.g. above
  window "A" by 6)

  NOTE: "Margin" is either added to ("lmAbove" and "lmLeftOf") or subtracted
        from("lmBelow" and "lmRightOf") depending on the relationship

  "lmSameAs" can be used with either edges or sizes, and it doesn't offset
  by 1 pixel like the above four relationships did. it also uses "Value"
  (e.g. same width as window "A" plus 10)

  NOTE: "Value" is always *added*. use a negative number if you want the
        effect to be subtractive
  }

  TLayoutConstraint = record
    RelWin : TControl;         { relative control, lmParent for parent }

    MyEdge : TEdge;            { edge or size (width/height) }
    Units : TMeasurementUnits;
    OtherEdge : TEdge;         { edge or size (width/height) }

    { this variant part is just for naming convenience }
    case Relationship : TRelationship of
      lmAbove, lmBelow       : (Margin  : Integer);
      lmSameAs, lmAbsolute   : (Value   : Integer);
      lmPercentOf            : (Percent : Integer);
  end; {TLayoutConstraint}

  {  record TLayoutMetrics
     ----- --------------
     when specifying the layout metrics for a window you specify four layout
     constraints  }

  TLayoutMetrics = record
    X      : TLayoutConstraint;  { Horz1 can be lmLeft, lmCenter, lmRight }
    Y      : TLayoutConstraint;  { Vert1 can be lmTop, lmCenter, lmBottom }
    Width  : TLayoutConstraint;  { Horz2 can be lmWidth, lmCenter, lmRight }
    Height : TLayoutConstraint;  { Vert2 can be lmHeight, lmCenter, lmBottom }
    MinW   : integer;            { Minimal width }
    MaxW   : integer;            { Maximal width }
    MinH   : integer;            { Minimal height }
    MaxH   : integer;            { Maximal height }
    { defaults each co: RelWin=0, MyEdge=(1st from above), Relationship=AsIs }
  end; {TLayoutMetrics}

  { classes and records used by TLayoutPanel }

  TChildMetrics = class;
  TConstraint = class;
  TLayoutPanel = class;

  { Variable represents control's Left, Top, Right or Bottom edge }
  TVariable = record
    Value : integer;
    DeterminedBy : TConstraint;  { 0 if variable is constant }
    Resolved : Boolean;
  end; {TVariable}

  PLayoutMetrics = ^TLayoutMetrics;
  PLayoutConstraint = ^TLayoutConstraint;
  PVariable = ^TVariable;
  TEdgeVariables = array [0..3] of TVariable;

  {
  constraints can have up to three input variables

  the method for solving the constraint is represented as an ordered linear
  combination of the inputs and the constant with the constant expressed last
  }
  TConstraint = class
  public
    Inputs : array [0..2] of PVariable; {three inputs in the equation}
    Output : PVariable;          {Variable to be resolved in a constraint}
    Match : PVariable;           {Match variable: for left - right, for top - bottom}
    Left : Boolean;              {Is this variable left (top) or right (bottom)}
    MinSize : integer;           {Minimal size}
    MaxSize : integer;           {Minimal size}
    OrderedCombination : array [0..3] of single; {coeficients}

    constructor Create;

    function IsResolved : Boolean;  { if its inputs have been resolved }
    function Evaluate : integer;
    function NumActualInputs : integer;
  end; {TConstraint}

  {
  the layout metrics represent four equations. for equations that are
  "absolute" or "as is" we don't add a constraint and just set the variable
  value directly(and mark the variable as constant); otherwise we produce an
  ordered linear combination from the equation and add a constraint
  }
  TChildMetrics = class
  private
    FGeneratedConstraints : boolean;
    FChild : TControl;
    FMetrics : TLayoutMetrics;
    FVariables : TEdgeVariables;  { x => 0, y => 1, right => 2, bottom => 3 }

  public
    Name : string; { NB! Set in ReadMetrics or SetVacantPointers
                     Essential only during the first Layout after ReadMetrics }
    Child : TControl;
    Metrics : TLayoutMetrics;
    Variables : TEdgeVariables;  { x => 0, y => 1, right => 2, bottom => 3 }
    property GeneratedConstraints : boolean read FGeneratedConstraints write FGeneratedConstraints stored false;

    constructor Create( chld : TControl; var metr : TLayoutMetrics );
    procedure ReadFromStream(Reader: TReader);
    procedure WriteToStream(Writer: TWriter; Panel : TLayoutPanel);
  end; {TChildMetrics}

  {
  class TLayoutPanel
  ----- -------------

  when specifying the layout metrics for a window, there are several options:
  e.g. in the horizontal direction,

  Two Edge Constraints in X and Width
    1. left edge and right edge
    2. center edge and right edge
    3. left edge and center edge

  Edge Constraint and Size constraint in X and Width
    4. left edge and size
    5. right edge and size
    6. center edge and size

  the same holds true in the vertical direction for Y and Height

  it is also possible to specify "lmAsIs" in which case we use the windows
  current value

  specifying "lmAbsolute" means that we will use whatever is in data member
  "Value"

  we just name the fields "X" and "Width" and "Y" and "Height",
  although its okay to place a right or center edge constraint in the
  "Width" field and its also okay to place a right edge constraint in
  the "X" field (i.e. option #3)

  however, it's NOT okay to place a width constraint in the "X" or
  "Height" fields or a height constraint in the "Y" or "Width" fields.
  }

  TWhichConstraint = (XConstraint, YConstraint, WidthConstraint, HeightConstraint);

  TLayoutPanel = class (TPanel)
  private
    FBeforeLayout: TNotifyEvent;
    FAfterLayout: TNotifyEvent;
    FFirstAfterRead: boolean;
    FMetrics : TList;
    FConstraints : TList;
    FPlan : TList;
    FVariables : TEdgeVariables;
    FPlanIsDirty : Boolean;
    FFontHeight : integer;
    FLayoutLevel : integer;
    procedure ReadMetrics(Reader: TReader);
    procedure WriteMetrics(Writer: TWriter);

    function  GetChildMetricsByChild( child : TControl) : TChildMetrics;
    function  GetChildMetricsIndexByChild( child : TControl) : integer;

    procedure AddConstraint( metrics : TChildMetrics;
                             c : PLayoutConstraint;
                             whichConstraint : TWhichConstraint );
    procedure BuildConstraints( chldMetrics : TChildMetrics );
    procedure RemoveConstraints( chldMetrics : TChildMetrics );
    procedure RemoveAllConstraints;

    procedure BuildPlan;
    procedure ExecutePlan;
    procedure ClearPlan;

    function LayoutUnitsToPixels( units : integer ) : integer;
    procedure GetFontHeight;

    function GetChildMetrics(Index: Integer): TChildMetrics;
    function GetChildMetricsCount : integer;
    function GetConstraint(Index: Integer): TConstraint;
    function GetPlanItem(Index: Integer): TConstraint;
    procedure Empty( m : TList);
    procedure SetLayoutEnabled( en : boolean );
    function IsLayoutEnabled : boolean;
    function WereAsIsMoved : boolean;
    procedure SetVacantPointers;
    function HasData( Filer : TFiler ): boolean;
    function EqualMetrics( Panel: TLayoutPanel ): boolean;
    function EqualConstraints( var c1, c2: TLayoutConstraint ): boolean;

  protected
    procedure AlignControls(AControl: TControl; var Rect: TRect); override;
    procedure RemoveSuperfluousMetrics(RemoveNils : boolean);
    procedure RemoveFoundChildLayoutMetrics( chldMetrics : TChildMetrics );
    procedure SetRelWinLayoutMetrics( var constraint : TLayoutConstraint );

    property Constraints[Index: Integer]: TConstraint read GetConstraint;
    property Plan[Index: Integer]: TConstraint read GetPlanItem;
    procedure DefineProperties( Filer: TFiler ); override;
    function GetRelControlByNum( N : integer ) : TControl;
    function GetRelControlNum( var Control : TControl ) : integer;

  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;

    { Pair of functions incrementing or decrementing Layout level }
    procedure DisableLayout;
    procedure EnableLayout;

    {
    causes the receiver to size/position its children according to the
    specified layout metrics

    if you change the layout metrics for a child window call Layout
    to have the changes take effect
    }
    procedure Layout;

    procedure SetChildLayoutMetrics( child : TControl; var metrics : TLayoutMetrics );
    function GetControlByName(const AName: string): TControl;
    function GetChildLayoutMetrics( child : TControl; var metrics : TLayoutMetrics) : Boolean;
    function RemoveChildLayoutMetrics( child : TControl ) : Boolean;
    function IsChildControl( child : Pointer ) : integer;

    property ChildMetricsCount: Integer read GetChildMetricsCount;
    property ChildMetrics[Ind: Integer]: TChildMetrics read GetChildMetrics;
    property PlanIsDirty : Boolean read FPlanIsDirty;

  published
    property Metrics: TList read FMetrics write Empty stored False;
    property LayoutEnabled: boolean read IsLayoutEnabled write SetLayoutEnabled;
    property BeforeLayout: TNotifyEvent read FBeforeLayout write FBeforeLayout;
    property AfterLayout: TNotifyEvent read FAfterLayout write FAfterLayout;
  end; {TLayoutPanel}

procedure SetConstraint( var constraint : TLayoutConstraint; edge : TEdge;
                         rel : TRelationship; otherWin : TControl;
                         otherEdge : TEdge; value : integer );
procedure AssignConstraint( var constraint : TLayoutConstraint;
                            otherConstraint : TLayoutConstraint );
procedure LeftOfConstraint(  var constraint : TLayoutConstraint;
                             sibling : TControl; margin : integer );
procedure RightOfConstraint(  var constraint : TLayoutConstraint;
                              sibling : TControl; margin : integer );
procedure AboveConstraint( var constraint : TLayoutConstraint;
                           sibling : TControl; margin : integer );
procedure BelowConstraint( var constraint : TLayoutConstraint;
                           sibling : TControl; margin : integer );
procedure SameAsConstraint( var constraint : TLayoutConstraint;
                            otherWin : TControl; edge : TEdge );
procedure PercentOfConstraint( var constraint : TLayoutConstraint;
                               otherWin : TControl; edge : TEdge;
                               percent : integer );
procedure AbsoluteConstraint( var constraint : TLayoutConstraint;
                              edge : TEdge; val : integer );
procedure ReadConstraint( var constraint : TLayoutConstraint;
                          Reader : TReader );
procedure WriteConstraint( var constraint : TLayoutConstraint;
                           Writer : TWriter; Panel : TLayoutPanel);
procedure AssignLayoutMetrics( var metrics : TLayoutMetrics;
                               otherMetrics : TLayoutMetrics );
procedure InitLayoutMetrics( var metrics : TLayoutMetrics );
procedure SetMeasurementUnits( var metrics : TLayoutMetrics;
                               units : TMeasurementUnits );

procedure Flush( Lst : TList );

var
  { This flag works at design time only
    It prevents function SetChildLayoutMetrics from testing
    the existance and setting AsIs LayoutMetrics for relative control
    This flag is set in TLayoutForm.FormClose to guarantee the right
    order in FMetrics list (at design time FMetrics[i] must belong to Control[i])
    }
  DontCareAboutRelMetrics : boolean;


implementation

{DDDDebug
uses ShowMetr;}

var
  Format : integer;

{ Common functions }

{--------------------------------------- SetConstraint }
{ Setting arbitrary edge constraints }
procedure SetConstraint( var constraint : TLayoutConstraint; edge : TEdge;
                         rel : TRelationship; otherWin : TControl;
                         otherEdge : TEdge; value : integer );
begin
  constraint.RelWin := otherWin;
  constraint.MyEdge := edge;
  constraint.Relationship := rel;
  constraint.OtherEdge := otherEdge;
  constraint.Value := value;
end; {SetConstraint}

{--------------------------------------- AssignConstraint }
{ Assign one constraint to anoter one }
procedure AssignConstraint( var constraint : TLayoutConstraint;
                            otherConstraint : TLayoutConstraint );
begin
  SetConstraint( constraint, otherConstraint.MyEdge,
                 otherConstraint.Relationship,
                 otherConstraint.RelWin,
                 otherConstraint.OtherEdge,
                 otherConstraint.Value );
end; {SetConstraint}

{
 these four procedures can be used to position your window with
 respective to a sibling window. you specify the sibling window and an
 optional margin between the two windows
}
{--------------------------------------- LeftOfConstraint }
procedure LeftOfConstraint( var constraint : TLayoutConstraint;
                            sibling : TControl; margin : integer );
begin
  SetConstraint(constraint, lmRight, lmLeftOf, sibling, lmLeft, margin);
end; {LeftOfConstraint}

{--------------------------------------- RightOfConstraint }
procedure RightOfConstraint(  var constraint : TLayoutConstraint;
                              sibling : TControl; margin : integer );
begin
  SetConstraint(constraint, lmLeft, lmRightOf, sibling, lmRight, margin);
end; {RightOfConstraint}

{--------------------------------------- AboveConstraint }
procedure AboveConstraint( var constraint : TLayoutConstraint;
                           sibling : TControl; margin : integer );
begin
  SetConstraint(constraint, lmBottom, lmAbove, sibling, lmTop, margin);
end; {AboveConstraint}

{--------------------------------------- BelowConstraint }
procedure BelowConstraint( var constraint : TLayoutConstraint;
                           sibling : TControl; margin : integer );
begin
  SetConstraint(constraint, lmTop, lmBelow, sibling, lmBottom, margin);
end; {BelowConstraint}

{ these two work on the same edge, e.g. "SameAs(win, lmLeft)" means
  that your left edge should be the same as the left edge of "otherWin" }

{--------------------------------------- SameAsConstraint }
procedure SameAsConstraint( var constraint : TLayoutConstraint;
                            otherWin : TControl; edge : TEdge );
begin
  SetConstraint(constraint, edge, lmSameAs, otherWin, edge, 0);
end; {SameAsConstraint}

{--------------------------------------- PercentOfConstraint }
procedure PercentOfConstraint( var constraint : TLayoutConstraint;
                               otherWin : TControl; edge : TEdge;
                               percent : integer );
begin
  SetConstraint(constraint, edge, lmPercentOf, otherWin, edge, percent);
end; {PercentOfConstraint}

{--------------------------------------- AbsoluteConstraint }
{ setting an edge to a fixed value }
procedure AbsoluteConstraint( var constraint : TLayoutConstraint;
                              edge : TEdge; val : integer );
begin
  with constraint do
  begin
    MyEdge := edge; Value := val; Relationship := lmAbsolute;
  end; {with}
end; {AbsoluteConstraint}

{--------------------------------------- ReadConstraint }
{ read constraint from stream }
procedure ReadConstraint( var constraint : TLayoutConstraint;
                          Reader : TReader );
begin
  with constraint, Reader do
  begin
    longint( RelWin ) := ReadInteger;  { In fact, sequential number of corresponding
                                         child metrics }
    MyEdge  := TEdge( ReadInteger );    { edge or size (width/height) }
    Units := TMeasurementUnits( ReadInteger );
    OtherEdge := TEdge( ReadInteger );    { edge or size (width/height) }
    Relationship := TRelationship( ReadInteger );
    Value := ReadInteger;

  end; {with}
end; { ReadConstraint }

{--------------------------------------- WriteConstraint }
{ write constraint from stream }
procedure WriteConstraint( var constraint : TLayoutConstraint;
                           Writer : TWriter; Panel : TLayoutPanel );
begin
  with constraint, Writer do
  begin
    WriteInteger( Panel.GetRelControlNum( RelWin ) );   { In fact, number of relative control }
    WriteInteger( ord( MyEdge ) );       { edge or size (width/height) }
    WriteInteger( ord( Units ) );
    WriteInteger( ord( OtherEdge ) );
    WriteInteger( ord( Relationship ) );
    WriteInteger( Value );
  end; {with}
end; { WriteConstraint }

{--------------------------------------- AssignLayoutMetrics }
{ Assign values to the fields of TLayoutMetrics }
procedure AssignLayoutMetrics( var metrics : TLayoutMetrics;
                               otherMetrics : TLayoutMetrics );
begin
  AssignConstraint( metrics.X, otherMetrics.X );
  AssignConstraint( metrics.Y, otherMetrics.Y );
  AssignConstraint( metrics.Width, otherMetrics.Width );
  AssignConstraint( metrics.Height, otherMetrics.Height );
  metrics.MinW := otherMetrics.MinW;
  metrics.MaxW := otherMetrics.MaxW;
  metrics.MinH := otherMetrics.MinH;
  metrics.MaxH := otherMetrics.MaxH;
end; {AssignLayoutMetrics}

{--------------------------------------- InitLayoutMetrics }
{ Initialize layout metrtics }
procedure InitLayoutMetrics( var metrics : TLayoutMetrics );
begin
  with metrics do
  begin
    X.RelWin := nil;
    X.MyEdge := lmLeft;
    X.OtherEdge := lmLeft;
    X.Relationship := lmAsIs;
    X.Units := lmPixels;
    X.Value := 0;

    Y.RelWin := nil;
    Y.MyEdge := lmTop;
    Y.OtherEdge := lmTop;
    Y.Relationship := lmAsIs;
    Y.Units := lmPixels;
    Y.Value := 0;

    Width.RelWin := nil;
    Width.MyEdge := lmWidth;
    Width.OtherEdge := lmWidth;
    Width.Relationship := lmAsIs;
    Width.Units := lmPixels;
    Width.Value := 0;

    Height.RelWin := nil;
    Height.MyEdge := lmHeight;
    Height.OtherEdge := lmHeight;
    Height.Relationship := lmAsIs;
    Height.Units := lmPixels;
    Height.Value := 0;

    MinW := 0;
    MaxW := 0;
    MinH := 0;
    MaxH := 0;
  end; {with}
end; {InitLayoutMetrics}

{--------------------------------------- SetMeasurementUnits }
procedure SetMeasurementUnits( var metrics : TLayoutMetrics;
                               units : TMeasurementUnits );
begin
  with metrics do
  begin
    X.Units := units;
    Y.Units := units;
    Width.Units := units;
    Height.Units := units;
  end; {with}
end; {SetMeasurementUnits}

{--------------------------------------- Flush }
{ Destroy all items in the list if
  they are proved to be Objects }
procedure Flush( Lst : TList );
var Obj : TObject;
begin
  with Lst do
    while Count > 0 do
    begin
      Obj := TObject( Last );
      Delete(Count - 1);
      Obj.Destroy;
    end; { while }
end; {Flush}

{ TChildMetrics class implementation }

{--------------------------------------- TChildMetrics.Create }
{ initialize fields }
constructor TChildMetrics.Create( chld : TControl; var metr : TLayoutMetrics );
begin
  Child := chld;
  AssignLayoutMetrics( Metrics, metr );
  GeneratedConstraints := False;
  Name := ''; { Always empty because it means nothing when Child <> nil }
end; {TChildMetrics.Create}

{--------------------------------------- TChildMetrics.ReadFromStream }
{ Read metrics from stream }
procedure TChildMetrics.ReadFromStream( Reader: TReader );
begin
  Child := nil; { no child yet }

  { Check if it is a string. Interpret it as child name }
  if Format <= Format10 then
    Name := Reader.ReadString
  else
    Name := '';

  with Metrics do
  begin
    ReadConstraint( X, Reader );
    ReadConstraint( Y, Reader );
    ReadConstraint( Metrics.Width, Reader );
    ReadConstraint( Metrics.Height, Reader );
    MinW := Reader.ReadInteger;
    MaxW := Reader.ReadInteger;
    MinH := Reader.ReadInteger;
    MaxH := Reader.ReadInteger;
  end; {with}
end; {TChildMetrics.ReadFromStream}

{--------------------------------------- TChildMetrics.WriteToStream }
{ Write metrics to stream }
procedure TChildMetrics.WriteToStream(Writer: TWriter; Panel : TLayoutPanel);
begin
  { Store child name }
  if Child <> nil then
    Writer.WriteString( Child.Name );

  with Metrics do
  begin
    WriteConstraint( X, Writer, Panel );
    WriteConstraint( Y, Writer, Panel );
    WriteConstraint( Metrics.Width, Writer, Panel );
    WriteConstraint( Metrics.Height, Writer, Panel );
    Writer.WriteInteger( MinW );
    Writer.WriteInteger( MaxW );
    Writer.WriteInteger( MinH );
    Writer.WriteInteger( MaxH );
  end;
end; {TChildMetrics.ReadFromStream}

{ TConstraint class implementation }

{--------------------------------------- TConstraint.Create }
{ initialize fields }
constructor TConstraint.Create;
begin
  Inputs[0] := nil;
  Inputs[1] := nil;
  Inputs[2] := nil;
  OrderedCombination[0] := 1.0;
  OrderedCombination[1] := 1.0;
  OrderedCombination[2] := 1.0;
  OrderedCombination[3] := 0;
  Output := nil;
  Left := false;
  Match := nil;
  MinSize := 0;
  MaxSize := 0;
end; { TConstraint.Create }

{--------------------------------------- TConstraint.IsResolved }
{ True if every input variable in it is resolved }
function TConstraint.IsResolved : Boolean;
begin
  Result := ( ( Inputs[0] = nil ) or ( Inputs[0]^.Resolved ) ) and
            ( ( Inputs[1] = nil ) or ( Inputs[1]^.Resolved ) ) and
            ( ( Inputs[2] = nil ) or ( Inputs[2]^.Resolved ) );
end; {TConstraint.IsResolved}

{--------------------------------------- TConstraint.Evaluate }
{ Evaluate the constraint by calculating linear combination
  of inputs and constant  }
function TConstraint.Evaluate : integer;
var
  value : single;
begin
  value := OrderedCombination[3];  { initialize to constant part }

  if Inputs[0] <> nil then
    value := value + OrderedCombination[0] * Inputs[0]^.Value;

  if Inputs[1] <> nil then
    value := value + OrderedCombination[1] * Inputs[1]^.Value;

  if Inputs[2] <> nil then
    value := value + OrderedCombination[2] * Inputs[2]^.Value;

  Result := Round( value );

  {Check for extremal values}

  if ( MinSize > 0 ) and
     ( Abs( Result - Match^.Value + 1 ) < MinSize ) then
  begin
    if Left then
      Result := Match^.Value - MinSize + 1
    else
      Result := Match^.Value + MinSize - 1;
  end; {if}

  if ( MaxSize > 0 ) and
     ( Abs( Result - Match^.Value + 1 ) > MaxSize ) then
  begin
    if Left then
      Result := Match^.Value - MaxSize + 1
    else
      Result := Match^.Value + MaxSize - 1;
  end; {if}

end; { TConstraint.Evaluate }

{--------------------------------------- TConstraint.NumActualInputs }
{ Only several first inputs can be not nil }
function TConstraint.NumActualInputs : integer;
var
  i : integer;
begin
  Result := 3;

  for i := 0 to 2 do
    if Inputs[i] = nil then
      Result := i;
end; { TConstraint.NumActualInputs }


{ TLayoutPanel class implementation }

{--------------------------------------- TLayoutPanel.Create }
{ Initialize variables }
constructor TLayoutPanel.Create(AOwner: TComponent);
begin
  { Initialize virtual bases, in case the derived-most used default ctor }
  inherited Create( AOwner );

  FPlanIsDirty := false;
  FLayoutLevel := 0;
  FFirstAfterRead := false; {Panel can be created dynamically, not read from the stream}

  {
  allocate variables for the parent's left, top, right, and bottom and
  mark them as resolved
  }
  FVariables[0].Resolved := true;
  FVariables[1].Resolved := true;
  FVariables[2].Resolved := true;
  FVariables[3].Resolved := true;

  FMetrics := TList.Create;
  FConstraints := TList.Create;
  FPlan := TList.Create;

{DDDDebug
  if ShowMetricsForm = nil then
  begin
    ShowMetricsForm := TShowMetricsForm.Create( Application );
    ShowMetricsForm.Show;
  end; {if}
end; {TLayoutPanel.Create}

{--------------------------------------- TLayoutPanel.Destroy }
{ Destroy all lists that were created }
destructor TLayoutPanel.Destroy;
begin
  Flush( FMetrics );
  Flush( FConstraints );
  Flush( FPlan );
  FMetrics.Free;
  FConstraints.Free;
  FPlan.Free;

  inherited Destroy;
end; { TLayoutPanel.Destroy }

{--------------------------------------- TLayoutPanel.AlignControls }
{ Some controls could be removed since the last call
  This can happen when AControl = nil }
procedure TLayoutPanel.AlignControls(AControl: TControl; var Rect: TRect);
begin
  inherited AlignControls(AControl, Rect);

  if AControl = nil then
    RemoveSuperfluousMetrics(false);

  Layout;
end; {TLayoutPanel.AlignControls}

{--------------------------------------- TLayoutPanel.RemoveSuperfluousMetrics }
{ If metrics has no corresponding control remove it }
procedure TLayoutPanel.RemoveSuperfluousMetrics(RemoveNils : boolean);
var
  chldMetrics, nextMetrics : TChildMetrics;
begin
{DDDDebug
  ShowMetricsForm.Memo1.Lines.Add(Parent.Name+' RemoveSuperfluousMetrics' + IntToStr( integer(RemoveNils) ) );
  ShowMetricsForm.ShowMetrics(Self, false);}

  nextMetrics := ChildMetrics[0];

  while nextMetrics <> nil do
  begin
    chldMetrics := nextMetrics;
    nextMetrics := ChildMetrics[FMetrics.IndexOf( chldMetrics )+1];

    if ( chldMetrics.Child <> nil ) then
    begin
      if ( IsChildControl( chldMetrics.Child ) < 0 ) then
        RemoveFoundChildLayoutMetrics( chldMetrics );
    end {if ( chldMetrics.Child <> nil )}
    else
      if ( RemoveNils ) then
        RemoveFoundChildLayoutMetrics( chldMetrics );
  end; {while}

{DDDDebug
  ShowMetricsForm.Memo1.Lines.Add(Parent.Name+' After RemoveSuperfluousMetrics');
  ShowMetricsForm.ShowMetrics(Self, false);}
end; {TLayoutPanel.RemoveSuperfluousMetrics}

{--------------------------------------- TLayoutPanel.SetChildLayoutMetrics }
{ Set layout metrics for control }
procedure TLayoutPanel.SetChildLayoutMetrics( child : TControl;
                                              var metrics : TLayoutMetrics );
var
  chldMetrics : TChildMetrics;
  i : integer;
begin

  DisableLayout;
  FPlanIsDirty := true;

  { Traditional alignment can play only if
    all for edges are AsIs constrained }
{
  if ( metrics.X.Relationship <> lmAsIs )      or
     ( metrics.Y.Relationship <> lmAsIs )      or
     ( metrics.Width.Relationship <> lmAsIs )  or
     ( metrics.Height.Relationship <> lmAsIs ) then
    child.Align := alNone;
}

  for i := 0 to FMetrics.Count-1 do
    if ChildMetrics[i].Child = child then
    begin
      chldMetrics := ChildMetrics[i];
      AssignLayoutMetrics( chldMetrics.Metrics, metrics );
      {get rid of the old constraints}
      RemoveConstraints( chldMetrics );
      { Set AsIs constraints for metrics for the relative child
        if it has no metrics yet }
      SetRelWinLayoutMetrics( metrics.X );
      SetRelWinLayoutMetrics( metrics.Y );
      SetRelWinLayoutMetrics( metrics.Width );
      SetRelWinLayoutMetrics( metrics.Height );
      EnableLayout;
      Exit;
    end; {if chldMetrics.Child = child}

  FMetrics.Add( TChildMetrics.Create(child, metrics) );

  if ( not ( csdesigning in ComponentState ) ) or
     ( not DontCareAboutRelMetrics ) then
  begin
    { Set AsIs constraints for metrics for the relative child
      if it has no metrics yet }
    SetRelWinLayoutMetrics( metrics.X );
    SetRelWinLayoutMetrics( metrics.Y );
    SetRelWinLayoutMetrics( metrics.Width );
    SetRelWinLayoutMetrics( metrics.Height );
  end; { if }

  EnableLayout;
end; { TLayoutPanel.SetChildLayoutMetrics }

{--------------------------------------- TLayoutPanel.SetRelWinLayoutMetrics }
{ Set AsIs metrics for any relative that has no metrics yet }
procedure TLayoutPanel.SetRelWinLayoutMetrics( var constraint : TLayoutConstraint );
var
  chldMetrics : TChildMetrics;
  metrics : TLayoutMetrics;
  i : integer;
begin
  { Do nothing if relative window is unimportant }
  if ( constraint.RelWin = nil )                  or
     ( IsChildControl( constraint.RelWin ) < 0 ) or
     ( constraint.Relationship = lmAsIs )         or
     ( constraint.Relationship = lmAbsolute )     then
    Exit;

  { Try to find corresponding child metrics }
  for i := 0 to FMetrics.Count-1 do
    if constraint.RelWin = ChildMetrics[i].Child then
      Exit;

  { Set AsIs metrics }
  SetConstraint(metrics.X, lmLeft, lmAsIs, nil, lmLeft, 0);
  SetConstraint(metrics.Y, lmTop, lmAsIs, nil, lmRight, 0);
  SetConstraint(metrics.Width, lmRight, lmAsIs, nil, lmRight, 0);
  SetConstraint(metrics.Height, lmBottom, lmAsIs, nil, lmBottom, 0);

  { Do not call SetChildLayoutMetrics because it
    will search the list of metrics for another time }
  FMetrics.Add( TChildMetrics.Create(constraint.RelWin, metrics) );
end; { TLayoutPanel.SetRelWinLayoutMetrics }

{--------------------------------------- TLayoutPanel.RemoveChildLayoutMetrics }
{ Remove child (layout) metrics for a given child (if found) and update
  other children as necessary }
function TLayoutPanel.RemoveChildLayoutMetrics( child : TControl ) : Boolean;
var
  i : integer;
begin
  { Try to find corresponding child control }

  for i := 0 to FMetrics.Count-1 do
    if ChildMetrics[i].Child = child then
    begin
      RemoveFoundChildLayoutMetrics( ChildMetrics[i] );
      Result := True;
      Exit;
    end; {if}

  Result := False;
end; {TLayoutPanel.RemoveChildLayoutMetrics}

{--------------------------------------- TLayoutPanel.RemoveFoundChildLayoutMetrics }
{ Remove child (layout) metrics (that exists) and update
  other children as necessary }
procedure TLayoutPanel.RemoveFoundChildLayoutMetrics( chldMetrics : TChildMetrics );
var
  cm : TChildMetrics;
  i : integer;
begin
  { unlink target metrics from list & clean up a bit }
  RemoveConstraints( chldMetrics );

  {
  Update other child metrics now that removed metric is gone
  Check for case where new relWin is lmParent and adjust other edge
  to be what removed window was using. If an 'edge' is really a size,
  then give up & just leave it asis. If the removed window had an edge
  that was really a size, then use the other constraint in that
  dimension (X or Y)
  Attention !!! This must be done when chldMetrics.Child <> nil
  }
  if chldMetrics.Child <> nil then
    for i := 0 to FMetrics.Count-1 do
    begin
      cm := ChildMetrics[i];
      { Check if some constraint is relative to the removed child }

      if cm.Metrics.X.RelWin = chldMetrics.Child then
      begin
        RemoveConstraints(cm);
        cm.Metrics.X.RelWin := chldMetrics.Metrics.X.RelWin;

        if cm.Metrics.X.RelWin = lmParent then
          cm.Metrics.X.OtherEdge := chldMetrics.Metrics.X.OtherEdge;
      end; { if cm.Metrics.X.RelWin = }

      if cm.Metrics.Y.RelWin = chldMetrics.Child then
      begin
        RemoveConstraints(cm);
        cm.Metrics.Y.RelWin := chldMetrics.Metrics.Y.RelWin;

        if cm.Metrics.Y.RelWin = lmParent then
          cm.Metrics.Y.OtherEdge := chldMetrics.Metrics.Y.OtherEdge;
      end; { if cm.Metrics.Y.RelWin = }

      if cm.Metrics.Width.RelWin = chldMetrics.Child then
      begin
        RemoveConstraints(cm);

        if cm.Metrics.Width.MyEdge = lmWidth then
          cm.Metrics.Width.Relationship := lmAsIs
        else
        begin
          if chldMetrics.Metrics.Width.MyEdge = lmWidth then
          begin
            cm.Metrics.Width.RelWin := chldMetrics.Metrics.X.RelWin;

            if cm.Metrics.Width.RelWin = lmParent then
              cm.Metrics.Width.OtherEdge := chldMetrics.Metrics.X.OtherEdge;
          end { if chldMetrics.Metrics.Width.MyEdge = }
          else
          begin
            cm.Metrics.Width.RelWin := chldMetrics.Metrics.Width.RelWin;

            if cm.Metrics.Width.RelWin = lmParent then
              cm.Metrics.Width.OtherEdge := chldMetrics.Metrics.Width.OtherEdge;
          end {else}
        end {else}
      end; { if cm.Metrics.Width.RelWin = }

      if cm.Metrics.Height.RelWin = chldMetrics.Child then
      begin
        RemoveConstraints(cm);

        if cm.Metrics.Height.MyEdge = lmHeight then
          cm.Metrics.Height.Relationship := lmAsIs
        else
        begin
          if chldMetrics.Metrics.Height.MyEdge = lmHeight then
          begin
            cm.Metrics.Height.RelWin := chldMetrics.Metrics.Y.RelWin;

            if cm.Metrics.Height.RelWin = lmParent then
              cm.Metrics.Height.OtherEdge := chldMetrics.Metrics.Y.OtherEdge;
          end { if chldMetrics.Metrics.Height.MyEdge = }
          else
          begin
            cm.Metrics.Height.RelWin := chldMetrics.Metrics.Height.RelWin;

            if cm.Metrics.Height.RelWin = lmParent then
              cm.Metrics.Height.OtherEdge := chldMetrics.Metrics.Height.OtherEdge;
          end {else}
        end {else}
      end; { if cm.Metrics.Height.RelWin = }

    end; {for}

  { finaly, delete target metrics }
  FMetrics.Remove( chldMetrics );
  chldMetrics.Free;
end; { TLayoutPanel.RemoveFoundChildLayoutMetrics }

{--------------------------------------- TLayoutPanel.IsChildControl }
{ Check if a pointer is child control.
  Returns control index or -1 if not found }
function TLayoutPanel.IsChildControl( child : Pointer ) : integer;
var
  i : integer;
begin
  Result := -1; { assume yes }

  for i := 0 to ControlCount-1 do
    if Controls[i] = child then
    begin
      Result := i;
      Exit;
    end; {if}
end; { TLayoutPanel.IsChildControl }

{--------------------------------------- TLayoutPanel.GetChildMetricsByChild }
{ Return child metrics corresponding to the control
  If metrics is not found return nil }
function TLayoutPanel.GetChildMetricsByChild( child : TControl ) : TChildMetrics;
var
  i : integer;
begin
  { Try to find corresponding child control }
  for i := 0 to FMetrics.Count-1 do
    if ChildMetrics[i].Child = child then
    begin
      Result := ChildMetrics[i];
      Exit;
    end; {if}

  Result := nil;
end; { TLayoutPanel.GetChildMetricsByChild }

{--------------------------------------- TLayoutPanel.GetChildMetricsIndexByChild }
{ Return child metrics index corresponding to the control
  If metrics is not found return nil }
function TLayoutPanel.GetChildMetricsIndexByChild( child : TControl ) : integer;
var
  i : integer;
begin
  { Try to find corresponding child control }
  for i := 0 to FMetrics.Count-1 do
    if ChildMetrics[i].Child = child then
    begin
      Result := i;
      Exit;
    end; {if}

  Result := -1;
end; { TLayoutPanel.GetChildMetricsIndexByChild }

{--------------------------------------- TLayoutPanel.GetChildLayoutMetrics }
{ Return layout metrics corresponding to the control
  If metrics is not found return false }
function TLayoutPanel.GetChildLayoutMetrics( child : TControl;
                                             var metrics : TLayoutMetrics) : Boolean;
var
  i : integer;
begin
  { Try to find corresponding child control }
  for i := 0 to FMetrics.Count-1 do
    if ChildMetrics[i].Child = child then
    begin
      AssignLayoutMetrics( metrics, ChildMetrics[i].Metrics );
      Result := true;
      Exit;
    end; {if}

  Result := false;
end; { TLayoutPanel.GetChildLayoutMetrics }

{--------------------------------------- TLayoutPanel.LayoutUnitsToPixels }
{ Calculate pixels value by layout units }
function TLayoutPanel.LayoutUnitsToPixels( units : integer ) : integer;
begin
  Result := Trunc( (units * FFontHeight + 4) / 8.0 );
end; { TLayoutPanel.LayoutUnitsToPixels }

{--------------------------------------- TLayoutPanel.ExecutePlan }
{ Execute plan that was built before }
procedure TLayoutPanel.ExecutePlan;
var
  i : integer;
begin
  { Try to find corresponding child control }
  for i := 0 to FPlan.Count-1 do
    Plan[i].Output^.Value := Plan[i].Evaluate;

end; {TLayoutPanel.ExecutePlan}

{--------------------------------------- TLayoutPanel.ClearPlan }
{ Clear plan that was built before }
procedure TLayoutPanel.ClearPlan;
var
  c : TConstraint;
begin
  while FPlan.Count > 0 do
  begin
    c := FPlan.Last;
    FPlan.Remove(c);
    FConstraints.Add(c);
  end; {while}
end; {TLayoutPanel.ClearPlan}

{--------------------------------------- TLayoutPanel.BuildPlan }
{ Try to resolve more and more variables }
procedure TLayoutPanel.BuildPlan;
var
  chldMetrics : TChildMetrics;
  lastInPlan : TConstraint;
  variable : PVariable;
  i, j : integer;
  foundOne : boolean;
  c : TConstraint;
  previous : TConstraint;
  resolved : TConstraint;

begin
  lastInPlan := nil;
  ClearPlan;

  { mark all variables that aren't determined by a constraint as resolved }
  for i := 0 to FMetrics.Count-1 do
    for j := 0 to 3 do
    begin
      variable := @ChildMetrics[i].Variables[j];
      variable^.Resolved := variable^.DeterminedBy = nil;
    end; {for}

  {
  uses local propagation as much as possible (because it's fast)
  if cycles exist then we will end up with constraints that haven't been
  added to the plan. we convert the remaining constraints into simultaneous
  linear equations which we solve using Gaussian elimination

  look for constraints that have all their input variables resolved and
  append them to the plan
  }
  foundOne := true;

  while foundOne do
  begin
    c := Constraints[0];
    foundOne := false;

    while c <> nil do
    begin

      if c.IsResolved then
      begin
        resolved := c;

        c.Output^.Resolved := True;

        { If match variable is not resolved yet set to 0 extremal sizes
          because we can move only edge that was resolved last }
        if not c.Match^.Resolved then
        begin
          c.MinSize := 0;
          c.MaxSize := 0;
        end; { if not c.Match^.Resolved }

        foundOne := True;

        { extract the constraint from the list of constraints }
        c := Constraints[ FConstraints.IndexOf(c)+1 ];
        FConstraints.Remove(resolved);

        { append the constraint to the plan }
        FPlan.Add(resolved);
      end { if c.IsResolved }
      else
        c := Constraints[ FConstraints.IndexOf(c)+1 ];
    end; { while c }
  end; { while foundOne }

  { Gaussian elimination not currently supported--give up }

  if FConstraints.Count > 0 then
    raise ELayoutIncomplete.Create( 'Too complicated layout metrics !' );
  {  THROW( TXWindow(this, IDS_LAYOUTINCOMPLETE) );}
end; { TLayoutPanel.BuildPlan }

{--------------------------------------- FindInput }
{ Internal function. Returns the number of variable
  in constraint or -1 if variable is not there }
function FindInput( SimpConstr : TConstraint ; input : PVariable ) : integer;
var
  i : integer;
begin
  Result := -1;

  for i := 0 to 2 do
    if SimpConstr.Inputs[i] = input then
      Result := i;
end; {FindInput}

{--------------------------------------- Simplify }
{
simplify constraint "SimpConstr" by substituting constraint "_using"

we do this when the two constraints are defined in terms of each other
  1. the output of "SimpConstr" is an input of "_using"
  2. the output of "_using" is an input of "SimpConstr"

we do this to aprocedure a layout cycle

"output" is the output variable for constraint "_using"
}
procedure Simplify( SimpConstr : TConstraint;
                    output : PVariable;
                    _using : TConstraint);
var
  outputOfSimplify, target : integer;
  numInputsOfUsing, i : integer;
  numInputsOfSimplify, newInputs : integer;
  commonInputs : array [0..2] of integer;
  m, f : single;

begin
  if SimpConstr = nil then
    Exit;

  outputOfSimplify := FindInput( _using, SimpConstr.Output);  { check #1 }
  target := FindInput( SimpConstr, output );  { check #2 }

  if ( outputOfSimplify > -1 ) and
     ( target > -1 )           then
  begin
    numInputsOfUsing := _using.NumActualInputs;

    { count how many inputs are common between "SimpConstr" and "_using" }
    for i := 0 to numInputsOfUsing-1 do
      commonInputs[i] := FindInput( SimpConstr, _using.Inputs[i]);

    {
    since constraints only have room for 3 inputs we can not SimpConstr if the
    total number of the existing inputs minus the input we are going to back
    substitute for plus the number of inputs added by "_using" (i.e. inputs
    not common between the two constraints) exceeds 3
    }
    numInputsOfSimplify := SimpConstr.NumActualInputs - 1;
    newInputs := 0;

    { compute the number of additional inputs contributed by "_using" }
    for i := 0 to numInputsOfUsing-1 do
      if ( commonInputs[i] = -1 ) and
         ( i <> outputOfSimplify ) then
        Inc(newInputs);

    if numInputsOfSimplify + newInputs > 3 then
      Exit;

    m := SimpConstr.OrderedCombination[target];

    { adjust the constant part }
    SimpConstr.OrderedCombination[3] := SimpConstr.OrderedCombination[3] +
                                       m * _using.OrderedCombination[3];

    { merge the common inputs }
    for i := 0 to numInputsOfUsing-1 do
      if commonInputs[i] > -1 then
        SimpConstr.OrderedCombination[commonInputs[i]] :=
          SimpConstr.OrderedCombination[commonInputs[i]] + m * _using.OrderedCombination[i];

    SimpConstr.Inputs[target] := nil;  { input has been back substituted out }

    {
    if necessary shift the inputs following "output" (and their associated
    mutiplier) left by one...
    }
    for i := target + 1 to 2 do
      if SimpConstr.Inputs[i] <> nil then
      begin
        SimpConstr.Inputs[i - 1] := SimpConstr.Inputs[i];
        SimpConstr.Inputs[i] := nil;
        SimpConstr.OrderedCombination[i - 1] := SimpConstr.OrderedCombination[i];
      end; { if SimpConstr.Inputs[i] }

    { add the new inputs }
    for i := 0 to numInputsOfUsing-1 do
      if ( commonInputs[i] = -1 ) and
         ( i <> outputOfSimplify ) then
      begin
        SimpConstr.Inputs[numInputsOfSimplify] := _using.Inputs[i];
        SimpConstr.OrderedCombination[numInputsOfSimplify] :=
          m * _using.OrderedCombination[i];
        Inc(numInputsOfSimplify);
      end; {if ( commonInputs[i] = -1 )}

    { now scale things back so that the output of "SimpConstr" is 1 }
    f := 1 - m;

    SimpConstr.OrderedCombination[3] := SimpConstr.OrderedCombination[3] / f;
    for i := 0 to numInputsOfSimplify-1 do
      SimpConstr.OrderedCombination[i] := SimpConstr.OrderedCombination[i] / f;
  end; {if ( outputOfSimplify > -1 )}
end; {Simplify}

{--------------------------------------- SetOutputAndMatch }
function SetOutputAndMatch( var output : PVariable;
                            var match : PVariable;
                            metrics : TChildMetrics;
                            index : integer ) : Boolean;
begin
  output := @metrics.Variables[index];

  if index > 1 then {not left}
  begin
    match := @metrics.Variables[index-2];
    Result := False;
  end {if}
  else
  begin
    match := @metrics.Variables[index+2];
    Result := True;
  end {if}
end; { SetOutputAndMatch }

{--------------------------------------- TLayoutPanel.AddConstraint }
{ Add a consraint by metrics and type of consraint }
procedure TLayoutPanel.AddConstraint( metrics : TChildMetrics;
                                      c : PLayoutConstraint;
                                      whichConstraint : TWhichConstraint );
var
  index, i : integer;
  output : PVariable;
  match : PVariable;
  left : Boolean;
  reslt : TConstraint;
  variables : ^TEdgeVariables;
  relWinMetrics : TChildMetrics;
  input : PVariable;
  value : integer;
  percent : single;

begin
  reslt := TConstraint.Create;

  { set the output variable }
  if ( whichConstraint = XConstraint ) and
     ( metrics.Metrics.X.MyEdge = lmRight ) then
    left := SetOutputAndMatch( output, match, metrics, 2 )
  else
  if ( whichConstraint = YConstraint ) and
     ( metrics.Metrics.Y.MyEdge = lmBottom ) then
    left := SetOutputAndMatch( output, match, metrics, 3 )
  else
    left := SetOutputAndMatch( output, match, metrics, ord(whichConstraint) );

  { set the inputs based on the edge }
  if ( c^.Relationship <> lmAbsolute ) and
     ( c^.Relationship <> lmAsIs ) then
  begin
    if c^.RelWin = lmParent then
      variables := @FVariables
    else
    begin
      relWinMetrics := GetChildMetricsByChild(c^.RelWin);

      if relWinMetrics = nil then
      begin
        reslt.Free;
        raise ELayoutBadRelCtl.Create( 'Relative control has no layout metrics !' );
{
        THROW( TXWindow(this, IDS_LAYOUTBADRELWIN) );
}
      end; {if relWinMetrics = nil}

      variables := @relWinMetrics.Variables;
    end; {else}

    case c^.OtherEdge of
      lmLeft..lmBottom :
        reslt.Inputs[0] := @variables^[ord(c^.OtherEdge)];

      lmWidth, lmHeight:
      begin
        { width => right - left + 1
          height => bottom - top + 1 }
        reslt.Inputs[0] := @variables^[ord(c^.OtherEdge) - ord(lmWidth) + ord(lmRight)];
        reslt.Inputs[1] := @variables^[ord(c^.OtherEdge) - ord(lmWidth) + ord(lmLeft)];
        reslt.OrderedCombination[1] := -1;
        reslt.OrderedCombination[3] := 1;
      end; {lmWidth, lmHeight:}

      lmCenter:
      begin
        case whichConstraint of
          XConstraint, WidthConstraint:
          begin
            { center => (left + right) / 2 }
            reslt.Inputs[0] := @variables^[0];
            reslt.Inputs[1] := @variables^[2];
          end; {XConstraint, WidthConstraint:}

          YConstraint, HeightConstraint:
          begin
            { center => (top + bottom) / 2 }
            reslt.Inputs[0] := @variables^[1];
            reslt.Inputs[1] := @variables^[3];
          end; {YConstraint, HeightConstraint:}
        end; {case whichConstraint}
        reslt.OrderedCombination[0] := 0.5;
        reslt.OrderedCombination[1] := 0.5;
      end; {lmCenter:}
    end; { case c^.OtherEdge }
  end; {if ( c^.Relationship <> lmAbsolute )}

  { now store the constant term as the last of the ordered linear combination
    we must do this after setting the inputs

    NOTE: we cannot assume that the constant part is 0, because it might have
          been set above
  }
  case c^.Relationship of
    lmAsIs:
    begin
      if ( whichConstraint = WidthConstraint ) or
         ( whichConstraint = XConstraint ) then
        { Divide if it applied to center }
        if c^.MyEdge = lmCenter then
          reslt.OrderedCombination[3] := metrics.Child.Left +
            ( reslt.OrderedCombination[3] + metrics.Child.Width ) * 0.5
        else { lmWidth }
          reslt.OrderedCombination[3] := reslt.OrderedCombination[3] + metrics.Child.Width
      else
        { Divide if it applied to center }
        if c^.MyEdge = lmCenter then
          reslt.OrderedCombination[3] := metrics.Child.Top +
            ( reslt.OrderedCombination[3] + metrics.Child.Height ) * 0.5
        else { lmHeight }
          reslt.OrderedCombination[3] := reslt.OrderedCombination[3] + metrics.Child.Height;
    end; { lmAsIs: }

    lmAbsolute, lmSameAs, lmBelow, lmAbove:
    begin
      if c^.Units = lmPixels then
        value :=  c^.Value
      else
        value :=  LayoutUnitsToPixels(c^.Value);

      if c^.Relationship = lmAbove then
        value := -value - 1
      else
      if c^.Relationship = lmBelow then
        Inc(value);

      reslt.OrderedCombination[3] := reslt.OrderedCombination[3] + value;
    end; { lmAbsolute, lmSameAs, lmBelow, lmAbove }

    lmPercentOf:
    begin
      percent := c^.Percent;

      percent := percent / 100;
      reslt.OrderedCombination[0] := reslt.OrderedCombination[0] * percent;
      reslt.OrderedCombination[3] := reslt.OrderedCombination[3] * percent;

      case c^.OtherEdge of
        lmWidth, lmHeight, lmCenter:
          reslt.OrderedCombination[1] := reslt.OrderedCombination[1] * percent;
      end; {case c^.OtherEdge}
    end; {lmPercentOf:}
  end; {if ( c^.Relationship <> lmAbsolute )}

  { now handle cases where the left hand side is width, height, or center
    this must be done last... }
  if reslt.Inputs[0] <> nil then
    if reslt.Inputs[1] <> nil then
      index := 2
    else
      index := 1
  else
    index := 0;

  case c^.MyEdge of
    lmWidth:
    begin
      if ( whichConstraint = XConstraint) or
         ( metrics.Metrics.X.MyEdge = lmRight ) then
      begin
        { rewrite "right - left + 1 := " as "left := right - (...) + 1" }
        for i := 0 to index-1 do
          reslt.OrderedCombination[i] := -reslt.OrderedCombination[i];

        reslt.OrderedCombination[3] := -reslt.OrderedCombination[3];
        reslt.OrderedCombination[3] := reslt.OrderedCombination[3] + 1;
        reslt.Inputs[index] := @metrics.Variables[2];

        if whichConstraint = WidthConstraint then
          left := SetOutputAndMatch( output, match, metrics, ord(XConstraint) );
      end {if ( whichConstraint = XConstraint)}
      else
      begin
        { rewrite "right - left + 1 := " as "right := left + ... - 1" }
        reslt.Inputs[index] := @metrics.Variables[0];
        reslt.OrderedCombination[3] := reslt.OrderedCombination[3] - 1;

        Simplify( metrics.Variables[0].DeterminedBy, output, reslt );
      end {else}
    end; {lmWidth:}

    lmHeight:
      if ( whichConstraint = YConstraint ) or
         ( metrics.Metrics.Y.MyEdge = lmBottom ) then
      begin
        { rewrite "bottom - top + 1 := " as "top := bottom - (...) + 1" }
        for i := 0 to index-1 do
          reslt.OrderedCombination[i] := -reslt.OrderedCombination[i];

        reslt.OrderedCombination[3] := -reslt.OrderedCombination[3];
        reslt.OrderedCombination[3] := reslt.OrderedCombination[3] + 1;
        reslt.Inputs[index] := @metrics.Variables[3];

        if whichConstraint = HeightConstraint then
          left := SetOutputAndMatch( output, match, metrics, ord(YConstraint) );
      end {if ( whichConstraint = YConstraint )}
      else
      begin
        { rewrite "bottom - top + 1 := " as "bottom := top + ... - 1" }
        reslt.Inputs[index] := @metrics.Variables[1];
        reslt.OrderedCombination[3] := reslt.OrderedCombination[3] - 1;

        Simplify(metrics.Variables[1].DeterminedBy, output, reslt);
      end; {else}

    lmCenter:
    begin
      input := @metrics.Variables[0];  { left }

      case whichConstraint of
        XConstraint:
          { rewrite "(left + right) / 2 := " as "left := -right + 2 * (...)" }
          input := @metrics.Variables[2];  { right }

        YConstraint:
          { rewrite "(top + bottom) / 2 := " as "top := -bottom + 2 * (...)" }
          input := @metrics.Variables[3];  { bottom }

        WidthConstraint:
          { rewrite "(left + right) / 2 := " as "right := -left + 2 * (...)" or
            "left := -right + 2 * (...)" depending on whether the "x" constraint
            is left or right }
          if metrics.Metrics.X.MyEdge = lmRight then
          begin
            input := @metrics.Variables[2];  { right }
            left := SetOutputAndMatch( output, match, metrics, ord(XConstraint) );
          end; {if metrics.Metrics.X.MyEdge}

        HeightConstraint:
          { rewrite "(top + bottom) / 2 := " as "bottom := -top + 2 * (...)" or
            "top := -bottom + 2 * (...)" depending on whether the "y" constraint
            is top or bottom }
          if metrics.Metrics.Y.MyEdge <> lmBottom then
            input := @metrics.Variables[1]  { top }
          else
          begin
            input := @metrics.Variables[3];  { bottom }
            left := SetOutputAndMatch( output, match, metrics, ord(YConstraint) );
                                                    {!!!???!!! was XConstraint}
          end; {if metrics.Metrics.X.MyEdge}
      end; {case whichConstraint}

      reslt.Inputs[index] := input;

      for i := 0 to index-1 do
        reslt.OrderedCombination[i] := reslt.OrderedCombination[i] * 2;

      reslt.OrderedCombination[3] := reslt.OrderedCombination[3] * 2;
      reslt.OrderedCombination[index] := -1;
    end; {lmCenter}
  end; {case c^.MyEdge}

  { now set the constraint output }
  output^.DeterminedBy := reslt;
  reslt.Output := output;
  reslt.Match := match;
  reslt.Left := left;

  { Set extremal sizes }
  if ( whichConstraint = XConstraint ) or
     ( whichConstraint = WidthConstraint ) then
  begin
    reslt.MinSize := metrics.Metrics.MinW;
    reslt.MaxSize := metrics.Metrics.MaxW;
  end {if ( whichConstraint = XConstraint )}
  else
  begin
    reslt.MinSize := metrics.Metrics.MinH;
    reslt.MaxSize := metrics.Metrics.MaxH;
  end; {else}

  { add the constraint to the list of constraints }
  FConstraints.Add(reslt);
end; {TLayoutPanel.AddConstraint }

{--------------------------------------- TLayoutPanel.RemoveConstraint }
{ Remove constraints for a given child metrics }
procedure TLayoutPanel.RemoveConstraints( chldMetrics : TChildMetrics );
var
  variable : PVariable;
  constraint, c : TConstraint;
  i : integer;
begin

  FPlanIsDirty := true;
  ClearPlan;
  chldMetrics.GeneratedConstraints := false;

  for i := 0 to 3 do
  begin
    variable := @chldMetrics.Variables[i];
    constraint := variable^.DeterminedBy;

    variable^.Value := 0;

    if constraint <> nil then
    begin
      { remove the constraint from the list of constraints }
      FConstraints.Remove( constraint );
      constraint.Free;
      variable^.DeterminedBy := nil;
    end; {if constraint <> nil}
  end; {for i := 0}
end; {TLayoutPanel.RemoveConstraints}

{--------------------------------------- TLayoutPanel.RemoveAllConstraints }
{ Remove all constraints}
procedure TLayoutPanel.RemoveAllConstraints;
var
  i : integer;
begin
  for i := 0 to FMetrics.Count-1 do
    RemoveConstraints( FMetrics[i] );
end; {TLayoutPanel.RemoveAllConstraints}

{--------------------------------------- TLayoutPanel.BuildConstraint }
{ Build constraints for a gien child metrics }
procedure TLayoutPanel.BuildConstraints( chldMetrics : TChildMetrics );
var
  c : PLayoutConstraint;
  value : integer;

begin
  {
  NOTE: to get uniformity we consider the window edges to sit on pixels
        and not between pixels. so our idea of right is left + width - 1
        and not left + width
  }
  if not chldMetrics.GeneratedConstraints then
  begin
    c := @chldMetrics.Metrics.X;

    chldMetrics.GeneratedConstraints := true;

    { "x" can be one of: left, right, center }
    if c^.Relationship = lmAsIs then
      case c^.MyEdge of
        lmLeft :
          chldMetrics.Variables[0].Value := chldMetrics.Child.Left;
        lmRight :
          chldMetrics.Variables[2].Value := chldMetrics.Child.Left +
                                            chldMetrics.Child.Width - 1;
      else {lmCenter}
        AddConstraint(chldMetrics, c, XConstraint);
      end {case c^.MyEdge}
    else
    if ( c^.Relationship = lmAbsolute ) and
       ( c^.MyEdge <> lmCenter) then
    begin
      if c^.Units = lmPixels then
        value := c^.Value
      else
        value := LayoutUnitsToPixels(c^.Value);

      if c^.MyEdge = lmLeft then
        chldMetrics.Variables[ 0 ].Value := value
      else
        chldMetrics.Variables[ 2 ].Value := value;
    end {if ( c^.Relationship = lmAbsolute )}
    else
      AddConstraint(chldMetrics, c, XConstraint);

    { "y" can be one of: top, bottom, center }
    c := @chldMetrics.Metrics.Y;

    if c^.Relationship = lmAsIs then
      case c^.MyEdge of
        lmTop :
          chldMetrics.Variables[1].Value := chldMetrics.Child.Top;
        lmBottom :
          chldMetrics.Variables[3].Value := chldMetrics.Child.Top +
                                            chldMetrics.Child.Height - 1;
      else {lmCenter}
        AddConstraint(chldMetrics, c, YConstraint);
      end {case c^.MyEdge}
    else
    if ( c^.Relationship = lmAbsolute) and
       ( c^.MyEdge <> lmCenter) then
    begin
      if c^.Units = lmPixels then
        value := c^.Value
      else
        value := LayoutUnitsToPixels(c^.Value);

      if c^.MyEdge = lmTop then
        chldMetrics.Variables[ 1 ].Value := value
      else
        chldMetrics.Variables[ 3 ].Value := value;
    end {if ( c^.Relationship = lmAbsolute)}
    else
      AddConstraint(chldMetrics, c, YConstraint);

    { "width" can be one of: width, right, center }
    c := @chldMetrics.Metrics.Width;

    if ( c^.MyEdge = lmRight ) and
       ( ( c^.Relationship = lmAsIs ) or
         ( c^.Relationship = lmAbsolute ) ) then
    begin
      if c^.Relationship = lmAsIs then
        chldMetrics.Variables[2].Value := chldMetrics.Child.Left +
                                          chldMetrics.Child.Width - 1
      else
      if c^.Units = lmPixels then
        chldMetrics.Variables[2].Value := c^.Value
      else
        chldMetrics.Variables[2].Value := LayoutUnitsToPixels(c^.Value);
    end {if ( c^.MyEdge = lmRight )}
    else
      AddConstraint(chldMetrics, c, WidthConstraint);

    { "height" can be one of: height, bottom, center }
    c := @chldMetrics.Metrics.Height;

    if ( c^.MyEdge = lmBottom ) and
       ( ( c^.Relationship = lmAsIs ) or
         ( c^.Relationship = lmAbsolute ) ) then
      if c^.Relationship = lmAsIs then
        chldMetrics.Variables[3].Value := chldMetrics.Child.Top +
                                          chldMetrics.Child.Height - 1
      else
      if c^.Units = lmPixels then
        chldMetrics.Variables[3].Value := c^.Value
      else
        chldMetrics.Variables[3].Value := LayoutUnitsToPixels(c^.Value)
    else
      AddConstraint(chldMetrics, c, HeightConstraint);
  end; {if chldMetrics.GeneratedConstraints = nil}
end; {TLayoutPanel.BuildConstraints}

{--------------------------------------- TLayoutPanel.GetFontHeight }
{ Get current font height }
procedure TLayoutPanel.GetFontHeight;
begin
  FFontHeight := Canvas.Font.Height;
end; { TLayoutPanel.GetFontHeight }

{--------------------------------------- TLayoutPanel.WereAsIsMoved }
{ Check if edges having AsIs constraints were moved
  since last building a plan }
function TLayoutPanel.WereAsIsMoved : boolean;
var
  chldMetrics : TChildMetrics;
  i : integer;

  { Was an edge having AsIs constraint moved ? }
  function WasAsIsMoved( var lc : TLayoutConstraint;
                         var LVar, RVar : TVariable;
                         X, Width : integer ) : boolean;
  begin
    Result := false;

    { check if this is really AsIs constraint }
    if lc.Relationship <> lmAsIs then
      Exit;

    if ( lc.MyEdge = lmLeft ) or
       ( lc.MyEdge = lmTop ) then
      Result := ( LVar.Value <> X )
    else
    if ( lc.MyEdge = lmRight ) or
       ( lc.MyEdge = lmBottom ) then
      Result := ( RVar.Value <> X+Width-1 )
    else
    if lc.MyEdge = lmCenter then
      Result := ( Round( LVar.Value * 0.5 + RVar.Value * 0.5 ) <>
                  Round( X * 0.5 + ( X + Width - 1 ) * 0.5 ) )
    else   { Width or height }
      Result := ( RVar.Value - LVar.Value + 1 <> Width );
  end; { WasAsIsMoved }

begin
  Result := false;

  for i := 0 to FMetrics.Count-1 do
  begin
    chldMetrics := FMetrics[i];

    if chldMetrics.Child <> nil then
      with chldMetrics do
        if WasAsIsMoved( Metrics.X, Variables[0], Variables[2], Child.Left, Child.Width ) or
           WasAsIsMoved( Metrics.Y, Variables[1], Variables[3], Child.Top, Child.Height ) or
           WasAsIsMoved( Metrics.Width, Variables[0], Variables[2], Child.Left, Child.Width ) or
           WasAsIsMoved( Metrics.Height, Variables[1], Variables[3], Child.Top, Child.Height ) then
        begin
           Result := true;
           Exit;
        end; {if}
  end; {for}

end; { TLayoutPanel.WereAsIsMoved }

{--------------------------------------- TLayoutPanel.Layout }
{
causes the receiver to size/position its children according to the
specified layout metrics

if you change the layout metrics for a child window call Layout
to have the changes take effect
}
procedure TLayoutPanel.Layout;
var
  cxBorder, cyBorder : integer;
  win : TControl;
  variables : ^TEdgeVariables;
  i : integer;

begin
  { Check if layout is disabled }
  if FLayoutLevel > 0 then
    Exit;

  if FMetrics.Count < 1 then
    Exit;

  { If it is the first layout after ReadMetrics any child metrics has Child field = nil
    then try to set it to the correspondent control }
  if FFirstAfterRead then
  begin
    FFirstAfterRead := false;

    SetVacantPointers;
    RemoveSuperfluousMetrics(true); {remove all nil pointers}
  end; {if}

  if Assigned( FBeforeLayout ) then
    FBeforeLayout( Self );

  GetFontHeight;

  { initialize the parent's variables }
  FVariables[0].Value := ClientRect.Left;
  FVariables[1].Value := ClientRect.Top;
  FVariables[2].Value := ClientRect.Right-1;
  FVariables[3].Value := ClientRect.Bottom-1;

  { if some  }
  if not FPlanIsDirty then
  begin
    FPlanIsDirty := WereAsIsMoved;

    { Remove all constraints }
    if FPlanIsDirty then
      RemoveAllConstraints;
  end; { if  not FPlanIsDirty }

  { Rebuild layout plan if necessary }
  if FPlanIsDirty then
  begin
    FPlanIsDirty := false;

    for i := 0 to FMetrics.Count-1 do
      BuildConstraints(ChildMetrics[i]);
    BuildPlan;
  end; {if FPlanIsDirty}

  { Use the plan to calculate actual child window position values }
  ExecutePlan;

  { Disable layout and align for it can be caused by changing positions and sizes for children }
  DisableAlign;
  DisableLayout;

  { now do the actual resizing of the windows }
  for i := 0 to FMetrics.Count-1 do
  begin
{DDDDebug
    ShowMetricsForm.Memo1.Lines.Add(Parent.Name+' ChildMetrics['+IntToStr(i)+']='+IntToStr(integer(ChildMetrics[i].Child)));}
    win := ChildMetrics[i].Child;
    variables := @ChildMetrics[i].Variables;

    {!!!!!!!!!!!! MIK's CHANGE !!!!!!!!!!!!}
    { added IFs }
    if win.Left <> variables^[0].Value then
      win.Left := variables^[0].Value;
    if win.Top <> variables^[1].Value then
      win.Top := variables^[1].Value;
    if win.Width <> variables^[2].Value - variables^[0].Value + 1 then
      win.Width := variables^[2].Value - variables^[0].Value + 1;
    if win.Height <> variables^[3].Value - variables^[1].Value + 1 then
      win.Height := variables^[3].Value - variables^[1].Value + 1;
    {!!!!!!!!!!!! MIK's CHANGE !!!!!!!!!!!!}
  end; { for }

  EnableLayout;
  EnableAlign;

  if Assigned( FAfterLayout ) then
    FAfterLayout( Self );

{DDDDebug
  ShowMetricsForm.Memo1.Lines.Add(Parent.Name+' Layout: exit');}
end; { TLayoutPanel.Layout }

{--------------------------------------- TLayoutPanel.SetVacantPointers }
{ If any child metrics has Child field = nil
  then try to set it to the correspondent control }
procedure TLayoutPanel.SetVacantPointers;
type
  { types to store a dynamic array of Boolean }
  TLotOfBoolean = array[0..64535 div sizeof(Boolean)] of Boolean;
  PLotOfBoolean = ^TLotOfBoolean;

var
  i, Last: integer;
  chldMetrics: TChildMetrics;
  SetRelWin: PLotOfBoolean;

begin
  Last := ControlCount-1;

  if FMetrics.Count < ControlCount then
    Last := FMetrics.Count-1;

  if Last < 0 then
    Exit;

  { Allocate memory for flags }
  GetMem( SetRelWin, Sizeof(Boolean)*(Last+1) );

  { Set true pointers for all children - this must be done first }
  for i := 0 to Last do
  begin
    chldMetrics := FMetrics[i];

    with chldMetrics do
      if Child = nil then
      begin
        SetRelWin^[i] := true; {Relative controls pointers must be set}

        { try to get control pointer by control name }
        if Format <= Format10 then
          Child := GetControlByName( chldMetrics.Name )
        else { old format }
          Child := Controls[i];
      end {if}
      else
        SetRelWin^[i] := false;
  end; {for}

  { Set true pointers for relative controls }
  for i := 0 to Last do
    { Relative controls pointers must be set only if
      corresponding Child was newly set }
    if SetRelWin^[i] then
    begin
      chldMetrics := FMetrics[i];

      with chldMetrics.Metrics do
      begin
        X.RelWin := GetRelControlByNum( integer(X.RelWin) );
        Y.RelWin := GetRelControlByNum( integer(Y.RelWin) );
        Width.RelWin := GetRelControlByNum( integer(Width.RelWin) );
        Height.RelWin := GetRelControlByNum( integer(Height.RelWin) );
      end; {with}
    end; {if}

{ Free allocated memory }
FreeMem( SetRelWin, Sizeof(Boolean)*(Last+1) );
RemoveSuperfluousMetrics(false);
end; { TLayoutPanel.SetVacantPointers }

{--------------------------------------- TLayoutPanel.GetChildMetrics }
{ Get child metrics by index }
function TLayoutPanel.GetChildMetrics(Index: Integer): TChildMetrics;
begin
  if ( Index < FMetrics.Count ) and
     ( Index >= 0 ) then
    Result := TChildMetrics( FMetrics[ Index ] )
  else
    Result := nil;
end; { TLayoutPanel.ChildMetrics }

{--------------------------------------- TLayoutPanel.GetConstraint }
{ Get constraint by index }
function TLayoutPanel.GetConstraint(Index: Integer): TConstraint;
begin
  if ( Index < FConstraints.Count ) and
     ( Index >= 0 ) then
    Result := TConstraint(FConstraints[ Index ])
  else
    Result := nil;
end; { TLayoutPanel.GetConstraint }

{--------------------------------------- TLayoutPanel.GetPlanItem }
{ Get constraint of plan by index }
function TLayoutPanel.GetPlanItem(Index: Integer): TConstraint;
begin
  if ( Index < FPlan.Count ) and
     ( Index >= 0 ) then
    Result := TConstraint(FPlan[ Index ])
  else
    Result := nil;
end; { TLayoutPanel.GetPlanItem }

{--------------------------------------- TLayoutPanel.GetChildMetricsCount }
{ Returns child metrics count }
function TLayoutPanel.GetChildMetricsCount : integer;
begin
  Result := FMetrics.Count;
end; { TLayoutPanel.GetChildMetricsCount }

{--------------------------------------- TLayoutPanel.Empty }
{ To publish a property }
procedure TLayoutPanel.Empty( m : TList);
begin
end; { TLayoutPanel.Empty }

{--------------------------------------- TLayoutPanel.DisableLayout }
{ Increment Layout level to disable layout }
procedure TLayoutPanel.DisableLayout;
begin
  Inc(FLayoutLevel);
end; { TLayoutPanel.DisableLayout }

{--------------------------------------- TLayoutPanel.EnableLayout }
{ Decrement Layout level to enable layout }
procedure TLayoutPanel.EnableLayout;
begin
  if FLayoutLevel > 0 then
    Dec(FLayoutLevel);
end; { TLayoutPanel.EnableLayout }

{--------------------------------------- TLayoutPanel.SetLayoutEnabled }
{ Set enable to true or false and
  force rebuilding constraints and  layout
  State can be written directly only at design time }
procedure TLayoutPanel.SetLayoutEnabled( en : boolean );
begin
  if csdesigning in ComponentState then
    if en then
    begin
      FLayoutLevel := 0;
      FPlanIsDirty := true;
      ClearPlan;
      RemoveAllConstraints;
    end {if}
    else
      FLayoutLevel := 1;
end; { TLayoutPanel.SetLayoutEnabled }

{--------------------------------------- TLayoutPanel.IsLayoutEnabled }
{ Is layout enabled ? }
function TLayoutPanel.IsLayoutEnabled : boolean;
begin
  { LayoutEnabled property must go as True to stream }
  Result := ( csWriting in ComponentState ) or ( FLayoutLevel = 0 );
end; { TLayoutPanel.IsLayoutEnabled }

{--------------------------------------- TLayoutPanel.GetControlByName }
{ Get control by name }
function TLayoutPanel.GetControlByName(const AName: string): TControl;
var
  i: Integer;
begin
  if (AName <> '') and (ControlCount > 0) then
    for i := 0 to ControlCount - 1 do
    begin
      Result := Controls[i];
      if CompareText(Result.Name, AName) = 0 then
        Exit;
    end; { for }

  Result := nil;
end; {TLayoutPanel.GetControlByName}

{--------------------------------------- TLayoutPanel.GetRelControlByNum }
{ Get relative control by index (lmParent - maximal or =ParentNum)
  This index is not a control number but coresponding metrics number
  All messing in this code comes from working with old storing format
  (one without controls' names)
}
function TLayoutPanel.GetRelControlByNum( N : integer ) : TControl;
var
  i: integer;
  chldMetrics : TChildMetrics;
begin
  if (N >= ControlCount) or (N = ParentNum) then
    Result := lmParent
  else
  begin
    chldMetrics := FMetrics[N];
    Result := chldMetrics.Child;

    if Result = nil then
    begin
      if Format <= Format10 then
        { try to get control pointer by control name }
        Result := GetControlByName( chldMetrics.Name )
      else
        { The last attempt - try to interpret it as control number }
        Result := Controls[N];
    end; {if Result = nil}

    if Result = nil then
      Result := lmParent;
  end; {else}
end;  { TLayoutPanel.GetRelControlByNum }

{--------------------------------------- TLayoutPanel.GetRelControlNum }
{ Get index of relative control (lmParent = ParentNum)
  This index is not a control number but coresponding metrics number
  All messing in this code comes from working with old storing format
  (one without controls' names)
}
function TLayoutPanel.GetRelControlNum( var Control : TControl ) : integer;
begin
  Result := -100;

  if Control = lmParent then
    Result := ParentNum
  else
  begin
    Result := GetChildMetricsIndexByChild( Control );

    if Result < 0 then {Something is wrong!}
      raise ELayoutBadRelCtl.Create( 'Relative control has no layout metrics !' );
  end; {else}

end;  { TLayoutPanel.GetRelControlNum }

{ Procedures for storing and reading Metrics }

{--------------------------------------- TLayoutPanel.DefineProperties }
{ Define function for reading and writing }
procedure TLayoutPanel.DefineProperties(Filer: TFiler);
begin
  inherited DefineProperties(Filer);
  Filer.DefineProperty('ChildMetrics', ReadMetrics, WriteMetrics, HasData(Filer));
end; { TLayoutPanel.DefineProperties }

{--------------------------------------- TLayoutPanel.ReadMetrics }
{ Read metrics from stream }
procedure TLayoutPanel.ReadMetrics(Reader: TReader);
var
  metr : TChildMetrics;
  lometr : TLayoutMetrics;
  N : integer;
begin
  InitLayoutMetrics( lometr );

  try
    ClearPlan;
    RemoveAllConstraints;
    Flush( FMetrics ); { clear any existing metrics }
    FPlanIsDirty := true;

    { read start-of-list marker }
    Reader.ReadListBegin;

    N := Reader.ReadInteger;

    { If format is new N must be Format10 }
    if N <= Format10 then
    begin
      Format := N;
      { Read number of metrics }
      N := Reader.ReadInteger;
    end {if N <= Format10 }
    else
      Format := 0;

    while N > 0 do	{ as long as there is still data... }
    begin
      metr := TChildMetrics.Create( nil, lometr );
      metr.ReadFromStream( Reader );
      FMetrics.Add( metr );
      Dec( N );
    end; {while}

    { read end-of-list marker }
    Reader.ReadListEnd;
  finally
    FFirstAfterRead := true;
  end; {try}

  { If any child metrics has Child field = nil
    then try to set it to the correspondent control.
    It is very important to do it right now when form
    ancestory is involved. This must be done for old format only }
  if Format > Format10 then
    SetVacantPointers;

  if ChildMetricsCount <= ControlCount then
    Layout; { This is important to layout descendant form (already visible)
              when Ancestor changes its metrics at design time }

end; { TLayoutPanel.ReadMetrics }

{--------------------------------------- TLayoutPanel.WriteMetrics }
{ Write metrics to stream }
procedure TLayoutPanel.WriteMetrics(Writer: TWriter);
var
  i : integer;
  metr : TChildMetrics;
  lometr : TLayoutMetrics;
begin
  RemoveSuperfluousMetrics(false);
  InitLayoutMetrics( lometr );

  { write start-of-list marker }
  Writer.WriteListBegin;

  { Write everything except controls pointers }
  Writer.WriteInteger(Format10);
  Writer.WriteInteger(ChildMetricsCount);

  for i := 0 to ChildMetricsCount-1 do
  begin
    ChildMetrics[i].WriteToStream( Writer, Self );
{
    metr := GetChildMetricsByChild( Controls[i] );

    if metr = nil then
    begin
      SetChildLayoutMetrics( Controls[i], lometr );
      metr := GetChildMetricsByChild( Controls[i] );
    end; {if}
{    metr.WriteToStream( Writer, Self );
}
  end; {for}

  { write end-of-list marker }
  Writer.WriteListEnd;

end; { TLayoutPanel.ReadMetrics }

{--------------------------------------- TLayoutPanel.HasData }
{ Determines at run time whether the
  ChildMetrics property has data to store.
  Differs for Delphi16 and Delphi32.
  In Delphi32 compares the Ancesor's (if any) ChildMetrics
  with native one and returns True if there are some
  differences between them }
function TLayoutPanel.HasData( Filer: TFiler ): boolean;
var
  Ancestor: TLayoutPanel;

begin
  RemoveSuperfluousMetrics(false);
  Result := ChildMetricsCount > 0;

{$IFDEF WIN32}
  { In any case if no metrics - nothing to write }
  if not Result then
    Exit;

  { Does an ancestor exist? }
  Ancestor := Filer.Ancestor as TLayoutPanel;

  if Ancestor <> nil then
    Result := not EqualMetrics( Ancestor );
{$ENDIF}

end; { TLayoutPanel.HasData }

{--------------------------------------- TLayoutPanel.EqualMetrics }
{ This procedure is used for comparison
  Ancestor's metrics with native metrics.
  Children with equal names are considered
  to be equal }
function TLayoutPanel.EqualMetrics( Panel: TLayoutPanel ): boolean;
var
  chldMetrics, aChldMetrics : TChildMetrics;
  i : integer;
begin
  { They are different very often }
  Result := false;

  { Metrics must be of the same size }
  if ChildMetricsCount <> Panel.ChildMetricsCount then
    Exit;

  { Compare metrics }
  for i := 0 to ChildMetricsCount-1 do
  begin
    chldMetrics := ChildMetrics[i];
    aChldMetrics := Panel.ChildMetrics[i];

    with chldMetrics do
    begin
      if AnsiCompareStr( Child.Name, aChldMetrics.Child.Name ) <> 0 then
        Exit;

      { Compare constraints }
      if (not EqualConstraints( Metrics.X,      aChldMetrics.Metrics.X )) or
         (not EqualConstraints( Metrics.Y,      aChldMetrics.Metrics.Y )) or
         (not EqualConstraints( Metrics.Width,  aChldMetrics.Metrics.Width )) or
         (not EqualConstraints( Metrics.Height, aChldMetrics.Metrics.Height )) then
        Exit;

      if (Metrics.MinW <> aChldMetrics.Metrics.MinW) or
         (Metrics.MaxW <> aChldMetrics.Metrics.MaxW) or
         (Metrics.MinH <> aChldMetrics.Metrics.MinH) or
         (Metrics.MaxH <> aChldMetrics.Metrics.MaxH) then
        Exit;
    end; {for}
  end; {with}

  { If we have reached this point then metrics are equal }
  Result := true;
end; { TLayoutPanel.HasData }

{--------------------------------------- TLayoutPanel.EqualConstraints }
{ Compare constraints }
function TLayoutPanel.EqualConstraints( var c1, c2: TLayoutConstraint ): boolean;
begin
  { They are different very often }
  Result := false;

  { Do not compare measurement units - they are not implemented yet}
  if ( c1.Relationship <> c2.Relationship ) or
     ( c1.MyEdge <> c2.MyEdge ) or
     ( c1.OtherEdge <> c2.OtherEdge ) then
    Exit;

  { Compare values only if they are essential }
  if ( c1.Relationship <> lmAsIs ) and
     ( c1.Relationship <> lmSameAs ) and
     ( c1.Value <> c2.Value ) then
    Exit;

  { Compare relative controls only if they are essential }
  if ( c1.Relationship <> lmAsIs ) and
     ( c1.Relationship <> lmAbsolute ) then
  begin
    { Relative controls must:
      1. be lmParent for both constraints
      2. have the same names }
    if ( ( c1.RelWin = lmParent ) and ( c2.RelWin <> lmParent ) ) or
       ( ( c2.RelWin = lmParent ) and ( c1.RelWin <> lmParent ) ) then
      Exit;

    if ( c1.RelWin <> lmParent ) and
       ( AnsiCompareStr( c1.RelWin.Name, c2.RelWin.Name ) <> 0 ) then
      Exit;
  end; {if ( c1.Relationship <> lmAsIs )}

  { If we have reached this point then metrics are equal }
  Result := true;
end; {TLayoutPanel.EqualConstraints}

initialization

Format := 0;
DontCareAboutRelMetrics := False;

end.
