(*
   Console Component for Delphi 2.0
   Version 01.00.00
   (C) Glen Why, 1996

Version history

*)


unit Console;

interface

{$IFDEF CONSOLE}
 You are compiling console application.
 Windows will allocate console automatically.
{$ENDIF}

uses
 Windows, Classes, SysUtils, Forms;



type

  EConsoleError = class( Exception );

  TCustomConsole = class( TComponent )
  private
    FActive :Boolean;
    FAutoOpen :Boolean;
    FOnOpen :TNotifyEvent;
    FOnClose :TNotifyEvent;
  protected
    function GetTitle :String; virtual;
    procedure OpenConsole; dynamic;
    procedure CloseConsole; dynamic;
    procedure CreateConsole; virtual; abstract;
    procedure DestroyConsole; virtual; abstract;
    procedure UpdateCOnsoleTitle; virtual; abstract;
    property Title :string
      read GetTitle;
    property OnOpen :TNotifyEvent
      read FOnOpen write FOnOpen;
    property OnClose :TNotifyEvent
      read FOnClose write FOnClose;
    property AutoOpen :Boolean
      read FAutoOpen write FAutoOpen;
  public
    constructor Create( anOwner :TComponent ); override;
    destructor Destroy; override;
    procedure WriteString( const S :String ); virtual; abstract;
    function ReadString( cbLength :Integer ) :String; virtual; abstract;
    procedure WriteLine( const L :String );
    function ReadLine( const Prompt :String ) :String;
    procedure Open; virtual;
    procedure Close; virtual;
    property Active :Boolean read FActive;
  end;

  TStdConsole = class( TCustomConsole )
  private
    FOnCtrlBreak :TNotifyEvent;
    function GetHandle( Index :Integer ):THandle;
  protected
    procedure CreateConsole; override;
    procedure DestroyConsole; override;
    procedure UpdateCOnsoleTitle; override;
    procedure CtrlBreak; dynamic;
  public
    constructor Create( anOwner :TComponent ); override;
    procedure WriteString( const S :String ); override;
    function ReadString( cbLength :Integer ) :String; override;
    procedure ClearInput;
    property InputHandle :THandle index STD_INPUT_HANDLE read GetHandle;
    property OutputHandle :THandle index STD_OUTPUT_HANDLE read GetHandle;
  published
    property OnCtrlBreak :TNotifyEvent
      read FOnCtrlBreak write FOnCtrlBreak;
    property OnClose;
    property OnOpen;
    property AutoOpen;
  end;


var AppConsole :TCustomConsole;

implementation

{$RESOURCE CONSOLE.RES}

{$INCLUDE CONSOLE.INC}

{ TCustomConsole }

procedure ConsoleError( ID :Cardinal );
begin
 raise EConsoleError.CreateRes( ID );
end;

const

LineBreak = #13#10;
MaxInputLineLen = 255;

procedure TCustomConsole.WriteLine( const L :String );
begin
 WriteString( L + LineBreak );
end;

function TCustomConsole.ReadLine( const Prompt :String ) :String;
var s :integer;
begin
 if Prompt <> '' then WriteString( Prompt );
 result := ReadString( MaxInputLineLen );
 S := Length( result );
 if ( S >= Length( LineBreak ) ) then SetLength( result, S - Length( LineBreak ) );
end;

function TCustomConsole.GetTitle :String;
begin
 result := format( LoadStr( SConsoleTitle),
  [ ExtractFileName( Application.ExeName ) ] );
end;

procedure TCustomConsole.OpenConsole;
begin
 if assigned( FOnOpen ) then FOnOpen( Self );
end;

procedure TCustomConsole.CloseConsole;
begin
 if assigned( FOnClose ) then FOnClose( Self );
end;

constructor TCustomConsole.Create( anOwner :TComponent );
begin
 if AppConsole <> Nil then ConsoleError( SMultiInstance );
 inherited Create( anOwner );
 FActive := false;
 FAutoOpen := false;
 FOnOpen := nil;
 FOnClose := nil;
 AppConsole := Self;
end;

destructor TCustomConsole.Destroy;
begin
 if Active then Close;
 inherited;
 AppConsole := Nil;
end;

procedure TCustomConsole.Open;
begin
 if not Active
   then begin
     CreateConsole;
     UpdateConsoleTitle;
     FActive := true;
     OpenConsole;
   end;
end;

procedure TCustomConsole.Close;
begin
 if Active
   then begin
     DestroyConsole;
     FActive := false;
     CloseConsole;
   end;
end;

{ TStdConsole }

procedure TStdConsole.UpdateCOnsoleTitle;
begin
  SetConsoleTitle( PChar( Title ) );
end;

function COnsoleHandler( dwCtrlType :Longint ):Bool; stdcall;
begin
  result := true;
  case dwCtrlType of
    CTRL_BREAK_EVENT : TStdConsole( AppConsole ).CtrlBreak;
    else result := false;
  end;
end;  


procedure TStdConsole.CreateConsole;
begin
 if not AllocConsole then ConsoleError( SAllocConsole );
 try
   if not SetConsoleMode( InputHandle, ENABLE_LINE_INPUT OR
    ENABLE_ECHO_INPUT OR ENABLE_PROCESSED_INPUT  )
     then ConsoleError( SAllocConsole );
   if not SetConsoleMode( OutputHandle, ENABLE_PROCESSED_OUTPUT
    OR ENABLE_WRAP_AT_EOL_OUTPUT )
     then ConsoleError( SAllocConsole );
   if not SetConsoleCtrlHandler( @ConsoleHandler, true )
     then ConsoleError( SAllocConsole );
 except
   DestroyConsole;
   raise;
 end;
end;

procedure TStdConsole.DestroyConsole;
begin
  if not FreeConsole then ConsoleError( SFreeConsole );
end;

procedure TStdConsole.WriteString( const S :String );
var C :Integer;
begin
 if not active then if AutoOpen then open else exit;
 if not WriteConsole( OutputHandle, Pointer( S ),
  Length( S ), C, nil ) then ConsoleError( SWriteError );
end;

function TStdConsole.ReadString( cbLength :Integer ) :String;
var S :Integer;
begin
 if not active then if AutoOpen then open else exit;
 SetLength( result, cbLength );
 if not ReadConsole( InputHandle, Pointer( result ),
  cbLength, S, nil ) then ConsoleError( SReadError );
 SetLength( result, s );
end;

function TStdConsole.GetHandle( Index :Integer ):THandle;
begin
 result := GetStdHandle( index );
end;

procedure TStdConsole.CtrlBreak;
begin
 if Assigned( FOnCtrlBreak ) then FOnCtrlBreak( Self );
end;

constructor TStdConsole.Create( anOwner :TComponent );
begin
 inherited Create( anOwner );
 FOnCtrlBreak := Nil;
end;

procedure TStdConsole.ClearInput;
begin
 if active then
  if not FlushConsoleInputBuffer( InputHandle )
   then ConsoleError( SFlushInputError );
end;


initialization
begin
  AppConsole := nil;
end;

finalization
begin
  if AppConsole <> Nil then AppConsole.Free;
end;

end.
