(*
     Tiny Shell Component for Delphi 2.0
     Version 00.00.12 (Test Version)
     by Glen Why


Tiny Shell - is a simple command shell for GUI
applications. GUI application can use this shell for
debugging purposes or in any other situation when an
alternate interface for a running programm need.
This is a test version, I would not recommend you to
rely on it in a commercial application. Anyway, it's
fully functional, and further changes will not affect
an existing interface (I hope).

Using Tiny Shell
----------------

This version of the shell has only three predefined commands:
"exit", "help", and "close". "exit" and "close" close the shell.
You can also close the shell by calling shell's Close method.
"help" lists all registered commands and their descriptions.
Use AddCommand to add new commands. Shell provides OnInitialize
event, that is the best place for registering shell commands.
Use Open method to open the shell window. See source for details.


*)

unit TinyShell;

interface
uses
 windows, classes, sysutils, console, forms;

type

  TTinyShellCOmmand = procedure( Sender :TObject;
    const Params :String ) of object;

  ETinyShellError = class( EConsoleError );

  TTinyShell = Class( TStdConsole )
  private
    FOnInitialize :TNotifyEvent;
    FCommands :TStringList;
    FShellThread :TThread;
    FPrompt :String;
    function GetCommandCount :Integer;
    function GetCommandNames( index :integer ):string;
    function GetCommandHints( index :integer ):string;
    function GetCOmmands( index :integer ):TTinyShellCommand;
    procedure InitShellThread;
    procedure DoneShellThread;
    procedure CmdExit( Sender :TObject; const Params :String );
    procedure CmdHelp( Sender :TObject; const Params :String );
  protected
    procedure CommandCycle;
    procedure ClearCommands;
    procedure DeleteCommand( index :integer );
    function FindCommand( const CmdName :string ):integer;
    procedure OpenConsole; override;
    procedure Loaded; override;
    function GetTitle :string; override;
    procedure Initialize; dynamic;
    property CommandCount :Integer
      read GetCommandCount;
    property Commands [ index :integer ] :TTinyShellCommand
      read GetCommands;
    property CommandNames [ index :integer ] :string
      read GetCommandNames;
    property CommandHints [ index :integer ] :string
      read GetCommandHints;
  public
    constructor Create( anOwner :TComponent ); override;
    destructor Destroy; override;
    procedure AddCommand( const CmdName :string;
     CmdHintResID :Integer; CmdProc :TTinyShellCommand );
    procedure RemoveCommand( const CmdName :String );
    procedure Open; override;
    procedure Close; override;
  published
    property OnInitialize :TNotifyEvent
      read FOnInitialize write FOnInitialize;
    property Prompt :String
      read FPrompt write FPrompt;
    property OnOpen;
    property OnClose;
    property OnCtrlBreak;
  end;


procedure Register;

implementation

{$INCLUDE CONSOLE.INC}

procedure Register;
begin
 RegisterComponents( 'more...', [ TTinyShell ] );
end;


{ TTinyShell }

procedure ShellError( ID :Integer );
begin
 raise ETinyShellError.CreateRes( ID );
end;

type

 TCommandData = class
   HintID :Integer;
   Command :TTinyShellCOmmand;
   constructor Create( aHintID :Integer; aCommand :TTinyShellCOmmand );
 end;


 TShellThread = class( TThread )
   FShell :TTinyShell;
   procedure Execute; override;
   procedure Command;
   constructor Create( Shell :TTinyShell; Susp :Boolean );
 end;


constructor TShellThread.Create( Shell :TTinyShell; Susp :Boolean );
begin
 inherited Create( Susp );
 FShell := Shell;
end;

procedure TShellThread.Execute;
begin
 WaitForSingleObject( FShell.OutputHandle, INFINITE );
 while not Terminated do if FShell.Active then COmmand else sleep( 0 );
end;

procedure TShellThread.Command;
begin
 FShell.CommandCycle ;
end;

procedure TTinyShell.InitShellThread;
begin
 FShellThread := TShellThread.Create( Self, false );
 FShellThread.FreeOnTerminate := true;
end;

procedure TTinyShell.DoneShellThread;
begin
  FShellThread.Terminate;
  if ( FShellThread.ThreadID <> GetCurrentThreadID )
   then begin
    TerminateThread( FShellThread.Handle, 0 );
    FShellThread.Free;
   end;
end;


constructor TCommandData.Create( aHintID :Integer; aCommand :TTinyShellCOmmand );
begin
 inherited Create;
 HintID := aHintID;
 Command := aCOmmand;
 if not assigned( COmmand ) then ShellError( SNoCommand );
end;


function TTinyShell.GetCommandCount :Integer;
begin
 if ( FCommands <> Nil ) then result := FCommands.Count else result := 0;
end;

function TTinyShell.GetCommandNames( index :integer ):string;
begin
 if ( FCommands = Nil ) then result := ''
  else result := FCOmmands.Strings[ index ];
end;

function TTinyShell.GetCommandHints( index :integer ):string;
var ID :Integer;
begin
 result := '';
 if ( FCommands <> Nil )
   then begin
     ID := TCommandData( FCommands.Objects[ index ] ).HintID;
     if ( ID <> 0 ) then result := LoadStr( ID );
   end;
end;

function TTinyShell.GetCOmmands( index :integer ):TTinyShellCommand;
begin
 result := TCommandData( FCommands.Objects[ index ] ).Command;
end;

procedure TTinyShell.OpenConsole;
begin
  WriteLine( LoadStr( STinyShellInfo ) );
  inherited;
end;

function TTinyShell.GetTitle :string;
begin
 result := format( LoadStr( STinyShellTitle ),
  [ ExtractFileName( Application.exeName ) ] );
end;


procedure TTinyShell.Initialize;
begin
 AddCommand( 'exit', SShellExitHint, CmdExit );
 AddCOmmand( 'help', SShellHelpHint, CmdHelp );
 AddCOmmand( 'close', SShellExitHint, CmdExit );
 if assigned( FOnInitialize ) then FOnInitialize( Self );
end;

constructor TTinyShell.Create( anOwner :TComponent );
begin
 inherited Create( anOwner );
 FCommands := TStringList.Create;
 FCommands.Duplicates := dupError;
 FCOmmands.Sorted := true;
 FPrompt := '>';
 FShellThread := Nil;
 FOnInitialize := nil;
end;

destructor TTinyShell.Destroy;
begin
 if ( FCommands <> Nil ) then
   begin
     ClearCommands;
     FCommands.Free;
   end;
 inherited;
end;

procedure TTinyShell.AddCommand( const CmdName :string;
  CmdHintResID :Integer; CmdProc :TTinyShellCommand );
begin
  FCOmmands.AddObject( CmdName, TCommandData.Create( CmdHintResID, CmdProc ) );
end;

procedure TTinyShell.RemoveCommand( const CmdName :String );
begin
 DeleteCommand( FindCommand( CmdName ) );
end;

procedure TTinyShell.ClearCommands;
var i :integer;
begin
  for i := 0 to COmmandCount - 1 do DeleteCommand( 0 );
end;

procedure TTinyShell.DeleteCommand( index :integer );
begin
  with FCommands do
   begin
    Objects[ index ].Free;
    Delete( index );
   end;
end;

function TTinyShell.FindCommand( const CmdName :string ):integer;
begin
 if not FCommands.Find( CmdName, result ) then result := -1;
end;

procedure TTinyShell.Loaded;
begin
  inherited;
  if not ( csDesigning in ComponentState ) then Initialize;
end;


procedure TTinyShell.CommandCycle;
var C, P :String; D :Integer; CMD :TTinyShellCOmmand;
begin
  C := ReadLine( Prompt );
  C := Trim( C );
  if ( C <> '' ) then
    begin
      D := Pos( ' ', C );
      if ( D > 0 )
        then begin
          P := Copy( C, D, Length( C ) );
          SetLength( C, D - 1 );
        end
        else P := '';
      D := FindCommand( C );
      if ( D < 0 )
        then WriteLine( Format( LoadStr( SUnknownCommand ), [ C ] ) )
        else begin
          CMD := Commands[ D ];
          CMD( Self, P );
        end;
    end;
end;

procedure TTinyShell.CmdExit( Sender :TObject; const Params :String );
begin
  Close;
end;

procedure TTinyShell.CmdHelp( Sender :TObject; const Params :String );
var i :integer;
begin
  for i := 0 to CommandCount - 1 do
   WriteLine( format( '"%s" '#9' - %s', [ CommandNames[ i ], CommandHints[ i ] ] ) );
end;

procedure TTinyShell.Open;
begin
 if Active then exit;
 inherited;
 InitShellThread;
end;

procedure TTinyShell.Close;
begin
 if not Active then exit;
 inherited;
 DoneShellThread;
end;


end.
