/* TC_GRAPH.C */

/* Turbo C example screen save and restore functions for use
   when using Steph in graphics modes. */

/* Supplied with Steph 1.0 */

/* The problem with the Turbo C getimage and putimage functions
   from the graphics library is that they can only store images
   of 64Kb size and less.  Some of the screen saves that you
   need to do when using Steph in graphics modes are considerably
   larger than 64K... */

/* The solution used here is to save the screen as a list of either
   1, 2, 4 or 8 blocks, each less than 64K in size. */


/* This structure stores pointers to screen blocks */

typedef struct {
   char number;             /* number of blocks */
   unsigned pixheight;      /* height of block in pixels */
   void _far *pointer[8];   /* array of pointers to upto 8 blocks */
}screen_store;


void my_screen_save( void )
{
   /* TC can only store in 64K blocks, so... */

   /* ysize is the height of the area to be saved.
      bsize is the height of the blocks we are going to save */

   screen_store *info;
   unsigned ysize = _pix_to_save.y2 - _pix_to_save.y1 + 1;
   unsigned bsize = ysize;

   unsigned thissize;

   char count;
   unsigned isize;

   /* Create a screen_store structure and put a pointer to it
      in .data */

   info = (screen_store *)malloc( sizeof( screen_store ) );
   _pix_to_save.data = (char *)info;
   if( info == NULL )
      return;

   /* Try just one block to start with */
   info->number = 1;

   while (
      ( isize = imagesize( _pix_to_save.x1, _pix_to_save.y1,
      _pix_to_save.x2, _pix_to_save.y1 + bsize - 1 ) )
      == 0xFFFF )
   {
      /* Couldn't fit the block in 64K, so let's use twice as
         many blocks, each half the size. */

      info->number = info->number * 2;
      bsize = (bsize + 1) / 2;
   }

   /* Now we know how many blocks we need to save. */

   /* Store the height of a block, this will be useful when
      we restore them... */

   info->pixheight = bsize;

   /* Loop for each block to be saved... */

   for( count = 0; count < info->number; count++ )
   {
      thissize = bsize;      /* the block size */

      /* Make the last block smaller if necessary. */

      if( ((count+1)*bsize) > ysize )
         thissize = ysize - count * bsize;

      /* Then allocate a block of far storage space, and put
         the pointer in the block table. */

      isize = imagesize( _pix_to_save.x1,
            _pix_to_save.y1 + count * bsize, _pix_to_save.x2,
            _pix_to_save.y1 + count * bsize + thissize - 1 );

      info->pointer[count] = farmalloc( (unsigned long)isize );

      if( info->pointer[count] == (void _far *)NULL )
      {
         /* The allocation failed to allocate enough blocks, so exit */

         /* Tidy up all the blocks allocated so far. */

         while( count > 0 )
            farfree( info->pointer[(--count)] );

         free( info );
         _pix_to_save.data = NULL;
         return;
      }

      /* the allocation was OK so store the screen block */

      getimage(_pix_to_save.x1, _pix_to_save.y1 + count * bsize,
            _pix_to_save.x2,_pix_to_save.y1 + count * bsize + thissize - 1,
            info->pointer[count] );

      /* Now loop back until all the blocks have been saved. */
   }
}



void my_screen_restore( void )
{
   /* Restore the screen given a pointer to a screen_store structure. */

   screen_store *info = (screen_store *)_pix_to_save.data;
   char count;

   /* Go through all the blocks and redraw each. */

   for( count = 0; count < info->number; count++ )
   {
      /* Restore the block and free the space it used. */

      putimage( _pix_to_save.x1, _pix_to_save.y1 + count * info->pixheight,
            info->pointer[count], COPY_PUT  );

      farfree( info->pointer[count] );
   }

   /* Free the space used for the screen_store structure. */
   free( info );
}


