/* Settings.c -- User interface settings for Jeu de Taquin */

#include <windows.h>
#include "taquin.h"
#include "resource.h"

static SETTINGS Settings;   /* the one and only instance of settings */

void Settings_EnableSound( void )    { Settings.bSound = TRUE;  }
void Settings_DisableSound( void )   { Settings.bSound = FALSE; }
BOOL Settings_IsSoundEnabled( void ) { return Settings.bSound;  }

void Settings_SetFont ( char *szNewFontName, int iNewFontSize, BOOL bIsFontBold, BOOL bIsFontItalic )
{
    lstrcpy( Settings.szFontName, szNewFontName );
    Settings.iFontSize = iNewFontSize;
    Settings.bFontBold = bIsFontBold;
    Settings.bFontItalic = bIsFontItalic;
}

int Settings_GetFontSize ( void ) { return Settings.iFontSize; }
BOOL Settings_IsFontBold ( void ) { return Settings.bFontBold; }
BOOL Settings_IsFontItalic ( void ) { return Settings.bFontItalic; }

void Settings_GetFontName ( char *szFontNameBuffer )
{
    lstrcpy( szFontNameBuffer, Settings.szFontName );
}

void Settings_SetSoundName ( GAMESOUND gsnd, char *szNewSoundName )
{
    switch ( gsnd )
    {
        case GSND_MOVE:
            lstrcpy( Settings.szMoveSound, szNewSoundName );
            break;
        case GSND_BLOCKED:
            lstrcpy( Settings.szBlockedSound, szNewSoundName );
            break;
        case GSND_SOLVED:
            lstrcpy( Settings.szSolvedSound, szNewSoundName );
            break;
        default:
            break;
    }
}

void Settings_GetSoundName ( GAMESOUND gsnd, char *szSoundNameBuffer )
{
    switch ( gsnd )
    {
        case GSND_MOVE:
            lstrcpy( szSoundNameBuffer, Settings.szMoveSound );
            break;
        case GSND_BLOCKED:
            lstrcpy( szSoundNameBuffer, Settings.szBlockedSound );
            break;
        case GSND_SOLVED:
            lstrcpy( szSoundNameBuffer, Settings.szSolvedSound );
            break;
        default:
            break;
    }
}
            
#define BASE    "Software\\Jere Kpyaho\\Jeu de Taquin\\1.0"
#define OPTIONS "Options"
#define SOUNDS  "Sounds"

void UseDefaultSettings( void );

BOOL Settings_Read ( void )
{
    HKEY hkey;
    TCHAR szKey[256];
	DWORD dwDataType, dwValue, dwDataSize;
    TCHAR szFontName[LF_FACESIZE];
    int iFontSize;
    BOOL bFontBold, bFontItalic;
    TCHAR szSoundName[_MAX_PATH];

    /* Enforce the default settings. */
    UseDefaultSettings();

    /* Try to open HKLKM\CompanyName\AppName\Version.
       If it fails, use the defaults. Otherwise we know
       we have some settings to read in HKCU. */
    lstrcpy( szKey, BASE );
    if ( RegOpenKeyEx( HKEY_LOCAL_MACHINE, szKey, 0,
                       KEY_QUERY_VALUE, &hkey )
        != ERROR_SUCCESS )
    {
        return FALSE;
    }
    RegCloseKey( hkey );

	/* Process the Options subkey. */
    wsprintf( szKey, "%s\\%s", BASE, OPTIONS );
    if ( RegOpenKeyEx( HKEY_CURRENT_USER, szKey, 0,
				  KEY_QUERY_VALUE, &hkey ) == ERROR_SUCCESS )
    {
        /* We have found our user-specific settings. Now read them. */
    	dwDataType = REG_DWORD;
	    dwValue = 0;
	    dwDataSize = sizeof(DWORD);
	    RegQueryValueEx( hkey, "Sound", NULL, &dwDataType,
		    (LPBYTE) &dwValue, &dwDataSize );
    	if ( dwValue != 0 )
        {
            Settings_EnableSound();
        }
        else
        {
            Settings_DisableSound();
        }

    	RegQueryValueEx( hkey, "FontSize", NULL, &dwDataType,
	        (LPBYTE) &dwValue, &dwDataSize );
	    iFontSize = dwValue;
        if ( iFontSize > TILEWIDTH )
            iFontSize = TILEWIDTH;

        RegQueryValueEx( hkey, "FontBold", NULL, &dwDataType,
	        (LPBYTE)&dwValue, &dwDataSize );
    	bFontBold = ( dwValue != 0 ) ? TRUE : FALSE;

        RegQueryValueEx( hkey, "FontItalic", NULL, &dwDataType,
	  	    (LPBYTE) &dwValue, &dwDataSize );
    	bFontItalic = ( dwValue != 0 ) ? TRUE : FALSE;

        dwDataType = REG_SZ;
	    dwDataSize = sizeof(szFontName);
        RegQueryValueEx( hkey, "FontName", NULL, &dwDataType,
	        (LPBYTE)szFontName, &dwDataSize );

        Settings_SetFont( szFontName, iFontSize, bFontBold, bFontItalic );

	    RegCloseKey( hkey );
    }
    else
    {
        return FALSE;
    }

	/* Process the Sounds subkey. */
    wsprintf( szKey, "%s\\%s", BASE, SOUNDS );
    if ( RegOpenKeyEx( HKEY_CURRENT_USER, szKey, 0,
                  KEY_QUERY_VALUE, &hkey ) == ERROR_SUCCESS )
    {
        dwDataType = REG_SZ;
	    dwDataSize = sizeof(szSoundName);
        RegQueryValueEx( hkey, "MoveSound", NULL, &dwDataType,
		    (LPBYTE)szSoundName, &dwDataSize );
    	Settings_SetSoundName( GSND_MOVE, szSoundName );

        dwDataType = REG_SZ;
	    dwDataSize = sizeof(szSoundName);
	    RegQueryValueEx( hkey, "BlockedSound", NULL, &dwDataType,
		    (LPBYTE)szSoundName, &dwDataSize );
	    Settings_SetSoundName( GSND_BLOCKED, szSoundName );

        dwDataType = REG_SZ;
	    dwDataSize = sizeof(szSoundName);
	    RegQueryValueEx( hkey, "SolvedSound", NULL, &dwDataType,
		    (LPBYTE)szSoundName, &dwDataSize );
    	Settings_SetSoundName( GSND_SOLVED, szSoundName );

	    RegCloseKey( hkey );
    }
    else /* use the defaults enforced earlier */
    {
        return FALSE;
    }
	return TRUE;
}

BOOL Settings_Write ( void )
{
    TCHAR szKey[256];
    HKEY hkey;
    DWORD dwDisposition, dwValue, dwType, dwDataSize;
    TCHAR szValue[256];

    if ( !g_bSettingsChanged ) /* no need to write settings */
    {
        return FALSE;
    }

	/* 'Settings' contains either the setting we read from the
	   registry, or the default settings. If the key doesn't
	   exist, RegCreateKeyEx creates it. If it exists, it is
	   opened. */
    wsprintf( szKey, "%s\\%s", BASE, OPTIONS );
    if ( RegCreateKeyEx( HKEY_CURRENT_USER, szKey, 0, NULL, 
        REG_OPTION_NON_VOLATILE, KEY_SET_VALUE, NULL, &hkey,
        &dwDisposition ) == ERROR_SUCCESS )
    {
		dwType = REG_DWORD;
		dwValue = (Settings_IsSoundEnabled() == TRUE) ? 1 : 0;
		dwDataSize = sizeof(DWORD);
		RegSetValueEx( hkey, "Sound", 0, dwType, (CONST BYTE *)&dwValue, dwDataSize );

		dwValue = Settings_GetFontSize();
		RegSetValueEx( hkey, "FontSize", 0, dwType, (CONST BYTE *)&dwValue, dwDataSize );

		dwValue = Settings_IsFontBold() ? 1 : 0;
		RegSetValueEx( hkey, "FontBold", 0, dwType, (CONST BYTE *)&dwValue, dwDataSize );

		dwValue = Settings_IsFontItalic() ? 1 : 0;
		RegSetValueEx( hkey, "FontItalic", 0, dwType, (CONST BYTE *)&dwValue, dwDataSize );

		dwType = REG_SZ;
		Settings_GetFontName( szValue );
		dwDataSize = lstrlen(szValue) + 1;
		RegSetValueEx( hkey, "FontName", 0, dwType, szValue, dwDataSize );

		RegCloseKey( hkey );
    }

	/* Process the Sounds subkey. */
    wsprintf( szKey, "%s\\%s", BASE, SOUNDS );
    if ( RegCreateKeyEx( HKEY_CURRENT_USER, szKey, 0, NULL,
            REG_OPTION_NON_VOLATILE, KEY_SET_VALUE, NULL, &hkey,
            &dwDisposition ) == ERROR_SUCCESS )
    {
		dwType = REG_SZ;
        Settings_GetSoundName( GSND_MOVE, szValue );
		dwDataSize = lstrlen(szValue) + 1;
        RegSetValueEx( hkey, "MoveSound", 0, dwType, (CONST BYTE *)szValue, dwDataSize );

        Settings_GetSoundName( GSND_BLOCKED, szValue );
		dwDataSize = lstrlen(szValue) + 1;
        RegSetValueEx( hkey, "BlockedSound", 0, dwType, (CONST BYTE *)szValue, dwDataSize );

        Settings_GetSoundName( GSND_SOLVED, szValue );
		dwDataSize = lstrlen(szValue) + 1;
        RegSetValueEx( hkey, "SolvedSound", 0, dwType, (CONST BYTE *)szValue, dwDataSize );

        RegCloseKey( hkey );
    }

    return TRUE;
}

void UseDefaultSettings( void )
{
    char szSoundName[_MAX_PATH];

    Settings_EnableSound();
    Settings_SetFont( "Arial", 18, TRUE, FALSE );

    LoadString( g_hInst, IDS_DEFAULTMOVESOUND, szSoundName, sizeof(szSoundName) );
    Settings_SetSoundName( GSND_MOVE, szSoundName );

	LoadString( g_hInst, IDS_DEFAULTBLOCKEDSOUND, szSoundName, sizeof(szSoundName) );
	Settings_SetSoundName( GSND_BLOCKED, szSoundName );

	LoadString( g_hInst, IDS_DEFAULTSOLVEDSOUND, szSoundName, sizeof(szSoundName) );
	Settings_SetSoundName( GSND_SOLVED, szSoundName );
}
