{***************************************************************************
File:      dws.pas
Version:   1.00
Tab stops: none
Project:   DiamondWare's Sound ToolKit for Windows
Copyright: 1994-1996 DiamondWare, Ltd.  All rights reserved.
Written:   by David Bollinger (based on dws.h for WIN-STK)
Purpose:   Contains declarations for the DW Sound ToolKit for Windows
History:   96/02/24 DB Started
           96/03/27 JCL Finalized for 1.0
           96/04/14 JCL Finalized for 1.01
           96/05/13 JCL Finalized for 1.1 (no changes)
***************************************************************************}



Unit DWS;


interface

uses WinTypes;

const

{***************************************************************************}
  {
   . If a call to a dws_ function returns 0, then this is an
   . indication that an error has occured.  Call dws_ErrNo to see which
   . one.  The following series of constants is the complete list of
   . possible return values for dws_ErrNo.
  }
  dws_EZERO                      =   0;  { no error }

  { The following 6 errors may be triggered by any dws_ function }
  dws_NOTINITTED                 =   1;
  dws_ALREADYINITTED             =   2;
  dws_NOTSUPPORTED               =   3;
  dws_INTERNALERROR              =   4;
  dws_INVALIDPOINTER             =   5;
  dws_RESOURCEINUSE              =   6;
  dws_MEMORYALLOCFAILED          =   7;
  dws_SETEVENTFAILED             =   8;

  {
   . dws_BUSY may be triggered during a call to a dws_ function, if any
   . dws_ function is already executing.  Ordinarily, this should never
   . occur.
  }
  dws_BUSY                       =   9;

  { This error may be triggered only by dws_Init }
  dws_Init_BUFTOOSMALL           = 101;

  { The following 3 errors may be triggered by any dws_D (dig) function }
  dws_D_NOTADWD                  = 201;
  dws_D_NOTSUPPORTEDVER          = 202;
  dws_D_BADDPLAY                 = 203;

  { The following error may be triggered by dws_DPlay }
  dws_DPlay_NOSPACEFORSOUND      = 251;

  { These 2 errors may be triggered by dws_WAV2DWD }
  dws_WAV2DWD_NOTAWAVE           = 301;
  dws_WAV2DWD_UNSUPPORTEDFORMAT  = 302;

  { The following error may be triggered by any dws_M (music) function }
  dws_M_BADMPLAY                 = 401;


{-----------------------------------------------------------------------------}
  {
   . This section defines bitfields which are used by various dws_
   . functions.  Each bit in a bitfield, by definition, may be set/reset
   . independantly of all other bits.
  }

  { flags for muscaps value of dws_DETECTRESULTS record }
  dws_muscap_NONE      = $0000; { no music playback }
  dws_muscap_MIDIPORT  = $0001; { output port }
  dws_muscap_SYNTH     = $0002; { generic int synth }
  dws_muscap_SQSYNTH   = $0004; { square wave int synth }
  dws_muscap_FMSYNTH   = $0008; { FM int synth }
  dws_muscap_MAPPER    = $0010; { MIDI mapper }
  dws_muscap_ANY       = $001F; { mask of all available modes }

  dws_digcap_NONE        = $0000; { no digitized sound playback }
  dws_digcap_11025_08_1  = $0001; { 11kHz, 8-bit, mono }
  dws_digcap_11025_08_2  = $0002; { 11kHz, 8-bit, stereo }
  dws_digcap_11025_16_1  = $0004; { 11kHz, 16-bit, mono }
  dws_digcap_11025_16_2  = $0008; { 11kHz, 16-bit, stereo }
  dws_digcap_22050_08_1  = $0010; { 22kHz, 8-bit, mono }
  dws_digcap_22050_08_2  = $0020; { 22kHz, 8-bit, stereo }
  dws_digcap_22050_16_1  = $0040; { 22kHz, 16-bit, mono }
  dws_digcap_22050_16_2  = $0080; { 22kHz, 16-bit, stereo }
  dws_digcap_44100_08_1  = $0100; { 44kHz, 8-bit, mono }
  dws_digcap_44100_08_2  = $0200; { 44kHz, 8-bit, stereo }
  dws_digcap_44100_16_1  = $0400; { 44kHz, 16-bit, mono }
  dws_digcap_44100_16_2  = $0800; { 44kHz, 16-bit, stereo }
  dws_digcap_ANY         = $0FFF; { mask of all available modes }

  {
   . The following constants are bitfields used in the flags field in the
   . dws_IDEAL record.
   .
   . SWAPLR should be used to compensate for downstream electronics which
   . switch the left and right field.
   .
   . Since pitch and volume change both require some additional CPU time,
   . you may want to disable them for slow machines and enable for fast
   . machines.  This is an easy way for you to do this, without changing
   . your program around.
   .
   . Notes:
   .   1) Pitch change is slower than volume change.
   .   2) Pitch decrease (dws_DPLAY.pitch > dws_IDENTITY) is slower than
   .      pitch increase
   .   3) When the source sound, or the STK, is 8-bit, raising the volume
   .      is slower than lowering it.  Otherwise both are equal.
  }
  dws_ideal_NONE           = $0000;  { this is the normal case }
  dws_ideal_SWAPLR         = $0001;  { swap Left & Right on playback }
  dws_ideal_DISABLEPITCH   = $0002;  { disable the pitch feature }
  dws_ideal_DISABLEVOLUME  = $0004;  { disable the volume feature }
  dws_ideal_MAXSPEED       = $0006;  { use this for very slow machines }

  {
   . These are the flag bitfields for the dws_DPLAY record.
   . Each corresponds to one field in the record.
  }
  dws_dplay_SND         = $00000001;
  dws_dplay_COUNT       = $00000002;
  dws_dplay_PRIORITY    = $00000004;
  dws_dplay_PRESND      = $00000008;
  dws_dplay_SOUNDNUM    = $00000010;
  dws_dplay_LVOL        = $00000020;
  dws_dplay_RVOL        = $00000040;
  dws_dplay_PITCH       = $00000080;
  dws_dplay_CALLBACK    = $00000100;
  dws_dplay_SYNCHRONOUS = $00000200;

  { The following 2 consts indicate the status of music playback }
  dws_MSONGSTATUSPLAYING           =   $0001;
  dws_MSONGSTATUSPAUSED            =   $0002;


{---------------------------------------------------------------------------}
  {
   . When the WIN-STK sends a message to your window, the lParam field
   . indicates the purpose of the message.
  }
  dws_event_SOUNDCOMPLETE  = $0001;
  dws_event_SOUNDSTARTED   = $0002;
  dws_event_SOUNDABORTED   = $0003;


  { For no change in volume level or pitch }
  dws_IDENTITY  = $100;

  { Default priority, if not specified in call to dws_DPlay }
  dws_NORMALPRIORITY  = $1000;



{*****************************************************************************}

type

  { For some reason Delphi 1.0 does not define a DWORD in WinTypes! }
  {$ifdef WIN32}
     PDWORD = ^DWORD;
  {$else}
     DWORD = longint;
     PDWORD = ^DWORD;
  {$endif}



  {
   . This section declares the record types used by the STK.  In each
   . case, the user must create an instance of the record prior to making
   . a call to any STK function which takes a pointer to it.  The STK does
   . not keep a pointer to any of these records internally; after the call
   . returns, you may deallocate it, if you wish.
   .
   . NB: The STK _does_ keep pointers to digitized sound buffers!
  }

  {
   . A pointer to this record is passed to dws_DetectHardWare, which fills
   . it in.  It should then be passed to dws_Init.  If you plan on writing
   . this record out to a file, it's important that you write the entire
   . contents.  There is information (for internal STK use only) in the
   . reserved[] field!
  }
  dws_DETECTRESULTS = packed record
    { This field indicates which music capabilities are available, if any }
    muscaps : DWORD;     { see dws_muscap_xxxxxx constants above }

    { This field indicates which dig capabilities are available, if any }
    digcaps : DWORD;     { see dws_digcap_xxxxxx constants above }

    { The next 4 fields should not be modified, except with great caution }
    digbfr : DWORD;      { minimum buffer size }

    reserved : array [0..19] of BYTE;
  end;


  {
   . A pointer to this record is passed as a parameter to dws_Init.  This
   . record allows the program to let the STK know what capabilities it
   . will actually use.
  }
  dws_IDEAL = packed record
    flags : DWORD;        { use combination of dws_ideal_xxxx }

    mustyp : DWORD;       { use dws_muscap_xxxxxx (mode) to select }

    digtyp : DWORD;       { use dws_digcap_xxxxxx (mode) to select }
    dignvoices : WORD;    { maximum number of digitized voices you'll use }

    reserved : array [0..17] of BYTE;
  end;


  {
   . A pointer to this record is passed to dws_DPlay.
   .
   . NB: The soundnum field is filled in by dws_DPlay as a return value.
  }
  dws_DPLAYREC = packed record
    flags : DWORD;       { flag indicates active fields in record         }
    snd : PBYTE;         { pointer to buffer which holds a .DWD file      }
    count : WORD;        { number of times to play, or 0=infinite loop    }
    priority : WORD;     { higher numbers mean higher priority            }
    presnd : WORD;       { soundnum to sequence sound _after_             }
    soundnum : WORD;     { dws_DPlay returns a snd number from 10-65535   }
    lvol : WORD;         { 0-65535, 0 is off, 256 is dws_IDENTITY         }
    rvol : WORD;         { if the output is mono lvol & rvol are averaged }
    pitch : WORD;        { 1-65535, 256 is dws_IDENTITY (0 is unuseable)  }
    dummy : WORD;        { added to insure DWORD alignment                }
    hwndmsg : HWND;      { handle of window to which to send msg          }
    wmessage : WORD;     { message sent to window hwndmsg                 }

    {$ifdef WIN32}
      reserved : array [0..15] of BYTE;
    {$else}
      reserved : array [0..19] of BYTE;
    {$endif}
  end;

  { A pointer to this record is passed to dws_MPlay. }
  dws_MPLAYREC = packed record
    track : PBYTE;        { ptr to NULL-terminated .MID file name string }
    count : WORD;         { number of times to play, or 0=infinite loop }

    reserved : array [0..9] of BYTE;
  end;

{$ifdef WIN32}
{***************************************************************************}
  { This section prototypes the WIN-STK functions }



  {
   . This function is callable at any time.  It returns the number of the
   . last error which occured.
  }
  function dws_ErrNo : WORD; stdcall;

{*****************************************************************************}


  {
   . Each function in this section has a boolean return value.  A 0
   . indicates that the function failed in some way.  In this case, call
   . dws_ErrNo to get the specific error.  Otherwise, a return value of 1
   . indicates that all is well.
  }
  function dws_DetectHardWare(var dr: dws_DETECTRESULTS) : WORDBOOL; stdcall;

  function dws_Init(var dr: dws_DETECTRESULTS;
                    var ideal: dws_IDEAL) : WORDBOOL; stdcall;

  {
   . If the program has called dws_Init, it _MUST_ call dws_Kill before it
   . terminates.
  }
  function dws_Kill : WORDBOOL; stdcall;


  {
   . The following function is the digital mixer of the STK.  A
   . value of 0 is off; dws_IDENTITY is normal, and 0xffff is maximum
   . volume (grossly distorted).
  }
  function dws_XDig(lvolume, rvolume: WORD) : WORDBOOL; stdcall;


  {
   . The following 9 functions comprise the digitized sound functions of
   . the STK.  See the documentation for complete details.
  }
  function dws_DPlay(var dplay: dws_DPLAYREC) : WORDBOOL; stdcall;

  {
   . Allows you to change the following items:
   .   count
   .   priority
   .   lvol
   .   rvol
   .   pitch
   .   hwndmsg and message (simultaneously)
  }
  function dws_DSetInfo(var dplay1 : dws_DPLAYREC;
                        var dplay2 : dws_DPLAYREC) : WORDBOOL; stdcall;

  {
   . Takes a ptr to a dws_DPLAY record; soundnum specifies which sound.
   .
   . Can retrieve the current values for any or all of the following:
   .   snd
   .   count
   .   priority
   .   presnd
   .   lvol
   .   rvol
   .   pitch
   .   hwndmsg and message (simultaneously)
  }
  function dws_DGetInfo(var dplay1 : dws_DPLAYREC;
                        var dplay2 : dws_DPLAYREC) : WORDBOOL; stdcall;

  { Callable at any time--even before dws_Init or after dws_Kill }
  function dws_DGetRateFromDWD(snd: PBYTE; var result: WORD) : WORDBOOL; stdcall;

  function dws_DDiscard(soundnum: WORD) : WORDBOOL; stdcall;

  function dws_DDiscardAO(snd: PBYTE) : WORDBOOL; stdcall;

  function dws_DClear : WORDBOOL; stdcall;         { global }

  function dws_DPause : WORDBOOL; stdcall;         { global }

  function dws_DUnPause : WORDBOOL; stdcall;       { global }

  {
   . Converts a .WAV buffer to a .DWD buffer
   .
   . This function has two usages.  In the first, pass the wave pointer, the
   . wave length in *len and a NIL pointer for dwd.  The routine will return
   . the length of the buffer required to hold the resulting DWD in *len.  In
   . the second usage, allocate a buffer of the correct size and pass its
   . address in dwd.  Make sure you pass the wave pointer in wave and the size
   . of the wave in *len.  It will then perform the conversion, returning the
   . number of bytes used in *len.
  }
  function dws_WAV2DWD(wave: PBYTE; var len: DWORD;
                       snd: PBYTE) : WORDBOOL; stdcall;

  { The following 5 functions comprise the music functions of the STK. }
  function dws_MPlay(var mplay: dws_MPLAYREC) : WORDBOOL; stdcall;

  function dws_MSongStatus(var result: WORD) : WORDBOOL; stdcall;

  function dws_MClear : WORDBOOL; stdcall;

  function dws_MPause : WORDBOOL; stdcall;

  function dws_MUnPause : WORDBOOL; stdcall;

  {
   . If your 16-bit application can't yield CPU time to the system, it must
   . call this function periodically to prevent the sound from pausing.
   . This is not required for 32-bit applications, but the function is
   . supported for compatibility.
  }
  function dws_Update : WORDBOOL; stdcall;

implementation

const
  DWSDLL = 'DWSW32';

  function dws_ErrNo : WORD; stdcall;
           external DWSDLL;
  function dws_DetectHardWare(var dr: dws_DETECTRESULTS) : WORDBOOL; stdcall;
           external DWSDLL;
  function dws_Init(var dr: dws_DETECTRESULTS;
                    var ideal: dws_IDEAL) : WORDBOOL; stdcall;
           external DWSDLL;
  function dws_Kill : WORDBOOL; stdcall;
           external DWSDLL;
  function dws_XDig(lvolume, rvolume: WORD) : WORDBOOL; stdcall;
           external DWSDLL;
  function dws_DPlay(var dplay: dws_DPLAYREC) : WORDBOOL; stdcall;
           external DWSDLL;
  function dws_DSetInfo(var dplay1 : dws_DPLAYREC;
                        var dplay2 : dws_DPLAYREC) : WORDBOOL; stdcall;
           external DWSDLL;
  function dws_DGetInfo(var dplay1 : dws_DPLAYREC;
                        var dplay2 : dws_DPLAYREC) : WORDBOOL; stdcall;
           external DWSDLL;
  function dws_DGetRateFromDWD(snd: PBYTE; var result: WORD) : WORDBOOL; stdcall;
           external DWSDLL;
  function dws_DDiscard(soundnum: WORD) : WORDBOOL; stdcall;
           external DWSDLL;
  function dws_DDiscardAO(snd: PBYTE) : WORDBOOL; stdcall;
           external DWSDLL;
  function dws_DClear : WORDBOOL; stdcall;
           external DWSDLL;
  function dws_DPause : WORDBOOL; stdcall;
           external DWSDLL;
  function dws_DUnPause : WORDBOOL; stdcall;
           external DWSDLL;
  function dws_WAV2DWD(wave: PBYTE; var len: DWORD;
                       snd: PBYTE) : WORDBOOL; stdcall;
           external DWSDLL;
  function dws_MPlay(var mplay: dws_MPLAYREC) : WORDBOOL; stdcall;
           external DWSDLL;
  function dws_MSongStatus(var result: WORD) : WORDBOOL; stdcall;
           external DWSDLL;
  function dws_MClear : WORDBOOL; stdcall;
           external DWSDLL;
  function dws_MPause : WORDBOOL; stdcall;
           external DWSDLL;
  function dws_MUnPause : WORDBOOL; stdcall;
           external DWSDLL;
  function dws_Update : WORDBOOL; stdcall;
           external DWSDLL;

{$else}
{***************************************************************************}
  { This section prototypes the WIN-STK functions }



  {
   . This function is callable at any time.  It returns the number of the
   . last error which occured.
  }
  function dws_ErrNo : WORD;

{*****************************************************************************}


  {
   . Each function in this section has a boolean return value.  A 0
   . indicates that the function failed in some way.  In this case, call
   . dws_ErrNo to get the specific error.  Otherwise, a return value of 1
   . indicates that all is well.
  }
  function dws_DetectHardWare(var dr: dws_DETECTRESULTS) : WORDBOOL;

  function dws_Init(var dr: dws_DETECTRESULTS;
                    var ideal: dws_IDEAL) : WORDBOOL;

  {
   . If the program has called dws_Init, it _MUST_ call dws_Kill before it
   . terminates.
  }
  function dws_Kill : WORDBOOL;


  {
   . The following function is the digital mixer of the STK.  A
   . value of 0 is off; dws_IDENTITY is normal, and 0xffff is maximum
   . volume (grossly distorted).
  }
  function dws_XDig(lvolume, rvolume: WORD) : WORDBOOL;


  {
   . The following 9 functions comprise the digitized sound functions of
   . the STK.  See the documentation for complete details.
  }
  function dws_DPlay(var dplay: dws_DPLAYREC) : WORDBOOL;

  {
   . Allows you to change the following items:
   .   count
   .   priority
   .   lvol
   .   rvol
   .   pitch
   .   hwndmsg and message (simultaneously)
  }
  function dws_DSetInfo(var dplay1 : dws_DPLAYREC;
                        var dplay2 : dws_DPLAYREC) : WORDBOOL;

  {
   . Takes a ptr to a dws_DPLAY record; soundnum specifies which sound.
   .
   . Can retrieve the current values for any or all of the following:
   .   snd
   .   count
   .   priority
   .   presnd
   .   lvol
   .   rvol
   .   pitch
   .   hwndmsg and message (simultaneously)
  }
  function dws_DGetInfo(var dplay1 : dws_DPLAYREC;
                        var dplay2 : dws_DPLAYREC) : WORDBOOL;

  { Callable at any time--even before dws_Init or after dws_Kill }
  function dws_DGetRateFromDWD(snd: PBYTE; var result: WORD) : WORDBOOL;

  function dws_DDiscard(soundnum: WORD) : WORDBOOL;

  function dws_DDiscardAO(snd: PBYTE) : WORDBOOL;

  function dws_DClear : WORDBOOL;         { global }

  function dws_DPause : WORDBOOL;         { global }

  function dws_DUnPause : WORDBOOL;       { global }

  {
   . Converts a .WAV buffer to a .DWD buffer
   .
   . This function has two usages.  In the first, pass the wave pointer, the
   . wave length in *len and a NIL pointer for dwd.  The routine will return
   . the length of the buffer required to hold the resulting DWD in *len.  In
   . the second usage, allocate a buffer of the correct size and pass its
   . address in dwd.  Make sure you pass the wave pointer in wave and the size
   . of the wave in *len.  It will then perform the conversion, returning the
   . number of bytes used in *len.
  }
  function dws_WAV2DWD(wave: PBYTE; var len: DWORD;
                       snd: PBYTE) : WORDBOOL;

  { The following 5 functions comprise the music functions of the STK. }
  function dws_MPlay(var mplay: dws_MPLAYREC) : WORDBOOL;

  function dws_MSongStatus(var result: WORD) : WORDBOOL;

  function dws_MClear : WORDBOOL;

  function dws_MPause : WORDBOOL;

  function dws_MUnPause : WORDBOOL;

  {
   . If your 16-bit application can't yield CPU time to the system, it must
   . call this function periodically to prevent the sound from pausing.
   . This is not required for 32-bit applications, but the function is
   . supported for compatibility.
  }
  function dws_Update : WORDBOOL;

implementation

const
  DWSDLL = 'DWSW16';

  function dws_ErrNo : WORD;
           external DWSDLL;
  function dws_DetectHardWare(var dr: dws_DETECTRESULTS) : WORDBOOL;
           external DWSDLL;
  function dws_Init(var dr: dws_DETECTRESULTS;
                    var ideal: dws_IDEAL) : WORDBOOL;
           external DWSDLL;
  function dws_Kill : WORDBOOL;
           external DWSDLL;
  function dws_XDig(lvolume, rvolume: WORD) : WORDBOOL;
           external DWSDLL;
  function dws_DPlay(var dplay: dws_DPLAYREC) : WORDBOOL;
           external DWSDLL;
  function dws_DSetInfo(var dplay1 : dws_DPLAYREC;
                        var dplay2 : dws_DPLAYREC) : WORDBOOL;
           external DWSDLL;
  function dws_DGetInfo(var dplay1 : dws_DPLAYREC;
                        var dplay2 : dws_DPLAYREC) : WORDBOOL;
           external DWSDLL;
  function dws_DGetRateFromDWD(snd: PBYTE; var result: WORD) : WORDBOOL;
           external DWSDLL;
  function dws_DDiscard(soundnum: WORD) : WORDBOOL;
           external DWSDLL;
  function dws_DDiscardAO(snd: PBYTE) : WORDBOOL;
           external DWSDLL;
  function dws_DClear : WORDBOOL;
           external DWSDLL;
  function dws_DPause : WORDBOOL;
           external DWSDLL;
  function dws_DUnPause : WORDBOOL;
           external DWSDLL;
  function dws_WAV2DWD(wave: PBYTE; var len: DWORD;
                       snd: PBYTE) : WORDBOOL;
           external DWSDLL;
  function dws_MPlay(var mplay: dws_MPLAYREC) : WORDBOOL;
           external DWSDLL;
  function dws_MSongStatus(var result: WORD) : WORDBOOL;
           external DWSDLL;
  function dws_MClear : WORDBOOL;
           external DWSDLL;
  function dws_MPause : WORDBOOL;
           external DWSDLL;
  function dws_MUnPause : WORDBOOL;
           external DWSDLL;
  function dws_Update : WORDBOOL;
           external DWSDLL;
{$endif}

{***************************************************************************}


end.
