/******************************************************************************
File:      playstk.c
Version:   1.00
Tab stops: every 2 columns
Project:   DiamondWare's Sound ToolKit for Windows
Copyright: 1996 DiamondWare, Ltd.  All rights reserved.*
Written:   95/12/11 by John Lundy
Purpose:   Contains sample application using the WIN-STK
History:   95/12/11 JCL Started
           96/03/27 KW & JCL finalized for 1.0
           96/05/13 JCL finalized for 1.1 (no changes)

*Permission is expressely granted to use this program or any derivitive made
 from it to registered users of the WIN-STK.
******************************************************************************/

#include <windows.h>
#include <commdlg.h>
#include <ctype.h>
#include <stdlib.h>
#include <string.h>

#include "resource.h"

#include "dws.h"



#ifdef  __BORLANDC__
  #pragma warn -par
#endif



#define SOUNDTOTAL 16



static WORD              volleft, volright, pitch;

static int               presentsnd=0, previoussnd=0;

static LPBYTE            sndptr[SOUNDTOTAL];
static HGLOBAL           hsndptr[SOUNDTOTAL];

static HWND              sbvolleft;
static HWND              sbvolright;
static HWND              sbpitch;
static HWND              cbswaplr;

/* variables used by the WIN-STK */
static dws_DETECTRESULTS dres;
static dws_DPLAY         dplay;
static dws_IDEAL         ideal;



static void DisplayErr(char *comment)
{
  char  totstr[256];
  char *errstr;

  switch (dws_ErrNo())
  {
    case dws_EZERO:
    {
      errstr = "dws_EZERO (Why am I here?): %s";

      break;
    }
    case dws_NOTINITTED:
    {
      errstr = "dws_NOTINITTED: %s";

      break;
    }
    case dws_ALREADYINITTED:
    {
      errstr = "dws_ALREADYINITTED: %s";

      break;
    }
    case dws_NOTSUPPORTED:
    {
      errstr = "dws_NOTSUPPORTED: %s";

      break;
    }
    case dws_INTERNALERROR:
    {
      errstr = "dws_INTERNALERROR: %s";

      break;
    }
    case dws_INVALIDPOINTER:
    {
      errstr = "dws_INVALIDPOINTER: %s";

      break;
    }
    case dws_RESOURCEINUSE:
    {
      errstr = "dws_RESOURCEINUSE: %s";

      break;
    }
    case dws_MEMORYALLOCFAILED:
    {
      errstr = "dws_MEMORYALLOCFAILED: %s";

      break;
    }
    case dws_SETEVENTFAILED:
    {
      errstr = "dws_SETEVENTFAILED: %s";

      break;
    }
    case dws_BUSY:
    {
      errstr = "dws_BUSY: %s";

      break;
    }
    case dws_Init_BUFTOOSMALL:
    {
      errstr = "dws_Init_BUFTOOSMALL: %s";

      break;
    }
    case dws_D_NOTADWD:
    {
      errstr = "dws_D_NOTADWD: %s";

      break;
    }
    case dws_D_NOTSUPPORTEDVER:
    {
      errstr = "dws_D_NOTSUPPORTEDVER: %s";

      break;
    }
    case dws_D_BADDPLAY:
    {
      errstr = "dws_D_BADDPLAY: %s";

      break;
    }
    case dws_DPlay_NOSPACEFORSOUND:
    {
      errstr = "dws_DPlay_NOSPACEFORSOUND: %s";

      break;
    }
    case dws_WAV2DWD_NOTAWAVE:
    {
      errstr = "dws_WAV2DWD_NOTAWAVE: %s";

      break;
    }
    case dws_WAV2DWD_UNSUPPORTEDFORMAT:
    {
      errstr = "dws_WAV2DWD_UNSUPPORTEDFORMAT: %s";

      break;
    }
    default:
    {
      errstr = "DEFAULT (unknown error!): %s";

      break;
    }
  }

  wsprintf(totstr, errstr, comment);
  MessageBox(NULL, totstr, "Sound ToolKit Error", MB_ICONSTOP | MB_OK);

  /* whether it needs it, or not... */
  dws_Kill();

  /* In our simple scheme, all errors are fatal. <g> */
  exit(-1);
}


/* Shutdown WIN-STK */
static void Kill(void)
{
  if (!dws_DClear())  /* stop all playing sounds */
  {
    DisplayErr("dws_DClear During Kill");
  }

  if (!dws_MClear())  /* stop any playing music */
  {
    DisplayErr("dws_MClear During Kill");
  }

  if (!dws_Kill())    /* stop everything else */
  {
    DisplayErr("dws_Kill During Kill");
  }
}


/* Reinitialize WIN-STK with new digitized sampling rate */
static void ChangeRate(DWORD digtyp)
{
  Kill();

  ideal.digtyp = digtyp;

  /* Restart it with the new rate */
  if (!dws_Init(&dres, &ideal))
  {
    DisplayErr("dws_Init During ChangeRate");
  }
}


static DWORD LoadFile(char *sndfile)
{
  char       txt[128];
  DWORD      filesize;
  #ifdef  WIN32
    DWORD    tmp;
    HANDLE   hfile;
  #else
    HFILE    hfile;
    OFSTRUCT openbuff;
  #endif

  if (hsndptr[presentsnd])
  {
    GlobalUnlock(hsndptr[presentsnd]);
    GlobalFree(hsndptr[presentsnd]);
  }

  #ifdef  WIN32
    hfile = CreateFile(sndfile, GENERIC_READ, FILE_SHARE_READ, NULL,
                       OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);

    if (hfile)
    {
      filesize = GetFileSize(hfile, NULL);

      hsndptr[presentsnd] = GlobalAlloc(GHND, filesize);
      sndptr[presentsnd] = (LPBYTE)GlobalLock(hsndptr[presentsnd]);

      ReadFile(hfile, sndptr[presentsnd], filesize, &tmp, NULL);
      CloseHandle(hfile);
    }
  #else
    hfile = OpenFile(sndfile, &openbuff, OF_READ);

    if (hfile != HFILE_ERROR)
    {
      filesize = _llseek(hfile, 0L, 2);

      hsndptr[presentsnd] = GlobalAlloc(GHND, filesize);
      sndptr[presentsnd] = (LPBYTE)GlobalLock(hsndptr[presentsnd]);

      _llseek(hfile, 0L, 0);
      _hread(hfile, sndptr[presentsnd], filesize);

      _lclose(hfile);
    }
  #endif
  else
  {
    lstrcpy(txt, "Could not load ");
    lstrcat(txt, sndfile);

    MessageBox(NULL,txt,"Sound ToolKit Error",MB_OK);

    return (0);
  }

  return (filesize);
}


/* Convert WAV to DWD */
static int ConvertWave(char *wavefile)
{
  BYTE   *wavedwd;
  WORD    status;
  DWORD   wavesize, len, tmp;
  HGLOBAL hwavedwd;

  wavesize = LoadFile(wavefile);

  if (!wavesize)
  {
    return (0);
  }

  /* convert WAV to DWD */
  tmp = len = wavesize;
  status = dws_WAV2DWD(sndptr[presentsnd], &tmp, NULL);

  if (!status)
  {
    DisplayErr("Get Wave Size");

    return (0);
  }

  hwavedwd = GlobalAlloc(GHND, tmp);
  wavedwd = (BYTE*)GlobalLock(hwavedwd);
  status = dws_WAV2DWD(sndptr[presentsnd], &len, wavedwd);

  if (!status)
  {
    GlobalUnlock(hwavedwd);
    GlobalFree(hwavedwd);
    DisplayErr("Wave Conversion");

    return (0);
  }

  GlobalUnlock(hsndptr[presentsnd]);
  GlobalFree(hsndptr[presentsnd]);

  hsndptr[presentsnd] = hwavedwd;
  sndptr[presentsnd]  = wavedwd;

  return (1);
}


/* Get a new WAV, DWD, or MID file */
static VOID DoNew(HWND hwnd)
{
  char         buffer[_MAX_PATH]="";
  OPENFILENAME of;

  /* set up the OPENFILE structure, then use the appropriate common dialog */
  of.lStructSize       = sizeof(OPENFILENAME);
  of.hwndOwner         = NULL;

  #ifdef  WIN32
    of.hInstance       = (HINSTANCE)GetWindowLong(hwnd, GWL_HINSTANCE);
  #else
    of.hInstance       = (HINSTANCE)GetWindowWord(hwnd, GWW_HINSTANCE);
  #endif

  of.lpstrFilter       = "Sounds and Music\000*.WAV;*.DWD;*.MID\000\000";
  of.lpstrCustomFilter = NULL;
  of.nMaxCustFilter    = 0;
  of.nFilterIndex      = 0;
  of.lpstrFile         = buffer;
  of.nMaxFile          = _MAX_PATH;
  of.lpstrFileTitle    = NULL;
  of.nMaxFileTitle     = 0;
  of.lpstrInitialDir   = NULL;
  of.lpstrTitle        = "Sound ToolKit - Open";
  of.Flags             = OFN_HIDEREADONLY;
  of.nFileOffset       = 0;
  of.nFileExtension    = 0;
  of.lpstrDefExt       = NULL;
  of.lCustData         = 0;
  of.lpfnHook          = NULL;
  of.lpTemplateName    = NULL;

  if (GetOpenFileName(&of))
  {
    /* Put the filename in the list box. */
    SendDlgItemMessage(hwnd, DID_LISTBOX, LB_ADDSTRING, 0, (LONG) buffer);
  }
}


/* Play the selected listbox item: WAV, DWD, or MID */
static VOID DoPlay(HWND hwnd)
{
  static char textselection[_MAX_PATH]="";
  char        buffer[_MAX_PATH], *ptr;
  WORD        status;
  static int  newselection=1;
  int         sel;
  dws_MPLAY   mplay;

  /* determine which item is selected in the list box, and get the text */
  sel = (int)SendDlgItemMessage(hwnd, DID_LISTBOX, LB_GETCURSEL, 0, 0);

  if (sel == LB_ERR)
  {
    MessageBox(NULL,
               "No listbox item is selected.",
               "Sound ToolKit Error",
               MB_OK);
    return;
  }

  SendDlgItemMessage(hwnd, DID_LISTBOX, LB_GETTEXT, (int)sel, (LONG)buffer);

  /* don't reload if the same file is replayed */
  if (lstrcmp(buffer, textselection) != 0)
  {
    newselection = 1;
    lstrcpy(textselection, buffer);
  }

  ptr = strrchr(buffer, '.');

  if (!ptr)
  {
    MessageBox(NULL,
               "File name format not known",
               "Sound ToolKit Error",
               MB_OK);
  }

  /* convert the retrieved text to lowercase */
  for (sel=0 ; buffer[sel] ; sel++)
  {
    buffer[sel] = (char)tolower(buffer[sel]);
  }

  if (lstrcmp(ptr, ".wav") == 0 || lstrcmp(ptr, ".dwd") == 0)
  {
    if (newselection)
    {
      if (lstrcmp(ptr, ".wav") == 0)
      {
        if (!ConvertWave(buffer))
        {
          return;
        }
      }
      else if (lstrcmp(ptr, ".dwd") == 0)
      {
        if (!LoadFile(buffer))
        {
          return;
        }
      }

      dplay.snd = sndptr[presentsnd];
    }
    else
    {
      dplay.snd = sndptr[previoussnd];
    }

    dplay.count = 1;
    if (volleft >= 8)
    {
      dplay.lvol = (WORD)((volleft - 7) * 256);
    }
    else
    {
      dplay.lvol = (WORD)(volleft * 32);
    }

    if (volright >= 8)
    {
      dplay.rvol = (WORD)((volright - 7) * 256);
    }
    else
    {
      dplay.rvol = (WORD)(volright * 32);
    }

    if (pitch >= 8)
    {
      dplay.pitch = (WORD)((pitch - 7) * 256);
    }
    else
    {
      dplay.pitch = (WORD)(pitch * 32);
    }

    dplay.flags = dws_dplay_SND|dws_dplay_COUNT|
                  dws_dplay_LVOL|dws_dplay_RVOL|
                  dws_dplay_PITCH;
    status = dws_DPlay(&dplay);

    if (!status)
    {
      DisplayErr("dws_DPlay During DoPlay");
    }

    if (newselection)
    {
      previoussnd = presentsnd++;
      if (presentsnd >= SOUNDTOTAL) presentsnd = 0;
    }

    newselection = 0;
  }
  else if (lstrcmp(ptr, ".mid") == 0)
  {
    mplay.track = (BYTE*)buffer;
    mplay.count = 1;
    status = dws_MPlay(&mplay);

    if (!status)
    {
      DisplayErr("dws_MPlay During DoPlay");
    }
  }
}


/* Stop the music and sound */
static VOID DoStop(void)
{
  if (!dws_DClear())  /* stop all playing sounds */
  {
    DisplayErr("DoStop of Digitized Sound");
  }

  if (!dws_MClear())  /* stop any playing music */
  {
    DisplayErr("DoStop of MIDI Music");
  }
}


/* Remove selected filename from list box */
static VOID DoRemove(HWND hwnd)
{
  char buffer[_MAX_PATH];
  int  sel;

  /* determine which item is selected in the list box, and get the text */
  sel = (int)SendDlgItemMessage(hwnd, DID_LISTBOX, LB_GETCURSEL, 0, 0);

  if (sel == LB_ERR)
  {
    MessageBox(NULL,
               "No listbox item is selected.",
               "Sound ToolKit Error",
               MB_OK);
    return;
  }

  SendDlgItemMessage(hwnd, DID_LISTBOX, LB_GETTEXT, (int)sel, (LONG)buffer);
  SendDlgItemMessage(hwnd, DID_LISTBOX, LB_DELETESTRING, sel, 0);
}


static VOID ScrollControl(HWND hwnd, HWND sb, int npos, int nbar)
{
  int val;

  val = GetScrollPos(sb, SB_CTL);

  switch (nbar)
  {
    case SB_LINEUP:
    {
      if (val)
      {
        val--;
      }

      break;
    }
    case SB_LINEDOWN:
    {

      if (val < 16)
      {
        val++;
      }

      break;
    }
    case SB_PAGEUP:
    {
      val -= 2;

      if (val < 0)
      {
        val = 0;
      }

      break;
    }
    case SB_PAGEDOWN:
    {
      val += 2;

      if (val > 16)
      {
        val = 16;
      }

      break;
    }
    case SB_THUMBPOSITION:
    {
      val = npos;
      break;
    }
    case SB_THUMBTRACK:
    {
      val = npos;
      break;
    }
    case SB_ENDSCROLL:
    {
      break;
    }
  }

  /* set the scrollbar to its new position */
  SetScrollPos(sb, SB_CTL, val, TRUE);

  /* want to Get and Set this information in playing sound */
  dplay.flags = dws_dplay_SOUNDNUM|dws_dplay_LVOL|dws_dplay_RVOL|dws_dplay_PITCH;

  /*
   . depending upon which scrollbar was moved, calculate the new
   . volume or pitch value
  */
  if (sb == sbvolleft)
  {
    volleft = (WORD)(16 - val);

    if (!dws_DGetInfo(&dplay, NULL))
    {
      DisplayErr("dws_DGetInfo During Left Volume Change");
    }

    if (volleft >= 8)
    {
      dplay.lvol = (WORD)((volleft - 7) * 256);
    }
    else
    {
      dplay.lvol = (WORD)(volleft * 32);
    }

    dplay.flags = dws_dplay_SOUNDNUM|dws_dplay_LVOL;

    if (!dws_DSetInfo(&dplay, NULL))
    {
      DisplayErr("dws_DSetInfo During Left Volume Change");
    }
  }
  else if (sb == sbvolright)
  {
    volright = (WORD)(16 - val);

    if (!dws_DGetInfo(&dplay, NULL))
    {
      DisplayErr("dws_DGetInfo During Right Volume Change");
    }

    if (volright >= 8)
    {
      dplay.rvol = (WORD)((volright - 7) * 256);
    }
    else
    {
      dplay.rvol = (WORD)(volright * 32);
    }

    dplay.flags = dws_dplay_SOUNDNUM|dws_dplay_RVOL;

    if (!dws_DSetInfo(&dplay, NULL))
    {
      DisplayErr("dws_DSetInfo During Right Volume Change");
    }
  }
  else if (sb == sbpitch)
  {
    pitch = (WORD)val;

    if (!pitch)
    {
      pitch++;
    }

    if (!dws_DGetInfo(&dplay, NULL))
    {
      DisplayErr("dws_DGetInfo During Pitch Change");
    }

    if (pitch >= 8)
    {
      dplay.pitch = (WORD)((pitch - 7) * 256);
    }
    else
    {
      dplay.pitch = (WORD)(pitch * 32);
    }

    dplay.flags = dws_dplay_SOUNDNUM|dws_dplay_PITCH;

    if (!dws_DSetInfo(&dplay, NULL))
    {
      DisplayErr("dws_DSetInfo During Pitch Change");
    }
  }
  else
  {
    MessageBox(NULL,
               "Unknown control reference in ScrollControl",
               "Sound ToolKit Error",
               MB_OK);
  }
}


static LRESULT CALLBACK MainDlgProc(HWND hwnd, UINT message,
                                    WPARAM wParam, LPARAM lParam)
{
  WORD result;
  int  i;
  UINT swaplr;

  switch (message)
  {
    case WM_INITDIALOG:
    {
      sbvolleft = GetDlgItem(hwnd, IDC_VOL_LEFT);
      SetScrollRange(sbvolleft, SB_CTL, 0, 16, FALSE);
      SetScrollPos(sbvolleft, SB_CTL, 8, FALSE);

      sbvolright = GetDlgItem(hwnd, IDC_VOL_RIGHT);
      SetScrollRange(sbvolright, SB_CTL, 0, 16, FALSE);
      SetScrollPos(sbvolright, SB_CTL, 8, FALSE);

      sbpitch = GetDlgItem(hwnd, IDC_PITCH);
      SetScrollRange(sbpitch, SB_CTL, 1, 16, FALSE);
      SetScrollPos(sbpitch, SB_CTL, 8, FALSE);

      cbswaplr = GetDlgItem(hwnd, IDC_SWAPLR);
      CheckDlgButton(hwnd, IDC_SWAPLR, FALSE);

      CheckDlgButton(hwnd, IDC_RATE_11025, TRUE);
      CheckDlgButton(hwnd, IDC_RATE_22050, FALSE);
      CheckDlgButton(hwnd, IDC_RATE_44100, FALSE);

      volleft  = 8;
      volright = 8;
      pitch    = 8;

      if (!dws_DetectHardWare(&dres))
      {
        DisplayErr("dws_DetectHardWare During Initdialog");
      }
      else if (!dres.digcaps)
      {
        MessageBox(NULL,
                   "Your computer does not support sound playback.",
                   "Sound ToolKit Error", MB_OK);

        exit (-1);
      }
      else if (!(dres.digcaps & dws_digcap_11025_08_2))
      {
        MessageBox(NULL,
                   "DiamondWare's Sound ToolKit for Windows\n"
                   "supports sound playback on your computer.\n"
                   "However, this demo requires 8-bit stereo,\n"
                   "which your computer does not support."
                   "Your sound hardware does not support\n"
                   "11025Hz, two channel, 8 bit sound.\n"
                   "This demo will not run properly on\n"
                   "your computer",
                   "Sound ToolKit Error", MB_OK);

        exit (-1);
      }

      /* Ordered from "best" to "worst" choice for music devices */
      if (dres.muscaps & dws_muscap_MAPPER)
      {
        result = dws_muscap_MAPPER;
      }
      else if (dres.muscaps & dws_muscap_FMSYNTH)
      {
        result = dws_muscap_FMSYNTH;
      }
      else if (dres.muscaps & dws_muscap_SYNTH)
      {
        result = dws_muscap_SYNTH;
      }
      else if (dres.muscaps & dws_muscap_SQSYNTH)
      {
        result = dws_muscap_SQSYNTH;
      }
      else if (dres.muscaps & dws_muscap_MIDIPORT)
      {
        result = dws_muscap_MIDIPORT;
      }
      else
        result = dws_muscap_NONE;

      ideal.mustyp     = result;
      ideal.digtyp     = dws_digcap_11025_08_2;
      ideal.dignvoices = 6;                     //6 voices is plenty

      if (!dws_Init(&dres, &ideal))
      {
        DisplayErr("dws_Init During Init");
      }
      else
      {
        if (!dws_XDig(128, 128))    /*  half volume */
        {
          DisplayErr("dws_XDig During Initdialog");
        }
      }

      return (TRUE);
    }
    case WM_SYSCOMMAND:
    {
      if (wParam == SC_CLOSE)
      {
        Kill();

        for (i=0 ; i < SOUNDTOTAL ; i++)
        {
          if (hsndptr[i])
          {
            GlobalUnlock(hsndptr[i]);
            GlobalFree(hsndptr[i]);
          }
        }

        EndDialog(hwnd, TRUE);

        return (TRUE);
      }

      return (FALSE); /* end WM_SYSCOMMAND */
    }
    case WM_COMMAND:
    {
      switch (LOWORD(wParam))   //which control ID?
      {
        case DID_LISTBOX:
        {
          #ifdef  WIN32
            if (HIWORD(wParam) == LBN_DBLCLK)
          #else
            if (HIWORD(lParam) == LBN_DBLCLK)
          #endif
          {
            DoPlay(hwnd);
          }

          break;
        }
        case DID_NEW:
        {
          DoNew(hwnd);

          break;
        }
        case DID_PLAY:
        {
          DoPlay(hwnd);

          break;
        }
        case DID_STOP:
        {
          DoStop();

          break;
        }
        case DID_REMOVE:
        {
          DoRemove(hwnd);

          break;
        }
        case IDC_SWAPLR:
        {
          #ifdef  WIN32
            if ((HWND)lParam == cbswaplr)
          #else
            if ((HWND)LOWORD(lParam) == cbswaplr)
          #endif
          {
            Kill();

            swaplr = 1 - IsDlgButtonChecked(hwnd, IDC_SWAPLR);
            CheckDlgButton(hwnd, IDC_SWAPLR, swaplr);
            ideal.flags = swaplr ? dws_ideal_SWAPLR : 0;

            if (!dws_Init(&dres, &ideal))
            {
              DisplayErr("dws_Init During Init");
            }
          }

          break;
        }
        case IDC_RATE_11025:
        {
          ChangeRate(dws_digcap_11025_08_2);

          break;
        }
        case IDC_RATE_22050:
        {
          ChangeRate(dws_digcap_22050_08_2);

          break;
        }
        case IDC_RATE_44100:
        {
          ChangeRate(dws_digcap_44100_08_2);

          break;
        }
      }

      return (TRUE);
    }
    case WM_VSCROLL:
    {
      #ifdef  WIN32
        ScrollControl(hwnd, (HWND)lParam, HIWORD(wParam), LOWORD(wParam));
      #else
        ScrollControl(hwnd, (HWND)HIWORD(lParam), LOWORD(lParam), wParam);
      #endif

      break;
    }
    default:
    {
      break;
    }
  }

  return (FALSE);
}


int PASCAL WinMain(HINSTANCE hinstCurrent, HINSTANCE hPrevInstance,
                   LPSTR lpCmdLine, int nCmdShow)
{
  int result;

  #if defined(WIN32) && defined(_MSC_VER)
    UNREFERENCED_PARAMETER(hPrevInstance);
    UNREFERENCED_PARAMETER(lpCmdLine);
    UNREFERENCED_PARAMETER(nCmdShow);
  #endif

  result = DialogBox(hinstCurrent, MAKEINTRESOURCE(DID_SAMPLEDLG),
                     NULL, (DLGPROC)MainDlgProc);

  return (result);
}
