/*
    Copyright (C) 1998 by Jorrit Tyberghein

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef COMMAND_H
#define COMMAND_H

#include <stdio.h>
#include "igraph3d.h"
#include "cstools/simpcons.h"

class csSystemDriver;
class csWorld;
class csCamera;
class csConsole;
interface ISystem;

/**
 * This class represents a command that can be send thru the console
 * or by pressing a key from within the engine.
 * It is a static class which contains only static data.
 */
class Command : public csSimpleCommand
{
private:
  /// The shared instance.
  static Command* shared_instance;
  /// The world that this command interpreter is using.
  static csWorld* world;
  /// The corresponding camera object for this command interpreter.
  static csCamera* camera;
  /// The IGraphics3D interface to use.
  static IGraphics3D* g3d;
  /// The ISystem interface to use.
  static ISystem* system;
  /// The console.
  static csConsole* console;

  /**
   * If this variable is non-NULL there is a running script from
   * which commands are read and executed (one command every frame).
   */
  static FILE* script;

public:
  /// Call this first.
  static void Initialize (csWorld* world, csCamera* camera, IGraphics3D* g3d, csConsole* console, ISystem* system);

  /// Return the shared instance.
  static Command* SharedInstance();

  /// Perform the command and return true if it was a valid command.
  static bool perform (char* cmd, char* arg = NULL);

  /**
   * Perform the command line (split in command and args),
   * perform the command and return true if it was a valid command.
   */
  static bool perform_line (char* line);

  /// Virtual override of csSimpleCommand.
  virtual bool PerformLine (char* line);

  /// Start a script of commands from a file.
  static bool start_script (char* scr);

  /**
   * Check if a script is running and if so, return one command from the
   * script. This routine will return false if there is no script, or
   * if the last command from the script has been read. The script will
   * be closed automatically in the last case.
   */
  static bool get_script_line (char* buf, int max_size);


  /// Change boolean variable value
  static void change_boolean (char* arg, bool* value, char* what);
  /// Change boolean variable value specifically for G3D.
  static bool change_boolean_gfx3d (char* arg, G3D_RENDERSTATEOPTION op, char* what);
  /// Change value to one of listed in choices[]
  static void change_choice (char* arg, int* value, char* what, char** choices, int num);
  /// Change float variable value
  static bool change_float (char* arg, float* value, char* what, float min, float max);
  /// Change int variable value
  static bool change_int (char* arg, int* value, char* what, int min, int max);

  /**
   * Additional command handler. If not NULL, this handler will be called
   * BEFORE internal handler, giving your handler a chance to override
   * default behaviour. Should return TRUE if command has been recognized.
   */
  typedef bool (*CmdHandler) (char* cmd, char* arg);
  static CmdHandler ExtraHandler;
};

#endif
