/*  MCEstd.h

  matrix management functions:
  manipulations on matrices of data type "eComplex"
  (extended-precision complex numbers)

  Copyright (c) 1996-1998 by Martin Sander
  All Rights Reserved.
*/

#if !defined( __MATLIB_H )
   #include <MatLib.h>
#endif
#if !defined( __VCESTD_H )
   #include <VCEstd.h>
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*************   Dynamic Generation of Matrices   ************************/

ceMatrix __vf  MCE_matrix(  unsigned ht, unsigned len );
ceMatrix __vf  MCE_matrix0( unsigned ht, unsigned len );
    /*  Notice that, in the memory model HUGE,
        neither len nor ht may exceed 1637            */

/***************************************************************************
 *  The following definitions ensure compatibility between dynamically     *
 *  and statically allocated matrices. The definitions are somewhat        *
 *  cumbersome, but the result for you is that you need not care about     *
 *  the differences between the two types.                                 *
 *  (Internally, the address of the first element of any matrix is needed; *
 *  the expression "MA[0]" is evaluated in a different way for both types, *
 *  but yields in either case the correct address to be passed to the      *
 *  function you wish to call.)                                            *
 *  Only in the rare case that you need to pass the address of one of      *
 *  these functions as an argument to another function, you have to use    *
 *  the actual run-time functions defined further below. Be careful with   *
 *  this: future development of compilers may allow us to avoid this un-   *
 *  handy scheme of macros. So future versions of MatrixLib may no longer  *
 *  use these run-time names.                                              *
 ***************************************************************************/


/***  Addressing single elements of dynamically allocated matrices: ******
     These two functions are for compatibility with Pascal
     (where elements of dynamically allocated matrices are not directly
     accessible), and for getting around the pointer arithmetics bug in
     some versions of Borland C++.                                     */

#define MCE_Pelement( MA, ht, len, m, n ) MCEPelement( MA[0], ht, len, m, n )
                     /* returns a pointer to MA[m][n]. */
#define MCE_element( MA, ht, len, m, n ) *MCEPelement( MA[0], ht, len, m, n )
                     /* dereferenced pointer */

 /****************  Initialization  ***************************************

    To initialize all elements of a matrix with the same value,
    or to perform arithmetic operations on all elements simultaneously,
    refer to the functions of VectorLib, declared in <VCEstd.h>, <VCEmath.h>.
    In order to use the VectorLib functions, utilize the feature that
    the whole matrix occupies one contiguous area in memory: pass the
    address of the first row to the desired vector function, the size
    of the "vector" being len * ht.
    For example, initialize all elements of the matrix MA with {1.0, 0.0}
    (this is *NOT* the identity matrix)  by calling
        VCE_equ1( MA[0], len * ht );
*/

#define MCE_equ1( MA, len )                MCEequ1( MA[0], len )
                       /* this is the identity matrix */
#define MCE_outerprod( MA, X, Y, ht, len ) MCEouterprod( MA[0], X, Y, ht, len )
                       /* sizX=ht, sizY=len */
#define MCE_Row_equC( MA, ht, len, iRow, C ) \
                                        MCERow_equC( MA[0], ht, len, iRow, C )
#define MCE_Col_equC( MA, ht, len, iCol, C ) \
                                        MCECol_equC( MA[0], ht, len, iCol, C )
#define MCE_Dia_equC( MA, len, C )      MCEDia_equC( MA[0], len, C )

#define MCE_Row_equV( MA, ht, len, iRow, X ) \
                                        MCERow_equV( MA[0], ht, len, iRow, X )
#define MCE_Col_equV( MA, ht, len, iCol, X ) \
                                        MCECol_equV( MA[0], ht, len, iCol, X )
#define MCE_Dia_equV( MA, len, X )      MCEDia_equV( MA[0], len, X )

#define MCE_equM( MB, MA, ht, len )  VCE_equV( MB[0], MA[0], ((ui)(len))*(ht) )

#define MCE_UequL( MA, len ) MCEUequL( MA[0], len )
#define MCE_LequU( MA, len ) MCELequU( MA[0], len )
         /* copy lower-diagonal elements into upper-diagonal
           (or vice versa) by index-reflection, so as to
           get a symmetric matrix    */

            /* data-type conversions:  */
#define M_CEtoCF( MCF, MCE, ht, len ) V_CEtoCF( MCF[0], MCE[0], ((ui)ht)*len )
#define M_CFtoCE( MCE, MCF, ht, len ) V_CFtoCE( MCE[0], MCF[0], ((ui)ht)*len )
#define M_CEtoCD( MCD, MCE, ht, len ) V_CEtoCD( MCD[0], MCE[0], ((ui)ht)*len )
#define M_CDtoCE( MCE, MCD, ht, len ) V_CDtoCE( MCE[0], MCD[0], ((ui)ht)*len )

/********  Extracting a submatrix and copying a submatrix back  *********/

#define MCE_submatrix( MSub, subHt, subLen, \
                       MSrce, srceHt, srceLen, \
                       firstRowInCol, sampInCol, firstColInRow, sampInRow ) \
               MCEsubmatrix(  MSub[0], subHt, subLen, \
                              MSrce[0], srceHt, srceLen, \
                              firstRowInCol, sampInCol, firstColInRow, sampInRow )

#define MCE_submatrix_equM( MDest, destHt, destLen, \
                            firstRowInCol, sampInCol, firstColInRow, sampInRow, \
                            MSrce, srceHt, srceLen ) \
               MCEsubmatrix_equM(  MDest[0], destHt, destLen, \
                             firstRowInCol, sampInCol, firstColInRow, sampInRow, \
                             MSrce[0], srceHt, srceLen )

/*****   Extracting a single row or a single column or the diagonal  ******
 *       and storing it into a vector                                     */

#define MCE_Row_extract( Y, MA, ht, len, iRow ) \
                                     MCERow_extract( Y, MA[0], ht, len, iRow )
#define MCE_Col_extract( Y, MA, ht, len, iCol ) \
                                     MCECol_extract( Y, MA[0], ht, len, iCol )
#define MCE_Dia_extract( Y, MA, len ) MCEDia_extract( Y, MA[0], len )


/*****************    Basic arithmetic operations *********************
                      performed on one single row,
                      or one single column of any matrix,
                      or on the diagonal of a square matrix

    Note: In contrast to the analogous VectorLib functions, the operations
    are performed in-place, i.e. the input matrix itself is changed  */

#define MCE_Row_addC( MA, ht, len, iRow, C ) \
                                     MCERow_addC( MA[0], ht, len, iRow, C )
#define MCE_Col_addC( MA, ht, len, iCol, C ) \
                                     MCECol_addC( MA[0], ht, len, iCol, C )
#define MCE_Dia_addC( MA, len, C )   MCEDia_addC( MA[0], len, C )

#define MCE_Row_addV( MA, ht, len, iRow, X ) \
                                     MCERow_addV( MA[0], ht, len, iRow, X )
#define MCE_Col_addV( MA, ht, len, iCol, X ) \
                                     MCECol_addV( MA[0], ht, len, iCol, X )
#define MCE_Dia_addV( MA, len, X )   MCEDia_addV( MA[0], len, X )

#define MCE_Row_subC( MA, ht, len, iRow, C ) \
                                     MCERow_addC( MA[0], ht, len, iRow, (-C) )
#define MCE_Col_subC( MA, ht, len, iCol, C ) \
                                     MCECol_addC( MA[0], ht, len, iCol, (-C) )
#define MCE_Dia_subC( MA, len, C )   MCEDia_addC( MA[0], len, (-C) )

#define MCE_Row_subV( MA, ht, len, iRow, X ) \
                                     MCERow_subV( MA[0], ht, len, iRow, X )
#define MCE_Col_subV( MA, ht, len, iCol, X ) \
                                     MCECol_subV( MA[0], ht, len, iCol, X )
#define MCE_Dia_subV( MA, len, X )   MCEDia_subV( MA[0], len, X )

#define MCE_Row_subrC( MA, ht, len, iRow, C ) \
                                     MCERow_subrC( MA[0], ht, len, iRow, C )
#define MCE_Col_subrC( MA, ht, len, iCol, C ) \
                                     MCECol_subrC( MA[0], ht, len, iCol, C )
#define MCE_Dia_subrC( MA, len, C )  MCEDia_subrC( MA[0], len, C )

#define MCE_Row_subrV( MA, ht, len, iRow, X ) \
                                     MCERow_subrV( MA[0], ht, len, iRow, X )
#define MCE_Col_subrV( MA, ht, len, iCol, X ) \
                                     MCECol_subrV( MA[0], ht, len, iCol, X )
#define MCE_Dia_subrV( MA, len, X )  MCEDia_subrV( MA[0], len, X )

#define MCE_Row_mulC( MA, ht, len, iRow, C ) \
                                     MCERow_mulC( MA[0], ht, len, iRow, C )
#define MCE_Col_mulC( MA, ht, len, iCol, C ) \
                                     MCECol_mulC( MA[0], ht, len, iCol, C )
#define MCE_Dia_mulC( MA, len, C )   MCEDia_mulC( MA[0], len, C )

#define MCE_Row_mulV( MA, ht, len, iRow, X ) \
                                     MCERow_mulV( MA[0], ht, len, iRow, X )
#define MCE_Col_mulV( MA, ht, len, iCol, X ) \
                                     MCECol_mulV( MA[0], ht, len, iCol, X )
#define MCE_Dia_mulV( MA, len, X )   MCEDia_mulV( MA[0], len, X )

#define MCE_Row_divC( MA, ht, len, iRow, C ) \
                                     MCERow_divC( MA[0], ht, len, iRow, C )
#define MCE_Col_divC( MA, ht, len, iCol, C ) \
                                     MCECol_divC( MA[0], ht, len, iCol, C )
#define MCE_Dia_divC( MA, len, C )   MCEDia_divC( MA[0], len, C )

#define MCE_Row_divV( MA, ht, len, iRow, X ) \
                                      MCERow_divV( MA[0], ht, len, iRow, X )
#define MCE_Col_divV( MA, ht, len, iCol, X ) \
                                      MCECol_divV( MA[0], ht, len, iCol, X )
#define MCE_Dia_divV( MA, len, X )    MCEDia_divV( MA[0], len, X )

#define MCE_Row_divrC( MA, ht, len, iRow, C ) \
                                      MCERow_divrC( MA[0], ht, len, iRow, C )
#define MCE_Col_divrC( MA, ht, len, iCol, C ) \
                                      MCECol_divrC( MA[0], ht, len, iCol, C )
#define MCE_Dia_divrC( MA, len, C )   MCEDia_divrC( MA[0], len, C )

#define MCE_Row_divrV( MA, ht, len, iRow, X ) \
                                      MCERow_divrV( MA[0], ht, len, iRow, X )
#define MCE_Col_divrV( MA, ht, len, iCol, X ) \
                                      MCECol_divrV( MA[0], ht, len, iCol, X )
#define MCE_Dia_divrV( MA, len, X )   MCEDia_divrV( MA[0], len, X )


/******  One-dimensional vector operations **********************
         performed along all rows or all columns simultaneously,
         or along the diagonal of a square matrix                */

#define MCE_Rows_absmax( Y, MA, ht, len )  MCERows_absmax( Y, MA[0], ht, len )
#define MCE_Cols_absmax( Y, MA, ht, len )  MCECols_absmax( Y, MA[0], ht, len )
#define MCE_Dia_absmax( MA, len )          MCEDia_absmax( MA[0], len )
#define MCE_Rows_absmin( Y, MA, ht, len )  MCERows_absmin( Y, MA[0], ht, len )
#define MCE_Cols_absmin( Y, MA, ht, len )  MCECols_absmin( Y, MA[0], ht, len )
#define MCE_Dia_absmin( MA, len )          MCEDia_absmin( MA[0], len )

#define MCE_Rows_maxReIm( Y, MA, ht, len ) MCERows_maxReIm( Y, MA[0], ht, len )
#define MCE_Cols_maxReIm( Y, MA, ht, len ) MCECols_maxReIm( Y, MA[0], ht, len )
#define MCE_Dia_maxReIm( MA, len )         MCEDia_maxReIm( MA[0], len )
#define MCE_Rows_minReIm( Y, MA, ht, len ) MCERows_minReIm( Y, MA[0], ht, len )
#define MCE_Cols_minReIm( Y, MA, ht, len ) MCECols_minReIm( Y, MA[0], ht, len )
#define MCE_Dia_minReIm( MA, len )         MCEDia_minReIm( MA[0], len )

#define MCE_Rows_absmaxReIm( Y, MA, ht, len ) MCERows_absmaxReIm( Y, MA[0], ht, len )
#define MCE_Cols_absmaxReIm( Y, MA, ht, len ) MCECols_absmaxReIm( Y, MA[0], ht, len )
#define MCE_Dia_absmaxReIm( MA, len )         MCEDia_absmaxReIm( MA[0], len )
#define MCE_Rows_absminReIm( Y, MA, ht, len ) MCERows_absminReIm( Y, MA[0], ht, len )
#define MCE_Cols_absminReIm( Y, MA, ht, len ) MCECols_absminReIm( Y, MA[0], ht, len )
#define MCE_Dia_absminReIm( MA, len )         MCEDia_absminReIm( MA[0], len )

#define MCE_Rows_sum( Y, MA, ht, len )     MCERows_sum( Y, MA[0], ht, len )
#define MCE_Cols_sum( Y, MA, ht, len )     MCECols_sum( Y, MA[0], ht, len )
#define MCE_Dia_sum(  MA, len )            MCEDia_sum( MA[0], len )
#define MCE_Rows_prod( Y, MA, ht, len )    MCERows_prod( Y, MA[0], ht, len )
#define MCE_Cols_prod( Y, MA, ht, len )    MCECols_prod( Y, MA[0], ht, len )
#define MCE_Dia_prod(  MA, len )           MCEDia_prod( MA[0], len )

#define MCE_Rows_runsum( MA, ht, len )     MCERows_runsum( MA[0], ht, len )
#define MCE_Cols_runsum( MA, ht, len )     MCECols_runsum( MA[0], ht, len )
#define MCE_Rows_runprod( MA, ht, len )    MCERows_runprod( MA[0], ht, len )
#define MCE_Cols_runprod( MA, ht, len )    MCECols_runprod( MA[0], ht, len )

#define MCE_Rows_rotate( MA, ht, len, pos) MCERows_rotate( MA[0], ht, len, pos )
#define MCE_Cols_rotate( MA, ht, len, pos) MCECols_rotate( MA[0], ht, len, pos )

/********  Operations involving two rows or two colums of one matrix  *****/

#define MCE_Rows_exchange( MA, ht, len, row1, row2 ) \
                             MCERows_exchange( MA[0], ht, len, row1, row2 )
#define MCE_Cols_exchange( MA, ht, len, col1, col2 ) \
                             MCECols_exchange( MA[0], ht, len, col1, col2 )

#define MCE_Rows_add( MA, ht, len, destRow, srceRow ) \
                             MCERows_add( MA[0], ht, len, destRow, srceRow )
#define MCE_Cols_add( MA, ht, len, destCol, srceCol ) \
                             MCECols_add( MA[0], ht, len, destCol, srceCol )
                         /* dest[i] += source[i]  */

#define MCE_Rows_sub( MA, ht, len, destRow, srceRow ) \
                             MCERows_sub( MA[0], ht, len, destRow, srceRow )
#define MCE_Cols_sub( MA, ht, len, destCol, srceCol ) \
                             MCECols_sub( MA[0], ht, len, destCol, srceCol )
                         /* dest[i] -= source[i]  */

#define MCE_Rows_Cadd( MA, ht, len, destRow, srceRow, C ) \
                          MCERows_Cadd( MA[0], ht, len, destRow, srceRow, C )
#define MCE_Cols_Cadd( MA, ht, len, destCol, srceCol, C ) \
                          MCECols_Cadd( MA[0], ht, len, destCol, srceCol, C )
                         /* dest[i] += C * source[i]  */

#define MCE_Rows_lincomb( MA, ht, len, destRow, destC, srceRow, srceC ) \
            MCERows_lincomb( MA[0], ht, len, destRow, destC, srceRow, srceC )
#define MCE_Cols_lincomb( MA, ht, len, destCol, destC, srceCol, srceC ) \
            MCECols_lincomb( MA[0], ht, len, destCol, destC, srceCol, srceC )
  /*  linear combination: dest[i] = destC * dest[i] + sourceC * source[i] */


/*************************  Transposing a matrix **********************/

#define MCE_transpose( MTr, MA, htTr, lenTr ) \
             MCEtranspose( MTr[0], MA[0], htTr, lenTr )
       /*  the dimensions htTr, lenTr are those of the transposed matrix,
           not of the original!                */


/************************ Matrix Multiplication *************************/

#define MCE_mulV( Y, MA, X, htA, lenA ) \
                    MCEmulV( Y, MA[0], X, htA, lenA )
                          /*  Y = MA * X.  sizX=lenA, sizY=htA
                              both X and Y are column-vectors    */
#define VCE_mulM( Y, X, MA, sizX, lenA ) \
                    VCEmulM( Y, X, MA[0], 1, sizX, lenA )
                         /*  Y = X * MA.  htA=sizX, sizY=lenA
                             both X and Y are row-vectors.
                             Mind the prefix: VCE_ (not MCE_)    */
#define MCE_mulM( MC, MA, MB, htA, lenA, lenB ) \
                    MCEmulM( MC[0], MA[0], MB[0], htA, lenA, lenB )
                         /*  MC = MA * MB.  htB=lenA, htC=htA, lenC=lenB */

/***************** Two-Dimensional Fourier-Transform Methods ************/

#define MCEl_FFT( MY, MX, ht, len, dir )     MCElFFT( MY[0], MX[0], ht, len, dir )
#define MCEl_filter( MY, MX, MFlt, ht, len ) MCElfilter( MY[0], MX[0], MFlt[0], ht, len )

#define MCEs_FFT( MY, MX, ht, len, dir )     MCEsFFT( MY[0], MX[0], ht, len, dir )
#define MCEs_filter( MY, MX, MFlt, ht, len ) MCEsfilter( MY[0], MX[0], MFlt[0], ht, len )

#if defined( __LARGE__ ) || defined( __COMPACT__ ) || defined ( __HUGE__ )
   #define MCE_FFT         MCEl_FFT
   #define MCE_filter      MCEl_filter
#else
   #define MCE_FFT         MCEs_FFT
   #define MCE_filter      MCEs_filter
#endif


/**************************  Input and Output  **************************/

#define MCE_fprint( stream, MA, ht, len, linewidth ) \
                    MCEfprint( stream, MA[0], ht, len, linewidth )
#define MCE_print( MA, ht, len )  MCEfprint( stdout, MA[0], ht, len, 80 )
                    /*  MCE_print is usable only for DOS and EasyWin!  */
#ifndef _Windows
    #define MCE_cprint( MA, ht, len ) MCEcprint( MA[0], ht, len )
#endif

#define MCE_write( str, MA, ht, len )   MCEwrite( str, MA[0], ht, len )
#define MCE_read( MA, ht, len, str )    MCEread( MA[0], ht, len, str )
#define MCE_setWriteFormat              VCE_setWriteFormat
#define MCE_setWriteSeparate            VCE_setNWriteSeparate
                                   /* write and read in ascii format */
#define MCE_store( str, MA, ht, len ) \
                           VCE_store( str, MA[0], ((ui)(len))*(ht) );
#define MCE_recall( MA, ht, len, str) \
                           VCE_recall( MA[0], ((ui)(len))*(ht), str);
                                  /* store and recall in binary format */


/*************************************************************************
 * Here are now the actual declarations of the functions used internally.*
 * These declarations may change in future versions of MatrixLib!        *
 * You should not care too much about them, except in the case you need  *
 * the actual address of a run-time function (see above). Under all      *
 * "normal" circumstances, use only the names defined above in the       *
 * macro section!                                                        *
 *************************************************************************/

eComplex _VFAR * MCEPelement( cePMatrix X, unsigned ht, unsigned len,
                         unsigned m, unsigned n );
                  /* pointer is normalized in memory model HUGE */

void   __vf  MCEequ1( cePMatrix MA, unsigned len );  /* identity matrix */
void   __vf  MCEouterprod( cePMatrix MA, ceVector X,  ceVector Y,
                           unsigned ht, unsigned len );

void   __vf  MCERow_equC( cePMatrix MA, unsigned ht, unsigned len,
                         unsigned iRow, eComplex C );
void   __vf  MCECol_equC( cePMatrix MA, unsigned ht, unsigned len,
                         unsigned iCol, eComplex C );
void   __vf  MCEDia_equC( cePMatrix MA, unsigned len, eComplex C );

void   __vf  MCERow_equV( cePMatrix MA, unsigned ht, unsigned len,
                         unsigned iRow, ceVector X );
void   __vf  MCECol_equV( cePMatrix MA, unsigned ht, unsigned len,
                         unsigned iCol, ceVector X );
void   __vf  MCEDia_equV( cePMatrix MA, unsigned len, ceVector X );

void   __vf  MCEUequL( cePMatrix MA, unsigned len );
void   __vf  MCELequU( cePMatrix MA, unsigned len );

/********  Extracting a submatrix and copying a submatrix back  *********/

void  __vf  MCEsubmatrix( cePMatrix MSub,
                          unsigned subHt,  unsigned subLen,
                          cePMatrix MSrce,
                          unsigned srceHt,  unsigned srceLen,
                          unsigned firstRowInCol,  unsigned sampInCol,
                          unsigned firstColInRow,  unsigned sampInRow );

void  __vf  MCEsubmatrix_equM( cePMatrix MDest,
                               unsigned destHt,     unsigned destLen,
                               unsigned firstRowInCol,  unsigned sampInCol,
                               unsigned firstColInRow,  unsigned sampInRow,
                               cePMatrix MSrce,
                               unsigned srceHt,     unsigned srceLen );


/*****   Extracting a single row or a single column or the diagonal  *****
 *       and storing it into a vector                                    */

void __vf MCERow_extract( ceVector Y, cePMatrix MA, unsigned ht, unsigned len,
                           unsigned iRow );
void __vf MCECol_extract( ceVector Y, cePMatrix MA, unsigned ht, unsigned len,
                           unsigned iCol );
void __vf MCEDia_extract( ceVector Y, cePMatrix MA, unsigned len );


/*****************    Basic arithmetic operations ***********************
     performed on one single row,  or one single column of any matrix,
     or on the diagonal of a square matrix                              */

void   __vf  MCERow_addC( cePMatrix MA, unsigned ht, unsigned len,
                         unsigned iRow, eComplex C );
void   __vf  MCECol_addC( cePMatrix MA, unsigned ht, unsigned len,
                         unsigned iCol, eComplex C );
void   __vf  MCEDia_addC( cePMatrix MA, unsigned len, eComplex C );

void   __vf  MCERow_subC( cePMatrix MA, unsigned ht, unsigned len,
                         unsigned iRow, eComplex C );
void   __vf  MCECol_subC( cePMatrix MA, unsigned ht, unsigned len,
                         unsigned iCol, eComplex C );
void   __vf  MCEDia_subC( cePMatrix MA, unsigned len, eComplex C );

void   __vf  MCERow_addV( cePMatrix MA, unsigned ht, unsigned len,
                         unsigned iRow, ceVector X );
void   __vf  MCECol_addV( cePMatrix MA, unsigned ht, unsigned len,
                         unsigned iCol, ceVector X );
void   __vf  MCEDia_addV( cePMatrix MA, unsigned len, ceVector X );

void   __vf  MCERow_subV( cePMatrix MA, unsigned ht, unsigned len,
                         unsigned iRow, ceVector X );
void   __vf  MCECol_subV( cePMatrix MA, unsigned ht, unsigned len,
                         unsigned iCol, ceVector X );
void   __vf  MCEDia_subV( cePMatrix MA, unsigned len, ceVector X );

void   __vf  MCERow_subrC( cePMatrix MA, unsigned ht, unsigned len,
                         unsigned iRow, eComplex C );
void   __vf  MCECol_subrC( cePMatrix MA, unsigned ht, unsigned len,
                         unsigned iCol, eComplex C );
void   __vf  MCEDia_subrC( cePMatrix MA, unsigned len, eComplex C );

void   __vf  MCERow_subrV( cePMatrix MA, unsigned ht, unsigned len,
                         unsigned iRow, ceVector X );
void   __vf  MCECol_subrV( cePMatrix MA, unsigned ht, unsigned len,
                         unsigned iCol, ceVector X );
void   __vf  MCEDia_subrV( cePMatrix MA, unsigned len, ceVector X );

void   __vf  MCERow_mulC( cePMatrix MA, unsigned ht, unsigned len,
                         unsigned iRow, eComplex C );
void   __vf  MCECol_mulC( cePMatrix MA, unsigned ht, unsigned len,
                         unsigned iCol, eComplex C );
void   __vf  MCEDia_mulC( cePMatrix MA, unsigned len, eComplex C );

void   __vf  MCERow_mulV( cePMatrix MA, unsigned ht, unsigned len,
                         unsigned iRow, ceVector X );
void   __vf  MCECol_mulV( cePMatrix MA, unsigned ht, unsigned len,
                         unsigned iCol, ceVector X );
void   __vf  MCEDia_mulV( cePMatrix MA, unsigned len, ceVector X );

void   __vf  MCERow_divC( cePMatrix MA, unsigned ht, unsigned len,
                         unsigned iRow, eComplex C );
void   __vf  MCECol_divC( cePMatrix MA, unsigned ht, unsigned len,
                         unsigned iCol, eComplex C );
void   __vf  MCEDia_divC( cePMatrix MA, unsigned len, eComplex C );

void   __vf  MCERow_divV( cePMatrix MA, unsigned ht, unsigned len,
                         unsigned iRow, ceVector X );
void   __vf  MCECol_divV( cePMatrix MA, unsigned ht, unsigned len,
                         unsigned iCol, ceVector X );
void   __vf  MCEDia_divV( cePMatrix MA, unsigned len, ceVector X );

void   __vf  MCERow_divrC( cePMatrix MA, unsigned ht, unsigned len,
                         unsigned iRow, eComplex C );
void   __vf  MCECol_divrC( cePMatrix MA, unsigned ht, unsigned len,
                         unsigned iCol, eComplex C );
void   __vf  MCEDia_divrC( cePMatrix MA, unsigned len, eComplex C );

void   __vf  MCERow_divrV( cePMatrix MA, unsigned ht, unsigned len,
                         unsigned iRow, ceVector X );
void   __vf  MCECol_divrV( cePMatrix MA, unsigned ht, unsigned len,
                         unsigned iCol, ceVector X );
void   __vf  MCEDia_divrV( cePMatrix MA, unsigned len, ceVector X );


/******  One-dimensional vector operations **********************
         performed along all rows or all columns simultaneously,
         or along the diagonal of a square matrix                */

void     __vf  MCERows_absmax( eVector Y, cePMatrix MA, unsigned ht, unsigned len );
void     __vf  MCECols_absmax( eVector Y, cePMatrix MA, unsigned ht, unsigned len );
extended __vf  MCEDia_absmax(  cePMatrix MA, unsigned len );
void     __vf  MCERows_absmin( eVector Y, cePMatrix MA, unsigned ht, unsigned len );
void     __vf  MCECols_absmin( eVector Y, cePMatrix MA, unsigned ht, unsigned len );
extended __vf  MCEDia_absmin(  cePMatrix MA, unsigned len );

void     __vf  MCERows_maxReIm( ceVector Y, cePMatrix MA, unsigned ht, unsigned len );
void     __vf  MCECols_maxReIm( ceVector Y, cePMatrix MA, unsigned ht, unsigned len );
eComplex __vf  MCEDia_maxReIm(  cePMatrix MA, unsigned len );
void     __vf  MCERows_minReIm( ceVector Y, cePMatrix MA, unsigned ht, unsigned len );
void     __vf  MCECols_minReIm( ceVector Y, cePMatrix MA, unsigned ht, unsigned len );
eComplex __vf  MCEDia_minReIm(  cePMatrix MA, unsigned len );

void     __vf  MCERows_absmaxReIm( ceVector Y, cePMatrix MA, unsigned ht, unsigned len );
void     __vf  MCECols_absmaxReIm( ceVector Y, cePMatrix MA, unsigned ht, unsigned len );
eComplex __vf  MCEDia_absmaxReIm(  cePMatrix MA, unsigned len );
void     __vf  MCERows_absminReIm( ceVector Y, cePMatrix MA, unsigned ht, unsigned len );
void     __vf  MCECols_absminReIm( ceVector Y, cePMatrix MA, unsigned ht, unsigned len );
eComplex __vf  MCEDia_absminReIm(  cePMatrix MA, unsigned len );

void     __vf  MCERows_sum( ceVector Y, cePMatrix MA, unsigned ht, unsigned len );
void     __vf  MCECols_sum( ceVector Y, cePMatrix MA, unsigned ht, unsigned len );
eComplex __vf  MCEDia_sum(  cePMatrix MA, unsigned len );
void     __vf  MCERows_prod(ceVector Y, cePMatrix MA, unsigned ht, unsigned len );
void     __vf  MCECols_prod(ceVector Y, cePMatrix MA, unsigned ht, unsigned len );
eComplex __vf  MCEDia_prod( cePMatrix MA, unsigned len );


void  __vf  MCERows_runsum( cePMatrix MA, unsigned ht, unsigned len );
void  __vf  MCECols_runsum( cePMatrix MA, unsigned ht, unsigned len );
void  __vf  MCERows_runprod( cePMatrix MA, unsigned ht, unsigned len );
void  __vf  MCECols_runprod( cePMatrix MA, unsigned ht, unsigned len );

void  __vf  MCERows_rotate( cePMatrix MA, unsigned ht, unsigned len, int pos );
void  __vf  MCECols_rotate( cePMatrix MA, unsigned ht, unsigned len, int pos );

/********  Operations involving two rows or two colums of one matrix  *****/

void   __vf  MCERows_exchange( cePMatrix MA, unsigned ht, unsigned len,
                              unsigned i1, unsigned i2 );
void   __vf  MCECols_exchange( cePMatrix MA, unsigned ht, unsigned len,
                              unsigned i1, unsigned i2 );

void   __vf  MCERows_add( cePMatrix MA, unsigned ht, unsigned len,
                          unsigned destRow, unsigned sourceRow );
void   __vf  MCECols_add( cePMatrix MA, unsigned ht, unsigned len,
                          unsigned destCol, unsigned sourceCol );

void   __vf  MCERows_sub( cePMatrix MA, unsigned ht, unsigned len,
                          unsigned destRow, unsigned sourceRow );
void   __vf  MCECols_sub( cePMatrix MA, unsigned ht, unsigned len,
                          unsigned destCol, unsigned sourceCol );

void   __vf  MCERows_Cadd( cePMatrix MA, unsigned ht, unsigned len,
                           unsigned destRow, unsigned sourceRow, eComplex C );
void   __vf  MCECols_Cadd( cePMatrix MA, unsigned ht, unsigned len,
                           unsigned destCol, unsigned sourceCol, eComplex C );

void   __vf  MCERows_lincomb( cePMatrix MA, unsigned ht, unsigned len,
                              unsigned destRow,  eComplex  destC,
                              unsigned srceRow,  eComplex  srceC );
void   __vf  MCECols_lincomb( cePMatrix MA, unsigned ht, unsigned len,
                              unsigned destCol,  eComplex  destC,
                              unsigned srceCol,  eComplex  srceC );


/*************************  Transposing a matrix **********************/

void  __vf  MCEtranspose( cePMatrix MTr, cePMatrix MA,
                          unsigned htTr, unsigned lenTr );


/************************ Matrix Multiplication *************************/

void  __vf  MCEmulV( ceVector Y, cePMatrix MA, ceVector X,
                     unsigned htA, unsigned lenA );
void  __vf  VCEmulM( ceVector Y, ceVector X, cePMatrix MA,
                     unsigned sizX, unsigned lenA );
void  __vf  MCEmulM( cePMatrix MC, cePMatrix MA, cePMatrix MB,
                     unsigned htA, unsigned lenA, unsigned lenB );

/*************  Two-Dimensional Fourier-Transform Methods *****************/

void  __vf   MCElFFT( cePMatrix MY, cePMatrix MX,
                      unsigned ht, unsigned len, int dir );
void  __vf   MCElfilter( cePMatrix MY, cePMatrix MX, cePMatrix MFlt,
                         unsigned ht, unsigned len );
void  __vf   MCEsFFT( cePMatrix MY, cePMatrix MX,
                      unsigned ht, unsigned len, int dir );
void  __vf   MCEsfilter( cePMatrix MY, cePMatrix MX, cePMatrix MFlt,
                         unsigned ht, unsigned len );

      /*************  Input and Output  ****************/

void __vf MCEfprint( FILE _VFAR *stream, cePMatrix MA, unsigned ht,
                    unsigned len, unsigned linewidth );
#ifndef _Windows
    void __vf MCEcprint( cePMatrix MA, unsigned ht, unsigned len );
#endif
void  __vf    MCEwrite( FILE _VFAR *stream, cePMatrix X, unsigned ht, unsigned len  );
void  __vf    MCEread( cePMatrix X, unsigned ht, unsigned len, FILE _VFAR *stream );

#ifdef __cplusplus
}
#endif