(*  JAM Application Programming Interface for Virtual Pascal/2, Version 1.2  *)
(*    Copyright (C) 1996-1997 Dusk To Dawn Computing. All Rights Reserved.   *)
(*                    Information Structures and Functions                   *)
(* -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=- *)
(* History:                                                                  *)
(*                                                                           *)
(* 07 Jun 1996 0.01 Initial coding                                          *)
(* 12 Mar 1997 0.02 Conversion to use 0.91 object                           *)
(* 14 Jun 1997 1.00  Conversion to use 1.10 object                           *)
(* 16 Jun 1997 1.01  Conversion to use 1.22 object                           *)

unit JamInfo;

(* -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=- *)
                                   INTERFACE
(* -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=- *)

uses
   JAM,
   JAMMB;

const
   DateTimeFormat: String = 'ddd dd mmm yyyy hh:mm:ss AMPM';
   AttrNames:   array[1..32] of String[16] =
   ('Local',
   'InTransit',
   'Private',
   'Received',
   'Sent',
   'Kill',
   'Archive',
   'Hold',
   'Crash',
   'Immediate',
   'Dir',
   'Gate',
   'FileReq',
   'FileAtt',
   'Trunc/Sent',
   'Kill/Sent',
   'RRQ',
   'Confirm',
   'Orphan',
   'Encrypted',
   'Compressed',
   'Escaped',
   'ForcePickup',
   'LocalOnly',
   'EchoMail',
   'NetMail',
   'N/A_1',
   'N/A_2',
   'N/A_3',
   'NoDisplay',
   'Locked',
   'Deleted');

function AttrToStr(Attr: LongInt): String;

{ -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=- }

function SubFieldDesc(LoID: SmallWord): String;

{ -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=- }

function SubFieldStr(var Field: tJamSubField): String;

{ -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=- }

procedure DisplayHdrInfo(var APIObj: tJamBase);

{ -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=- }

procedure DisplayMsgHdr(var APIObj: tJamBase; MsgNo: LongInt);

{ -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=- }

procedure DisplayMsgSubFld(var APIObj: tJamBase);

{ -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=- }

procedure DisplayMsgTxt(var APIObj: tJamBase);

(* -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=- *)
                                 IMPLEMENTATION
(* -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=- *)

uses
   SysUtils,
   UnixTime;

{ -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=- }

function AttrToStr;

var
   TempNum: Byte;

begin
   Result := '';
   for TempNum := 1 to 32 do
   begin
      if (Attr and $01) <> 0 then
      begin
         if Length(Result) > 0 then
            Result := Result + ', ';
         Result := Result + AttrNames[TempNum];
      end;
      Attr := Attr shr 1;
   end;
end;

{ -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=- }

procedure DisplayHdrInfo;

var
   TotalMsgs:  LongInt;

begin
   WriteLn;
   WriteLn('Header Info:', #13#10);
   with APIObj do
   begin
      try
         GetHdrInfo;
      except
         on Exception do
            raise;
      end;
      with HdrInfo do
      begin
         WriteLn('Signature    : "', Signature, '"');
         WriteLn(Format('DateCreated  : %s ', [FormatUnixTime(DateTimeFormat, DateCreated)]));
         WriteLn(Format('ModCounter   : %d', [ModCounter]));
         WriteLn(Format('ActiveMsgs   : %d', [ActiveMsgs]));
         WriteLn(Format('Password CRC : %8.8x', [PasswordCRC]));
         WriteLn(Format('Base MsgNum  : %d', [BaseMsgNum]));
         WriteLn;
         WriteLn(Format('TotalMsgs    : %d', [TotalMsgs]));
         WriteLn(Format('DeletedMsgs  : %d', [TotalMsgs - ActiveMsgs]));
      end;
   end;
end;

{ -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=- }

procedure DisplayMsgHdr;

begin
   with APIObj do
   begin
      WriteLn;
      if (MsgNo - HdrInfo.BaseMsgNum) >= TotalMsgs then
      begin
         WriteLn('No more messages.');
         Exit;
      end;
      if MsgNo < HdrInfo.BaseMsgNum then
      begin
         WriteLn(Format('Invalid message number %d.', [MsgNo]));
         Exit;
      end;
      WriteLn(Format('Message #    : %d', [MsgNo]));
      try
         GetMsgHdr(MsgNo, False);
      except
         raise;
      end;
      with Hdr do
      begin
         WriteLn('Signature    : "', Signature, '"');
         WriteLn(Format('Revision     : %d', [Revision]));
         WriteLn(Format('SubFieldLen  : %d', [SubFieldLen]));
         WriteLn(Format('Times Read   : %d', [TimesRead]));
         WriteLn(Format('MsgID CRC32  : %x', [MsgIdCRC]));
         WriteLn(Format('Reply CRC32  : %x', [ReplyCRC]));
         WriteLn(Format('ReplyTo      : %d', [ReplyTo]));
         WriteLn(Format('Reply1st     : %d', [Reply1st]));
         WriteLn(Format('ReplyNext    : %d', [ReplyNext]));
         WriteLn(Format('DateWritten  : %s', [FormatUnixTime(DateTimeFormat, DateWritten)]));
         WriteLn(Format('DateReceived : %s', [FormatUnixTime(DateTimeFormat, DateReceived)]));
         WriteLn(Format('DateProcessed: %s', [FormatUnixTime(DateTimeFormat, DateProcessed)]));
         WriteLn(Format('MsgNum       : %d', [MsgNum]));
         WriteLn(Format('Attributes   : %s', [AttrToStr(Attribute)]));
         WriteLn(Format('Attributes 2 : %d', [Attribute2]));
         WriteLn(Format('Text Offset  : %d', [TxtOffset]));
         WriteLn(Format('Text Length  : %d', [TxtLen]));
         WriteLn(Format('Password CRC : %x', [PasswordCRC]));
         WriteLn(Format('Cost         : %d', [Cost]));
      end;
   end;
end;

{ -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=- }

procedure DisplayMsgTxt(var APIObj: tJamBase);

const
   First:   Boolean = TRUE;            { First read flag                   }

var
   TempPtr: ^Char;
   Column:  LongInt;
   TempVal: LongInt;

   function NextSpace(var APIObj: tJamBase; Position: LongInt): LongInt;

   var
      TmpPtr: ^Char;
      TmpVal: LongInt;

   begin
      TmpVal := 0;
      TmpPtr := Pointer(LongInt(APIObj.WorkBuf) + (Position + 1));
      while ((not ((TmpPtr^ = #32) or (TmpPtr^ = #13))) and (TmpVal < APIObj.WorkLen - Position)) do
      begin
         Inc(LongInt(TmpPtr));
         Inc(TmpVal);
      end;
      NextSpace := TmpVal;
   end;

begin
   First := TRUE;
   Column := 0;
   TempVal := 0;
   with APIObj do
   begin
      WriteLn;
      WriteLn(Format('Message #    : %d', [Hdr.MsgNum]));
      while True do
      begin
         try
            GetMsgTxt(First);
         except
            on ENoMoreText do
               Break;
            on E: Exception do
            begin
               WriteLn(E.Message);
               Exit;
            end;
         end;
         First := FALSE;
         LongInt(TempPtr) := LongInt(WorkBuf);
         while TempVal < WorkPos do
         begin
            case TempPtr^ of
            #13:
               begin
                  WriteLn;
                  Column := 0;
               end;
            #00..#12,#14..#31:
               begin
                  Write(Format('<%2d>', [TempPtr^]));
                  Inc(Column, 4);
               end;
            #32:
               begin
                  if (Column + NextSpace(APIObj, TempVal)) > 79 then { if Column >= 79 then }
                  begin
                     WriteLn;
                     Column := 0;
                  end
                  else
                  begin
                     Write(TempPtr^);
                     Inc(Column);
                  end;
               end;
            else
               begin
                  Write(TempPtr^);
                  Inc(Column);
               end;
            end;
            Inc(LongInt(TempPtr));
            Inc(TempVal);
         end;
      end;
   end;
end;

{ -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=- }

procedure DisplayMsgSubFld;

var
   Len:        LongInt;
   SubFldLen:  LongInt;
   pSubField:  pJamSubField;

begin
   with APIObj do
   begin
      SubFldLen := Hdr.SubFieldLen;
      WriteLn;
      WriteLn(Format('Message #%d', [Hdr.MsgNum]));
      WriteLn;
      try
         GetMsgHdr(Hdr.MsgNum, True);
      except
         on E: Exception do WriteLn(E.Message);
      end;
      if SubFldLen > WorkLen then
         SubFldLen := WorkLen;
      WriteLn('HiID LoID Name             Len Data');
      WriteLn('~~~~ ~~~~ ~~~~             ~~~ ~~~~');
      pSubField := pJamSubField(WorkBuf);
      while (pSubField^.DatLen + SizeOf(tJamBinSubfield) <= SubFldLen) do
      begin
         WriteLn(Format('%4.4d %4.4d %-16s %3.3d "%s"',
                       [pSubField^.HiID,
                        pSubField^.LoID,
                        SubFieldDesc(pSubField^.LoID),
                        pSubField^.DatLen,
                        SubFieldStr(pSubField^)]));
         Len := (pSubField^.DatLen + SizeOf(tJamBinSubfield));
         SubFldLen := SubFldLen - Len;
         Inc(LongInt(pSubField), Len);
      end;
   end;
end;

{ -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=- }

function SubFieldDesc(LoID: SmallWord): String;

begin
   case LoID of
      JSub_Origin:         Result := 'Origin';
      JSub_Dest:           Result := 'Destination';
      JSub_Sender:         Result := 'Sender';
      JSub_Receiver:       Result := 'Receiver';
      JSub_MsgID:          Result := 'MsgID';
      JSub_ReplyID:        Result := 'ReplyID';
      JSub_Subject:        Result := 'Subject';
      JSub_PID:            Result := 'PID';
      JSub_Trace:          Result := 'Trace';
      JSub_FileAttach:     Result := 'FileAttach';
      JSub_FileAlias:      Result := 'FileAttachAlias';
      JSub_FileReq:        Result := 'FileRequest';
      JSub_WildCardAttach: Result := 'WildCardAttach';
      JSub_IndirectAttach: Result := 'AttachIndirect';
      JSub_EMBINDAT:       Result := '"EMBINDAT"';
      JSub_FTSKludge:      Result := 'IFNA Kludge';
      JSub_SeenBy:         Result := 'Seen-By';
      JSub_Path:           Result := 'Path';
      JSub_Flags:          Result := 'Flags';
      JSub_TZInfo:         Result := 'TZ Info';
   else
      Result := 'Unknown Field';
   end;
end;

{ -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=- }

function SubFieldStr(var Field: tJamSubField): String;

var
   TempVal: LongInt;
   TempPtr: ^Char;
   BufLen:  LongInt;

begin
   TempPtr := @Field.Buffer;
   BufLen := Field.DatLen;
   if BufLen > 255 then
      BufLen := 255;
   Result[0] := Char(BufLen);
   for TempVal := 1 TO BufLen do
   begin
      Result[TempVal] := TempPtr^;
      Inc(LongInt(TempPtr));
   end;
end;

{ -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=- }

end.

