// ***********************************************************************
// CANVAS.HPP
// ***********************************************************************
//
// Written by Michael T. Duffy
//
// Canvas Graphics Object for 256 color modes.
//
// ***********************************************************************

#ifndef CANVAS_HPP
#define CANVAS_HPP

// ***********************************************************************
//  Error Processing
// ***********************************************************************

#define ERR_CNV   100


#define ERR_CNV_CREATE         (ERR_CNV + 1)



// ***********************************************************************
//  Constant Definitions
// ***********************************************************************

// These color definitions are for the first 16 colors in a standard VGA
//  palette.  Later we will move them to our palette module

#ifndef COLOR_DEFS
#define COLOR_DEFS
const BYTE         BLACK        = 0;
const BYTE         BLUE         = 1;
const BYTE         GREEN        = 2;
const BYTE         CYAN         = 3;
const BYTE         RED          = 4;
const BYTE         MAGENTA      = 5;
const BYTE         BROWN        = 6;
const BYTE         LIGHTGRAY    = 7;
const BYTE         DARKGRAY     = 8;
const BYTE         LIGHTBLUE    = 9;
const BYTE         LIGHTGREEN   = 10;
const BYTE         LIGHTCYAN    = 11;
const BYTE         LIGHTRED     = 12;
const BYTE         LIGHTMAGENTA = 13;
const BYTE         YELLOW       = 14;
const BYTE         WHITE        = 15;
#endif

// ***********************************************************************
//  Class Definitions
// ***********************************************************************

class Canvas
  {
  private:

    // Internal data
    USHORT           usLastErrorCode;

    // Canvas structure
    USHORT           usWidth;
    USHORT           usHeight;
    ULONG            ulSize;
    PBYTE            pbySurface;
    PBYTE *          apbyRowStart;
    LONG             lRowChange;

    LONG             lDwordWidth;
    BOOL             bTopDown;

    // Drawing parameters

    BYTE             byDrawColor;


  public:

    // Constructors
    Canvas
      (
      USHORT         usWidthIn    = 320,
      USHORT         usHeightIn   = 200,
      BOOL           bTopDownIn   = TRUE
      );

    // Destructor
    ~Canvas ();

    // Member Functions

    VOID SetDrawColor
      (
      BYTE             byColor
      );

    BYTE QueryDrawColor
      (
      VOID
      );

    SHORT QueryWidth
      (
      VOID
      );

    SHORT QueryHeight
      (
      VOID
      );

    PBYTE QuerySurface
      (
      VOID
      );

    PBYTE QueryRowAddress
      (
      ULONG            ulIndex
      );

    PBYTE * QueryRowAddressArray
      (
      VOID
      );

    LONG QueryRowIncrement
      (
      VOID
      );

    VOID Clear
      (
      BYTE             byColor    = 0
      );

    VOID PutPixel
      (
      SHORT            sXPos,
      SHORT            sYPos
      );

    VOID HorizLine
      (
      SHORT            sX1,
      SHORT            sX2,
      SHORT            sY
      );

    VOID VertLine
      (
      SHORT            sX,
      SHORT            sY1,
      SHORT            sY2
      );

    VOID Rectangle
      (
      SHORT            sX1,
      SHORT            sY1,
      SHORT            sX2,
      SHORT            sY2
      );

    VOID XORHorizLine
      (
      SHORT            sX1,
      SHORT            sX2,
      SHORT            sY
      );

    VOID XORVertLine
      (
      SHORT            sX,
      SHORT            sY1,
      SHORT            sY2
      );

    VOID XORRectangle
      (
      SHORT            sX1,
      SHORT            sY1,
      SHORT            sX2,
      SHORT            sY2
      );


    BOOL Validate
      (
      VOID
      );

    USHORT QueryLastErrorCode
      (
      VOID
      ) {return (usLastErrorCode);};

  // End of Canvas definition
  };

typedef Canvas *   PCANVAS;


// ***********************************************************************
//  Inline Functions
// ***********************************************************************

//........................................................................
inline VOID Canvas::SetDrawColor
//........................................................................
  (
  BYTE             byColor
  )
{
byDrawColor = byColor;
};

//........................................................................
inline BYTE Canvas::QueryDrawColor
//........................................................................
  (
  VOID
  )
{
return (byDrawColor);
};

//........................................................................
inline SHORT Canvas::QueryWidth
//........................................................................
  (
  VOID
  )
{
return (usWidth);
};

//........................................................................
inline SHORT Canvas::QueryHeight
//........................................................................
  (
  VOID
  )
{
return (usHeight);
};

//........................................................................
inline PBYTE Canvas::QuerySurface
//........................................................................
  (
  VOID
  )
{
return (pbySurface);
};

//........................................................................
inline PBYTE Canvas::QueryRowAddress
//........................................................................
  (
  ULONG            ulIndex
  )
{
return (apbyRowStart [ulIndex]);
};

//........................................................................
inline PBYTE * Canvas::QueryRowAddressArray
//........................................................................
  (
  VOID
  )
{
return (apbyRowStart);
};

//........................................................................
inline LONG Canvas::QueryRowIncrement
//........................................................................
  (
  VOID
  )
{
return (lRowChange);
};

//........................................................................
inline BOOL Canvas::Validate
//........................................................................
  (
  VOID
  )
{
// This routine simply makes sure the canvas exists with valid data
if ((pbySurface == NULL) ||
    (usWidth  == 0) ||
    (usHeight == 0) ||
    (ulSize  == 0))
  {
  return (FALSE);
  }
else
  {
  return (TRUE);
  };
};


#endif // CANVAS_HPP
