/* -----------------------------------------------------------------
                             CANVAS.CPP
     This module is for linear 256 color bitmaps under OS/2.  It is
  designed so that it can handle top-down or bottom-up bitmaps, yet
  still have all canvases approached as if they were top-down.

  Author: Michael T. Duffy
  Date: May 21st, 1995


  Copyright 1995 Michael T. Duffy. All Rights Reserved.

   ----------------------------------------------------------------- */

/* Useage Notes:

   * Canvases accessed with the following routines can be considered to be
      top down bitmaps, bitmaps with the origin in the top left corner.

*/

// ***********************************************************************
//  Compile Directives
// ***********************************************************************

#pragma pack (1);

// ***********************************************************************
//  Header Files
// ***********************************************************************

// Note: The __IBMC__ define is needed to get the Warp Toolkit to work with
//  Watcom compilers.
#define __IBMC__

#define INCL_WIN
#define INCL_GPI
#define INCL_DOS
#define INCL_ERRORS

// OS/2 API functions
#include <os2.h>

#include <stdlib.h>
#include <string.h>

#include "newtypes.hpp"
#include "canvas.hpp"

// Note: Inline assembly written for Watcom compilers.
//#define INLINE_ASSEMBLY



// ***********************************************************************
//  Code
// ***********************************************************************

//........................................................................
Canvas::Canvas
//........................................................................
  (
  USHORT           usWidthIn,
  USHORT           usHeightIn,
  BOOL             bTopDownIn
  )
{
ULONG              ulCanvasSize;
ULONG              ulOffset;
LONG               lRowSize;
ULONG              ulIndex;


usLastErrorCode = 0;

// Determine canvas size with width taken to the nearest dword.  Knowing that
//  the width is always on a dword boundary allows us to use dword copies.
usWidthIn = (USHORT) (((usWidthIn + 3) / 4) * 4);
ulCanvasSize = usWidthIn * usHeightIn;

// Allocate a buffer for the bitmap which we will call the surface
pbySurface        = (PBYTE) malloc (ulCanvasSize);

// If you have a buffer (given or allocation was successful)
if (pbySurface != NULL)
  {
  // Clear the buffer to 0
  memset (pbySurface, 0, ulCanvasSize);

  // Allocate memory for a list of row offsets
  if ((apbyRowStart = (PBYTE*)malloc (usHeightIn * sizeof (PBYTE))) !=
      NULL)
    {
    // Calculate row offsets

    lRowSize = usWidthIn;

    if (bTopDownIn)
      {
      // Top Down: Start the offset list at the beginning of the buffer
      ulOffset = 0;
      }
    else
      {
      // Bottom Up: Start the offset list at the end of the buffer
      ulOffset = (ULONG) lRowSize * (usHeightIn - 1);
      lRowSize = -lRowSize;
      };

    // loop through all the rows
    for (ulIndex = 0; ulIndex < usHeightIn; ++ulIndex)
      {
      // Calculate and store the offset into the buffer of that row.
      apbyRowStart [ulIndex] = pbySurface + ulOffset;
      ulOffset += lRowSize;
      };

    usWidth  = usWidthIn;
    usHeight = usHeightIn;
    ulSize   = ulCanvasSize;

    // The row change variable is for incrementing from one row to the next.
    //  For top down bitmaps, this value will be positive and adding it to your
    //  current position will move you forward one row int the buffer.  Bottom
    //  up bitmaps have a negative row change, so adding it will move you back
    //  in the buffer.
    lRowChange = usWidthIn;

    bTopDown = bTopDownIn;
    if (!bTopDown)
      {
      lRowChange = -lRowChange;
      };
    }
  else
    {
    // Error in allocating memory for row list.
    // Deallocate surface
    free (pbySurface);

    pbySurface = NULL;
    usWidth  = 0;
    usHeight = 0;

    usLastErrorCode = ERR_CNV_CREATE;
    return;
    };
  }
else
  {
  usLastErrorCode = ERR_CNV_CREATE;
  return;
  };

// The dword (32 bit word) value will be of use later, so we calcualte it now.
lDwordWidth    = abs (lRowChange / 4);
};

//........................................................................
Canvas::~Canvas
//........................................................................
  (
  VOID
  )
{

// Release the memory used for
free (apbyRowStart);

// Release the memory for the canvas if the canvas object allocated it.
if (pbySurface != NULL)
  {
  free(pbySurface);
  };
};

// Inline assembly for the Watcom compiler.  This is faster than a memset
//  because of less overhead and 32-bit copies.  This is possible because
//  canvases are allocated so that the row size always evenly divides into
//  dwords.
#ifdef INLINE_ASSEMBLY

VOID CnvMemSet (PBYTE pbyDest, BYTE VALUE, ULONG ulNumOfBytes);

  #pragma aux CnvMemSet = \
    "and  ebx,0x000000ff", \
    "mov  bh,bl", \
    "mov  eax,ebx", \
    "shl  ebx,16", \
    "or   eax,ebx", \
    "shr  ecx,2", \
    "rep  stosd", \
    MODIFY [eax] \
    PARM [edi][ebx][ecx] ;

#endif

//........................................................................
VOID Canvas::Clear
//........................................................................
  (
  BYTE             byColor
  )
{

// Make sure the canvas has a surface.
if (pbySurface != NULL)
  {
  #ifdef INLINE_ASSEMBLY

    CnvMemSet(pbySurface, byColor, ulSize);

  #else

    memset (pbySurface, byColor, ulSize);

  #endif
  };
};

//........................................................................
VOID Canvas::PutPixel
//........................................................................
  (
  SHORT            sXPos,
  SHORT            sYPos
  )
{
PBYTE              pbyTarget;


// bounds checking
if ((sXPos < 0) || (sYPos >= usWidth))  return;
if ((sYPos < 0) || (sYPos >= usHeight)) return;

// Calculate the pixel's location in the buffer
pbyTarget = apbyRowStart [sYPos] + sXPos;

// Write the pixel
*pbyTarget = byDrawColor;
};




//........................................................................
VOID Canvas::HorizLine
//........................................................................
  (
  SHORT            sX1,
  SHORT            sX2,
  SHORT            sY
  )
{
SHORT              sOrderedX1;
SHORT              sOrderedX2;
SHORT              sLength;
PBYTE              pbyTarget;


// Make sure X1 is to the left
if (sX1 < sX2)
  {
  sOrderedX1 = sX1;
  sOrderedX2 = sX2;
  }
else
  {
  sOrderedX1 = sX2;
  sOrderedX2 = sX1;
  };

// Perform clipping
sOrderedX1 = max (sOrderedX1, (SHORT)0);
sOrderedX2 = min (sOrderedX2, (SHORT)(usWidth - 1));
if ((sY < 0) || (sY >= usHeight)) return;

// Calculate range and starting address
sLength = (SHORT)(sOrderedX2 - sOrderedX1 + 1);
pbyTarget = apbyRowStart [sY] + sOrderedX1;

memset (pbyTarget, byDrawColor, sLength);
};

//........................................................................
VOID Canvas::VertLine
//........................................................................
  (
  SHORT            sX,
  SHORT            sY1,
  SHORT            sY2
  )
{
SHORT              sOrderedY1;
SHORT              sOrderedY2;
SHORT              sLength;
ULONG              ulIndex;
PBYTE              pbyTarget;
LONG               lRowIncrement;


// Make sure Y1 is to the top
if (sY1 < sY2)
  {
  sOrderedY1 = sY1;
  sOrderedY2 = sY2;
  }
else
  {
  sOrderedY1 = sY2;
  sOrderedY2 = sY1;
  };

// Perform clipping
sOrderedY1 = max (sOrderedY1, (SHORT)0);
sOrderedY2 = min (sOrderedY2, (SHORT)(usHeight - 1));
if ((sX < 0) || (sX >= usWidth)) return;

// Calculate range and starting address
sLength = (SHORT) (sOrderedY2 - sOrderedY1 + 1);
pbyTarget = apbyRowStart [sOrderedY1] + sX;

// Store lRowChange locally on the stack (as opposed to accessed with the
//  "this" pointer)
lRowIncrement = lRowChange;

for (ulIndex = 0; ulIndex < sLength; ++ulIndex)
  {
  *pbyTarget = byDrawColor;
  pbyTarget += lRowIncrement;
  };
};

//........................................................................
VOID Canvas::Rectangle
//........................................................................
  (
  SHORT            sX1,
  SHORT            sY1,
  SHORT            sX2,
  SHORT            sY2
  )
{
// Check for minimum Rectangle size (2x2)
if ((sX1 == sX2) || (sY1 == sY2))
  {
  return;
  };

HorizLine (sX1, sX2, sY1);
HorizLine (sX1, sX2, sY2);

VertLine (sX1, (SHORT)(sY1 + 1), (SHORT)(sY2 - 1));
VertLine (sX2, (SHORT)(sY1 + 1), (SHORT)(sY2 - 1));
};

//........................................................................
VOID Canvas::XORHorizLine
//........................................................................
  (
  SHORT            sX1,
  SHORT            sX2,
  SHORT            sY
  )
{
SHORT              sOrderedX1;
SHORT              sOrderedX2;
SHORT              sLength;
PBYTE              pbyTarget;
ULONG              ulIndex;


// Make sure X1 is to the left
if (sX1 < sX2)
  {
  sOrderedX1 = sX1;
  sOrderedX2 = sX2;
  }
else
  {
  sOrderedX1 = sX2;
  sOrderedX2 = sX1;
  };

// Perform clipping
sOrderedX1 = max (sOrderedX1, (SHORT)0);
sOrderedX2 = min (sOrderedX2, (SHORT)(usWidth - 1));
if ((sY < 0) || (sY >= usHeight) ||
    (sOrderedX1 >= usWidth) || (sOrderedX2 < 0)) return;

sLength = (SHORT)(sOrderedX2 - sOrderedX1 + 1);
pbyTarget = apbyRowStart [sY] + sOrderedX1;

for (ulIndex = 0; ulIndex < sLength; ++ulIndex)
  {
  *pbyTarget ^= byDrawColor;
  ++pbyTarget;
  };
};

//........................................................................
VOID Canvas::XORVertLine
//........................................................................
  (
  SHORT            sX,
  SHORT            sY1,
  SHORT            sY2
  )
{
SHORT              sOrderedY1;
SHORT              sOrderedY2;
SHORT              sLength;
ULONG              ulIndex;
PBYTE              pbyTarget;
LONG               lRowIncrement;


// Make sure Y1 is to the left
if (sY1 < sY2)
  {
  sOrderedY1 = sY1;
  sOrderedY2 = sY2;
  }
else
  {
  sOrderedY1 = sY2;
  sOrderedY2 = sY1;
  };

// Perform clipping
sOrderedY1 = max (sOrderedY1, (SHORT)0);
sOrderedY2 = min (sOrderedY2, (SHORT)(usHeight - 1));
if ((sX < 0) || (sX >= usWidth) ||
    (sOrderedY1 >= usHeight) || (sOrderedY2 < 0)) return;

sLength = (SHORT) (sOrderedY2 - sOrderedY1 + 1);
pbyTarget = apbyRowStart [sOrderedY1] + sX;
lRowIncrement = lRowChange;
for (ulIndex = 0; ulIndex < sLength; ++ulIndex)
  {
  *pbyTarget ^= byDrawColor;
  pbyTarget += lRowIncrement;
  };
};

//........................................................................
VOID Canvas::XORRectangle
//........................................................................
  (
  SHORT            sX1,
  SHORT            sY1,
  SHORT            sX2,
  SHORT            sY2
  )
{
// Check for minimum Rectangle size (2x2)
if ((sX1 == sX2) || (sY1 == sY2))
  {
  return;
  };

XORHorizLine (sX1, sX2, sY1);
XORHorizLine (sX1, sX2, sY2);

XORVertLine (sX1, (SHORT)(sY1 + 1), (SHORT)(sY2 - 1));
XORVertLine (sX2, (SHORT)(sY1 + 1), (SHORT)(sY2 - 1));
};

