/*****************************************************************************
 * RMX Patching Functions
 ***************************************************************************** 
 *
 * Source:   $Source$
 *
 * Overview: This file contains the implementation of the following
 *           functions: 
 *
 *           Exported
 *               RmxPatch
 *               RmxPatchIsPatched
 *               RmxPatchOpen
 *               RmxPatchReadDlls
 *
 * $Log$
 *
 ***************************************************************************** 
 * 
 * Copyright (c) 1994 Johan Wikman
 * 
 * Permission to use, copy, modify, distribute, and sell this software
 * and its documentation for any purpose is hereby granted without fee, 
 * provided that the above copyright notice appear in all copies and 
 * that both that copyright notice and this permission notice appear in 
 * supporting documentation.
 *
 * THERE IS NO WARRANTY FOR THIS SOFTWARE, TO THE EXTENT PERMITTED BY
 * APPLICABLE LAW. THE SOFTWARE IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY
 * KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE. THE ENTIRE RISK AS TO THE QUALITY AND PERFORMANCE OF THE PROGRAM
 * IS WITH YOU.  SHOULD THE PROGRAM PROVE DEFECTIVE, YOU ASSUME THE COST OF
 * ALL NECESSARY SERVICING, REPAIR OR CORRECTION. 
 *
 *****************************************************************************/

#define INCL_NOCOMMON
#define INCL_DOSERRORS
#define INCL_DOSFILEMGR
#include "rmxpatch.h"
#include <stdlib.h>
#include <string.h>
#include <os2.h>


/*****************************************************************************
 * MODULE TYPES
 *****************************************************************************/

struct RMXREPLACEMENT
{
  PCSZ os2;
  PCSZ rmx;
};


/*****************************************************************************
 * MODULE VARIABLES
 *****************************************************************************/

static CHAR achRxWin[] = "RXWIN";


static RMXREPLACEMENT replacements[] = 
{
  { "PMWIN",   "RXWIN"   },
  { "PMGPI",   "RXGPI"   },
  { "PMSHAPI", "RXSHAPI" },
  { "PMCTLS",  "RXCTLS"  },
  { "HELPMGR", "RXLPMGR" }
};

const ULONG ulPatchCount = sizeof(replacements)/sizeof(replacements[0]);

static PCSZ fatals[] =
{
  "PMMERGE",
  "VIOCALLS"
};

const int ulFatalCount = sizeof(fatals)/sizeof(fatals[0]);

static PCSZ warnings[] =
{
  "PM",
  "SOM"
};

const int ulWarnCount = sizeof(warnings)/sizeof(warnings[0]);


/*****************************************************************************
 * MODULE FUNCTIONS
 *****************************************************************************/

static ULONG PatchCopyInfo(ULONG flMode, ULONG ulCount, 
			   RMXMODULE* pModules, RMXDLL* pDlls);


/*****************************************************************************
 * EXPORTED FUNCTIONS
 *****************************************************************************
 *
 * FUNCTION: ULONG RMXENTRY RmxPatch(HFILE hFile, PCRMXDLL pcDll)
 *
 * INPUT:
 *    hFile: Handle of the file to be patched.
 *    pcDll: Description of the DLL to replace.
 *
 * RETURN:
 *    NO_ERROR
 *    ERROR_BAD_FORMAT    // Attempt to replace non-replacable DLL.
 *    ERROR_INVALID_NAME  // Name not as long as the original name.
 *
 *    All codes returned by:
 *
 *    DosSetFilePtr
 *    DosWrite
 *
 * NOTES:
 *    This function applies the patch described by the RMXDLL argument
 *    on the file referenced by the given file handle.
 *
 *****************************************************************************/

ULONG RMXENTRY RmxPatch(HFILE hFile, PCRMXDLL pcDll)
{
  switch (pcDll->flState)
    {
    case RMXDLL_REPLACABLE:
      break;
      
    case RMXDLL_TRANSPARENT:
    case RMXDLL_SUSPICIOUS:
      return NO_ERROR;
      
    case RMXDLL_USEDNONPURELY:
    case RMXDLL_NOTSUPPORTED:
      return ERROR_BAD_FORMAT;
    }
  
  ULONG
    ulLength = strlen(pcDll->pcszReplacement);
  
  if (strlen(pcDll->achOriginal) != ulLength)
    return ERROR_INVALID_NAME;
  
  ULONG
    ulPosition,
    rc;
  
  rc = DosSetFilePtr(hFile, pcDll->ulOffset, FILE_BEGIN, &ulPosition);
  
  if (rc != NO_ERROR)
    return rc;
  
  ULONG
    cBytes;
  
  rc = DosWrite(hFile, (PVOID) pcDll->pcszReplacement, ulLength, &cBytes);

  if (rc != NO_ERROR)
    return rc;

  if (ulLength != cBytes)
    return ERROR_WRITE_FAULT;
  
  return NO_ERROR;
}


/*****************************************************************************
 *
 * FUNCTION: ULONG RMXENTRY RmxPatchIsPatched(PCSZ pcszFile, BOOL* pbPatched);
 *
 * INPUT:
 *    pcszFile: the file to be checked whether it is patched or not.
 *
 * RETURN:
 *   NO_ERROR
 *
 *   All codes returned by:
 *
 *   RmxPatchOpen
 *   RmxPatchReadDlls
 *
 * NOTES:
 *   This function checks whether the given file has been patched for RMX.
 *
 *****************************************************************************/

ULONG RMXENTRY RmxPatchIsPatched(PCSZ pcszFile, BOOL* pbPatched)
{
  *pbPatched = FALSE;

  ULONG
    rc;
  HFILE
    hFile;
  
  rc = RmxPatchOpen(pcszFile, RMX_OPEN_QUERY, &hFile);

  if (rc != NO_ERROR)
    return rc;

  ULONG
    cDlls;
  RMXDLL
    *pDlls;

  rc = RmxPatchReadDlls(hFile, RMX_UNPATCH, &cDlls, &pDlls);
  
  if (rc != NO_ERROR)
    {
      DosClose(hFile);
      return rc;
    }
  
  for (ULONG i = 0; i < cDlls; i++)
    if (!strcmpi(pDlls[i].achOriginal, achRxWin))
      {
	*pbPatched = TRUE;
	break;
      }
  
  free(pDlls);
  DosClose(hFile);
  
  return NO_ERROR;
}


/*****************************************************************************
 *
 * FUNCTION: ULONG RmxPatchOpen(PCSZ pcszFile, ULONG flMode, HFILE* phFile);
 *
 * INPUT:
 *   pcszFile: Name of the file to be opened.
 *   flMode:   The mode to be used when opening the file.
 *
 * OUTPUT:
 *   phFile:    Pointer to variable where the handle will be returned.
 *
 * RETURN
 *   NO_ERROR
 *   ERROR_INVALID_PARAMETER  // The mode flag was invalid.
 *  
 *   All codes returned by DosOpen _except_
 *
 *   ERROR_TOO_MANY_OPEN_FILES
 *
 * NOTES
 *   This function opens the file using the correct sharing and access
 *   modes for patching. Use this function if you intend to patch the
 *   executable module, otherwise you can use open. 
 *
 *****************************************************************************/

ULONG RMXENTRY RmxPatchOpen(PCSZ pcszFile, ULONG flMode, HFILE* phFile)
{
  ULONG
    flOpenMode;
  
  switch (flMode)
    {
    case RMX_OPEN_PATCH:
      flOpenMode = OPEN_SHARE_DENYREADWRITE | OPEN_ACCESS_READWRITE;
      break;
      
    case RMX_OPEN_QUERY:
      flOpenMode = OPEN_SHARE_DENYWRITE | OPEN_ACCESS_READONLY;
      break;
      
    default:
      return ERROR_INVALID_PARAMETER;
    }
  
  ULONG
    rc;
  
  do
    {
      ULONG
	ulAction;
      
      rc = DosOpen(pcszFile,
		   phFile,
		   &ulAction,
		   0,
		   0,
		   OPEN_ACTION_OPEN_IF_EXISTS,
		   flOpenMode,
		   0);
      
      if (rc == ERROR_TOO_MANY_OPEN_FILES)
	{
	  LONG
	    cbReqCount = 1;
	  ULONG
	    cbCurMaxFH; // Not used
	  
	  DosSetRelMaxFH(&cbReqCount, &cbCurMaxFH);
	}
    }
  while (rc == ERROR_TOO_MANY_OPEN_FILES);

  return rc;
}


/*****************************************************************************
 *
 * FUNCTION: ULONG RmxPatchReadDlls(HFILE    hFile,
 *                                  ULONG    flMode,
 *                                  ULONG*   pcDlls,
 *                                  RPADLL** ppDlls);
 *
 * INPUT:
 *   hFile:  Handle to the executable file.
 *   flMode: RPA_PATCH   look for PM DLLs.
 *           RPA_UNPATCH look for RMX DLLs.
 *
 * OUTPUT:
 *   *pcDlls: The number of elements in the array dlls.
 *   *ppDlls: Pointer to array of RPADLL structs.
 *
 * RETURN
 *   NO_ERROR
 *   ERROR_INVALID_PARAMETER // The mode parameter was invalid.
 *
 *   All codes returned by:
 *
 *   RmxExeRead32Modules
 *   RmxAlloc
 *   PatchCopyInfo
 *
 * NOTES
 *   If the function returns RMXERR_NOT_PATCHABLE you have to examine each
 *   RpaDll element separately to find out exactly which element(s) is/are
 *   the cause.
 *
 *****************************************************************************/

ULONG RMXENTRY RmxPatchReadDlls(HFILE    hFile,
				 ULONG    flMode, 
				 ULONG*   pcDlls,
				 RMXDLL** ppDlls)
{
  switch (flMode)
    {
    case RMX_PATCH:
    case RMX_UNPATCH:
      break;

    default:
      return ERROR_INVALID_PARAMETER;
    }
  
  RMXMODULE
    *pModules;
  ULONG
    rc;

  rc = RmxExeRead32Modules(hFile, pcDlls, &pModules);

  if (rc != NO_ERROR)
    return rc;
  
  *ppDlls = (RMXDLL*) malloc(*pcDlls * sizeof(RMXDLL));
  
  if (!*ppDlls)
    {
      free(pModules);
      return rc;
    }

  memset(*ppDlls, 0, *pcDlls * sizeof(RMXDLL));

  rc = PatchCopyInfo(flMode, *pcDlls, pModules, *ppDlls);

  free(pModules);
  
  return rc;
}


/*****************************************************************************
 * MODULE FUNCTIONS
 *****************************************************************************
 *
 * FUNCTION: ULONG PatchCopyInfo(ULONG flMode, ULONG ulCount,
 *                               RMXMODULE* pModules, RMXDLL* pDlls);
 *
 * INPUT:
 *   flMode:   Whether we are looking for PM or RMX dlls.
 *   ulCount:  The count of modules/dlls.
 *   pModules: Pointer to array of RMXMODULE structures.
 *   pDlls:    Pointer to array of RMXDLL structures.
 *
 * RETURN
 *   NO_ERROR
 *   ERROR_BAD_EXE_FORMAT
 *
 * NOTES
 *   This function copies information from the module info to the dll info.
 *
 *****************************************************************************/

static ULONG PatchCopyInfo(ULONG flMode, ULONG ulCount, 
			   RMXMODULE* pModules, RMXDLL* pDlls)
{
  ULONG
    rc = NO_ERROR;
  PCSZ
    (RMXREPLACEMENT::*from) = 0,
    (RMXREPLACEMENT::*to)   = 0;

  switch (flMode)
    {
    case RMX_PATCH:
      from = &RMXREPLACEMENT::os2;
      to   = &RMXREPLACEMENT::rmx;
      break;
      
    case RMX_UNPATCH:
      from = &RMXREPLACEMENT::rmx;
      to   = &RMXREPLACEMENT::os2;
    }
      
  for (ULONG i = 0; i < ulCount; i++)
    {
      RMXMODULE
	*pModule = &pModules[i];
      RMXDLL
	*pDll = &pDlls[i];
      
      strcpy(pDll->achOriginal, pModule->achName);
      pDll->ulOffset = pModule->ulOffset;
      
      BOOL
	bFound = FALSE;
      
      // It is quite unnecessary to loop through everything each
      // time. Once a name has been matched, there won't be another
      // match on that name. But it really doesn't cost too much.
      
      for (ULONG j = 0; j < ulPatchCount; j++)
	{
	  if (!stricmp(pModule->achName, replacements[j].*from))
	    {
	      if (pModule->fixup8           ||
		  pModule->fixup16Selector  ||
		  pModule->fixup1616Pointer ||
		  pModule->fixup16Offset    ||
		  pModule->fixup1632Pointer)
		{
		  pDll->flState = RMXDLL_USEDNONPURELY;
		  rc = ERROR_BAD_FORMAT;
		}
	      else
		{
		  pDll->flState = RMXDLL_REPLACABLE;
		  pDll->pcszReplacement = replacements[j].*to;
		}
	      
	      bFound = TRUE;
	    }
	}

      if (bFound)
	continue;

      // Is the DLL among the fatal ones.
      
      for (j = 0; j < ulFatalCount; j++)
	{
	  if (!stricmp(pModule->achName, fatals[j]))
	    {
	      pDll->flState = RMXDLL_NOTSUPPORTED;
	      rc = ERROR_BAD_FORMAT;
	      bFound = TRUE;
	      break;
	    }
	}
      
      if (bFound)
	continue;

      // Is the DLL among the suspicuous ones.
      
      for (j = 0; j < ulWarnCount; j++)
	{
	  if (!strncmpi(pModule->achName, warnings[j], strlen(warnings[j])))
	    {
	      pDll->flState = RMXDLL_SUSPICIOUS;
	      bFound = TRUE;
	      break;
	    }
	}

      if (bFound)
	continue;
      
      // So it must be transparent.
      
      pDll->flState = RMXDLL_TRANSPARENT;
    }

  return rc;
}
